import http
import logging
from copy import copy
from typing import Callable

import click

from ._colourizedformatter import ColourizedFormatter
from ._colourizedformatter import format_colors


class AccessFormatter(ColourizedFormatter):
    __module__: str = 'libcanonical.utils.logging'
    status_code_colours: dict[int, Callable[..., str]] = {
        1: format_colors("bright_white"),
        2: format_colors("green"),
        3: format_colors("yellow"),
        4: format_colors("red"),
        5: format_colors("bright_red"),
    }

    def get_status_code(self, status_code: int) -> str:
        try:
            status_phrase = http.HTTPStatus(status_code).phrase
        except ValueError:
            status_phrase = ""
        status_and_phrase = f"{status_code} {status_phrase}"
        if self.use_colors:

            def default(code: int) -> str:
                return status_and_phrase  # pragma: no cover

            func = self.status_code_colours.get(status_code // 100, default)
            return func(status_and_phrase)
        return status_and_phrase

    def formatMessage(self, record: logging.LogRecord) -> str:
        client_addr: str
        method: str
        full_path: str
        http_version: str
        status_code: str
        recordcopy = copy(record)
        (
            client_addr,
            method,
            full_path,
            http_version,
            status_code,
        ) = recordcopy.args  # type: ignore[misc]
        status_code = self.get_status_code(int(status_code))  # type: ignore[arg-type]
        request_line = f"{method} {full_path} HTTP/{http_version}"
        if self.use_colors:
            request_line = click.style(request_line, bold=True)
        recordcopy.__dict__.update(
            {
                "client_addr": client_addr,
                "request_line": request_line,
                "status_code": status_code,
            }
        )
        return super().formatMessage(recordcopy)
