import click

from gama_cli.helpers import call, get_gama_version
import subprocess


class Misc:
    def __init__(self, cli: click.Group, dev_mode: bool = False):
        if dev_mode:

            @cli.command(name="lint")
            def lint():
                """Lints all the things"""
                call("pre-commit run --all")
                call("gama_cli vessel lint-ui")

            @cli.command(name="test")
            def test():
                """Tests all the things"""
                call("gama_cli lint")
                call("gama_cli gs test")
                call("gama_cli vessel test-ui")
                call("gama_cli vessel test-ros")

            @cli.command(name="test-e2e")
            def test_e2e():
                """Brings up all containers and runs the e2e tests"""
                call("gama_cli gs up")
                call("gama_cli vessel up --mode stubs")
                call("gama_cli vessel test-e2e")

        @cli.command(name="upgrade")
        @click.option("--version", help="The version to upgrade to.")
        def upgrade(version: str):
            """Upgrade GAMA CLI"""
            click.echo(f"Current version: {get_gama_version()}")
            result = click.prompt(
                "Are you sure you want to upgrade?", default="y", type=click.Choice(["y", "n"])
            )
            if result == "n":
                return

            if version:
                call(f"pip install --upgrade gama-cli=={version}")
            else:
                call("pip install --upgrade gama-cli")

            click.echo(click.style("Upgrade of GAMA CLI complete.", fg="green"))
            click.echo(
                click.style(
                    "Run `gama vessel install` or `gama gs install` to upgrade GAMA.", fg="green"
                )
            )

        @cli.command(name="authenticate")
        @click.option(
            "--username",
            help="The username to use for authentication.",
            required=True,
            prompt=True,
        )
        @click.option(
            "--token", help="The token to use for authentication.", required=True, prompt=True
        )
        def authenticate(username: str, token: str):
            """
            Authenticate with the GAMA package repository so that you can pull images.

            To get a username and token you'll need to contact a Greenroom Robotics employee.
            """
            call(f"echo {token} | docker login ghcr.io -u {username} --password-stdin")
