"""Groq API provider for gac."""

import logging
import os

import httpx

from gac.errors import AIError

logger = logging.getLogger(__name__)


def call_groq_api(model: str, messages: list[dict], temperature: float, max_tokens: int) -> str:
    """Call Groq API directly."""
    api_key = os.getenv("GROQ_API_KEY")
    if not api_key:
        raise AIError.model_error("GROQ_API_KEY not found in environment variables")

    url = "https://api.groq.com/openai/v1/chat/completions"
    headers = {"Authorization": f"Bearer {api_key}", "Content-Type": "application/json"}

    data = {"model": model, "messages": messages, "temperature": temperature, "max_tokens": max_tokens}

    try:
        response = httpx.post(url, headers=headers, json=data, timeout=120)
        response.raise_for_status()
        response_data = response.json()

        # Debug logging to understand response structure
        logger.debug(f"Groq API response: {response_data}")

        # Handle different response formats
        if "choices" in response_data and len(response_data["choices"]) > 0:
            choice = response_data["choices"][0]
            if "message" in choice and "content" in choice["message"]:
                content = choice["message"]["content"]
                logger.debug(f"Found content in message.content: {content}")
                return content
            elif "text" in choice:
                content = choice["text"]
                logger.debug(f"Found content in choice.text: {content}")
                return content
            else:
                logger.debug(f"Choice structure: {choice}")

        # If we can't find content in the expected places, raise an error
        raise AIError.model_error(f"Unexpected response format from Groq API: {response_data}")
    except httpx.HTTPStatusError as e:
        raise AIError.model_error(f"Groq API error: {e.response.status_code} - {e.response.text}") from e
    except Exception as e:
        raise AIError.model_error(f"Error calling Groq API: {str(e)}") from e
