"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lib_1 = require("../lib");
const cdk = require("@aws-cdk/core");
const kinesis = require("@aws-cdk/aws-kinesis");
const iam = require("@aws-cdk/aws-iam");
// import * as defaults from '@aws-solutions-constructs/core';
require("@aws-cdk/assert/jest");
const iotTopicRuleProps = {
    topicRulePayload: {
        description: "sends data to kinesis streams",
        sql: "SELECT * FROM 'solutions/constructs'",
        actions: []
    }
};
test('check iot topic rule properties', () => {
    const stack = new cdk.Stack();
    const props = {
        iotTopicRuleProps
    };
    const construct = new lib_1.IotToKinesisStreams(stack, 'test-iot-kinesisstreams', props);
    expect(stack).toHaveResource('AWS::IoT::TopicRule', {
        TopicRulePayload: {
            Actions: [
                {
                    Kinesis: {
                        StreamName: {
                            Ref: "testiotkinesisstreamsKinesisStreamA6FEF4AF"
                        },
                        RoleArn: {
                            "Fn::GetAtt": [
                                "testiotkinesisstreamsIotActionsRoleAE74F764",
                                "Arn"
                            ]
                        }
                    }
                }
            ],
            Description: "sends data to kinesis streams",
            RuleDisabled: false,
            Sql: "SELECT * FROM 'solutions/constructs'"
        }
    });
    expect(stack).toHaveResource('AWS::Kinesis::Stream', {
        ShardCount: 1,
        RetentionPeriodHours: 24,
        StreamEncryption: {
            EncryptionType: "KMS",
            KeyId: "alias/aws/kinesis"
        }
    });
    expect(construct.iotTopicRule).toBeDefined();
    expect(construct.iotActionsRole).toBeDefined();
    expect(construct.cloudwatchAlarms).toBeDefined();
    expect(construct.kinesisStream).toBeDefined();
});
test('check existing kinesis stream', () => {
    const stack = new cdk.Stack();
    const existingKinesisStream = new kinesis.Stream(stack, `existing-stream`, {
        shardCount: 2,
        streamName: 'testexistingstream',
        retentionPeriod: cdk.Duration.hours(25)
    });
    const props = {
        iotTopicRuleProps,
        existingStreamObj: existingKinesisStream,
        createCloudWatchAlarms: false
    };
    const construct = new lib_1.IotToKinesisStreams(stack, 'test-iot-kinesisstreams', props);
    expect(stack).toHaveResourceLike('AWS::Kinesis::Stream', {
        ShardCount: 2,
        RetentionPeriodHours: 25,
        Name: 'testexistingstream'
    });
    expect(stack).not.toHaveResource('AWS::CloudWatch::Alarm');
    expect(construct.iotTopicRule).toBeDefined();
    expect(construct.iotActionsRole).toBeDefined();
    expect(construct.cloudwatchAlarms).toBeUndefined();
    expect(construct.kinesisStream).toBeDefined();
});
test('check new kinesis stream with override props', () => {
    const stack = new cdk.Stack();
    const props = {
        iotTopicRuleProps,
        createCloudWatchAlarms: false,
        kinesisStreamProps: {
            shardCount: 5,
            streamName: 'testnewstream',
            retentionPeriod: cdk.Duration.hours(30)
        }
    };
    const construct = new lib_1.IotToKinesisStreams(stack, 'test-iot-kinesisstreams', props);
    expect(stack).toHaveResourceLike('AWS::Kinesis::Stream', {
        ShardCount: 5,
        RetentionPeriodHours: 30,
        Name: 'testnewstream'
    });
    const kinesisStream = construct.kinesisStream;
    expect(kinesisStream).toBeDefined();
});
test('check existing action in topic rule props', () => {
    const stack = new cdk.Stack();
    const existingKinesisStream = new kinesis.Stream(stack, `existing-stream`, {});
    const existingIamRole = new iam.Role(stack, 'existingRole', {
        assumedBy: new iam.ServicePrincipal('iot.amazonaws.com')
    });
    const existingKinesisActionProperty = {
        kinesis: {
            streamName: existingKinesisStream.streamName,
            roleArn: existingIamRole.roleArn
        }
    };
    const props = {
        iotTopicRuleProps: {
            topicRulePayload: {
                sql: "SELECT * FROM 'solutions/constructs'",
                actions: [existingKinesisActionProperty],
                ruleDisabled: true
            },
            ruleName: 'testiotrulename'
        },
        createCloudWatchAlarms: false,
        kinesisStreamProps: {
            shardCount: 5,
            streamName: 'testnewstream',
            retentionPeriod: cdk.Duration.hours(30)
        }
    };
    new lib_1.IotToKinesisStreams(stack, 'test-iot-kinesisstreams', props);
    // Check multiple actions in the Topic Rule
    expect(stack).toHaveResource('AWS::IoT::TopicRule', {
        TopicRulePayload: {
            Actions: [
                {
                    Kinesis: {
                        StreamName: {
                            Ref: "testiotkinesisstreamsKinesisStreamA6FEF4AF"
                        },
                        RoleArn: {
                            "Fn::GetAtt": [
                                "testiotkinesisstreamsIotActionsRoleAE74F764",
                                "Arn"
                            ]
                        }
                    }
                },
                {
                    Kinesis: {
                        RoleArn: {
                            "Fn::GetAtt": [
                                "existingRole3E995BBA",
                                "Arn"
                            ]
                        },
                        StreamName: {
                            Ref: "existingstream0A902451"
                        }
                    }
                }
            ],
            RuleDisabled: true,
            Sql: "SELECT * FROM 'solutions/constructs'"
        },
        RuleName: "testiotrulename"
    });
    expect(stack).toHaveResourceLike('AWS::Kinesis::Stream', {
        ShardCount: 5,
        RetentionPeriodHours: 30,
        Name: 'testnewstream'
    });
    expect(stack).toCountResources('AWS::Kinesis::Stream', 2);
});
test('check name confict', () => {
    const stack = new cdk.Stack();
    const props = {
        iotTopicRuleProps
    };
    new lib_1.IotToKinesisStreams(stack, 'test-iot-kinesisstreams1', props);
    new lib_1.IotToKinesisStreams(stack, 'test-iot-kinesisstreams2', props);
    expect(stack).toCountResources('AWS::Kinesis::Stream', 2);
});
test('check construct chaining', () => {
    const stack = new cdk.Stack();
    const props1 = {
        iotTopicRuleProps
    };
    const construct = new lib_1.IotToKinesisStreams(stack, 'test-iot-kinesisstreams1', props1);
    const props2 = {
        iotTopicRuleProps,
        existingStreamObj: construct.kinesisStream
    };
    new lib_1.IotToKinesisStreams(stack, 'test-iot-kinesisstreams2', props2);
    expect(stack).toCountResources('AWS::Kinesis::Stream', 1);
});
test('check error when stream props and existing stream is supplied', () => {
    const stack = new cdk.Stack();
    const existingKinesisStream = new kinesis.Stream(stack, `existing-stream`, {});
    const props = {
        iotTopicRuleProps,
        existingStreamObj: existingKinesisStream,
        kinesisStreamProps: {}
    };
    const app = () => {
        new lib_1.IotToKinesisStreams(stack, 'test-iot-kinesisstreams', props);
    };
    expect(app).toThrowError();
});
//# sourceMappingURL=data:application/json;base64,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