# -*- coding:utf-8 -*-
# !/usr/bin/env python
"""
Date: 2022/11/17 22:00
Desc: 上海黄金交易所-数据资讯-行情走势
https://www.sge.com.cn/sjzx/mrhq
上海黄金交易所-数据资讯-上海金基准价-历史数据
上海黄金交易所-数据资讯-上海银基准价-历史数据
"""
import pandas as pd
import requests


def spot_symbol_table_sge() -> pd.DataFrame:
    """
    上海黄金交易所-数据资讯-行情走势-品种表
    https://www.sge.com.cn/sjzx/mrhq
    :return: 品种表
    :rtype: pandas.DataFrame
    """
    url = "https://www.sge.com.cn/sjzx/mrhq_data"
    payload = {"date": "2022-11-17"}
    headers = {
        "Accept": "text/html, */*; q=0.01",
        "Accept-Encoding": "gzip, deflate, br",
        "Accept-Language": "zh-CN,zh;q=0.9,en;q=0.8",
        "Cache-Control": "no-cache",
        "Connection": "keep-alive",
        "Content-Length": "15",
        "Content-Type": "multipart/form-data; boundary=---011000010111000001101001",
        "Host": "www.sge.com.cn",
        "Origin": "https://www.sge.com.cn",
        "Pragma": "no-cache",
        "Referer": "https://www.sge.com.cn/sjzx/mrhq",
        "sec-ch-ua": '"Google Chrome";v="107", "Chromium";v="107", "Not=A?Brand";v="24"',
        "sec-ch-ua-mobile": "?0",
        "sec-ch-ua-platform": '"Windows"',
        "Sec-Fetch-Dest": "empty",
        "Sec-Fetch-Mode": "cors",
        "Sec-Fetch-Site": "same-origin",
        "User-Agent": "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/107.0.0.0 Safari/537.36",
        "X-Requested-With": "XMLHttpRequest",
    }
    r = requests.post(url, data=payload, headers=headers)
    data_text = r.text
    re_html = "<" + data_text.strip().strip("<!--").strip("-->") + ">"
    temp_df = pd.read_html(re_html)[0]
    temp_df.reset_index(inplace=True)
    temp_df = temp_df.iloc[:, :2]
    temp_df.columns = ["序号", "品种"]
    temp_df['序号'] = temp_df['序号'] + 1
    return temp_df


def spot_hist_sge(symbol: str = "Au99.99") -> pd.DataFrame:
    """
    上海黄金交易所-数据资讯-行情走势-历史数据
    https://www.sge.com.cn/sjzx/mrhq
    :param symbol: choice of {'Au99.99', 'Au99.95', 'Au100g', 'Pt99.95', 'Ag(T+D)', 'Au(T+D)', 'mAu(T+D)', 'Au(T+N1)', 'Au(T+N2)', 'Ag99.99', 'iAu99.99', 'Au99.5', 'iAu100g', 'iAu99.5', 'PGC30g', 'NYAuTN06', 'NYAuTN12'}; 可以通过 ak.spot_symbol_table_sge() 获取品种表
    :type symbol: str
    :return: 历史数据
    :rtype: pandas.DataFrame
    """
    url = "https://www.sge.com.cn/graph/Dailyhq"
    payload = {"instid": symbol}
    headers = {
        "Accept": "text/html, */*; q=0.01",
        "Accept-Encoding": "gzip, deflate, br",
        "Accept-Language": "zh-CN,zh;q=0.9,en;q=0.8",
        "Cache-Control": "no-cache",
        "Connection": "keep-alive",
        "Content-Length": "15",
        "Content-Type": "application/x-www-form-urlencoded; charset=UTF-8",
        "Host": "www.sge.com.cn",
        "Origin": "https://www.sge.com.cn",
        "Pragma": "no-cache",
        "Referer": "https://www.sge.com.cn/sjzx/mrhq",
        "sec-ch-ua": '"Google Chrome";v="107", "Chromium";v="107", "Not=A?Brand";v="24"',
        "sec-ch-ua-mobile": "?0",
        "sec-ch-ua-platform": '"Windows"',
        "Sec-Fetch-Dest": "empty",
        "Sec-Fetch-Mode": "cors",
        "Sec-Fetch-Site": "same-origin",
        "User-Agent": "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/107.0.0.0 Safari/537.36",
        "X-Requested-With": "XMLHttpRequest",
    }
    r = requests.post(url, data=payload, headers=headers)
    data_json = r.json()
    temp_df = pd.DataFrame(data_json["time"])
    temp_df.columns = [
        "date",
        "open",
        "close",
        "high",
        "low",
    ]

    temp_df["date"] = pd.to_datetime(temp_df["date"]).dt.date
    temp_df["open"] = pd.to_numeric(temp_df["open"])
    temp_df["close"] = pd.to_numeric(temp_df["close"])
    temp_df["high"] = pd.to_numeric(temp_df["high"])
    temp_df["low"] = pd.to_numeric(temp_df["low"])
    return temp_df


def spot_golden_benchmark_sge() -> pd.DataFrame:
    """
    上海黄金交易所-数据资讯-上海金基准价-历史数据
    https://www.sge.com.cn/sjzx/jzj
    :return: 历史数据
    :rtype: pandas.DataFrame
    """
    url = "https://www.sge.com.cn/graph/DayilyJzj"
    payload = {}
    headers = {
        "User-Agent": "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/107.0.0.0 Safari/537.36",
    }
    r = requests.post(url, data=payload, headers=headers)
    data_json = r.json()
    temp_df = pd.DataFrame(data_json["wp"])
    temp_df.columns = [
        "交易时间",
        "晚盘价",
    ]
    temp_df["交易时间"] = pd.to_datetime(temp_df["交易时间"], unit="ms").dt.date
    temp_zp_df = pd.DataFrame(data_json["zp"])
    temp_zp_df.columns = [
        "交易时间",
        "早盘价",
    ]
    temp_zp_df["交易时间"] = pd.to_datetime(temp_zp_df["交易时间"], unit="ms").dt.date
    temp_df["早盘价"] = temp_zp_df["早盘价"]
    temp_df["晚盘价"] = pd.to_numeric(temp_df["晚盘价"])
    temp_df["早盘价"] = pd.to_numeric(temp_df["早盘价"])
    return temp_df


def spot_silver_benchmark_sge() -> pd.DataFrame:
    """
    上海黄金交易所-数据资讯-上海银基准价-历史数据
    https://www.sge.com.cn/sjzx/mrhq
    :return: 历史数据
    :rtype: pandas.DataFrame
    """
    url = "https://www.sge.com.cn/graph/DayilyShsilverJzj"
    payload = {}
    headers = {
        "User-Agent": "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/107.0.0.0 Safari/537.36",
    }
    r = requests.post(url, data=payload, headers=headers)
    data_json = r.json()
    temp_df = pd.DataFrame(data_json["wp"])
    temp_df.columns = [
        "交易时间",
        "晚盘价",
    ]
    temp_df["交易时间"] = pd.to_datetime(temp_df["交易时间"], unit="ms").dt.date
    temp_zp_df = pd.DataFrame(data_json["zp"])
    temp_zp_df.columns = [
        "交易时间",
        "早盘价",
    ]
    temp_zp_df["交易时间"] = pd.to_datetime(temp_zp_df["交易时间"], unit="ms").dt.date
    temp_df["早盘价"] = temp_zp_df["早盘价"]
    temp_df["晚盘价"] = pd.to_numeric(temp_df["晚盘价"])
    temp_df["早盘价"] = pd.to_numeric(temp_df["早盘价"])
    return temp_df


if __name__ == "__main__":
    spot_symbol_table_sge_df = spot_symbol_table_sge()
    print(spot_symbol_table_sge_df)

    spot_hist_sge_df = spot_hist_sge(symbol="Au99.99")
    print(spot_hist_sge_df)

    spot_golden_benchmark_sge_df = spot_golden_benchmark_sge()
    print(spot_golden_benchmark_sge_df)

    for spot in spot_symbol_table_sge_df["品种"].tolist():
        spot_hist_sge_df = spot_hist_sge(symbol=spot)
        print(spot_hist_sge_df)
