# -*- coding: utf-8 -*-
r"""
GL(2,R)-orbit closure of translation surfaces

.. TODO::

    - Comparison of renf_elem with Python int not supported (but seem to work
      through the RealEmbeddedNumberField)

    - Theorem 1.9 of Alex Wright: the field of definition is contained in the field generated by
      the ratio of circumferences. We should provide a method, .reset_field_of_definition or
      something similar
"""
######################################################################
#  This file is part of sage-flatsurf.
#
#        Copyright (C) 2019-2020 Julian Rüth
#                      2020      Vincent Delecroix
#
#  sage-flatsurf is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 2 of the License, or
#  (at your option) any later version.
#
#  sage-flatsurf is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with sage-flatsurf. If not, see <https://www.gnu.org/licenses/>.
######################################################################

import cppyy
import gmpxxyy
from pyeantic import RealEmbeddedNumberField

import pyflatsurf
import pyflatsurf.vector

# TODO: it would be convenient to have Vertex directly available from
# pyflatsurf, ie being able to do
#     from pyflatsurf import Vertex
Vertex = cppyy.gbl.flatsurf.Vertex

from sage.all import VectorSpace, FreeModule, matrix, identity_matrix, ZZ, QQ, Unknown, vector, prod

from .subfield import subfield_from_elements
from .polygon import is_between, projectivization
from .translation_surface import TranslationSurface

class Decomposition:
    def __init__(self, gl2rorbit, decomposition, u):
        self.u = u
        self.orbit = gl2rorbit
        self.decomposition = decomposition

    def cylinders(self):
        r"""
        Return the cylinders (aka completely periodic components)

        EXAMPLES::

            sage: from flatsurf import translation_surfaces
            sage: x = polygen(QQ)
            sage: K.<a> = NumberField(x^3 - 2, embedding=AA(2)**(1/3))
            sage: S = translation_surfaces.mcmullen_L(1,1,1,a)

            sage: from flatsurf import GL2ROrbitClosure # optional: pyflatsurf
            sage: O = GL2ROrbitClosure(S) # optional: pyflatsurf
            sage: O.decomposition((1,2)).cylinders() # optional: pyflatsurf
            [Cylinder with perimeter [...]]
        """
        return self.decomposition.cylinders()

    def minimal_components(self):
        r"""
        Return the minimal components

        EXAMPLES::

            sage: from flatsurf import translation_surfaces
            sage: x = polygen(QQ)
            sage: K.<a> = NumberField(x^3 - 2, embedding=AA(2)**(1/3))
            sage: S = translation_surfaces.mcmullen_L(1,1,1,a)

            sage: from flatsurf import GL2ROrbitClosure # optional: pyflatsurf
            sage: O = GL2ROrbitClosure(S) # optional: pyflatsurf
            sage: O.decomposition((1,2)).minimal_components() # optional: pyflatsurf
            [Component Without Periodic Trajectory with perimeter [...]]
        """
        return self.decomposition.minimalComponents()

    def undetermined_components(self):
        r"""
        Return the undetermined components that could either be cylinder or minimal components.

        EXAMPLES::

            sage: from flatsurf import translation_surfaces
            sage: S = translation_surfaces.mcmullen_genus2_prototype(4,2,1,1,1/4)
            sage: l = S.base_ring().gen()

            sage: from flatsurf import  GL2ROrbitClosure # optional: pyflatsurf
            sage: O = GL2ROrbitClosure(S) # optional: pyflatsurf
            sage: dec = O.decomposition((8*l - 25, 16), 10) # optional: pyflatsurf
            sage: dec.undetermined_components() # optional: pyflatsurf
            [Component with perimeter [...]]

        Further refinement might change the status of undetermined components::

            sage: import pyflatsurf # optional: pyflatsurf
            sage: dec.decomposition.decompose(10r) # optional: pyflatsurf
            True
            sage: dec.undetermined_components() # optional: pyflatsurf
            []
        """
        return self.decomposition.undeterminedComponents()

    def num_cylinders_minimals_undetermined(self):
        r"""
        Return the triple of numbers of respectively cylinders, minimal
        components and undetermined components of this flow decomposition.

        EXAMPLES::

            sage: from flatsurf import translation_surfaces
            sage: S = translation_surfaces.mcmullen_genus2_prototype(4,2,1,1,1/4)
            sage: l = S.base_ring().gen()

            sage: from flatsurf import  GL2ROrbitClosure # optional: pyflatsurf
            sage: O = GL2ROrbitClosure(S) # optional: pyflatsurf
            sage: dec = O.decomposition((8*l - 25, 16), 100) # optional: pyflatsurf
            sage: dec.num_cylinders_minimals_undetermined() # optional: pyflatsurf
            (2, 0, 0)
        """
        ncyl = 0
        nmin = 0
        nund = 0
        for comp in self.decomposition.components():
            if comp.cylinder() == True:
                assert comp.withoutPeriodicTrajectory() == False
                ncyl += 1
            elif comp.cylinder() == False:
                assert comp.withoutPeriodicTrajectory() == True
                nmin += 1
            else:
                nund += 1
        return (ncyl, nmin, nund)

    def __repr__(self):
        ncyl, nmin, nund = self.num_cylinders_minimals_undetermined()
        return "Flow decomposition with %d cylinders, %d minimal components and %d undetermined components" % (ncyl, nmin, nund)

    def _spanning_tree_decomposition(self, sc_index, sc_comp):
        r"""
        Return

        - a list of indices of edges that form a basis
        - a matrix projection to the basis (modulo the face relations)
        """
        components = [c for c in self.decomposition.components()]

        n = len(sc_index)
        assert n % 2 == 0
        n //= 2

        for p in components[0].perimeter():
            break
        root = p.saddleConnection()
        t = {0: None} # face -> half edge to take to go to the root
        todo = [0]
        edges = []  # store edges in topological order to perform Gauss reduction
        while todo:
            i = todo.pop()
            c = components[i]
            for sc in c.perimeter():
                sc1 = -sc.saddleConnection()
                j = sc_comp[sc1]
                if j not in t:
                    t[j] = sc1
                    todo.append(j)
                    edges.append(sc1)

        # gauss reduction
        spanning_set = set(range(n))
        proj = identity_matrix(ZZ, n)
        edges.reverse()
        for sc1 in edges:
            i1 = sc_index[sc1]
            if i1 < 0:
                s1 = -1
                i1 = -i1-1
            else:
                s1 = 1
            comp = components[sc_comp[sc1]]
            proj[i1] = 0
            for p in comp.perimeter():
                sc = p.saddleConnection()
                if sc == sc1:
                    continue
                j = sc_index[sc]
                if j < 0:
                    s = -1
                    j = -j-1
                else:
                    s = 1
                proj[i1] = - s1 * s * proj[j]

            spanning_set.remove(i1)
            for j in range(n):
                assert proj[j,i1] == 0

        return (t, sorted(spanning_set), proj)

    def kontsevich_zorich_cocycle(self):
        r"""
        Base change for this flow decomposition.
        """
        components = [c for c in self.decomposition.components()]
        sc_pos = []
        sc_comp = {}
        sc_index = {}
        n = 0
        for i,comp in enumerate(components):
            for p in comp.perimeter():
                sc = p.saddleConnection()
                sc_comp[sc] = i
                if sc not in sc_index:
                    sc_index[sc] = n
                    sc_pos.append(sc)
                    sc_index[-sc] = -n-1
                    n += 1

        t, spanning_set, proj = self._spanning_tree_decomposition(sc_index, sc_comp)
        assert proj.rank() == len(spanning_set) == n - len(components) + 1, self.u
        proj = proj.transpose()
        proj = matrix(ZZ, [r for r in proj.rows() if not r.is_zero()])
        assert proj.nrows() == self.orbit.proj.nrows(), self.u

        # Write the base change V^*(T') -> V^*(T) relative to our bases
        A = matrix(ZZ, self.orbit.d)
        for i, sc in enumerate(spanning_set):
            sc = sc_pos[sc]
            c = sc.chain()
            v = [0] * self.orbit.d
            for edge in self.orbit._surface.edges():
                A[i] += ZZ(str(c[edge])) * self.orbit.proj.column(edge.index())
        assert A.det().is_unit()
        return A, sc_index, proj

    def parabolic(self):
        r"""
        Return whether this decomposition is completely periodic with cylinder with
        commensurable moduli.

        EXAMPLES::

            sage: from flatsurf import translation_surfaces
            sage: from flatsurf import GL2ROrbitClosure  # optional: pyflatsurf

        Veech surfaces have the property that any saddle connection direction is
        parabolic (one half of the Veech dichotomy)::

            sage: S = translation_surfaces.veech_double_n_gon(5)
            sage: O = GL2ROrbitClosure(S)  # optional: pyflatsurf
            sage: all(d.parabolic() for d in O.decompositions_depth_first(3))  # optional: pyflatsurf
            True

        For surfaces in rank one loci, even though they are completely periodic,
        they are generally not completely periodic::

            sage: S = translation_surfaces.mcmullen_genus2_prototype(4,2,1,1,1/4)
            sage: O = GL2ROrbitClosure(S)  # optional: pyflatsurf
            sage: all((d.decomposition.hasCylinder() == False) or d.parabolic() for d in O.decompositions(6))  # optional: pyflatsurf
            False
            sage: all((d.decomposition.completelyPeriodic() == True) or (d.decomposition.hasCylinder() == False) for d in O.decompositions(6))  # optional: pyflatsurf
            True
        """
        return self.decomposition.parabolic()

    def circumference(self, component, sc_index, proj):
        r"""
        Return the circumference of ``component`` in the coordinate of the original
        surface.
        """
        if component.cylinder() != True:
            raise ValueError

        A, sc_index, proj = self.kontsevich_zorich_cocycle()
        perimeters = [p for p in component.perimeter()]
        per = perimeters[0]
        assert not per.vertical()
        sc = per.saddleConnection()
        i = sc_index[sc]
        if i < 0:
            s = -1
            i = -i-1
        else:
            s = 1
        v = s * proj.column(i)
        circumference = -A.solve_right(v)

        # check
        hol = self.orbit.holonomy_dual(circumference)
        holbis = component.circumferenceHolonomy()
        holbis = self.orbit.V2._isomorphic_vector_space(self.orbit.V2(holbis))
        assert hol == holbis, (hol, holbis)

        return circumference

    def cylinder_deformation_subspace(self):
        r"""
        Return a subspace included in the tangent space to the GL(2,R)-orbit closure.

        From A. Wright cylinder deformation Theorem.
        """
        def eliminate_denominators(fractions):
            r"""
            Given a list of ``fractions``, pairs of numerators `n_i` and
            denominators `d_i`, return a list of fractions `c n_i/d_i` scaled
            uniformly such that the value can be represented in the underlying
            ring.
            """
            fractions = list(fractions)
            try:
                return [x.parent()(x / y) for x, y in fractions]
            except (ValueError, ArithmeticError, NotImplementedError):
                denominators = set([denominator for numerator, denominator in fractions])
                return [numerator * prod(
                    [d for d in denominators if denominator != d]
                ) for (numerator, denominator) in fractions]

        v = self.orbit.V()
        module_fractions = []
        vcyls = []
        A, sc_index, proj = self.kontsevich_zorich_cocycle()
        for component in self.decomposition.components():
            if component.cylinder() == False:
                continue
            elif component.cylinder() == True:
                vcyls.append(self.circumference(component, sc_index, proj))

                vertical = component.vertical()
                width = self.orbit.V2._isomorphic_vector_space.base_ring()(self.orbit.V2.base_ring()(component.width()))
                height = self.orbit.V2._isomorphic_vector_space.base_ring()(self.orbit.V2.base_ring()(component.vertical().project(component.circumferenceHolonomy())))
                module_fractions.append((width, height))
            else:
                return []

        if not module_fractions:
            return []

        modules = eliminate_denominators(module_fractions)

        if hasattr(modules[0], '_backend'):
            # Make sure all modules live in the same K-Module so that .coefficients() below produces coefficient lists of the same length.
            from functools import reduce
            parent = reduce(lambda m, n: m.span(m, n), [module._backend.module() for module in modules], modules[0]._backend.module())
            modules = [module.parent()(module._backend.promote(parent)) for module in modules]

        def to_rational_vector(x):
            r"""
            Return the rational coefficients of `x` over its implicit basis,
            e.g., if `x` is in a number field K, return the coefficients of x
            in `K` as a vector space over the rationals.
            """
            if x.parent() in [ZZ, QQ]:
                ret = [QQ(x)]
            elif hasattr(x, 'vector'):
                ret = x.vector()
            elif hasattr(x, 'renf_elem'):
                ret = x.parent().number_field(x).vector()
            elif hasattr(x, '_backend'):
                from itertools import chain
                ret = list(chain(*[to_rational_vector(self.orbit.V2.base_ring().base_ring()(self.orbit.V2.base_ring().base_ring()(c))) for c in x._backend.coefficients()]))
            else:
                raise NotImplementedError("cannot turn %s, i.e., a %s, into a rational vector yet"%(x,type(x)))

            assert all(y in QQ for y in ret)
            return ret

        assert all(module.parent() is modules[0].parent() for module in modules)
        M = matrix([to_rational_vector(module) for module in modules])
        assert M.base_ring() is QQ
        relations = M.left_kernel().matrix()
        assert len(vcyls) == len(module_fractions) == relations.ncols()

        vectors = [sum(t * vcyl * module[1] for (t, vcyl, module) in zip(relation, vcyls, module_fractions)) for relation in relations.right_kernel().basis()]

        assert all(v.base_ring() is self.orbit.V2._isomorphic_vector_space.base_ring() for v in vectors)

        return vectors

    def plot_completely_periodic(self):
        from sage.plot.all import polygon2d, Graphics, point2d, text
        O = self.orbit
        G = []
        u = self.u  # direction (that we put horizontal)
        m = matrix(2, [u[1], -u[0], u[1], u[0]])
        indices = {}
        xmin = xmax = ymin = ymax = 0
        for comp in self.decomposition.components():
            H = Graphics()
            x = O.V2._isomorphic_vector_space.zero()

            pts = [x]
            below = True
            for p in comp.perimeter():
                sc = p.saddleConnection()
                y = x + m * O.V2._isomorphic_vector_space(O.V2(p.saddleConnection().vector()))

                if p.vertical():
                    if sc in indices:
                        i = indices[sc]
                    else:
                        i = len(indices) // 2
                        indices[sc] = i
                        indices[-sc] = i
                    if below:
                        H += text(str(i), (x+y)/2, color='black')
                x = y
                xmin = min(xmin, x[0])
                xmax = max(xmax, x[0])
                ymin = min(ymin, x[1])
                ymax = max(ymax, x[1])
                pts.append(x)
            H += polygon2d(pts, color='blue', alpha=0.3)
            H += point2d(pts, color='red', pointsize=20)
            G.append(H)
        aspect_ratio = float(xmax - xmin) / float(ymax - ymin)
        for H in G:
            H.set_axes_range(xmin, xmax, ymin, ymax)
            H.axes(False)
            H.set_aspect_ratio(aspect_ratio)
        return G

class GL2ROrbitClosure:
    r"""
    Lower bound approximation to the tangent space of a GL(2,R)-orbit closure of a
    linear family of translation surfaces.

    EXAMPLES::

        sage: from flatsurf import polygons, similarity_surfaces
        sage: from flatsurf import GL2ROrbitClosure  # optional: pyflatsurf

        sage: T = polygons.triangle(3, 3, 5)
        sage: S = similarity_surfaces.billiard(T)
        sage: S = S.minimal_cover(cover_type="translation")
        sage: GL2ROrbitClosure(S)  # optional: pyflatsurf
        GL(2,R)-orbit closure of dimension at least 2 in H_5(4, 2^2) (ambient dimension 12)

    Computing an orbit closure over an exact real ring with transcendental elements::

        sage: from flatsurf import EquiangularPolygons
        sage: from pyexactreal import ExactReals  # optional: exactreal

        sage: E = EquiangularPolygons(1, 5, 5, 5)
        sage: R = ExactReals(E.base_ring())  # optional: exactreal
        sage: T = E(R(1), R.random_element(1/4))  # optional: exactreal
        sage: S = similarity_surfaces.billiard(T)  # optional: exactreal
        sage: S = S.minimal_cover(cover_type="translation")  # optional: exactreal
        sage: O = GL2ROrbitClosure(S); O  # optional: exactreal, pyflatsurf
        GL(2,R)-orbit closure of dimension at least 4 in H_7(4^3, 0) (ambient dimension 17)
        sage: bound = E.billiard_unfolding_stratum('half-translation', marked_points=True).dimension()
        sage: for decomposition in O.decompositions(1):  # long time, optional: exactreal, pyflatsurf
        ....:     O.update_tangent_space_from_flow_decomposition(decomposition)
        ....:     if O.dimension() == bound: break
        sage: O  # long time, optional: exactreal, pyflatsurf
        GL(2,R)-orbit closure of dimension at least 8 in H_7(4^3, 0) (ambient dimension 17)

    TESTS::

        sage: from flatsurf import translation_surfaces
        sage: x = polygen(QQ)
        sage: K = NumberField(x**3 - 2, 'a', embedding=AA(2)**QQ((1,3)))
        sage: a = K.gen()
        sage: S = translation_surfaces.mcmullen_genus2_prototype(2, 1, 0, -1, a/4)

        sage: from flatsurf import GL2ROrbitClosure  # optional: pyflatsurf
        sage: GL2ROrbitClosure(S)._U.base_ring() # optional: pyflatsurf
        Number Field in a with defining polynomial x^3 - 2 with a = 1.259921049894873?
    """
    def __init__(self, surface):
        if isinstance(surface, TranslationSurface):
            base_ring = surface.base_ring()
            from flatsurf.geometry.pyflatsurf_conversion import to_pyflatsurf
            self._surface = to_pyflatsurf(surface)
        else:
            from flatsurf.geometry.pyflatsurf_conversion import sage_base_ring
            base_ring, _ = sage_base_ring(surface)
            self._surface = surface

        # A model of the vector space R² in libflatsurf, e.g., to represent the
        # vector associated to a saddle connection.
        self.V2 = pyflatsurf.vector.Vectors(base_ring)

        # We construct a spanning set of edges, that is a subset of the
        # edges that form a basis of H_1(S, Sigma; Z)
        # It comes together with a projection matrix
        t, m = self._spanning_tree()
        assert set(t.keys()) == set(f[0] for f in self._faces())
        self.spanning_set = []
        v = set(t.values())
        for e in self._surface.edges():
            if e.positive() not in v and e.negative() not in v:
                self.spanning_set.append(e)
        self.d = len(self.spanning_set)
        assert 3*self.d - 3 == self._surface.size()
        assert m.rank() == self.d
        m = m.transpose()
        # projection matrix from Z^E to H_1(S, Sigma; Z) in the basis
        # of spanning edges
        self.proj = matrix(ZZ, [r for r in m.rows() if not r.is_zero()])

        self.Omega = self._intersection_matrix(t, self.spanning_set)

        self.V = FreeModule(self.V2.base_ring(), self.d)
        self.H = matrix(self.V2.base_ring(), self.d, 2)
        for i in range(self.d):
            s = self._surface.fromHalfEdge(self.spanning_set[i].positive())
            self.H[i] = self.V2._isomorphic_vector_space(self.V2(s))
        self.Hdual = self.Omega * self.H

        # Note that we don't use Sage vector spaces because they are usually
        # way too slow (in particular we avoid calling .echelonize())
        self._U = matrix(self.V2._algebraic_ring(), self.d)
        self._U_rank = 0
        self.update_tangent_space_from_vector(self.H.transpose()[0])
        self.update_tangent_space_from_vector(self.H.transpose()[1])

    def dimension(self):
        r"""
        Return the current real dimension of the GL(2,R)-orbit closure.

        EXAMPLES::

            sage: from flatsurf import EquiangularPolygons, similarity_surfaces
            sage: from flatsurf import GL2ROrbitClosure # optional: pyflatsurf
            sage: E = EquiangularPolygons(1, 3, 5)
            sage: T = E(1)
            sage: S = similarity_surfaces.billiard(T)
            sage: S = S.minimal_cover(cover_type="translation")
            sage: O = GL2ROrbitClosure(S) # optional: pyflatsurf
            sage: O.dimension() # optional: pyflatsurf
            2
            sage: bound = E.billiard_unfolding_stratum('half-translation', marked_points=True).dimension()
            sage: for decomposition in O.decompositions(1):  # long time, optional: pyflatsurf
            ....:     if O.dimension() == bound: break
            ....:     O.update_tangent_space_from_flow_decomposition(decomposition)
            ....:     print(O.dimension())
            3
            5
            5
            7
            9
            10
        """
        return self._U_rank

    def ambient_stratum(self):
        r"""
        Return the stratum of Abelian differentials this surface belongs to.

        EXAMPLES::

            sage: from flatsurf import EquiangularPolygons, similarity_surfaces
            sage: from flatsurf import GL2ROrbitClosure # optional: pyflatsurf
            sage: E = EquiangularPolygons(1, 3, 5)
            sage: T = E(1)
            sage: S = similarity_surfaces.billiard(T)
            sage: S = S.minimal_cover(cover_type="translation")
            sage: O = GL2ROrbitClosure(S) # optional: pyflatsurf
            sage: O.ambient_stratum() # optional: pyflatsurf
            H_3(4, 0^4)
        """
        from surface_dynamics import AbelianStratum
        surface = self._surface
        angles = [surface.angle(v) for v in surface.vertices()]
        return AbelianStratum([a-1 for a in angles])

    def base_ring(self):
        r"""
        Return the underlying base ring
        """
        return self._U.base_ring()

    def field_of_definition(self):
        r"""
        Return the field of definition of the current subspace.

        .. WARNING::

            This involves the computation of the echelon form of the matrix. It
            might be rather expensive if the computation of the tangent space is
            not terminated.

        EXAMPLES::

            sage: from flatsurf import polygons, similarity_surfaces, EquiangularPolygons
            sage: from flatsurf import GL2ROrbitClosure  # optional: pyflatsurf
            sage: from pyexactreal import ExactReals  # optional: exactreal
            sage: E = EquiangularPolygons(1, 5, 5, 5)
            sage: R = ExactReals(E.base_ring())  # optional: exactreal
            sage: T = E(R(1), R.random_element(1/4))  # optional: exactreal
            sage: S = similarity_surfaces.billiard(T)  # optional: exactreal
            sage: S = S.minimal_cover(cover_type="translation")  # optional: exactreal
            sage: O = GL2ROrbitClosure(S); O  # optional: exactreal, pyflatsurf
            GL(2,R)-orbit closure of dimension at least 4 in H_7(4^3, 0) (ambient dimension 17)
            sage: O.field_of_definition() # optional: exactreal, pyflatsurf
            Number Field in c0 with defining polynomial x^2 - 2 with c0 = 1.414213562373095?
            sage: bound = E.billiard_unfolding_stratum('half-translation', marked_points=True).dimension()
            sage: for decomposition in O.decompositions(1):  # long time, optional: exactreal, pyflatsurf
            ....:     if O.dimension() == bound: break
            ....:     O.update_tangent_space_from_flow_decomposition(decomposition)
            sage: O.field_of_definition()  # long time, optional: exactreal, pyflatsurf
            Rational Field

            sage: E = EquiangularPolygons(1, 3, 5)
            sage: T = E(1)
            sage: S = similarity_surfaces.billiard(T)
            sage: S = S.minimal_cover(cover_type="translation")
            sage: O = GL2ROrbitClosure(S) # optional: pyflatsurf
            sage: O.field_of_definition() # optional: pyflatsurf
            Number Field in c0 with defining polynomial x^3 - 3*x - 1 with c0 = 1.879385241571817?
            sage: bound = E.billiard_unfolding_stratum('half-translation', marked_points=True).dimension()
            sage: for decomposition in O.decompositions(1):  # long time, optional: pyflatsurf
            ....:     if O.dimension() == bound: break
            ....:     O.update_tangent_space_from_flow_decomposition(decomposition)
            sage: O.field_of_definition()  # long time, optional: pyflatsurf
            Rational Field
        """
        M = self._U.echelon_form()
        L, elts, phi = subfield_from_elements(M.base_ring(), M[:self._U_rank].list())
        return L

    def _half_edge_to_face(self, h):
        surface = self._surface
        h1 = h
        h2 = surface.nextInFace(h1)
        h3 = surface.nextInFace(h2)
        return min([h1, h2, h3], key=lambda x: x.index())

    def _faces(self):
        seen = set()
        faces = []
        surface = self._surface
        for e in surface.edges():
            for h1 in [e.positive(), e.negative()]:
                if h1 in seen:
                    continue
                h2 = surface.nextInFace(h1)
                h3 = surface.nextInFace(h2)
                faces.append((h1, h2, h3))
                seen.add(h1)
                seen.add(h2)
                seen.add(h3)
        return faces

    def __repr__(self):
        return "GL(2,R)-orbit closure of dimension at least %d in %s (ambient dimension %d)" % (self._U_rank, self.ambient_stratum(), self.d)

    def holonomy(self, v):
        r"""
        Return the holonomy of ``v`` (with respect to the basis)
        """
        return self.V(v) * self.H

    def holonomy_dual(self, v):
        return self.V(v) * self.Hdual

    def tangent_space_basis(self):
        return self._U[:self._U_rank].rows()

    def lift(self, v):
        r"""
        Given a vector in the "spanning set basis" return a vector on the full basis of
        edges.

        EXAMPLES::

            sage: from flatsurf import polygons, translation_surfaces, similarity_surfaces
            sage: from flatsurf import GL2ROrbitClosure  # optional: pyflatsurf

            sage: S = translation_surfaces.mcmullen_genus2_prototype(4,2,1,1,0)
            sage: O = GL2ROrbitClosure(S)  # optional: pyflatsurf
            sage: u0,u1 = O.tangent_space_basis()  # optional: pyflatsurf
            sage: v0 = O.lift(u0)  # optional: pyflatsurf
            sage: v1 = O.lift(u1)  # optional: pyflatsurf
            sage: span([v0, v1])  # optional: pyflatsurf
            Vector space of degree 9 and dimension 2 over Real Embedded Number Field in l with defining polynomial x^2 - x - 8 with l = 3.372281323269015?
            Basis matrix:
            [                         1                          0                         -1   (1/4*l-1/4 ~ 0.59307033) (-1/4*l+1/4 ~ -0.59307033)                          0 (-1/4*l+1/4 ~ -0.59307033)                          0 (-1/4*l+1/4 ~ -0.59307033)]
            [                         0                          1                         -1    (1/8*l+7/8 ~ 1.2965352) (-1/8*l+1/8 ~ -0.29653517)                         -1 (3/8*l-11/8 ~ -0.11039450) (-1/2*l+3/2 ~ -0.18614066) (-1/8*l+1/8 ~ -0.29653517)]

        This can be used to deform the surface::

            sage: T = polygons.triangle(3,4,13)
            sage: S = similarity_surfaces.billiard(T)
            sage: S = S.minimal_cover("translation").erase_marked_points() # optional: pyflatsurf
            sage: O = GL2ROrbitClosure(S)  # optional: pyflatsurf
            sage: for d in O.decompositions(4, 20):  # optional: pyflatsurf
            ....:     O.update_tangent_space_from_flow_decomposition(d)
            ....:     if O.dimension() == 4:
            ....:         break
            sage: d1,d2,d3,d4 = [O.lift(b) for b in O.tangent_space_basis()]  # optional: pyflatsurf
            sage: dreal = d1/132 + d2/227 + d3/280 - d4/201  # optional: pyflatsurf
            sage: dimag = d1/141 - d2/233 + d4/230 + d4/250  # optional: pyflatsurf
            sage: d = [O.V2((x,y)).vector for x,y in zip(dreal,dimag)]  # optional: pyflatsurf
            sage: S2 = O._surface + d  # optional: pyflatsurf

        TODO: ``GL2ROrbitClosure`` does not accept pyflatsurf surfaces as input::

            sage: O2 = GL2ROrbitClosure(S2)   # not tested
            sage: for d in O2.decompositions(4, 20, sector=((1,0),(5,1))):  # not tested
            ....:     O2.update_tangent_space_from_flow_decomposition(d)
        """
        # given the values on the spanning edges we reconstruct the unique vector that
        # vanishes on the boundary
        bdry = self.boundaries()
        n = self._surface.edges().size()
        k = len(self.spanning_set)
        assert k + len(bdry) == n + 1
        A = matrix(self.V2.base_ring(), n+1, n)
        for i,e in enumerate(self.spanning_set):
            A[i,e.index()] = 1
        for i,b in enumerate(bdry):
            A[k+i,:] = b
        u = vector(self.V2.base_ring(), n + 1)
        u[:k] = v
        return A.solve_right(u)

    def absolute_homology(self):
        vert_index = {v:i for i,v in enumerate(self._surface.vertices())}
        m = len(vert_index)
        if m == 1:
            return self.V
        rows = []
        for e in self.spanning_set:
            r = [0] * m
            i = vert_index[Vertex.target(e.positive(), self._surface.combinatorial())]
            j = vert_index[Vertex.source(e.positive(), self._surface.combinatorial())]
            if i != j:
                r[i] = 1
                r[j] = -1
            rows.append(r)
        return matrix(rows).left_kernel()

    def absolute_dimension(self):
        r"""
        EXAMPLES::

            sage: from flatsurf import polygons, similarity_surfaces
            sage: from flatsurf import GL2ROrbitClosure  # optional: pyflatsurf
            sage: T = polygons.triangle(1,3,4)  # Veech octagon
            sage: S = similarity_surfaces.billiard(T)
            sage: S = S.minimal_cover("translation")
            sage: O = GL2ROrbitClosure(S)  # optional: pyflatsurf
            sage: O.absolute_dimension()  # optional: pyflatsurf
            2

        The triangular billiard (5,6,7) belongs to the canonical double cover of
        the stratum Q(5,3,0^3) in genus 3. The orbit is dense and we can check
        that the absolute dimension is indeed `6 = 2 rank`::

            sage: T = polygons.triangle(5,6,7)
            sage: S = similarity_surfaces.billiard(T)
            sage: S = S.minimal_cover("translation")
            sage: O = GL2ROrbitClosure(S)  # optional: pyflatsurf
            sage: for d in O.decompositions(5, 100):  # optional: pyflatsurf
            ....:     O.update_tangent_space_from_flow_decomposition(d)
            ....:     if O.dimension() == 9:
            ....:         break
            sage: O.absolute_dimension()  # optional: pyflatsurf
            6
        """
        return (self.absolute_homology().matrix() * self._U[:self._U_rank].transpose()).rank()

    def _spanning_tree(self, root=None):
        r"""
        Return

        - a list of indices of edges that form a basis
        - a matrix projection to the basis (modulo the triangle relations)
        """
        if root is None:
            root = next(iter(self._surface.edges())).positive()

        root = self._half_edge_to_face(root)
        t = {root: None} # face -> half edge to take to go to the root
        todo = [root]
        edges = []  # store edges in topological order to perform Gauss reduction
        while todo:
            f = todo.pop()
            for _ in range(3):
                f1 = -f
                g = self._half_edge_to_face(f1)
                if g not in t:
                    t[g] = f1
                    todo.append(g)
                    edges.append(f1)

                f = self._surface.nextInFace(f)

        # gauss reduction
        n = self._surface.size()
        proj = identity_matrix(ZZ, n)
        edges.reverse()
        for f1 in edges:
            v = [0] * n
            f2 = self._surface.nextInFace(f1)
            f3 = self._surface.nextInFace(f2)
            assert self._surface.nextInFace(f3) == f1

            i1 = f1.index()
            s1 = -1 if i1%2 else 1
            i2 = f2.index()
            s2 = -1 if i2%2 else 1
            i3 = f3.index()
            s3 = -1 if i3%2 else 1
            i1 = f1.edge().index()
            i2 = f2.edge().index()
            i3 = f3.edge().index()
            proj[i1] = -s1*(s2*proj[i2] + s3*proj[i3])
            for j in range(n):
                assert proj[j,i1] == 0

        return (t, proj)

    def _intersection_matrix(self, t, spanning_set):
        r"""
        Given a spanning tree, compute the associated intersection matrix.

        It can be used to compute holonomies. (we can be off by a - sign)
        """
        d = len(spanning_set)
        h = spanning_set[0].positive()
        all_edges = set([e.positive() for e in spanning_set])
        all_edges.update([e.negative() for e in spanning_set])
        contour = []
        contour_inv = {}   # half edge -> position in contour
        while h not in contour_inv:
            contour_inv[h] = len(contour)
            contour.append(h)
            h = self._surface.nextAtVertex(-h)
            while h not in all_edges:
                h = self._surface.nextAtVertex(h)

        assert len(contour) == len(all_edges)

        # two curves intersect when their relative position in the contour
        # are x y x y or y x y x
        Omega = matrix(ZZ, d)
        for i in range(len(spanning_set)):
            ei = spanning_set[i]
            pi1 = contour_inv[ei.positive()]
            pi2 = contour_inv[ei.negative()]
            if pi1 > pi2:
                si = -1
                pi1, pi2 = pi2, pi1
            else:
                si = 1
            for j in range(i+1, len(spanning_set)):
                ej = spanning_set[j]
                pj1 = contour_inv[ej.positive()]
                pj2 = contour_inv[ej.negative()]
                if pj1 > pj2:
                    sj = -1
                    pj1, pj2 = pj2, pj1
                else:
                    sj = 1

                # pj1 pj2 pi1 pi2: pj2 < pi1
                # pi1 pi2 pj1 pj2: pi2 < pj1
                # pi1 pj1 pj2 pi2: pi1 < pj1 and pj2 < pi2
                # pj1 pi1 pi2 pj2: pj1 < pi1 and pi2 < pj2
                if (pj2 < pi1) or (pi2 < pj1) or \
                   (pj1 > pi1 and pj2 < pi2) or \
                   (pj1 < pi1 and pj2 > pi2):
                    # no intersection
                    continue

                if pi1 < pj1 < pi2:
                    # one sign
                    Omega[i,j] = si * sj
                else:
                    # other sign
                    assert pi1 < pj2 < pi2, (pi1, pi2, pj1, pj2)
                    Omega[i,j] = -si*sj
                Omega[j,i] = - Omega[i,j]
        return Omega

    def boundaries(self):
        r"""
        Return the list of boundaries (ie sum of edges around a triangular face).

        These are elements of H_1(S, Sigma; Z).

        TESTS::

            sage: from flatsurf import polygons, similarity_surfaces
            sage: from flatsurf import GL2ROrbitClosure  # optional: pyflatsurf

            sage: from itertools import product
            sage: for a in range(1,5):  # optional: pyflatsurf
            ....:     for b in range(a, 5):
            ....:         for c in range(b, 5):
            ....:             if gcd([a, b, c]) != 1 or (a,b,c) == (1,1,2):
            ....:                 continue
            ....:             T = polygons.triangle(a, b, c)
            ....:             S = similarity_surfaces.billiard(T)
            ....:             S = S.minimal_cover(cover_type="translation")
            ....:             O = GL2ROrbitClosure(S)
            ....:             for b in O.boundaries():
            ....:                 assert (O.proj * b).is_zero()
        """
        n = self._surface.size()
        V = FreeModule(ZZ, n)
        B = []
        for (f1,f2,f3) in self._faces():
            i1 = f1.index()
            s1 = -1 if i1%2 else 1
            i2 = f2.index()
            s2 = -1 if i2%2 else 1
            i3 = f3.index()
            s3 = -1 if i3%2 else 1
            i1 = f1.edge().index()
            i2 = f2.edge().index()
            i3 = f3.edge().index()
            v = [0] * n
            v[i1] = 1
            v[i2] = s1 * s2
            v[i3] = s1 * s3
            B.append(V(v))
            B[-1].set_immutable()

        return B

    def decomposition(self, v, limit=-1):
        v = self.V2(v)
        decomposition = pyflatsurf.flatsurf.makeFlowDecomposition(self._surface, v.vector)
        u = self.V2._isomorphic_vector_space(v)
        if limit != 0:
            decomposition.decompose(int(limit))
        return Decomposition(self, decomposition, u)

    def decompositions(self, bound, limit=-1, bfs=False):
        limit = int(limit)

        connections = self._surface.connections().bound(int(bound))
        if bfs:
            connections = connections.byLength()

        Vector = cppyy.gbl.flatsurf.Vector[type(self._surface).Coordinate]
        slopes = cppyy.gbl.std.set[Vector, Vector.CompareSlope]()

        for connection in connections:
            direction = connection.vector()
            if slopes.find(direction) != slopes.end():
                continue
            slopes.insert(direction)
            yield self.decomposition(direction, limit)

    def decompositions_depth_first(self, bound, limit=-1):
        return self.decompositions(bound, bfs=False, limit=limit)

    def decompositions_breadth_first(self, bound, limit=-1):
        return self.decompositions(bound, bfs=True, limit=limit)

    def is_teichmueller_curve(self, bound, limit=-1):
        r"""
        Return ``False`` when the program can find a direction which is either completely
        periodic with incomensurable moduli or a direction with at least one cylinder
        and at least one minimal component.

        EXAMPLES::

            sage: from flatsurf import polygons, similarity_surfaces
            sage: from flatsurf import GL2ROrbitClosure  # optional: pyflatsurf
            sage: for a in range(1,6):  # optional: pyflatsurf
            ....:     for b in range(a,6):
            ....:         for c in range(b,6):
            ....:             if a + b + c > 7 or gcd([a,b,c]) != 1:
            ....:                 continue
            ....:             T = polygons.triangle(a, b, c)
            ....:             S = similarity_surfaces.billiard(T)
            ....:             S = S.minimal_cover(cover_type="translation")
            ....:             O = GL2ROrbitClosure(S)
            ....:             if O.is_teichmueller_curve(3, 50) != False:
            ....:                 print(a,b,c)
            1 1 1
            1 1 2
            1 1 4
            1 2 2
            1 2 3
            1 3 3
        """
        if self.V2.base_ring in [ZZ, QQ]:
            # square tiled surface
            return True
        # TODO: implement simpler criterion based on the holonomy field
        # (e.g. one can compute the trace field and verify that it is
        #  totally real, of degree at most the genus and that the surface
        #  is algebraically completely periodic)
        for decomposition in self.decompositions_depth_first(bound, limit):
            if decomposition.parabolic() == False:
                return False
        # TODO: from there on one should run the program of Ronen Mukamel (or
        # something similar) to certify that we do have a Veech surface
        return Unknown

    def update_tangent_space_from_flow_decomposition(self, decomposition):
        r"""
        Update the current tangent space by using the cylinder deformation vectors from ``decomposition``.

        EXAMPLES::

            sage: from flatsurf import polygons, similarity_surfaces
            sage: from flatsurf import GL2ROrbitClosure  # optional: pyflatsurf

            sage: T = polygons.triangle(1, 2, 5)
            sage: S = similarity_surfaces.billiard(T)
            sage: S = S.minimal_cover(cover_type="translation")
            sage: O = GL2ROrbitClosure(S)  # optional: pyflatsurf
            sage: for d in O.decompositions(1):  # optional: pyflatsurf
            ....:     O.update_tangent_space_from_flow_decomposition(d)
            sage: assert O.dimension() == 2  # optional: pyflatsurf

        TESTS:

        A regression test for  https://github.com/flatsurf/sage-flatsurf/pull/69::

            sage: from itertools import islice
            sage: from flatsurf import polygons, similarity_surfaces
            sage: from flatsurf import GL2ROrbitClosure  # optional: pyflatsurf

            sage: for (a,b,c,dim) in [(3,2,2,7), (4,2,1,8), (4,4,3,11), (5,3,3,11), (5,4,4,13), (5,5,3,13)]: # long time, optional: pyflatsurf
            ....:     T = polygons.triangle(a, b, c)
            ....:     S = similarity_surfaces.billiard(T)
            ....:     S = S.minimal_cover(cover_type="translation")
            ....:     O = GL2ROrbitClosure(S)  # optional: pyflatsurf
            ....:     nsteps = 0
            ....:     for d in islice(O.decompositions(3,100), 10):  # optional: pyflatsurf
            ....:         O.update_tangent_space_from_flow_decomposition(d)
            ....:         nsteps += 1
            ....:         if O.dimension() == dim:
            ....:             break
            ....:     print("(%d, %d, %d): %d" % (a, b, c, nsteps))
            (3, 2, 2): 5
            (4, 2, 1): 4
            (4, 4, 3): 4
            (5, 3, 3): 4
            (5, 4, 4): 7
            (5, 5, 3): 4
        """
        if self._U_rank == self._U.nrows(): return
        for v in decomposition.cylinder_deformation_subspace():
            self.update_tangent_space_from_vector(v)
            if self._U_rank == self._U.nrows(): return

    def update_tangent_space_from_vector(self, v):
        if self._U_rank == self._U.nrows():
            return

        v = vector(v)

        if v.base_ring() is not self.V2._algebraic_ring():
            for gen, p in self.V2.decomposition(v):
                self.update_tangent_space_from_vector(p)
            return

        self._U[self._U_rank] = v
        r = self._U.rank()
        if r > self._U_rank:
            assert r == self._U_rank + 1
            self._U_rank += 1

    def __eq__(self, other):
        r"""
        Return whether ``other`` was built starting from the same surface than
        this orbit closure.

        EXAMPLES::

            sage: from flatsurf import polygons, similarity_surfaces
            sage: from flatsurf import GL2ROrbitClosure  # optional: pyflatsurf

            sage: T = polygons.triangle(1, 2, 5)
            sage: S = similarity_surfaces.billiard(T)
            sage: S = S.minimal_cover(cover_type="translation")
            sage: GL2ROrbitClosure(S) == GL2ROrbitClosure(S) # optional: pyflatsurf
            True
            
        """
        return self._surface == other._surface

    def __ne__(self, other):
        r"""
        Return whether ``other`` was not built starting from the same surface
        than this orbit closure.

        EXAMPLES::

            sage: from flatsurf import polygons, similarity_surfaces
            sage: from flatsurf import GL2ROrbitClosure  # optional: pyflatsurf

            sage: T = polygons.triangle(1, 2, 5)
            sage: S = similarity_surfaces.billiard(T)
            sage: S = S.minimal_cover(cover_type="translation")
            sage: GL2ROrbitClosure(S) != GL2ROrbitClosure(S) # optional: pyflatsurf
            False
            
        """
        return not (self == other)

    def __hash__(self):
        r"""
        Return a hash value of this object compatible with equality operators.

        EXAMPLES::

            sage: from flatsurf import polygons, similarity_surfaces
            sage: from flatsurf import GL2ROrbitClosure  # optional: pyflatsurf

            sage: T = polygons.triangle(1, 2, 5)
            sage: S = similarity_surfaces.billiard(T)
            sage: S = S.minimal_cover(cover_type="translation")
            sage: O = GL2ROrbitClosure(S) # optional: pyflatsurf
            sage: hash(O) == hash(O)
            True

        """
        return hash(self._surface)

    def __reduce__(self):
        r"""
        Return a serializable representation of this Orbit Closure.

        EXAMPLES::

            sage: from flatsurf import polygons, similarity_surfaces
            sage: from flatsurf import GL2ROrbitClosure  # optional: pyflatsurf

            sage: T = polygons.triangle(1, 2, 5)
            sage: S = similarity_surfaces.billiard(T)
            sage: S = S.minimal_cover(cover_type="translation")
            sage: O = GL2ROrbitClosure(S)  # optional: pyflatsurf
            sage: loads(dumps(O)) == O
            True

        """
        from flatsurf.geometry.pyflatsurf_conversion import from_pyflatsurf
        return (GL2ROrbitClosure, (self._surface,), {'_U': self._U, '_U_rank': self._U_rank})
