"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ComputeEnvironment = exports.AllocationStrategy = exports.ComputeResourceType = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("aws-cdk-lib/aws-ec2");
const iam = require("aws-cdk-lib/aws-iam");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_batch_1 = require("aws-cdk-lib/aws-batch");
/**
 * Property to specify if the compute environment
 * uses On-Demand, SpotFleet, Fargate, or Fargate Spot compute resources.
 */
var ComputeResourceType;
(function (ComputeResourceType) {
    /**
     * Resources will be EC2 On-Demand resources.
     */
    ComputeResourceType["ON_DEMAND"] = "EC2";
    /**
     * Resources will be EC2 SpotFleet resources.
     */
    ComputeResourceType["SPOT"] = "SPOT";
    /**
     * Resources will be Fargate resources.
     */
    ComputeResourceType["FARGATE"] = "FARGATE";
    /**
     * Resources will be Fargate Spot resources.
     *
     * Fargate Spot uses spare capacity in the AWS cloud to run your fault-tolerant,
     * time-flexible jobs at up to a 70% discount. If AWS needs the resources back,
     * jobs running on Fargate Spot will be interrupted with two minutes of notification.
     */
    ComputeResourceType["FARGATE_SPOT"] = "FARGATE_SPOT";
})(ComputeResourceType = exports.ComputeResourceType || (exports.ComputeResourceType = {}));
/**
 * Properties for how to prepare compute resources
 * that are provisioned for a compute environment.
 */
var AllocationStrategy;
(function (AllocationStrategy) {
    /**
     * Batch will use the best fitting instance type will be used
     * when assigning a batch job in this compute environment.
     */
    AllocationStrategy["BEST_FIT"] = "BEST_FIT";
    /**
     * Batch will select additional instance types that are large enough to
     * meet the requirements of the jobs in the queue, with a preference for
     * instance types with a lower cost per unit vCPU.
     */
    AllocationStrategy["BEST_FIT_PROGRESSIVE"] = "BEST_FIT_PROGRESSIVE";
    /**
     * This is only available for Spot Instance compute resources and will select
     * additional instance types that are large enough to meet the requirements of
     * the jobs in the queue, with a preference for instance types that are less
     * likely to be interrupted.
     */
    AllocationStrategy["SPOT_CAPACITY_OPTIMIZED"] = "SPOT_CAPACITY_OPTIMIZED";
})(AllocationStrategy = exports.AllocationStrategy || (exports.AllocationStrategy = {}));
/**
 * Batch Compute Environment.
 *
 * Defines a batch compute environment to run batch jobs on.
 */
class ComputeEnvironment extends aws_cdk_lib_1.Resource {
    constructor(scope, id, props = { enabled: true, managed: true }) {
        super(scope, id, {
            physicalName: props.computeEnvironmentName,
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_batch_alpha_ComputeEnvironmentProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, ComputeEnvironment);
            }
            throw error;
        }
        const isFargate = ComputeResourceType.FARGATE === props.computeResources?.type
            || ComputeResourceType.FARGATE_SPOT === props.computeResources?.type;
        ;
        this.validateProps(props, isFargate);
        const spotFleetRole = this.getSpotFleetRole(props);
        let computeResources;
        this.connections = this.buildConnections(props.computeResources?.vpc, props.computeResources?.securityGroups);
        // Only allow compute resources to be set when using MANAGED type
        if (props.computeResources && this.isManaged(props)) {
            computeResources = {
                bidPercentage: props.computeResources.bidPercentage,
                desiredvCpus: props.computeResources.desiredvCpus,
                ec2KeyPair: props.computeResources.ec2KeyPair,
                imageId: props.computeResources.image && props.computeResources.image.getImage(this).imageId,
                launchTemplate: props.computeResources.launchTemplate,
                maxvCpus: props.computeResources.maxvCpus || 256,
                placementGroup: props.computeResources.placementGroup,
                securityGroupIds: this.getSecurityGroupIds(),
                spotIamFleetRole: spotFleetRole?.roleArn,
                subnets: props.computeResources.vpc.selectSubnets(props.computeResources.vpcSubnets).subnetIds,
                tags: props.computeResources.computeResourcesTags,
                type: props.computeResources.type || ComputeResourceType.ON_DEMAND,
                ...(!isFargate ? {
                    allocationStrategy: props.computeResources.allocationStrategy
                        || (props.computeResources.type === ComputeResourceType.SPOT
                            ? AllocationStrategy.SPOT_CAPACITY_OPTIMIZED
                            : AllocationStrategy.BEST_FIT),
                    instanceRole: props.computeResources.instanceRole
                        ? props.computeResources.instanceRole
                        : new iam.CfnInstanceProfile(this, 'Instance-Profile', {
                            roles: [new iam.Role(this, 'Ecs-Instance-Role', {
                                    assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
                                    managedPolicies: [
                                        iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2ContainerServiceforEC2Role'),
                                    ],
                                }).roleName],
                        }).attrArn,
                    instanceTypes: this.buildInstanceTypes(props.computeResources.instanceTypes),
                    minvCpus: props.computeResources.minvCpus || 0,
                } : {}),
            };
        }
        const computeEnvironment = new aws_batch_1.CfnComputeEnvironment(this, 'Resource', {
            computeEnvironmentName: this.physicalName,
            computeResources,
            serviceRole: props.serviceRole?.roleArn
                ?? new iam.Role(this, 'Resource-Service-Instance-Role', {
                    managedPolicies: [
                        iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSBatchServiceRole'),
                    ],
                    assumedBy: new iam.ServicePrincipal('batch.amazonaws.com'),
                }).roleArn,
            state: props.enabled === undefined ? 'ENABLED' : (props.enabled ? 'ENABLED' : 'DISABLED'),
            type: this.isManaged(props) ? 'MANAGED' : 'UNMANAGED',
        });
        if (props.computeResources && props.computeResources.vpc) {
            this.node.addDependency(props.computeResources.vpc);
        }
        this.computeEnvironmentArn = this.getResourceArnAttribute(computeEnvironment.ref, {
            service: 'batch',
            resource: 'compute-environment',
            resourceName: this.physicalName,
        });
        this.computeEnvironmentName = this.getResourceNameAttribute(computeEnvironment.ref);
    }
    /**
     * Fetches an existing batch compute environment by its amazon resource name.
     *
     * @param scope
     * @param id
     * @param computeEnvironmentArn
     */
    static fromComputeEnvironmentArn(scope, id, computeEnvironmentArn) {
        const stack = aws_cdk_lib_1.Stack.of(scope);
        const computeEnvironmentName = stack.splitArn(computeEnvironmentArn, aws_cdk_lib_1.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
        class Import extends aws_cdk_lib_1.Resource {
            constructor() {
                super(...arguments);
                this.computeEnvironmentArn = computeEnvironmentArn;
                this.computeEnvironmentName = computeEnvironmentName;
            }
        }
        return new Import(scope, id);
    }
    isManaged(props) {
        return props.managed ?? true;
    }
    /**
     * Validates the properties provided for a new batch compute environment.
     */
    validateProps(props, isFargate) {
        if (props === undefined) {
            return;
        }
        if (!this.isManaged(props) && props.computeResources !== undefined) {
            throw new Error('It is not allowed to set computeResources on an AWS unmanaged compute environment');
        }
        if (this.isManaged(props) && props.computeResources === undefined) {
            throw new Error('computeResources is missing but required on a managed compute environment');
        }
        if (props.computeResources) {
            if (isFargate) {
                // VALIDATE FOR FARGATE
                // Bid percentage cannot be set for Fargate evnvironments
                if (props.computeResources.bidPercentage !== undefined) {
                    throw new Error('Bid percentage must not be set for Fargate compute environments');
                }
                // Allocation strategy cannot be set for Fargate evnvironments
                if (props.computeResources.allocationStrategy !== undefined) {
                    throw new Error('Allocation strategy must not be set for Fargate compute environments');
                }
                // Desired vCPUs cannot be set for Fargate evnvironments
                if (props.computeResources.desiredvCpus !== undefined) {
                    throw new Error('Desired vCPUs must not be set for Fargate compute environments');
                }
                // Image ID cannot be set for Fargate evnvironments
                if (props.computeResources.image !== undefined) {
                    throw new Error('Image must not be set for Fargate compute environments');
                }
                // Instance types cannot be set for Fargate evnvironments
                if (props.computeResources.instanceTypes !== undefined) {
                    throw new Error('Instance types must not be set for Fargate compute environments');
                }
                // EC2 key pair cannot be set for Fargate evnvironments
                if (props.computeResources.ec2KeyPair !== undefined) {
                    throw new Error('EC2 key pair must not be set for Fargate compute environments');
                }
                // Instance role cannot be set for Fargate evnvironments
                if (props.computeResources.instanceRole !== undefined) {
                    throw new Error('Instance role must not be set for Fargate compute environments');
                }
                // Launch template cannot be set for Fargate evnvironments
                if (props.computeResources.launchTemplate !== undefined) {
                    throw new Error('Launch template must not be set for Fargate compute environments');
                }
                // Min vCPUs cannot be set for Fargate evnvironments
                if (props.computeResources.minvCpus !== undefined) {
                    throw new Error('Min vCPUs must not be set for Fargate compute environments');
                }
                // Placement group cannot be set for Fargate evnvironments
                if (props.computeResources.placementGroup !== undefined) {
                    throw new Error('Placement group must not be set for Fargate compute environments');
                }
                // Spot fleet role cannot be set for Fargate evnvironments
                if (props.computeResources.spotFleetRole !== undefined) {
                    throw new Error('Spot fleet role must not be set for Fargate compute environments');
                }
            }
            else {
                // VALIDATE FOR ON_DEMAND AND SPOT
                if (props.computeResources.minvCpus) {
                    // minvCpus cannot be less than 0
                    if (props.computeResources.minvCpus < 0) {
                        throw new Error('Minimum vCpus for a batch compute environment cannot be less than 0');
                    }
                    // minvCpus cannot exceed max vCpus
                    if (props.computeResources.maxvCpus &&
                        props.computeResources.minvCpus > props.computeResources.maxvCpus) {
                        throw new Error('Minimum vCpus cannot be greater than the maximum vCpus');
                    }
                }
                // Check if both launchTemplateId and launchTemplateName are provided
                if (props.computeResources.launchTemplate &&
                    (props.computeResources.launchTemplate.launchTemplateId && props.computeResources.launchTemplate.launchTemplateName)) {
                    throw new Error('You must specify either the launch template ID or launch template name in the request, but not both.');
                }
                // Check if both launchTemplateId and launchTemplateName are missing
                if (props.computeResources.launchTemplate &&
                    (!props.computeResources.launchTemplate.launchTemplateId && !props.computeResources.launchTemplate.launchTemplateName)) {
                    throw new Error('You must specify either the launch template ID or launch template name in the request.');
                }
                // Setting a bid percentage is only allowed on SPOT resources +
                // Cannot use SPOT_CAPACITY_OPTIMIZED when using ON_DEMAND
                if (props.computeResources.type === ComputeResourceType.ON_DEMAND) {
                    // VALIDATE FOR ON_DEMAND
                    // Bid percentage is not allowed
                    if (props.computeResources.bidPercentage !== undefined) {
                        throw new Error('Setting the bid percentage is only allowed for SPOT type resources on a batch compute environment');
                    }
                    // SPOT_CAPACITY_OPTIMIZED allocation is not allowed
                    if (props.computeResources.allocationStrategy && props.computeResources.allocationStrategy === AllocationStrategy.SPOT_CAPACITY_OPTIMIZED) {
                        throw new Error('The SPOT_CAPACITY_OPTIMIZED allocation strategy is only allowed if the environment is a SPOT type compute environment');
                    }
                }
                else if (props.computeResources.type === ComputeResourceType.SPOT) {
                    // VALIDATE FOR SPOT
                    // Bid percentage must be from 0 - 100
                    if (props.computeResources.bidPercentage !== undefined &&
                        (props.computeResources.bidPercentage < 0 || props.computeResources.bidPercentage > 100)) {
                        throw new Error('Bid percentage can only be a value between 0 and 100');
                    }
                }
            }
        }
    }
    buildInstanceTypes(instanceTypes) {
        if (instanceTypes === undefined) {
            return [
                'optimal',
            ];
        }
        return instanceTypes.map((type) => type.toString());
    }
    buildConnections(vpc, securityGroups) {
        if (vpc === undefined) {
            return new ec2.Connections({});
        }
        if (securityGroups === undefined) {
            return new ec2.Connections({
                securityGroups: [
                    new ec2.SecurityGroup(this, 'Resource-Security-Group', { vpc }),
                ],
            });
        }
        return new ec2.Connections({ securityGroups });
    }
    ;
    getSecurityGroupIds() {
        if (this.connections === undefined) {
            return undefined;
        }
        return this.connections.securityGroups.map((group) => group.securityGroupId);
    }
    /**
     * Generates an AWS IAM role for provisioning spotfleet resources
     * if the allocation strategy is set to BEST_FIT or not defined.
     *
     * @param props - the compute environment construct properties
     */
    getSpotFleetRole(props) {
        if (props.computeResources?.allocationStrategy && props.computeResources.allocationStrategy !== AllocationStrategy.BEST_FIT) {
            return undefined;
        }
        if (props.computeResources) {
            if (props.computeResources.spotFleetRole) {
                return props.computeResources.spotFleetRole;
            }
            else if (props.computeResources.type === ComputeResourceType.SPOT) {
                return iam.Role.fromRoleArn(this, 'Resource-SpotFleet-Role', `arn:${this.stack.partition}:iam::${this.env.account}:role/aws-service-role/spotfleet.amazonaws.com/AWSServiceRoleForEC2SpotFleet`);
            }
        }
        return undefined;
    }
}
exports.ComputeEnvironment = ComputeEnvironment;
_a = JSII_RTTI_SYMBOL_1;
ComputeEnvironment[_a] = { fqn: "@aws-cdk/aws-batch-alpha.ComputeEnvironment", version: "2.37.1-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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