# DOE 
# Ryan Gosselin
# Version 2023

import numpy as np

##############################################

def doe_general(levels):
    """
    Créer un plan factoriel général 
    
    EXEMPLE
    
    n1 = 2 # niveaux du facteur 1
    n2 = 4 # niveaux du facteur 2
    n3 = 3 # niveaux du facteur 3
    plan = doe_general([n1,n2,n3])
               
    """
    n = len(levels)  # number of factors
    nb_lines = np.prod(levels)  # number of trial conditions
    H = np.zeros((nb_lines, n))
    
    level_repeat = 1
    range_repeat = np.prod(levels)
    for i in range(n):
        range_repeat //= levels[i]
        lvl = []
        for j in range(levels[i]):
            lvl += [j]*level_repeat
        rng = lvl*range_repeat
        level_repeat *= levels[i]
        H[:, i] = rng
     
    return H

##############################################

def doe_2k(n):
    """
    Créer un plan factoriel 2k 

    EXEMPLE
    k = 3 # nombre de facteurs
    plan = doe_2k(k)

    """
    return 2*doe_general([2]*n) - 1

##############################################

def generateurs(nb_factors, nb_generators):
    """
    Plans générateurs d'un plan 2k-p

    
    EXEMPLE
    k = 7 # nombre de facteurs
    p = 4 # nombre de générateurs
    generators = generateurs(k,p)

    """
    
    GEN = [
    [3,1,[[1,1]]],
    [4,1,[[1,1,1]]],
    [5,2,[[1,1,0],[1,0,1]]],
    [5,1,[[1,1,1,1]]],
    [6,3,[[1,1,0],[1,0,1],[0,1,1]]],
    [6,2,[[1,1,1,0],[0,1,1,1]]],
    [6,1,[[1,1,1,1,1]]],
    [7,4,[[1,1,0],[1,0,1],[0,1,1],[1,1,1]]],
    [7,3,[[1,1,1,0],[0,1,1,1],[1,0,1,1]]],
    [7,2,[[1,1,1,1,0],[1,1,1,0,1]]],
    [7,1,[[1,1,1,1,1,1]]],
    [8,4,[[1,1,1,0],[0,1,1,1],[1,0,1,1],[1,1,0,1]]],
    [8,3,[[1,1,1,0,0],[1,1,0,1,0],[0,1,1,1,1]]],
    [8,2,[[1,1,1,1,0,0],[1,1,0,0,1,1]]],
    [8,1,[[1,1,1,1,1,1,1]]],
    [9,5,[[1,1,1,0],[0,1,1,1],[1,0,1,1],[1,1,0,1],[1,1,1,1]]],
    [9,4,[[1,1,1,1,0],[1,1,1,0,1],[1,1,0,1,1],[1,0,1,1,1]]],
    [9,3,[[1,1,1,1,0,0],[1,0,1,0,1,1],[0,0,1,1,1,1]]],
    [9,2,[[1,1,1,1,1,1,0],[1,1,1,0,1,1,1]]],
    [9,1,[[1,1,1,1,1,1,1,1]]],
    [10,6,[[1,1,1,0],[0,1,1,1],[1,0,1,1],[1,1,0,1],[1,1,1,1],[1,1,0,0]]],
    [10,5,[[1,1,1,1,0],[1,1,1,0,1],[1,1,0,1,1],[1,0,1,1,1],[0,1,1,1,1]]],
    [10,4,[[1,1,1,1,0,0],[1,1,1,0,1,0],[1,0,0,1,1,1],[0,1,0,1,1,1]]],
    [10,3,[[1,1,1,0,0,0,1],[0,1,1,1,1,0,0],[1,0,1,1,0,1,0]]],
    [10,2,[[1,1,1,1,1,1,0,0],[1,1,1,1,0,0,1,1]]],
    [10,1,[[1,1,1,1,1,1,1,1,1]]],
    [11,7,[[1,1,1,0],[0,1,1,1],[1,0,1,1],[1,1,0,1],[1,1,1,1],[1,1,0,0],[1,0,1,0]]],
    [11,6,[[1,1,1,0,0],[0,1,1,1,0],[0,0,1,1,1],[1,0,1,1,0],[1,0,0,1,1],[0,1,0,1,1]]],
    [11,5,[[1,1,1,1,0,0],[1,1,1,0,1,0],[1,1,0,1,1,0],[1,0,1,1,1,1],[0,1,1,1,1,1]]],
    [11,4,[[1,1,1,0,0,0,1],[0,1,1,1,1,0,0],[1,0,1,1,0,1,0],[1,1,1,1,1,1,1]]],
    [11,3,[[1,1,1,1,1,1,0,0],[1,1,1,1,0,0,1,1],[1,1,0,1,0,1,1,1]]],
    [11,2,[[1,1,1,1,1,1,0,0,0],[1,1,1,0,0,0,1,1,1]]],
    [11,1,[[1,1,1,1,1,1,1,1,1,1]]],
    [12,8,[[1,1,1,0],[1,1,0,1],[1,0,1,1],[0,1,1,1],[1,1,1,1],[1,1,0,0],[1,0,1,0],[1,0,0,1]]],
    [12,7,[[1,1,1,0,0],[1,1,0,1,0],[1,0,1,1,0],[0,1,1,1,0],[1,1,0,0,1],[1,0,1,0,1],[1,0,0,1,1]]],
    [12,6,[[1,1,1,0,0,0],[1,1,0,1,0,0],[1,0,1,1,1,0],[1,0,1,1,0,1],[1,1,0,0,1,1],[0,1,1,1,1,1]]],
    [12,5,[[1,1,1,0,0,0,0],[1,0,0,1,1,1,0],[0,1,0,1,1,0,1],[0,0,1,1,0,1,1],[1,1,1,0,1,1,1]]],
    [12,4,[[1,1,1,1,1,0,0,0],[1,1,1,0,0,1,1,0],[1,1,0,1,0,0,1,1],[1,0,1,0,1,0,1,1]]],
    [12,3,[[1,1,1,1,1,1,0,0,0],[1,1,1,1,0,0,1,1,0],[1,1,0,0,1,1,1,1,1]]],
    [12,2,[[0,0,0,0,1,1,1,1,1,1],[1,1,1,1,1,1,0,0,0,0]]],
    [12,1,[[1,1,1,1,1,1,1,1,1,1,1]]],
    ]
    
    
    cc= True
    count = 0
    while cc:
        if((GEN[count][0]==nb_factors) and (GEN[count][1]==nb_generators)):
            cc = False
            generators = GEN[count][2]
        else:
            count = count + 1
            
    for j in range(len(generators)):
        gen = generators[j]
        output = ''
        for k in range(len(gen)):
            if gen[k] == 1: output = output+chr(65+k)
        print('Generateur',j+1,':',chr(65+len(gen)+j),'=',output)
    
    return generators
        
##############################################
   


def resolution(nb_factors, nb_generators):
    """
    Résolutiond'un plan 2k-p

    EXEMPLE
    res = resolution(k,p)

    """
    
    REZ = [
    [3,1,'III'],
    [4,1,'IV'],
    [5,2,'III'],
    [5,1,'V et +'],
    [6,3,'III'],
    [6,2,'IV'],
    [6,1,'V et +'],
    [7,4,'III'],
    [7,3,'IV'],
    [7,2,'IV'],
    [7,1,'V et +'],
    [8,4,'IV'],
    [8,3,'IV'],
    [8,2,'V et +'],
    [8,1,'V et +'],
    [9,5,'III'],
    [9,4,'IV'],
    [9,3,'IV'],
    [9,2,'V et +'],
    [9,1,'V et +'],
    [10,6,'III'],
    [10,5,'IV'],
    [10,4,'IV'],
    [10,3,'V et +'],
    [10,2,'V et +'],
    [10,1,'V et +'],
    [11,7,'III'],
    [11,6,'IV'],
    [11,5,'IV'],
    [11,4,'V et +'],
    [11,3,'V et +'],
    [11,2,'V et +'],
    [11,1,'V et +'],
    [12,8,'III'],
    [12,7,'IV'],
    [12,6,'IV'],
    [12,5,'IV'],
    [12,4,'V et +'],
    [12,3,'V et +'],
    [12,2,'V et +'],
    [12,1,'V et +'],
    ]
    
    cc= True
    count = 0
    while cc:
        if((REZ[count][0]==nb_factors) and (REZ[count][1]==nb_generators)):
            cc = False
            r = REZ[count][2]
        else:
            count = count + 1

    return r

##############################################
     
def doe_2kp(k,p):

    """
    Créer un plan factoriel 2k-p

    EXEMPLE
    k = 3 # nombre de facteurs
    plan = doe_2k(k)

    """

    # Full plan
    plan = doe_2k(k)

    generators = generateurs(k,p)

    # Partial plan
    plan_small = []
    
    for j in range(len(generators)): 
        gen = generators[j]
        length = len(gen)
        zeros = np.zeros((1,k-length))
        gen = np.array(gen)[np.newaxis]
        gen = np.concatenate((gen,zeros),axis=1)
        gen[0,length+j] = 1
    
        # Scan through lines and only keep ones where the product is 1
        for i in range(plan.shape[0]):    
            line = plan[i,:]*gen
            line = line[np.nonzero(line)]    # non-zero elements
            line = np.prod(line)
            if line == 1:
                plan_small.append(plan[i,:])
        
        plan = np.array(plan_small)
        plan_small = []
    
    
    plan = np.array(plan)    


    rez = resolution(k,p)
    print('\nRésolution du plan 2k-p:',rez)
    return plan    

##############################################

def table():
    
    """
    Créer un table des résolutions de plans factoriels 2k-p

    EXEMPLE
    table()

    """
    
    import matplotlib.pyplot as plt
    import matplotlib.cm as cm

    X = np.ones((6,11)) - np.eye(6,11)
    
    # Red squares
    X[0,1] = X[1,3:6] = X[2,7:] = 0.85
    
    # Yellow squares
    X[1,2] = X[2,4:7] = X[3,5:] = X[4,7:] = X[5,10] = 0.75
    
    # Green squares
    X[2,3] = X[3,4] = X[4,5:7] = X[5,6:10] = 0.45
    
    plt.figure(figsize=((12,6)))
    plt.imshow(X, cmap=cm.nipy_spectral, interpolation='nearest',alpha=0.15)
    
    #plt.colorbar()
    
    col_labels = ['2','3','4','5','6','7','8','9','10','11','12']
    plt.xticks(range(11), col_labels,fontsize=14)
    plt.xlabel('Facteurs',fontsize=16)
    
    row_labels = ['4','8','16','32','64','128']
    plt.yticks(range(6), row_labels,fontsize=14)
    plt.ylabel('Conditions expérimentales',fontsize=16)
    
    plt.title('Table de résolution de DOE partiels',fontsize=16)
    plt.tick_params(bottom = False)
    plt.tick_params(left = False)
    
    # Horizontal black lines
    plt.axhline(y=0.5,c='black',linewidth=0.5 )
    plt.axhline(y=1.5,c='black',linewidth=0.5 )
    plt.axhline(y=2.5,c='black',linewidth=0.5 )
    plt.axhline(y=3.5,c='black',linewidth=0.5 )
    plt.axhline(y=4.5,c='black',linewidth=0.5 )
    plt.axhline(y=5.5,c='black',linewidth=0.5 )
    
    # Vertical black lines
    plt.axvline(x=0.5,c='black',linewidth=0.5 )
    plt.axvline(x=1.5,c='black',linewidth=0.5 )
    plt.axvline(x=2.5,c='black',linewidth=0.5 )
    plt.axvline(x=3.5,c='black',linewidth=0.5 )
    plt.axvline(x=4.5,c='black',linewidth=0.5 )
    plt.axvline(x=5.5,c='black',linewidth=0.5 )
    plt.axvline(x=6.5,c='black',linewidth=0.5 )
    plt.axvline(x=7.5,c='black',linewidth=0.5 )
    plt.axvline(x=8.5,c='black',linewidth=0.5 )
    plt.axvline(x=9.5,c='black',linewidth=0.5 )
    
    # Box info: Diagonal
    plt.text(0, 0, '$2^2$',size=18,ha='center', va='center',color='black',weight='bold')
    plt.text(1, 1, '$2^3$',size=18,ha='center', va='center',color='black',weight='bold')
    plt.text(2, 2, '$2^4$',size=18,ha='center', va='center',color='black',weight='bold')
    plt.text(3, 3, '$2^5$',size=18,ha='center', va='center',color='black',weight='bold')
    plt.text(4, 4, '$2^6$',size=18,ha='center', va='center',color='black',weight='bold')
    plt.text(5, 5, '$2^7$',size=18,ha='center', va='center',color='black',weight='bold')
    
    # Box info: Line 4 tests
    plt.text(1, 0, '$2^{3-1}_{III}$',size=18,ha='center', va='center',color='red',weight='bold')
    
    # Box info: Line 8 tests
    plt.text(2, 1, '$2^{4-1}_{IV}$',size=18,ha='center', va='center',color='orange',weight='bold')
    plt.text(3, 1, '$2^{5-2}_{III}$',size=18,ha='center', va='center',color='red',weight='bold')
    plt.text(4, 1, '$2^{6-3}_{III}$',size=18,ha='center', va='center',color='red',weight='bold')
    plt.text(5, 1, '$2^{7-4}_{III}$',size=18,ha='center', va='center',color='red',weight='bold')
    
    # Box info: Line 16 tests
    plt.text(3, 2, '$2^{5-1}_{V+}$',size=18,ha='center', va='center',color='green',weight='bold')
    plt.text(4, 2, '$2^{6-2}_{IV}$',size=18,ha='center', va='center',color='orange',weight='bold')
    plt.text(5, 2, '$2^{7-3}_{IV}$',size=18,ha='center', va='center',color='orange',weight='bold')
    plt.text(6, 2, '$2^{8-4}_{IV}$',size=18,ha='center', va='center',color='orange',weight='bold')
    plt.text(7, 2, '$2^{9-5}_{III}$',size=18,ha='center', va='center',color='red',weight='bold')
    plt.text(8, 2, '$2^{10-6}_{III}$',size=18,ha='center', va='center',color='red',weight='bold')
    plt.text(9, 2, '$2^{11-7}_{III}$',size=18,ha='center', va='center',color='red',weight='bold')
    plt.text(10, 2, '$2^{12-8}_{III}$',size=18,ha='center', va='center',color='red',weight='bold')
    
    # Box info: Line 32 tests
    plt.text(4, 3, '$2^{6-1}_{V+}$',size=18,ha='center', va='center',color='green',weight='bold')
    plt.text(5, 3, '$2^{7-2}_{IV}$',size=18,ha='center', va='center',color='orange',weight='bold')
    plt.text(6, 3, '$2^{8-3}_{IV}$',size=18,ha='center', va='center',color='orange',weight='bold')
    plt.text(7, 3, '$2^{9-4}_{IV}$',size=18,ha='center', va='center',color='orange',weight='bold')
    plt.text(8, 3, '$2^{10-5}_{IV}$',size=18,ha='center', va='center',color='orange',weight='bold')
    plt.text(9, 3, '$2^{11-6}_{IV}$',size=18,ha='center', va='center',color='orange',weight='bold')
    plt.text(10, 3, '$2^{12-7}_{IV}$',size=18,ha='center', va='center',color='orange',weight='bold')
    
    # Box info: Line 64 tests
    plt.text(5, 4, '$2^{7-1}_{V+}$',size=18,ha='center', va='center',color='green',weight='bold')
    plt.text(6, 4, '$2^{8-2}_{V+}$',size=18,ha='center', va='center',color='green',weight='bold')
    plt.text(7, 4, '$2^{9-3}_{IV}$',size=18,ha='center', va='center',color='orange',weight='bold')
    plt.text(8, 4, '$2^{10-4}_{IV}$',size=18,ha='center', va='center',color='orange',weight='bold')
    plt.text(9, 4, '$2^{11-5}_{IV}$',size=18,ha='center', va='center',color='orange',weight='bold')
    plt.text(10, 4, '$2^{12-6}_{IV}$',size=18,ha='center', va='center',color='orange',weight='bold')
    
    # Box info: Line 128 tests
    plt.text(6, 5, '$2^{8-1}_{V+}$',size=18,ha='center', va='center',color='green',weight='bold')
    plt.text(7, 5, '$2^{9-2}_{V+}$',size=18,ha='center', va='center',color='green',weight='bold')
    plt.text(8, 5, '$2^{10-3}_{V+}$',size=18,ha='center', va='center',color='green',weight='bold')
    plt.text(9, 5, '$2^{11-4}_{V+}$',size=18,ha='center', va='center',color='green',weight='bold')
    plt.text(10, 5, '$2^{12-5}_{IV}$',size=18,ha='center', va='center',color='orange',weight='bold')
    
    plt.tight_layout()
    plt.savefig('table_resolution.png', dpi=500)
    plt.show()

    return









