/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.s3a;

import com.amazonaws.AmazonClientException;
import com.amazonaws.AmazonServiceException;
import com.amazonaws.auth.AWSCredentialsProvider;
import com.amazonaws.auth.EnvironmentVariableCredentialsProvider;
import com.amazonaws.auth.InstanceProfileCredentialsProvider;
import com.amazonaws.services.s3.model.AmazonS3Exception;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import java.io.Closeable;
import java.io.EOFException;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.net.URI;
import java.nio.file.AccessDeniedException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import org.apache.commons.lang.StringUtils;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.s3a.AWSClientIOException;
import org.apache.hadoop.fs.s3a.AWSCredentialProviderList;
import org.apache.hadoop.fs.s3a.AWSS3IOException;
import org.apache.hadoop.fs.s3a.AWSServiceIOException;
import org.apache.hadoop.fs.s3a.BasicAWSCredentialsProvider;
import org.apache.hadoop.fs.s3a.S3AFileStatus;
import org.apache.hadoop.fs.s3a.S3AFileSystem;
import org.apache.hadoop.fs.s3a.SharedInstanceProfileCredentialsProvider;
import org.apache.hadoop.fs.s3native.S3xLoginHelper;
import org.apache.hadoop.security.ProviderUtils;
import org.slf4j.Logger;

@InterfaceAudience.Private
@InterfaceStability.Evolving
public final class S3AUtils {
    private static final Logger LOG = S3AFileSystem.LOG;
    static final String CONSTRUCTOR_EXCEPTION = "constructor exception";
    static final String INSTANTIATION_EXCEPTION = "instantiation exception";
    static final String NOT_AWS_PROVIDER = "does not implement AWSCredentialsProvider";
    static final String ABSTRACT_PROVIDER = "is abstract and therefore cannot be created";
    static final String ENDPOINT_KEY = "Endpoint";
    static final String CREDENTIAL_PROVIDER_PATH = "hadoop.security.credential.provider.path";

    private S3AUtils() {
    }

    public static IOException translateException(String operation, Path path, AmazonClientException exception) {
        return S3AUtils.translateException(operation, path.toString(), exception);
    }

    public static IOException translateException(String operation, String path, AmazonClientException exception) {
        IOException ioe;
        String message = String.format("%s%s: %s", new Object[]{operation, path != null ? " on " + path : "", exception});
        if (!(exception instanceof AmazonServiceException)) {
            if (S3AUtils.containsInterruptedException(exception)) {
                return (IOException)new InterruptedIOException(message).initCause(exception);
            }
            return new AWSClientIOException(message, exception);
        }
        AmazonServiceException ase = (AmazonServiceException)((Object)exception);
        AmazonS3Exception s3Exception = ase instanceof AmazonS3Exception ? (AmazonS3Exception)((Object)ase) : null;
        int status = ase.getStatusCode();
        switch (status) {
            case 301: {
                if (s3Exception != null) {
                    if (s3Exception.getAdditionalDetails() != null && s3Exception.getAdditionalDetails().containsKey(ENDPOINT_KEY)) {
                        message = String.format("Received permanent redirect response to endpoint %s.  This likely indicates that the S3 endpoint configured in %s does not match the AWS region containing the bucket.", s3Exception.getAdditionalDetails().get(ENDPOINT_KEY), "fs.s3a.endpoint");
                    }
                    ioe = new AWSS3IOException(message, s3Exception);
                    break;
                }
                ioe = new AWSServiceIOException(message, ase);
                break;
            }
            case 401: 
            case 403: {
                ioe = new AccessDeniedException(path, null, message);
                ioe.initCause(ase);
                break;
            }
            case 404: 
            case 410: {
                ioe = new FileNotFoundException(message);
                ioe.initCause(ase);
                break;
            }
            case 416: {
                ioe = new EOFException(message);
                break;
            }
            default: {
                ioe = s3Exception != null ? new AWSS3IOException(message, s3Exception) : new AWSServiceIOException(message, ase);
            }
        }
        return ioe;
    }

    public static IOException extractException(String operation, String path, ExecutionException ee) {
        Throwable cause = ee.getCause();
        IOException ioe = cause instanceof AmazonClientException ? S3AUtils.translateException(operation, path, (AmazonClientException)cause) : (cause instanceof IOException ? (IOException)cause : new IOException(operation + " failed: " + cause, cause));
        return ioe;
    }

    static boolean containsInterruptedException(Throwable thrown) {
        if (thrown == null) {
            return false;
        }
        if (thrown instanceof InterruptedException || thrown instanceof InterruptedIOException) {
            return true;
        }
        return S3AUtils.containsInterruptedException(thrown.getCause());
    }

    public static String stringify(AmazonServiceException e) {
        StringBuilder builder = new StringBuilder(String.format("%s: %s error %d: %s; %s%s%n", e.getErrorType(), e.getServiceName(), e.getStatusCode(), e.getErrorCode(), e.getErrorMessage(), e.isRetryable() ? " (retryable)" : ""));
        String rawResponseContent = e.getRawResponseContent();
        if (rawResponseContent != null) {
            builder.append(rawResponseContent);
        }
        return builder.toString();
    }

    public static String stringify(AmazonS3Exception e) {
        StringBuilder builder = new StringBuilder(S3AUtils.stringify((AmazonServiceException)((Object)e)));
        Map details = e.getAdditionalDetails();
        if (details != null) {
            builder.append('\n');
            for (Map.Entry d : details.entrySet()) {
                builder.append((String)d.getKey()).append('=').append((String)d.getValue()).append('\n');
            }
        }
        return builder.toString();
    }

    public static S3AFileStatus createFileStatus(Path keyPath, S3ObjectSummary summary, long blockSize, String owner) {
        if (S3AUtils.objectRepresentsDirectory(summary.getKey(), summary.getSize())) {
            return new S3AFileStatus(true, keyPath, owner);
        }
        return new S3AFileStatus(summary.getSize(), S3AUtils.dateToLong(summary.getLastModified()), keyPath, blockSize, owner);
    }

    public static boolean objectRepresentsDirectory(String name, long size) {
        return !name.isEmpty() && name.charAt(name.length() - 1) == '/' && size == 0L;
    }

    public static long dateToLong(Date date) {
        if (date == null) {
            return 0L;
        }
        return date.getTime();
    }

    /*
     * WARNING - void declaration
     */
    public static AWSCredentialProviderList createAWSCredentialProviderSet(URI binding, Configuration conf, URI fsURI) throws IOException {
        Class[] awsClasses;
        AWSCredentialProviderList credentials = new AWSCredentialProviderList();
        try {
            awsClasses = conf.getClasses("fs.s3a.aws.credentials.provider", new Class[0]);
        }
        catch (RuntimeException e) {
            Throwable c = e.getCause() != null ? e.getCause() : e;
            throw new IOException("From option fs.s3a.aws.credentials.provider " + c, c);
        }
        if (awsClasses.length == 0) {
            S3xLoginHelper.Login creds = S3AUtils.getAWSAccessKeys(binding, conf);
            credentials.add(new BasicAWSCredentialsProvider(creds.getUser(), creds.getPassword()));
            credentials.add((AWSCredentialsProvider)new EnvironmentVariableCredentialsProvider());
            credentials.add((AWSCredentialsProvider)SharedInstanceProfileCredentialsProvider.getInstance());
        } else {
            for (Class clazz : awsClasses) {
                void var8_11;
                if (clazz == InstanceProfileCredentialsProvider.class) {
                    LOG.debug("Found {}, but will use {} instead.", (Object)clazz.getName(), (Object)SharedInstanceProfileCredentialsProvider.class.getName());
                    Class<SharedInstanceProfileCredentialsProvider> clazz2 = SharedInstanceProfileCredentialsProvider.class;
                }
                credentials.add(S3AUtils.createAWSCredentialProvider(conf, var8_11, fsURI));
            }
        }
        return credentials;
    }

    static AWSCredentialsProvider createAWSCredentialProvider(Configuration conf, Class<?> credClass, URI uri) throws IOException {
        String className;
        AWSCredentialsProvider credentials;
        block14: {
            AWSCredentialsProvider aWSCredentialsProvider;
            block15: {
                Constructor<?> cons;
                block12: {
                    AWSCredentialsProvider aWSCredentialsProvider2;
                    block13: {
                        block10: {
                            AWSCredentialsProvider aWSCredentialsProvider3;
                            block11: {
                                credentials = null;
                                className = credClass.getName();
                                if (!AWSCredentialsProvider.class.isAssignableFrom(credClass)) {
                                    throw new IOException("Class " + credClass + " " + NOT_AWS_PROVIDER);
                                }
                                if (Modifier.isAbstract(credClass.getModifiers())) {
                                    throw new IOException("Class " + credClass + " " + ABSTRACT_PROVIDER);
                                }
                                LOG.debug("Credential provider class is {}", (Object)className);
                                cons = S3AUtils.getConstructor(credClass, URI.class, Configuration.class);
                                if (cons == null) break block10;
                                aWSCredentialsProvider3 = credentials = (AWSCredentialsProvider)cons.newInstance(uri, conf);
                                if (credentials == null) break block11;
                                LOG.debug("Using {} for {}.", (Object)credentials, (Object)uri);
                            }
                            return aWSCredentialsProvider3;
                        }
                        Method factory = S3AUtils.getFactoryMethod(credClass, AWSCredentialsProvider.class, "getInstance");
                        if (factory == null) break block12;
                        aWSCredentialsProvider2 = credentials = (AWSCredentialsProvider)factory.invoke(null, new Object[0]);
                        if (credentials == null) break block13;
                        LOG.debug("Using {} for {}.", (Object)credentials, (Object)uri);
                    }
                    return aWSCredentialsProvider2;
                }
                cons = S3AUtils.getConstructor(credClass, new Class[0]);
                if (cons == null) break block14;
                aWSCredentialsProvider = credentials = (AWSCredentialsProvider)cons.newInstance(new Object[0]);
                if (credentials == null) break block15;
                LOG.debug("Using {} for {}.", (Object)credentials, (Object)uri);
            }
            return aWSCredentialsProvider;
        }
        try {
            try {
                throw new IOException(String.format("%s constructor exception.  A class specified in %s must provide a public constructor accepting URI and Configuration, or a public factory method named getInstance that accepts no arguments, or a public default constructor.", className, "fs.s3a.aws.credentials.provider"));
            }
            catch (IllegalArgumentException | ReflectiveOperationException e) {
                throw new IOException(className + " " + INSTANTIATION_EXCEPTION + ".", e);
            }
        }
        catch (Throwable throwable) {
            if (credentials != null) {
                LOG.debug("Using {} for {}.", credentials, (Object)uri);
            }
            throw throwable;
        }
    }

    public static S3xLoginHelper.Login getAWSAccessKeys(URI name, Configuration conf) throws IOException {
        S3xLoginHelper.Login login = S3xLoginHelper.extractLoginDetailsWithWarnings(name);
        Configuration c = ProviderUtils.excludeIncompatibleCredentialProviders((Configuration)conf, S3AFileSystem.class);
        String accessKey = S3AUtils.getPassword(c, "fs.s3a.access.key", login.getUser());
        String secretKey = S3AUtils.getPassword(c, "fs.s3a.secret.key", login.getPassword());
        return new S3xLoginHelper.Login(accessKey, secretKey);
    }

    static String getPassword(Configuration conf, String key, String val) throws IOException {
        return StringUtils.isEmpty((String)val) ? S3AUtils.lookupPassword(conf, key, "") : val;
    }

    static String lookupPassword(Configuration conf, String key, String defVal) throws IOException {
        try {
            char[] pass = conf.getPassword(key);
            return pass != null ? new String(pass).trim() : defVal;
        }
        catch (IOException ioe) {
            throw new IOException("Cannot find password option " + key, ioe);
        }
    }

    public static String stringify(S3ObjectSummary summary) {
        StringBuilder builder = new StringBuilder(summary.getKey().length() + 100);
        builder.append(summary.getKey()).append(' ');
        builder.append("size=").append(summary.getSize());
        return builder.toString();
    }

    static int intOption(Configuration conf, String key, int defVal, int min) {
        int v = conf.getInt(key, defVal);
        Preconditions.checkArgument((v >= min ? 1 : 0) != 0, (Object)String.format("Value of %s: %d is below the minimum value %d", key, v, min));
        LOG.debug("Value of {} is {}", (Object)key, (Object)v);
        return v;
    }

    static long longOption(Configuration conf, String key, long defVal, long min) {
        long v = conf.getLong(key, defVal);
        Preconditions.checkArgument((v >= min ? 1 : 0) != 0, (Object)String.format("Value of %s: %d is below the minimum value %d", key, v, min));
        LOG.debug("Value of {} is {}", (Object)key, (Object)v);
        return v;
    }

    static long longBytesOption(Configuration conf, String key, long defVal, long min) {
        long v = conf.getLongBytes(key, defVal);
        Preconditions.checkArgument((v >= min ? 1 : 0) != 0, (Object)String.format("Value of %s: %d is below the minimum value %d", key, v, min));
        LOG.debug("Value of {} is {}", (Object)key, (Object)v);
        return v;
    }

    public static long getMultipartSizeProperty(Configuration conf, String property, long defVal) {
        long partSize = conf.getLongBytes(property, defVal);
        if (partSize < 0x500000L) {
            LOG.warn("{} must be at least 5 MB; configured value is {}", (Object)property, (Object)partSize);
            partSize = 0x500000L;
        }
        return partSize;
    }

    public static int ensureOutputParameterInRange(String name, long size) {
        if (size > Integer.MAX_VALUE) {
            LOG.warn("s3a: {} capped to ~2.14GB (maximum allowed size with current output mechanism)", (Object)name);
            return Integer.MAX_VALUE;
        }
        return (int)size;
    }

    private static Constructor<?> getConstructor(Class<?> cl, Class<?> ... args) {
        try {
            Constructor<?> cons = cl.getDeclaredConstructor(args);
            return Modifier.isPublic(cons.getModifiers()) ? cons : null;
        }
        catch (NoSuchMethodException | SecurityException e) {
            return null;
        }
    }

    private static Method getFactoryMethod(Class<?> cl, Class<?> returnType, String methodName) {
        try {
            Method m = cl.getDeclaredMethod(methodName, new Class[0]);
            if (Modifier.isPublic(m.getModifiers()) && Modifier.isStatic(m.getModifiers()) && returnType.isAssignableFrom(m.getReturnType())) {
                return m;
            }
            return null;
        }
        catch (NoSuchMethodException | SecurityException e) {
            return null;
        }
    }

    public static Configuration propagateBucketOptions(Configuration source, String bucket) {
        Preconditions.checkArgument((boolean)StringUtils.isNotEmpty((String)bucket), (Object)"bucket");
        String bucketPrefix = "fs.s3a.bucket." + bucket + '.';
        LOG.debug("Propagating entries under {}", (Object)bucketPrefix);
        Configuration dest = new Configuration(source);
        for (Map.Entry entry : source) {
            String key = (String)entry.getKey();
            String value = (String)entry.getValue();
            if (!key.startsWith(bucketPrefix) || bucketPrefix.equals(key)) continue;
            String stripped = key.substring(bucketPrefix.length());
            if (stripped.startsWith("bucket.") || "impl".equals(stripped)) {
                LOG.debug("Ignoring bucket option {}", (Object)key);
                continue;
            }
            String generic = "fs.s3a." + stripped;
            LOG.debug("Updating {}", (Object)generic);
            dest.set(generic, value, key);
        }
        return dest;
    }

    static void patchSecurityCredentialProviders(Configuration conf) {
        Collection customCredentials = conf.getStringCollection("fs.s3a.security.credential.provider.path");
        Collection hadoopCredentials = conf.getStringCollection(CREDENTIAL_PROVIDER_PATH);
        if (!customCredentials.isEmpty()) {
            ArrayList all = Lists.newArrayList((Iterable)customCredentials);
            all.addAll(hadoopCredentials);
            String joined = StringUtils.join((Collection)all, (char)',');
            LOG.debug("Setting {} to {}", (Object)CREDENTIAL_PROVIDER_PATH, (Object)joined);
            conf.set(CREDENTIAL_PROVIDER_PATH, joined, "patch of fs.s3a.security.credential.provider.path");
        }
    }

    public static void closeAll(Logger log, Closeable ... closeables) {
        for (Closeable c : closeables) {
            if (c == null) continue;
            try {
                if (log != null) {
                    log.debug("Closing {}", (Object)c);
                }
                c.close();
            }
            catch (Exception e) {
                if (log == null || !log.isDebugEnabled()) continue;
                log.debug("Exception in closing {}", (Object)c, (Object)e);
            }
        }
    }
}

