# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['dataclass_mapper', 'dataclass_mapper.assignments']

package_data = \
{'': ['*']}

extras_require = \
{'pydantic': ['pydantic>=1.9.0,<2.0.0']}

setup_kwargs = {
    'name': 'dataclass-mapper',
    'version': '1.5.1',
    'description': 'Autogenerate mappings between dataclasses',
    'long_description': 'dataclass-mapper\n================\n\n|pypi| |support| |licence| |readthedocs| |build| |coverage|\n\n.. |pypi| image:: https://img.shields.io/pypi/v/dataclass-mapper.svg?style=flat-square\n    :target: https://pypi.org/project/dataclass-mapper/\n    :alt: pypi version\n\n.. |support| image:: https://img.shields.io/pypi/pyversions/dataclass-mapper.svg?style=flat-square\n    :target: https://pypi.org/project/dataclass-mapper/\n    :alt: supported Python version\n\n.. |build| image:: https://github.com/dataclass-mapper/dataclass-mapper/actions/workflows/test.yml/badge.svg\n    :target: https://github.com/dataclass-mapper/dataclass-mapper/actions\n    :alt: build status\n\n.. |coverage| image:: https://codecov.io/gh/dataclass-mapper/dataclass-mapper/branch/main/graphs/badge.svg?branch=main\n    :target: https://codecov.io/gh/dataclass-mapper/dataclass-mapper?branch=main\n    :alt: Code coverage\n\n.. |licence| image:: https://img.shields.io/pypi/l/dataclass-mapper.svg?style=flat-square\n    :target: https://pypi.org/project/dataclass-mapper/\n    :alt: licence\n\n.. |readthedocs| image:: https://img.shields.io/readthedocs/dataclass-mapper/latest.svg?style=flat-square&label=Read%20the%20Docs\n   :alt: Read the documentation at https://dataclass-mapper.readthedocs.io/en/latest/\n   :target: https://dataclass-mapper.readthedocs.io/en/latest/\n\nWriting mapper methods between two similar dataclasses is boring, need to be actively maintained and are error-prone.\nMuch better to let this library auto-generate them for you.\n\nThe focus of this library is:\n\n* **Concise and easy syntax:**\n  \n  * using it has to be a lot less overhead than writing the mappers by hand\n  * trivial mappings should not require code\n  * identical syntax for mapping between dataclasses and Pydantic models\n\n* **Safety:**\n\n  * using this library must give equal or more type safety than writing the mappers by hand\n  * the types between source and target classes must matches (including optional checks)\n  * all target fields must be actually initialized\n  * mappings cannot reference non-existing fields\n  * in case of an error a clean exception must be raised\n\n* **Performance:**\n\n  * mapping an object using this library must be the same speed than mapping using a custom mapper function\n  * the type checks shouldn\'t slow down the program\n  * because of the first two points, all type checks and the generation of the mapper functions happen during the definition of the classes\n\nMotivation\n----------\n\nA couple of example usecases, that show why this library might be useful.\n\n* Given an API with multiple, different interfaces (e.g. different API versions), that are all connected to a common algorithm with some common datamodel.\n  All the different API models needs to be mapped to the common datamodel, and afterwards mapped back to the API model.\n* Given an API that has a `POST` and a `GET` endpoint.\n  Both models (`POST` request body model and `GET` response body model) are almost the same, but there are some minor differences.\n  E.g. response model has an additional `id` parameter.\n  You need a way of mapping the request model to a response model.\n\nInstallation\n------------\n\n``dataclass-mapper`` can be installed using:\n\n.. code-block:: bash\n\n   pip install dataclass-mapper\n   # or for Pydantic support\n   pip install \'dataclass-mapper[pydantic]\'\n\nSmall Example\n-------------\n\nWe have the following target data structure, a class called ``Person``.\n\n.. code-block:: python\n\n   >>> from dataclasses import dataclass\n\n   >>> @dataclass\n   ... class Person:\n   ...     first_name: str\n   ...     second_name: str\n   ...     age: int\n\n\nWe want to have a mapper from the source data structure, a class called ``ContactInfo``.\nNotice that the attribute ``second_name`` of `Person` is called ``surname`` in `ContactInfo`.\nOther than that, all the attribute names are the same.\n\nInstead of writing a mapper `to_Person` by hand:\n\n.. code-block:: python\n\n   >>> @dataclass\n   ... class ContactInfo:\n   ...     first_name: str\n   ...     surname: str\n   ...     age: int\n   ...\n   ...     def to_Person(self) -> Person:\n   ...         return Person(\n   ...             first_name=self.first_name,\n   ...             second_name=self.surname,\n   ...             age=self.age,\n   ...         )\n\n   >>> contact = ContactInfo(first_name="Henry", surname="Kaye", age=42)\n   >>> contact.to_Person()\n   Person(first_name=\'Henry\', second_name=\'Kaye\', age=42)\n\nyou can let the mapper autogenerate with:\n\n.. code-block:: python\n\n   >>> from dataclass_mapper import map_to, mapper\n   >>>\n   >>> @mapper(Person, {"second_name": "surname"})\n   ... @dataclass\n   ... class ContactInfo:\n   ...     first_name: str\n   ...     surname: str\n   ...     age: int\n   >>>\n   >>> contact = ContactInfo(first_name="Henry", surname="Kaye", age=42)\n   >>> map_to(contact, Person)\n   Person(first_name=\'Henry\', second_name=\'Kaye\', age=42)\n\nThe ``dataclass-mapper`` library autogenerated some a mapper, that can be used with the ``map_to`` function.\nAll we had to specify was the name of the target class, and optionally specify which fields map to which other fields.\nNotice that we only had to specify that the ``second_name`` field has to be mapped to ``surname``,\nall other fields were mapped automatically because the field names didn\'t change.\n\nAnd the ``dataclass-mapper`` library will perform a lot of checks around this mapping.\nIt will check if the data types match, if some fields would be left uninitialized, etc.\n\nFeatures\n--------\n\nThe current version has support for:\n\n* Python\'s ``dataclass`` (with recursive models, custom initializers, optional types, ...): see `Supported features <https://dataclass-mapper.readthedocs.io/en/latest/features.html>`_ for the full list and examples\n* Mappings between Enum classes:  see `Enum mappings <https://dataclass-mapper.readthedocs.io/en/latest/enums.html>`_\n* Pydantic models:  see `Pydanitc support <https://dataclass-mapper.readthedocs.io/en/latest/pydantic.html>`_\n* Type/Value checks:  see `Type safety <https://dataclass-mapper.readthedocs.io/en/latest/type_safety.html>`_\n\nLicense\n-------\n\nThe project is released under the `MIT license <https://github.com/dataclass-mapper/dataclass-mapper/blob/main/LICENSE.md>`_.\n',
    'author': 'Jakob Kogler',
    'author_email': 'jakob.kogler@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://dataclass-mapper.readthedocs.io',
    'packages': packages,
    'package_data': package_data,
    'extras_require': extras_require,
    'python_requires': '>=3.9,<4.0',
}


setup(**setup_kwargs)
