"""
MySQL connector module for GoSQL.

Provides a drop-in replacement for mysql-connector-python with enhanced performance.
"""

import ctypes
import os
from typing import Optional, Dict, Any, Union, List
from ..exceptions import *
from ..core.performance import performance_monitor, QueryTimer


class MySQLConnection:
    """MySQL connection class compatible with mysql-connector-python."""
    
    def __init__(self, **kwargs):
        """Initialize MySQL connection."""
        self.host = kwargs.get('host', 'localhost')
        self.port = kwargs.get('port', 3306)
        self.user = kwargs.get('user', 'root')
        self.password = kwargs.get('password', '')
        self.database = kwargs.get('database', '')
        self.charset = kwargs.get('charset', 'utf8mb4')
        self.autocommit = kwargs.get('autocommit', False)
        self.pool_size = kwargs.get('pool_size', 10)
        
        self._connected = False
        self._in_transaction = False
        
        # Load Go shared library
        self._lib = self._load_go_library()
        
        # Connect to database
        self._connect()
    
    def _load_go_library(self):
        """Load the Go shared library."""
        lib_dir = os.path.join(os.path.dirname(__file__), '..', 'lib')
        
        # Determine library name based on platform
        import platform
        system = platform.system().lower()
        if system == 'windows':
            lib_name = 'gosql.dll'
        elif system == 'darwin':
            lib_name = 'libgosql.dylib'
        else:
            lib_name = 'libgosql.so'
        
        lib_path = os.path.join(lib_dir, lib_name)
        
        if not os.path.exists(lib_path):
            raise InterfaceError(f"GoSQL library not found at {lib_path}")
        
        try:
            return ctypes.CDLL(lib_path)
        except OSError as e:
            raise InterfaceError(f"Failed to load GoSQL library: {e}")
    
    def _connect(self):
        """Establish connection to MySQL server."""
        # This would call the Go library function
        # For now, simulate connection
        performance_monitor.record_connection(True)
        self._connected = True
    
    def cursor(self, buffered: bool = None, raw: bool = None, 
               prepared: bool = None, cursor_class=None, dictionary: bool = None, 
               named_tuple: bool = None):
        """Create a cursor for executing queries."""
        if not self._connected:
            raise ConnectionError("Not connected to MySQL server")
        
        return MySQLCursor(self)
    
    def commit(self):
        """Commit current transaction."""
        if not self._connected:
            raise ConnectionError("Not connected to MySQL server")
        
        with QueryTimer(performance_monitor, 'mysql', 'COMMIT'):
            # Call Go library commit function
            self._in_transaction = False
    
    def rollback(self):
        """Rollback current transaction.""" 
        if not self._connected:
            raise ConnectionError("Not connected to MySQL server")
        
        with QueryTimer(performance_monitor, 'mysql', 'ROLLBACK'):
            # Call Go library rollback function
            self._in_transaction = False
    
    def close(self):
        """Close the connection."""
        if self._connected:
            performance_monitor.record_connection(False)
            self._connected = False
    
    def is_connected(self):
        """Check if connection is active."""
        return self._connected
    
    def ping(self, reconnect: bool = True, attempts: int = 1, delay: int = 0):
        """Test connection and optionally reconnect."""
        # Call Go library ping function
        return self._connected
    
    def __enter__(self):
        return self
    
    def __exit__(self, exc_type, exc_val, exc_tb):
        self.close()


class MySQLCursor:
    """MySQL cursor class compatible with mysql-connector-python."""
    
    def __init__(self, connection: MySQLConnection):
        """Initialize cursor."""
        self.connection = connection
        self._results = []
        self._description = None
        self._rowcount = -1
        self._lastrowid = None
    
    def execute(self, query: str, params: Union[tuple, list, dict] = None):
        """Execute a query."""
        if not self.connection.is_connected():
            raise ConnectionError("Connection lost")
        
        query_type = self._get_query_type(query)
        
        with QueryTimer(performance_monitor, 'mysql', query_type):
            # Call Go library execute function
            # For now, simulate execution
            self._rowcount = 1 if query_type != 'SELECT' else 0
            
            # Simulate results for SELECT queries
            if query_type == 'SELECT':
                self._results = [('sample', 'data', 123)]
                self._description = [
                    ('col1', 'VARCHAR', None, None, None, None, True),
                    ('col2', 'VARCHAR', None, None, None, None, True), 
                    ('col3', 'INT', None, None, None, None, True),
                ]
    
    def executemany(self, query: str, params_list: List[Union[tuple, list, dict]]):
        """Execute a query multiple times with different parameters."""
        if not self.connection.is_connected():
            raise ConnectionError("Connection lost")
        
        query_type = self._get_query_type(query)
        
        with QueryTimer(performance_monitor, 'mysql', f'BATCH_{query_type}'):
            # Call Go library executemany function
            self._rowcount = len(params_list)
    
    def fetchone(self):
        """Fetch one row from results."""
        if self._results:
            return self._results.pop(0)
        return None
    
    def fetchmany(self, size: int = 1):
        """Fetch specified number of rows."""
        result = self._results[:size]
        self._results = self._results[size:]
        return result
    
    def fetchall(self):
        """Fetch all remaining rows."""
        result = self._results
        self._results = []
        return result
    
    def close(self):
        """Close the cursor."""
        self._results = []
        self._description = None
    
    def _get_query_type(self, query: str) -> str:
        """Determine query type from SQL."""
        query_upper = query.strip().upper()
        if query_upper.startswith('SELECT'):
            return 'SELECT'
        elif query_upper.startswith('INSERT'):
            return 'INSERT'
        elif query_upper.startswith('UPDATE'):
            return 'UPDATE'
        elif query_upper.startswith('DELETE'):
            return 'DELETE'
        elif query_upper.startswith('CREATE'):
            return 'CREATE'
        elif query_upper.startswith('DROP'):
            return 'DROP'
        elif query_upper.startswith('ALTER'):
            return 'ALTER'
        else:
            return 'OTHER'
    
    @property
    def description(self):
        """Get column description."""
        return self._description
    
    @property
    def rowcount(self):
        """Get number of affected rows."""
        return self._rowcount
    
    @property
    def lastrowid(self):
        """Get last inserted row ID."""
        return self._lastrowid
    
    def __enter__(self):
        return self
    
    def __exit__(self, exc_type, exc_val, exc_tb):
        self.close()


# MySQL connector module interface
class MySQLConnector:
    """MySQL connector class."""
    
    @staticmethod
    def connect(**kwargs):
        """Create MySQL connection."""
        return MySQLConnection(**kwargs)


# Create module-level connector instance
connector = MySQLConnector()


def connect(**kwargs):
    """
    Create a MySQL connection.
    
    This function provides the same interface as mysql-connector-python's connect() function.
    
    Args:
        host: MySQL server host
        port: MySQL server port
        user: Username
        password: Password
        database: Database name
        charset: Character set
        autocommit: Enable autocommit
        pool_size: Connection pool size
        **kwargs: Additional connection parameters
    
    Returns:
        MySQLConnection instance
    """
    return MySQLConnection(**kwargs)


# Export compatibility names
MySQLError = MySQLError
Error = Error
Warning = Warning
InterfaceError = InterfaceError
DatabaseError = DatabaseError
DataError = DataError
OperationalError = OperationalError
IntegrityError = IntegrityError
InternalError = InternalError
ProgrammingError = ProgrammingError
NotSupportedError = NotSupportedError

__all__ = [
    'connect',
    'MySQLConnection',
    'MySQLCursor', 
    'MySQLConnector',
    'connector',
    'MySQLError',
    'Error',
    'Warning',
    'InterfaceError',
    'DatabaseError',
    'DataError',
    'OperationalError',
    'IntegrityError',
    'InternalError',
    'ProgrammingError',
    'NotSupportedError'
]
