"""Defintion of the TinyFlux Index.

Class descriptions for Index and IndexResult.  An Index acts like a singleton,
and is initialized at creation time with the TinyFlux instance. It provides
efficient in-memory data structures and getters for TinyFlux operations. An
Index instance is not a part of the TinyFlux interface.

An IndexResult returns the indicies of revelant TinyFlux queries for further
handling, usually as an input to a storage retrieval.
"""
from datetime import datetime, timezone
import operator
from typing import Dict, Iterable, List, Optional, Set, Tuple, Union

from tinyflux.queries import SimpleQuery, CompoundQuery, Query
from .point import FieldSet, FieldValue, Point, TagSet
from .utils import find_eq, find_lt, find_le, find_gt, find_ge


class IndexResult:
    """Returns indicies of TinyFlux queries that are handled by an Index.

    IndexResults instances are generated by an Index.

    Arritributes:
        items: A set of indicies as ints.

    Usage:
        >>> IndexResult(items=set(), index_count=0)
    """

    _items: Set[int]
    _index_count: int

    def __init__(self, items: Set[int], index_count: int):
        """Init IndexResult.

        Args:
            items: Matching items from a query as indicies.
            index_count: Number of items in the index..
        """
        self._items = items
        self._index_count = index_count

    @property
    def items(self) -> Set[int]:
        """Return query result items."""
        return self._items

    def __invert__(self) -> "IndexResult":
        """Return the complement list.

        Returns:
            An IndexResult in which the items are the complement.

        Usage:
            >>> ~IndexResult()
        """
        return IndexResult(
            set(range(self._index_count)).difference(self._items),
            self._index_count,
        )

    def __and__(self, other: "IndexResult") -> "IndexResult":
        """Return the intersection of two IndexResults as one IndexResult.

        Args:
            other: Another IndexResult.

        Returns:
            An IndexResult in which the items are the intersection.

        Usage:
            >>> IndexResult() & IndexResult()
        """
        return IndexResult(
            self._items.intersection(other._items),
            self._index_count,
        )

    def __or__(self, other: "IndexResult") -> "IndexResult":
        """Return the union of two IndexResults as one IndexResult.

        Args:
            other: Another IndexResult.

        Returns:
            An IndexResult in which the items are the union.

        Usage:
            >>> IndexResult() | IndexResult()
        """
        return IndexResult(
            self._items.union(other._items),
            self._index_count,
        )


class Index:
    """An in-memory index for the storage instance.

    Provides efficient data structures and searches for TinyFlux data. An Index
    instance is created and its lifetime is handled by a TinyFlux instance.

    Attributes:
        empty: Index contains no items (used in testing).
        valid: Index represents current state of TinyFlux.
    """

    _num_items: int
    _tags: Dict[str, Dict[Union[None, str], List[int]]]
    _fields: Dict[str, List[Tuple[int, Optional[float]]]]
    _measurements: Dict[str, List[int]]
    _timestamps: List[float]
    _valid: bool
    _storage_pos_sorted_by_ts: List[int]

    def __init__(self, valid: bool = True) -> None:
        """Initialize an Index.

        Accepts a list of Points sorted by time.

        Args:
            valid: Index represents current state of TinyFlux.
        """
        self._num_items = 0
        self._tags = {}
        self._fields = {}
        self._measurements = {}
        self._timestamps = []
        self._valid = valid
        self._storage_pos_sorted_by_ts = []

    @property
    def empty(self) -> bool:
        """Return True if index is empty."""
        return (
            not self._num_items
            and not self._tags
            and not self._fields
            and not self._measurements
            and not self._timestamps
        )

    @property
    def valid(self) -> bool:
        """Return an empty index."""
        return self._valid

    @property
    def lateset_time(self) -> datetime:
        """Return the lastest time in the index."""
        return datetime.fromtimestamp(self._timestamps[-1]).astimezone(
            timezone.utc
        )

    def __len__(self) -> int:
        """Return number of items in the index."""
        return self._num_items

    def __repr__(self) -> str:
        """Return printable representation of Index."""
        args = [
            f"_tags={len(self._tags.keys())}",
            f"_measurements={len(self._measurements.keys())}",
            f"_timestamps={len(self._timestamps)}",
        ]

        return f'<{type(self).__name__} {", ".join(args)}>'

    def build(self, points: Iterable[Point]) -> None:
        """Build the index from scratch.

        Args:
            points: The collection of points to build the Index from.

        Usage:
            >>> i = Index().build([Point()])
        """
        self._reset()

        # A buffer for the new timestamps and their storage positions.
        timestamp_buffer: List[Tuple[float, int]] = []

        for idx, point in enumerate(points):
            self._num_items += 1
            self._insert_measurements(idx, point.measurement)
            self._insert_tags(idx, point.tags)
            self._insert_fields(idx, point.fields)

            if not point.time:  # pragma: no cover
                raise ValueError

            timestamp_buffer.append((point.time.timestamp(), idx))

        # Sort the timestamp buffer by timestamp.
        timestamp_buffer.sort(key=lambda x: x[0])

        # Split up the tuple into their own lists.
        # We have to do this because the bisect library does not work on
        # containers prior to py310.
        self._timestamps = [i[0] for i in timestamp_buffer]
        self._storage_pos_sorted_by_ts = [i[1] for i in timestamp_buffer]

        return

    def get_field_keys(self, measurement: Optional[str] = None) -> Set[str]:
        """Get field keys from this index, optionally filtered by measurement.

        Args:
            measurement: Optional measurement to filter by.

        Returns:
            Set of field keys.
        """
        # No measurement specified.
        if not measurement:
            return set(list(self._fields.keys()))

        # Measurement specified.
        rst: Set[str] = set({})

        # No measurement in the DB.
        if measurement not in self._measurements:
            return rst

        # If there is a measurement in the DB, we intersect.
        measurement_items = set(self._measurements[measurement])

        for field_key, items in self._fields.items():
            if measurement_items.intersection(set([i[0] for i in items])):
                rst.add(field_key)

        return rst

    def get_field_values(
        self, field_key: str, measurement: Optional[str] = None
    ) -> List[FieldValue]:
        """Get field values from this index, optionally filter by measurement.

        Args:
            field_key: Field key to get field values for.
            measurement: Optional measurement to filter by.

        Returns:
            List of field values.
        """
        # No measurement specified.
        if not measurement:
            if field_key in self._fields:
                field_values = [i[1] for i in self._fields[field_key]]
                return field_values
            else:
                return []

        # Measurement specified.
        rst: List[FieldValue] = []

        # No measurement in the DB.
        if measurement not in self._measurements:
            return rst

        # If there is a measurement in the DB, we intersect.
        measurement_items = set(self._measurements[measurement])

        for fk, items in self._fields.items():
            # Not a matching field key.
            if fk != field_key:
                continue

            # The field items are the first element of the tuple.
            field_items = [i[0] for i in items]

            # Check to see if there is overlap, if so, extend results.
            if measurement_items.intersection(set(field_items)):
                field_values = [i[1] for i in items]
                rst.extend(field_values)

        return rst

    def get_measurements(self) -> Set[str]:
        """Get the names of all measurements in the Index.

        Returns:
            Unique names of measurements as a set.

        Usage:
            >>> n = Index().build([Point()]).get_measurements()
        """
        return set(self._measurements.keys())

    def get_tag_keys(self, measurement: Optional[str] = None) -> Set[str]:
        """Get tag keys from this index, optionally filtered by measurement.

        Args:
            measurement: Optional measurement to filter by.

        Returns:
            Set of field keys.
        """
        # No measurement specified.
        if not measurement:
            return set(list(self._tags.keys()))

        # Measurement specified.
        rst: Set[str] = set({})

        # No measurement in the DB.
        if measurement not in self._measurements:
            return rst

        # If there is a measurement in the DB, we intersect.
        measurement_items = set(self._measurements[measurement])

        for tag_key, tag_values in self._tags.items():
            for items in tag_values.values():
                if measurement_items.intersection(set(items)):
                    rst.add(tag_key)

        return rst

    def get_tag_values(
        self, tag_keys: List[str] = [], measurement: Optional[str] = None
    ) -> Dict[str, Set[Optional[str]]]:
        """Get all tag values from the index.

        Args:
            tag_keys: Optional list of tag keys to get associated values for.
            measurement: Optional measurement to filter by.

        Returns:
            Mapping of tag_keys to associated tag values as a set.
        """
        # Return set.
        rst: Dict[str, Set[Optional[str]]] = {}

        # 1. No measurement, no tag keys. Return all.
        if not measurement and not tag_keys:
            for tag_key, tag_values in self._tags.items():
                rst[tag_key] = set({})

                for tag_value in tag_values:
                    rst[tag_key].add(tag_value)

            return rst

        # 2. Measurement, no tag keys.
        elif measurement and not tag_keys:
            if measurement in self._measurements:
                measurement_items = set(self._measurements[measurement])
            else:
                return rst

            for tag_key, tag_values in self._tags.items():
                for tag_value, items in self._tags[tag_key].items():
                    if measurement_items.intersection(set(items)):
                        if tag_key not in rst:
                            rst[tag_key] = set([tag_value])
                        else:
                            rst[tag_key].add(tag_value)

            return rst

        # 3. No measurement, tag keys.
        elif not measurement and tag_keys:
            rst = {i: set({}) for i in tag_keys}

            for tag_key, tag_values in self._tags.items():
                if tag_key in rst:
                    for tag_value in tag_values:
                        rst[tag_key].add(tag_value)

            return rst

        # 4. Measurement, tag keys.
        else:
            rst = {i: set({}) for i in tag_keys}

            if measurement in self._measurements:
                measurement_items = set(self._measurements[measurement])
            else:
                return rst

            for tag_key, tag_values in self._tags.items():
                for tag_value, items in self._tags[tag_key].items():
                    if tag_key in rst and measurement_items.intersection(
                        set(items)
                    ):
                        rst[tag_key].add(tag_value)

            return rst

    def get_timestamps(self, measurement: Optional[str] = None) -> List[float]:
        """Get timestamps from the index.

        Args:
            measurement: Optional measurement to filter by.

        Returns:
            List of timestamps.
        """
        # No measurement specified.
        if not measurement:
            zipped = [
                (i, j)
                for i, j in zip(
                    self._timestamps, self._storage_pos_sorted_by_ts
                )
            ]
            return [i[0] for i in sorted(zipped, key=lambda x: x[1])]

        # No measurement in the DB.
        if measurement not in self._measurements:
            return []

        # If there is a measurement in the DB, we intersect.
        zipped = [
            (i, j)
            for i, j in zip(self._timestamps, self._storage_pos_sorted_by_ts)
            if j in set(self._measurements[measurement])
        ]
        return [i[0] for i in sorted(zipped, key=lambda x: x[1])]

    def insert(self, points: List[Point] = []) -> None:
        """Update index with new points.

        Accepts new points to add to an Index.  Points are assumed to be passed
        to this method in non-descending time order.

        Args:
            points: List of tinyflux.Point instances.

        Usage:
            >>> Index().insert([Point()])
        """
        start_idx = len(self._timestamps)

        for idx, point in enumerate(points):
            new_idx = start_idx + idx

            self._num_items += 1

            if not point.time:  # pragma: no cover
                raise ValueError

            self._insert_time(point.time)
            self._insert_tags(new_idx, point.tags)
            self._insert_fields(new_idx, point.fields)
            self._insert_measurements(new_idx, point.measurement)

        return

    def invalidate(self) -> None:
        """Invalidate an Index.

        This method is invoked when the Index no longer represents the
        current state of TinyFlux and its Storage instance.

        Usage:
            >>> i = Index()
            >>> i.invalidate()
        """
        # Empty out the index.
        self._reset()

        # Set 'valid' to False.
        self._valid = False

        return

    def remove(self, r_items: Set[int]) -> None:
        """Remove items from the index."""
        self._remove_timestamps(r_items)
        self._remove_measurements(r_items)
        self._remove_tags(r_items)
        self._remove_fields(r_items)
        self._num_items -= len(r_items)

        return

    def search(self, query: Query) -> IndexResult:
        """Handle a TinyFlux query.

        Parses the query, generates a new IndexResult, and returns it.

        Args:
            query: A tinyflux.queries.Query.

        Returns:
            An IndexResult instance.

        Usage:
            >>> i = Index().build([Point()])
            >>> q = TimeQuery() < datetime.now(timezone.utc)
            >>> r = i.search(q)
        """
        return self._search_helper(query)

    def update(self, u_items: Dict[int, int]) -> None:
        """Update the index.

        Args:
            u_items: A mapping of old indices to update indices.
        """
        self._update_measurements(u_items)
        self._update_tags(u_items)
        self._update_fields(u_items)

        return

    def _insert_fields(self, idx: int, fields: FieldSet) -> None:
        """Index a field value.

        Args:
            idx: Index of the point.
            fields: Dict of Field key/vals.
        """
        for field_key, field_value in fields.items():
            if field_key not in self._fields:
                self._fields[field_key] = [(idx, field_value)]
            else:
                self._fields[field_key].append((idx, field_value))

        return

    def _insert_measurements(self, idx: int, measurement: str) -> None:
        """Index a measurement value.

        Args:
            idx: Index of the point.
            measurement: Name of measurement.
        """
        if measurement not in self._measurements:
            self._measurements[measurement] = [idx]
        else:
            self._measurements[measurement].append(idx)

        return

    def _insert_tags(self, idx: int, tags: TagSet) -> None:
        """Index a tag value.

        Args:
            idx: Index of the point.
            tags: Dict of Tag key/vals.
        """
        for tag_key, tag_value in tags.items():
            if tag_key not in self._tags:
                self._tags[tag_key] = {}

            if tag_value not in self._tags[tag_key]:
                self._tags[tag_key][tag_value] = [idx]
            else:
                self._tags[tag_key][tag_value].append(idx)

        return

    def _insert_time(self, time: datetime) -> None:
        """Index a time value.

        Args:
            time: Time to index.
        """
        # Add the new storage position.
        self._storage_pos_sorted_by_ts.append(len(self._timestamps))

        # Add the timestamp.
        self._timestamps.append(time.timestamp())

        return

    def _reset(self) -> None:
        """Reset the index.

        Empty the index out.
        """
        self._num_items = 0
        self._tags = {}
        self._fields = {}
        self._measurements = {}
        self._timestamps = []

        self._valid = True

        return

    def _search_fields(self, query: SimpleQuery) -> Set[int]:
        """Search the index for field matches.

        Args:
            query: A SimpleQuery instance.

        Returns:
            A list of candidates by index value.
        """
        rst_items: Set[int] = set([])

        for field_key, items in self._fields.items():
            # Transform the key. We're only concerned with whether or not a
            # storage item has a relevant field key before testing its value.
            # It if does, then we the values, and add these items to results.
            try:
                query._path_resolver({field_key: 0.0})
            except Exception:
                continue

            for idx, test_value in items:
                if query._test(test_value):
                    rst_items.add(idx)

        return rst_items

    def _search_helper(self, query: Optional[Query]) -> IndexResult:
        """Return an IndexResult from a parsed query.

        This method is recursive in order to handle a CompoundQuery.

        Args:
            query: A Query.

        Returns:
            An IndexResult instance.
        """
        if isinstance(query, CompoundQuery):
            if query.operator == operator.and_:
                rst1 = self._search_helper(query.query1)
                rst2 = self._search_helper(query.query2)
                return rst1 & rst2

            if query.operator == operator.or_:
                rst1 = self._search_helper(query.query1)
                rst2 = self._search_helper(query.query2)
                return rst1 | rst2

            if query.operator == operator.not_:
                rst = self._search_helper(query.query1)

                # For logical-NOT with a FieldQuery, we have to check every
                # single item in storage :(
                if (
                    isinstance(query.query1, SimpleQuery)
                    and query.query1._point_attr == "_fields"
                ):
                    rst._items = set(range(self._num_items))
                    return rst
                else:
                    return ~rst

        if isinstance(query, SimpleQuery):
            if query.point_attr == "_time":
                return IndexResult(
                    self._search_timestamps(query), self._num_items
                )

            if query.point_attr == "_measurement":
                return IndexResult(
                    self._search_measurement(query), self._num_items
                )

            if query.point_attr == "_tags":
                return IndexResult(self._search_tags(query), self._num_items)

            if query.point_attr == "_fields":
                return IndexResult(self._search_fields(query), self._num_items)

        raise TypeError("Query must be SimpleQuery or CompoundQuery.")

    def _search_measurement(self, query: SimpleQuery) -> Set[int]:
        """Search the index for measurement matches.

        Args:
            query: A SimpleQuery instance.

        Returns:
            A list of matches by index value.
        """
        rst_items: Set[int] = set([])

        for key, items in self._measurements.items():
            # Transform the key.
            test_value = query._path_resolver(key)

            # If it matches, update the list.
            if query._test(test_value):
                rst_items = rst_items.union(set(items))

        return rst_items

    def _search_tags(self, query: SimpleQuery) -> Set[int]:
        """Search the index for tag matches.

        Args:
            query: A SimpleQuery instance.

        Returns:
            A list of matches as index values.
        """
        rst_items: Set[int] = set([])

        for tag_key, tag_values in self._tags.items():
            for value, items in tag_values.items():
                # Transform the key.
                try:
                    test_value = query._path_resolver({tag_key: value})
                except Exception:
                    continue

                # If it matches, update the list.
                if query._test(test_value):
                    rst_items = rst_items.union(set(items))

        return rst_items

    def _search_timestamps(self, query: SimpleQuery) -> Set[int]:
        """Search for a timestamp.

        Search function for searching the timestamp index.

        Args:
            func: The operator or test of a query.
            rhs: The right-hand-side of the operator.

        Returns:
            A list of matches as a list of indices.
        """
        op = query._operator
        rhs = query._rhs

        # Exact timestamp match.
        if op == operator.eq:
            # Find the exact match, or return empty set if None.
            match = find_eq(self._timestamps, rhs.timestamp())
            if match is None:
                return set([])

            # Find the other timestamps with same value.
            results = set([self._storage_pos_sorted_by_ts[match]])
            match += 1

            while match < len(self._timestamps):
                if self._timestamps[match] != rhs.timestamp():
                    break

                results.add(self._storage_pos_sorted_by_ts[match])
                match += 1

            return results

        # Anything except exact timestamp match.
        elif op == operator.ne:
            # Find the exact match, or return full set if None.
            match = find_eq(self._timestamps, rhs.timestamp())
            if match is None:
                return set(self._storage_pos_sorted_by_ts)

            # Find the other timestamps with same value.
            results = set([self._storage_pos_sorted_by_ts[match]])
            match += 1

            while match < len(self._timestamps):
                if self._timestamps[match] != rhs.timestamp():
                    break

                results.add(self._storage_pos_sorted_by_ts[match])
                match += 1

            return set(self._storage_pos_sorted_by_ts).difference(results)

        # Everything less than rhs.
        elif op == operator.lt:
            match = find_lt(self._timestamps, rhs.timestamp())
            if match is None:
                return set([])

            return set(self._storage_pos_sorted_by_ts[: match + 1])

        # Every less than or equal to rhs.
        elif op == operator.le:
            match = find_le(self._timestamps, rhs.timestamp())
            if match is None:
                return set([])

            return set(self._storage_pos_sorted_by_ts[: match + 1])

        # Everything greater than rhs.
        elif op == operator.gt:
            match = find_gt(self._timestamps, rhs.timestamp())
            if match is None:
                return set([])

            return set(self._storage_pos_sorted_by_ts[match:])

        # Everything greater than or equal to rhs.
        elif op == operator.ge:
            match = find_ge(self._timestamps, rhs.timestamp())
            if match is None:
                return set([])

            return set(self._storage_pos_sorted_by_ts[match:])

        # All other operators.
        else:
            items = set([])
            for idx, timestamp in zip(
                self._storage_pos_sorted_by_ts, self._timestamps
            ):
                if query._test(
                    query._path_resolver(
                        datetime.fromtimestamp(timestamp).astimezone(
                            timezone.utc
                        )
                    )
                ):
                    items.add(idx)

            return items

    def _remove_fields(self, r_items: Set[int]) -> None:
        """Remove indices from fields index.

        Args:
            r_items: A set of indices to remove.
        """
        new_fields = {}

        for field_key, old_items in self._fields.items():
            new_items = [i for i in old_items if i[0] not in r_items]
            if new_items:
                new_fields[field_key] = new_items

        self._fields = new_fields

        return

    def _remove_measurements(self, r_items: Set[int]) -> None:
        """Remove indices from measurement index.

        Args:
            r_items: A set of indices to remove.
        """
        new_measurements = {}

        for m in self._measurements.keys():
            new_items = [i for i in self._measurements[m] if i not in r_items]
            if new_items:
                new_measurements[m] = new_items

        self._measurements = new_measurements

        return

    def _remove_tags(self, r_items: Set[int]) -> None:
        """Remove indices from tags index.

        Args:
            r_items: A set of indices to remove.
        """
        new_tags = {}

        for tag_key, tag_values in self._tags.items():
            for value, old_items in tag_values.items():
                new_items = [i for i in old_items if i not in r_items]

                if not new_items:
                    continue

                if tag_key not in new_tags:
                    new_tags[tag_key] = {value: new_items}
                else:
                    new_tags[tag_key][value] = new_items

        self._tags = new_tags

        return

    def _remove_timestamps(self, r_items: Set[int]) -> None:
        """Remove indices from timestamps index.

        Args:
            r_items: A set of indices to remove.
        """
        new_timestamps = []

        for i, ts in enumerate(self._timestamps):
            if i not in r_items:
                new_timestamps.append(ts)

        self._timestamps = new_timestamps

        return

    def _update_fields(self, u_items: Dict[int, int]) -> None:
        """Update fields index.

        Args:
            u_items: A mapping of old indices to new indices.
        """
        for field_key, old_items in self._fields.items():
            self._fields[field_key] = [
                (u_items[i[0]], i[1]) if i[0] in u_items else i
                for i in old_items
            ]

        return

    def _update_measurements(self, u_items: Dict[int, int]) -> None:
        """Update measurements index.

        Args:
            u_items: A mapping of old indices to new indices.
        """
        for measurement, old_items in self._measurements.items():
            self._measurements[measurement] = [
                u_items[i] if i in u_items else i for i in old_items
            ]

        return

    def _update_tags(self, u_items: Dict[int, int]) -> None:
        """Update tags index.

        Args:
            u_items: A mapping of old indices to new indices.
        """
        for tag_key, tag_values in self._tags.items():
            for value, old_items in tag_values.items():
                self._tags[tag_key][value] = [
                    u_items[i] if i in u_items else i for i in old_items
                ]
        return
