"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RESERVED_TUNNEL_INSIDE_CIDR = exports.VpnConnection = exports.VpnGateway = exports.VpnConnectionType = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const net = require("net");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const core_1 = require("@aws-cdk/core");
const ec2_generated_1 = require("./ec2.generated");
/**
 * The VPN connection type.
 */
var VpnConnectionType;
(function (VpnConnectionType) {
    /**
     * The IPsec 1 VPN connection type.
     */
    VpnConnectionType["IPSEC_1"] = "ipsec.1";
    /**
     * Dummy member
     * TODO: remove once https://github.com/aws/jsii/issues/231 is fixed
     */
    VpnConnectionType["DUMMY"] = "dummy";
})(VpnConnectionType = exports.VpnConnectionType || (exports.VpnConnectionType = {}));
/**
 * The VPN Gateway that shall be added to the VPC
 *
 * @resource AWS::EC2::VPNGateway
 */
class VpnGateway extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_VpnGatewayProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, VpnGateway);
            }
            throw error;
        }
        // This is 'Default' instead of 'Resource', because using 'Default' will generate
        // a logical ID for a VpnGateway which is exactly the same as the logical ID that used
        // to be created for the CfnVPNGateway (and 'Resource' would not do that).
        const vpnGW = new ec2_generated_1.CfnVPNGateway(this, 'Default', props);
        this.gatewayId = vpnGW.ref;
    }
}
exports.VpnGateway = VpnGateway;
_a = JSII_RTTI_SYMBOL_1;
VpnGateway[_a] = { fqn: "@aws-cdk/aws-ec2.VpnGateway", version: "1.163.1" };
/**
 * Define a VPN Connection
 *
 * @resource AWS::EC2::VPNConnection
 */
class VpnConnection extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_VpnConnectionProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, VpnConnection);
            }
            throw error;
        }
        if (!props.vpc.vpnGatewayId) {
            props.vpc.enableVpnGateway({
                type: 'ipsec.1',
                amazonSideAsn: props.asn,
            });
        }
        if (!core_1.Token.isUnresolved(props.ip) && !net.isIPv4(props.ip)) {
            throw new Error(`The \`ip\` ${props.ip} is not a valid IPv4 address.`);
        }
        const type = VpnConnectionType.IPSEC_1;
        const bgpAsn = props.asn || 65000;
        const customerGateway = new ec2_generated_1.CfnCustomerGateway(this, 'CustomerGateway', {
            bgpAsn,
            ipAddress: props.ip,
            type,
        });
        this.customerGatewayId = customerGateway.ref;
        this.customerGatewayAsn = bgpAsn;
        this.customerGatewayIp = props.ip;
        // Validate tunnel options
        if (props.tunnelOptions) {
            if (props.tunnelOptions.length > 2) {
                throw new Error('Cannot specify more than two `tunnelOptions`');
            }
            if (props.tunnelOptions.length === 2 && props.tunnelOptions[0].tunnelInsideCidr === props.tunnelOptions[1].tunnelInsideCidr) {
                throw new Error(`Same ${props.tunnelOptions[0].tunnelInsideCidr} \`tunnelInsideCidr\` cannot be used for both tunnels.`);
            }
            props.tunnelOptions.forEach((options, index) => {
                if (options.preSharedKey && options.preSharedKeySecret) {
                    throw new Error("Specify at most one of 'preSharedKey' and 'preSharedKeySecret'.");
                }
                if (options.preSharedKey && !core_1.Token.isUnresolved(options.preSharedKey) && !/^[a-zA-Z1-9._][a-zA-Z\d._]{7,63}$/.test(options.preSharedKey)) {
                    /* eslint-disable max-len */
                    throw new Error(`The \`preSharedKey\` ${options.preSharedKey} for tunnel ${index + 1} is invalid. Allowed characters are alphanumeric characters and ._. Must be between 8 and 64 characters in length and cannot start with zero (0).`);
                    /* eslint-enable max-len */
                }
                if (options.tunnelInsideCidr) {
                    if (exports.RESERVED_TUNNEL_INSIDE_CIDR.includes(options.tunnelInsideCidr)) {
                        throw new Error(`The \`tunnelInsideCidr\` ${options.tunnelInsideCidr} for tunnel ${index + 1} is a reserved inside CIDR.`);
                    }
                    if (!/^169\.254\.\d{1,3}\.\d{1,3}\/30$/.test(options.tunnelInsideCidr)) {
                        /* eslint-disable-next-line max-len */
                        throw new Error(`The \`tunnelInsideCidr\` ${options.tunnelInsideCidr} for tunnel ${index + 1} is not a size /30 CIDR block from the 169.254.0.0/16 range.`);
                    }
                }
            });
        }
        const vpnConnection = new ec2_generated_1.CfnVPNConnection(this, 'Resource', {
            type,
            customerGatewayId: customerGateway.ref,
            staticRoutesOnly: props.staticRoutes ? true : false,
            vpnGatewayId: props.vpc.vpnGatewayId,
            vpnTunnelOptionsSpecifications: props.tunnelOptions?.map(t => ({
                preSharedKey: t.preSharedKeySecret?.unsafeUnwrap() ?? t.preSharedKey,
                tunnelInsideCidr: t.tunnelInsideCidr,
            })),
        });
        this.vpnId = vpnConnection.ref;
        if (props.staticRoutes) {
            props.staticRoutes.forEach(route => {
                new ec2_generated_1.CfnVPNConnectionRoute(this, `Route${route.replace(/[^\d]/g, '')}`, {
                    destinationCidrBlock: route,
                    vpnConnectionId: this.vpnId,
                });
            });
        }
    }
    /**
     * Return the given named metric for all VPN connections in the account/region.
     */
    static metricAll(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/VPN',
            metricName,
            ...props,
        });
    }
    /**
     * Metric for the tunnel state of all VPN connections in the account/region.
     *
     * @default average over 5 minutes
     */
    static metricAllTunnelState(props) {
        return this.metricAll('TunnelState', { statistic: 'avg', ...props });
    }
    /**
     * Metric for the tunnel data in of all VPN connections in the account/region.
     *
     * @default sum over 5 minutes
     */
    static metricAllTunnelDataIn(props) {
        return this.metricAll('TunnelDataIn', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the tunnel data out of all VPN connections.
     *
     * @default sum over 5 minutes
     */
    static metricAllTunnelDataOut(props) {
        return this.metricAll('TunnelDataOut', { statistic: 'sum', ...props });
    }
}
exports.VpnConnection = VpnConnection;
_b = JSII_RTTI_SYMBOL_1;
VpnConnection[_b] = { fqn: "@aws-cdk/aws-ec2.VpnConnection", version: "1.163.1" };
exports.RESERVED_TUNNEL_INSIDE_CIDR = [
    '169.254.0.0/30',
    '169.254.1.0/30',
    '169.254.2.0/30',
    '169.254.3.0/30',
    '169.254.4.0/30',
    '169.254.5.0/30',
    '169.254.169.252/30',
];
//# sourceMappingURL=data:application/json;base64,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