#!/usr/bin/env python3
from collections import namedtuple
import pytest
from autopilot_tools.mavlink_params import (
    deserialize_param_value,
    serialize_param_value,
    integer_to_float,
)
from autopilot_tools.enums import AutopilotTypes

MavlinkMessage = namedtuple("MavlinkMessage", "param_id param_type param_value")

def test_deserialize_param_value_px4_int():
    test_case_int = MavlinkMessage("SYS_AUTOSTART", 6, 1.8216880036222622e-41)
    _, _, recv_param_value = deserialize_param_value(AutopilotTypes.PX4, test_case_int)
    assert recv_param_value == 13000


def test_deserialize_param_value_px4_float():
    test_case_float = MavlinkMessage("CAL_GYRO0_XOFF", 9, 1.0)
    _, _, recv_param_value = deserialize_param_value(AutopilotTypes.PX4, test_case_float)
    assert recv_param_value == 1.0


@pytest.mark.parametrize(
    "msg, expected",
    [
        (MavlinkMessage("SCR_ENABLE,", 2, 1.0), 1),
        (MavlinkMessage("some_param,", 4, 42.0), 42),
        (MavlinkMessage("COMPASS_DEV_ID,", 6, 76291.0), 76291),
        (MavlinkMessage("COMPASS_DIA_X,", 9, 0.999), 0.999),
    ],
)
def test_deserialize_param_value_ardupilot(msg, expected):
    _, _, recv_param_value = deserialize_param_value(AutopilotTypes.ArduPilot, msg)
    assert recv_param_value == expected

def test_serialize_param_value_int():
    value, param_type = serialize_param_value(13000)
    assert value == pytest.approx(1.8216880036222622e-41)
    assert param_type == 6

def test_serialize_param_value_float():
    value, param_type = serialize_param_value(1.0)
    assert value == 1.0
    assert param_type == 9

def test_integer_to_float():
    assert integer_to_float(13000) == pytest.approx(1.8216880036222622e-41)
    # Uncomment when behavior for negatives is defined
    # assert integer_to_float(-1) == pytest.approx(-1.8216880036222622e-41)
