#!/usr/bin/env python3
# This software is distributed under the terms of the MIT License.
# Copyright (c) 2023-2024 Dmitry Ponomarev.
# Author: Dmitry Ponomarev <ponomarevda96@gmail.com>

from datetime import date
from typing import Optional
from dataclasses import dataclass, fields

@dataclass
class FlightReport:  # pylint: disable=too-many-instance-attributes
    date:                   Optional[str]=None
    takeoff_time:           Optional[str]=None
    landing_time:           Optional[str]=None
    duration:               Optional[str]=None
    distance:               Optional[float]=None
    height:                 Optional[float]=None
    flight_mode:            Optional[str]=None
    vehicle:                Optional[str]=None
    battery_min:            Optional[float]=None
    battery_remaining:      Optional[float]=None
    initial_fuel_mass:      Optional[float]=None
    remaining_fuel_mass:    Optional[float]=None
    temperature:            Optional[float]=None
    wind_speed:             Optional[float]=None
    wind_direction:         Optional[str]=None
    atmospheric_pressure:   Optional[float]=None
    humidity:               Optional[float]=None
    log_url:                Optional[str]=None

    def __str__ (self):
        report_lines = ["Flight report:"]
        for field in fields(self):
            value = getattr(self, field.name)
            if value is not None:
                report_lines.append(f"- {field.name}: {value}")
        return "\n".join(report_lines)

@dataclass
class SoftwareReport:
    autopilot:              Optional[str]=None
    ice:                    Optional[str]=None

    def __str__ (self):
        report_lines = ["Software report:"]
        for field in fields(self):
            value = getattr(self, field.name)
            if value is not None:
                report_lines.append(f"- {field.name}: {value}")
        return "\n".join(report_lines)

@dataclass
class HardwareReport:
    fmu:                    Optional[str]=None
    battery:                Optional[str]=None
    ice:                    Optional[str]=None
    esc:                    Optional[str]=None

    def __str__ (self):
        report_lines = ["Hardware report:"]
        for field in fields(self):
            value = getattr(self, field.name)
            if value is not None:
                report_lines.append(f"- {field.name}: {value}")
        return "\n".join(report_lines)

class TelegramReportCreator:
    def __init__(self, data: dict, url: str, known_vehicles: dict):
        self.data = data
        self.url = url
        self.known_vehicles = known_vehicles

    def create(self) -> str:
        today = date.today().strftime("%Y.%m.%d")
        flight_report = FlightReport(
            date=today,
            vehicle={self.known_vehicles.get(self.data["log_info"].sys_uuid)},
            log_url=self.url
        )
        software_report = SoftwareReport(
            autopilot=self.data["log_info"].full_sw_version(),
            ice=self.data["ice_status"].sw_stats()
        )
        hardware_report = HardwareReport(
            fmu=self.data["log_info"].ver_hw,
            # battery=data["battery_statuses"][0].hw_stats(),
            ice=self.data["ice_status"].hw_stats(),
            esc=self.data["esc_status"].hw_stats(),
        )
        text = f"{flight_report}\n\n{hardware_report}\n\n{software_report}"
        return text
