#!/usr/bin/env python3
# This software is distributed under the terms of the MIT License.
# Copyright (c) 2023-2024 Dmitry Ponomarev.
# Author: Dmitry Ponomarev <ponomarevda96@gmail.com>

import os
import sys
import logging
from pathlib import Path

def configure_logger(verbose) -> None:
    def get_log_path() -> Path:
        """Return platform-appropriate log path."""
        if os.name == "nt":  # Windows
            base = Path(os.getenv("APPDATA", Path.home() / "AppData" / "Roaming"))
        else:  # Linux / macOS
            base = Path.home()
        log_dir = base / ".autopilot_tools" if os.name != "nt" else base / "autopilot_tools"
        log_dir /= "Logs"
        log_dir.mkdir(parents=True, exist_ok=True)
        return log_dir / "log.txt"

    log_path = get_log_path()
    formatter = logging.Formatter("[%(asctime)s] [%(levelname)s] %(message)s", "%H:%M:%S")

    console_handler = logging.StreamHandler(sys.stdout)
    console_handler.setLevel(logging.DEBUG if verbose else logging.INFO)
    console_handler.setFormatter(formatter)

    file_handler = logging.FileHandler(log_path, mode="w", encoding="utf-8")
    file_handler.setLevel(logging.DEBUG)  # Always capture all messages
    file_handler.setFormatter(formatter)

    root_logger = logging.getLogger()
    root_logger.setLevel(logging.DEBUG)  # Let handlers decide what to show
    root_logger.handlers.clear()
    root_logger.addHandler(console_handler)
    root_logger.addHandler(file_handler)

    root_logger.info(f"Logging to {log_path}")

    # Turn down noisy third-party libraries
    for name in ("telegram", "telegram.ext", "httpx", "apscheduler"):
        logging.getLogger(name).setLevel(logging.WARNING)
