#!/usr/bin/env python3
# This software is distributed under the terms of the MIT License.
# Copyright (c) 2023-2024 Dmitry Ponomarev.
# Author: Dmitry Ponomarev <ponomarevda96@gmail.com>

import re
import sys
import logging
from typing import Optional
from dataclasses import dataclass

from utils import run_cmd

logger = logging.getLogger(__name__)

try:
    import pyulog  # pylint: disable=unused-import
except ModuleNotFoundError:
    logger.critical("pip install pyulog>=1.2.2")
    sys.exit(1)

try:
    import pandas as pd
except ModuleNotFoundError:
    logger.critical("pip install pandas")
    sys.exit(1)

def parse_topic(log_path : str, topic_name : str, idx=0):
    log_path_splitted = log_path.split("/")
    log_file_name = log_path_splitted[-1][:-4]

    cmd = ['ulog2csv', '-m', topic_name, '-o', '.', log_path]
    run_cmd(cmd)

    csv_file = f"{log_file_name}_{topic_name}_{idx}.csv"
    try:
        data_frame = pd.read_csv(csv_file)
    except FileNotFoundError:
        return None

    return data_frame

@dataclass
class BaseTopic:
    def __init__(self, name : str) -> None:
        self.name = name
    def hw_stats(self) -> str:
        return "❌"
    def sw_stats(self) -> str:
        return "❌"

@dataclass
class BatteryStatus:
    remaining: list
    current_a: list
    id: int

    @property
    def remaining_max_pct(self):
        return int(max(self.remaining) * 100)

    @property
    def remaining_min_pct(self):
        return int(min(self.remaining) * 100)

    @property
    def max_current(self):
        return int(max(self.current_a))

    @staticmethod
    def parse_log(log_path : str, topic_name="battery_status", idx=0):
        df = parse_topic(log_path, topic_name, idx=idx)
        if df is None:
            return None

        battery_status = BatteryStatus(
            remaining=df['remaining'],
            current_a=df['current_a'],
            id=df['id']
        )

        return battery_status

    def hw_stats(self) -> str:
        stats = (
            "Battery: "
            f"Remaining: {self.remaining_max_pct} -> {self.remaining_min_pct} %, "
            f"Max current: {self.max_current} A"
        )
        return stats

    def __str__(self) -> str:
        return (f"BatteryStatus("
                f"remaining_max_pct={self.remaining_max_pct}, "
                f"remaining_min_pct={self.remaining_min_pct})"
        )

@dataclass
class IceStatus:
    sw_git_hash: hex
    max_adc_temperature_ice: int
    max_rpm: int

    @staticmethod
    def parse_log(log_path : str, topic_name="internal_combustion_engine_status"):
        df = parse_topic(log_path, topic_name, idx=0)
        if df is None:
            return BaseTopic(name="ICE")

        sw_git_hash = hex(df['flags'][0])

        if 'adc_temperature_ice' in df:
            adc_temperature_ice = df['adc_temperature_ice']
        elif 'external_temperature' in df:
            adc_temperature_ice = df['external_temperature']
        elif 'intake_manifold_temperature' in df:
            adc_temperature_ice = df['intake_manifold_temperature'] - 273.15
        else:
            adc_temperature_ice = 0
        max_adc_temperature_ice = int(max(adc_temperature_ice))

        rpm = df['engine_speed_rpm']
        max_rpm = max(rpm)

        ice_status = IceStatus(
            sw_git_hash=sw_git_hash,
            max_adc_temperature_ice=max_adc_temperature_ice,
            max_rpm=max_rpm
        )
        return ice_status

    def hw_stats(self) -> str:
        stats = (
            f"Max temperature: {self.max_adc_temperature_ice} Celcius, "
            f"Max RPM: {self.max_rpm}"
        )
        return stats

    def sw_stats(self) -> str:
        return f"{self.sw_git_hash}"

@dataclass
class EscStatus:
    esc_0_rpm: float
    esc_1_rpm: float
    esc_2_rpm: float
    esc_3_rpm: float

    @staticmethod
    def parse_log(log_path : str, topic_name="esc_status"):
        df = parse_topic(log_path, topic_name, idx=0)
        if df is None:
            return BaseTopic(name="ESC")

        if df is not None:
            esc_rpm = (
                df['esc[0].esc_rpm'],
                df['esc[1].esc_rpm'],
                df['esc[2].esc_rpm'],
                df['esc[3].esc_rpm']
            )
        else:
            esc_rpm = ([-1], [-1], [-1], [-1])

        esc_status = EscStatus(
            esc_0_rpm=max(esc_rpm[0]),
            esc_1_rpm=max(esc_rpm[1]),
            esc_2_rpm=max(esc_rpm[2]),
            esc_3_rpm=max(esc_rpm[3]),
        )

        return esc_status

    def hw_stats(self) -> str:
        stats = (
            "Max RPM: "
            f"{self.esc_0_rpm}, "
            f"{self.esc_1_rpm}, "
            f"{self.esc_2_rpm}, "
            f"{self.esc_3_rpm}\n"
        )
        return stats

@dataclass
class UlogInfo:
    sys_uuid: str
    ver_hw: str
    ver_sw: str
    ver_sw_branch: str
    vehicle: str
    ver_sw_release: str
    ver_vendor_sw_release: Optional[str]

    @staticmethod
    def search(ulog_output: str, pattern: str) -> str:
        res = re.search(pattern, ulog_output)
        return res.group(1) if res is not None else None

    @staticmethod
    def parse_log(log_path : str):
        cmd = ['ulog_info', log_path]
        res = run_cmd(cmd)
        ulog_output = res.stdout

        ver_sw_release = UlogInfo.search(ulog_output, r'ver_sw_release:\s+(\d+)')
        if ver_sw_release is not None:
            ver_sw_release = UlogInfo.semver(int(ver_sw_release))

        ver_vendor_sw_release = UlogInfo.search(ulog_output, r'ver_vendor_sw_release:\s+(\d+)')
        if ver_vendor_sw_release is not None:
            ver_vendor_sw_release = UlogInfo.semver(int(ver_vendor_sw_release))

        ulog_info = UlogInfo(
            sys_uuid=UlogInfo.search(ulog_output, r'sys_uuid:\s+(\w+)'),
            ver_hw=UlogInfo.search(ulog_output, r'ver_hw:\s+(\w+)'),
            ver_sw=UlogInfo.search(ulog_output, r'ver_sw:\s+(\w+)'),
            ver_sw_branch=UlogInfo.search(ulog_output, r'ver_sw_branch:\s+([\w\.]+)'),
            vehicle="",
            ver_sw_release=ver_sw_release,
            ver_vendor_sw_release=ver_vendor_sw_release
        )

        return ulog_info

    def full_sw_version(self) -> str:
        try:
            if self.ver_vendor_sw_release is not None:
                version = (f"v{self.ver_sw_release}-{self.ver_vendor_sw_release}"
                           f"_{self.ver_sw[:8]} ({self.ver_sw_branch})")
            else:
                version = (f"v{self.ver_sw_release}_{self.ver_sw[:8]} "
                           f"({self.ver_sw_branch})")
            return version
        except TypeError:
            return ""

    @staticmethod
    def semver(release):
        major = (release >> 24) & 0xFF
        minor = (release >> 16) & 0xFF
        patch = (release >> 8) & 0xFF
        type_val = release & 0xFF

        type_str = ""
        if type_val == 255:
            type_str = ""
        elif type_val >= 192:
            type_str = "-RC"
        elif type_val >= 128:
            type_str = "-beta"
        elif type_val >= 64:
            type_str = "-alpha"
        else:
            type_str = ""

        return f"{major}.{minor}.{patch}{type_str}"

class LogParser:
    def __init__(self, log_path: str):
        self.log_path = log_path

    def parse(self) -> dict:
        log_info = UlogInfo.parse_log(self.log_path)
        battery_statuses = []
        for idx in range(4):
            battery_status = BatteryStatus.parse_log(self.log_path, idx=idx)
            if battery_status is not None:
                battery_statuses.append(battery_status)
            else:
                break
        ice_status = IceStatus.parse_log(self.log_path)
        esc_status = EscStatus.parse_log(self.log_path)

        data = {
            "log_info" : log_info,
            "battery_statuses" : battery_statuses,
            "ice_status" : ice_status,
            "esc_status" : esc_status,
        }

        return data
