#!/usr/bin/env python3
# This software is distributed under the terms of the MIT License.
# Copyright (c) 2023-2024 Dmitry Ponomarev.
# Author: Dmitry Ponomarev <ponomarevda96@gmail.com>

import os
from configparser import ConfigParser
from pathlib import Path
from typing import Optional

def get_default_config_path() -> Path:
    if os.name == "nt":
        base = Path(os.getenv("APPDATA", Path.home() / "AppData" / "Roaming"))
        config_dir = base / "autopilot_tools"
    else:
        base = Path.home()
        config_dir = base / ".autopilot_tools"

    config_dir.mkdir(parents=True, exist_ok=True)
    return config_dir / "config.ini"

def load_config(config_path: Optional[str] = None):
    if config_path is None:
        config_path = get_default_config_path()

    if not config_path.exists():
        raise FileNotFoundError(f"Config file not found: {config_path}")

    parser = ConfigParser()
    parser.optionxform = str
    parser.read(config_path, encoding="utf-8")

    try:
        bot_token = parser["telegram"]["bot_token"].strip()
    except KeyError as e:
        raise KeyError(f"Missing [telegram] bot_token in {config_path}") from e

    known_vehicles = {}
    if "vehicles" in parser:
        for sys_uuid, name in parser["vehicles"].items():
            sys_uuid = sys_uuid.strip()
            name = name.strip()
            if sys_uuid:
                known_vehicles[sys_uuid] = name

    return bot_token, known_vehicles
