#!/usr/bin/env python3
# This software is distributed under the terms of the MIT License.
# Copyright (c) 2023-2024 Dmitry Ponomarev.
# Author: Dmitry Ponomarev <ponomarevda96@gmail.com>
"""
I'm flight log handler. I can parse .ulog files and provide a detailed information about it.
Please, send me a flight log, and you will see a result.
"""
import os
import argparse

from application import Application
from bot import TelegramBotAdapter
from config import load_config, get_default_config_path
from logger import configure_logger

if __name__ == '__main__':
    parser = argparse.ArgumentParser(description=__doc__)

    default_telegram_base_url = TelegramBotAdapter.DEFAULT_TELEGRAM_BASE_URL
    local_telegram_base_url = TelegramBotAdapter.LOCAL_TELEGRAM_BASE_URL
    parser.add_argument(
        "--base-url",
        type=str,
        default=default_telegram_base_url,
        help=("Telegram Base URL. "
              f"If not specified, default Telegram Base URL is used: {default_telegram_base_url}. "
              f"You can override with use local Bot API server: {local_telegram_base_url}"),
    )

    default_config_path = get_default_config_path()
    parser.add_argument(
        "--config-path",
        type=str,
        default=default_config_path,
        help=("It is recommended to keep secrets such as bot_token outside of the project. "
              f"If not specified, look for the default config location: {default_config_path}. "
              f"Optionally, you can override the path."),
    )

    log_base_dir = os.path.expanduser("~/.autopilot_tools/LogFiles")
    parser.add_argument(
        "--log-base-dir",
        type=str,
        default=log_base_dir,
        help=f"Log base directory by default is {log_base_dir}",
    )

    parser.add_argument(
        "--log-out",
        action="store_true",
        help="Send logOut request to log out from the default server."
    )

    parser.add_argument(
        "--flight-review", action="store_true", help="Upload to PX4 Flight Review"
    )

    parser.add_argument(
        "-v", "--verbose", action="store_true", help="Enable verbose logging"
    )

    args = parser.parse_args()

    configure_logger(args.verbose)
    bot_token, known_vehicles = load_config(config_path=args.config_path)

    flbot = TelegramBotAdapter(args, bot_token)
    app = Application(flbot, known_vehicles, flight_review=args.flight_review)

    flbot.add_command_handler('start', app.start_command_handler)
    flbot.add_message_handler(app.message_handler)
    flbot.add_file_handler(app.handle_file)

    flbot.start_polling()
