#!/usr/bin/env python3
# This software is distributed under the terms of the MIT License.
# Copyright (c) 2023-2024 Dmitry Ponomarev.
# Author: Dmitry Ponomarev <ponomarevda96@gmail.com>
"""
Thin adapter around python-telegram-bot that exposes a callback-based API.
"""
import sys
import logging
from pathlib import Path
from typing import Callable, Awaitable, Optional, Dict, Any

from reactions import Reactions

logger = logging.getLogger(__name__)
MessageCallback = Callable[[int, str, Optional[int]], Dict[str, Any]]
CommandCallback = Callable[[int], str]
FileCallback = Callable[[int, int, dict], Awaitable[None]]

# Guard external modules import with human-readable hints
try:
    import requests
except ModuleNotFoundError:
    logger.critical("pip install requests")
    sys.exit(1)
try:
    from telegram import Update
    from telegram.ext import (
        ApplicationBuilder,
        CommandHandler,
        MessageHandler,
        filters,
        ContextTypes)
except ModuleNotFoundError:
    logger.critical("pip install python-telegram-bot")
    sys.exit(1)

class TelegramBotAdapter:
    DOWNLOAD_DIR = "app/downloads"
    MAGIC_OFFSET = 14   # is it len(DOWNLOAD_DIR)-1 ?

    # These are rather the Internet connection speed constants than Telegram Bot
    # properties, but let's keep them here since they are related anyway
    DOWNLOAD_SPEED_MB_PER_SEC = 1.61
    PARSING_SPEED_MB_PER_SEC = 8.33

    # Fast setup, but very limited log file size
    DEFAULT_TELEGRAM_BASE_URL = "https://api.telegram.org/bot"
    # Local Bot API server is required, but larget files are allowed
    LOCAL_TELEGRAM_BASE_URL = "http://localhost:8081/bot"

    def __init__(self, args, bot_token: str):
        if args.log_out:
            url = f'https://api.telegram.org/bot{bot_token}/logOut'
            response = requests.post(url, timeout=5)
            if response.status_code == 200:
                logger.info("Successfully logged out from the default server.")
            else:
                logger.error("Failed to log out. Response: %s", response.text)
            sys.exit(0)

        self._log_base_dir: Path = Path(args.log_base_dir)
        self._app = ApplicationBuilder().token(bot_token).base_url(args.base_url).build()
        self._bot = self._app.bot

    def add_command_handler(self, command: str, callback: CommandCallback) -> None:
        """
        Wrap app callback into a telegram command handler.
        Keeps telegram handler stable and decoupled from application logic.
        Catches exceptions and always sends a text reply.
        """
        async def command_handler(update: Update, _: ContextTypes.DEFAULT_TYPE):
            chat_id = update.effective_chat.id
            try:
                text = callback(chat_id)
            except Exception as err:  # pylint: disable=broad-exception-caught
                logger.exception("Error in command handler '%s'", command)
                text = f"{type(err).__name__}: {err}"
            await self.send_message(chat_id, text)

        self._app.add_handler(CommandHandler(command, command_handler))

    def add_message_handler(self, callback: MessageCallback) -> None:
        """
        Wrap app callback into a telegram message handler.
        Keeps telegram handler stable and decoupled from application logic.
        Catches exceptions and always sends a text reply.
        """
        async def handle_message(update: Update, context: ContextTypes.DEFAULT_TYPE):
            msg = update.message
            chat_id = update.effective_chat.id
            replied_message_id = msg.reply_to_message.message_id if msg.reply_to_message else None

            try:
                result = callback(chat_id, msg.text, replied_message_id)
                text = result.get("text", "")
                reaction = result.get("reaction")
            except Exception as err:  # pylint: disable=broad-exception-caught
                logger.exception("Error in message handler")
                text = f"{type(err).__name__}: {err}"
                reaction = Reactions.ERROR.value

            is_reply = msg.reply_to_message and msg.reply_to_message.from_user.id == context.bot.id

            if reaction is not None:
                await self.set_message_reaction(chat_id, msg.message_id, reaction)

            if is_reply:
                message_id = msg.reply_to_message.message_id
                await context.bot.set_message_reaction(chat_id, message_id, Reactions.OK.value)

            if text:
                await self.send_message(chat_id, text, msg.message_id)

            logger.debug("Handle message: %s, %s", chat_id, msg.text)

        message_handler = MessageHandler(filters.TEXT & (~filters.COMMAND), handle_message)
        self._app.add_handler(message_handler)

    def add_file_handler(self, callback: FileCallback) -> None:
        """
        Wrap app callback into a telegram file handler.
        Keeps telegram handler stable and decoupled from application logic.
        Catches exceptions and sends a text reply on error.
        """
        async def handle_file(update: Update, _: ContextTypes.DEFAULT_TYPE):
            msg = update.message
            chat_id = update.effective_chat.id
            msg_id = msg.message_id
            file_name = msg.document.file_name
            file_id = msg.document.file_id
            file_size = round(msg.document.file_size / 1024 / 1024, 1)

            try:
                file_preperties = {"name": file_name, "id": file_id, "size": file_size}
                await callback(chat_id, msg_id, file_preperties)
            except Exception as err:  # pylint: disable=broad-exception-caught
                logger.exception("Error in file handler")
                text = f"Sorry, I got an Exception: {type(err).__name__}."
                await self.send_message(chat_id, text, msg_id)

        file_handler = MessageHandler(filters.Document.ALL, handle_file)
        self._app.add_handler(file_handler)

    async def send_message(self, chat_id: int, text: str, reply_id: Optional[int] = None) -> int:
        try:
            msg = await self._bot.send_message(chat_id=chat_id,
                                               text=text,
                                               reply_to_message_id=reply_id)
            return msg.id
        except Exception:  # pylint: disable=broad-exception-caught
            logger.exception("Unexpected error in send_message")

    async def edit_message_text(self, chat_id: int, message_id: int, text: str) -> None:
        try:
            await self._bot.edit_message_text(chat_id=chat_id,
                                              message_id=message_id,
                                              text=text)
        except Exception:  # pylint: disable=broad-exception-caught
            logger.exception("Unexpected error in edit_message_text")

    async def delete_message(self, chat_id: int, message_id: int) -> None:
        try:
            await self._bot.deleteMessage(chat_id=chat_id,
                                          message_id=message_id)
        except Exception:  # pylint: disable=broad-exception-caught
            logger.exception("Unexpected error in delete_message")

    async def set_message_reaction(self, chat_id: int, msg_id: int, reaction: Any) -> None:
        try:
            await self._bot.set_message_reaction(chat_id=chat_id,
                                                 message_id=msg_id,
                                                 reaction=reaction)
        except Exception:  # pylint: disable=broad-exception-caught
            logger.exception("Unexpected error in set_message_reaction")

    async def get_file(self, file_id: str, file_name: str) -> str:
        """
        Exceptions:
        - ValueError - bad arguments
        - BadRequest - bad file ID
        """
        logger.debug("bot.get_file(file_id=%s, file_name=%s)...", file_id, file_name)
        if not isinstance(file_id, str):
            raise ValueError(f"bot.get_file: file_id ({file_id}) is not str")
        if not isinstance(file_name, str):
            raise ValueError(f"bot.get_file: file_name ({file_name}) is not str")

        telegram_file = await self._bot.get_file(file_id, read_timeout=120)

        # Example 1: https://api.telegram.org/file/bot***:***/documents/file_0.ulg
        # Example 2: Add Local Telegram API Server example
        server_file_path = telegram_file.file_path

        # If Local Telegram API Server
        if self.DOWNLOAD_DIR in server_file_path:
            idx = server_file_path.find(self.DOWNLOAD_DIR) + self.MAGIC_OFFSET
            local_path = server_file_path[idx:]
            log_path = str(self._log_base_dir / local_path)

        # If Default Telegram API Server
        # Debug with both default and custom Telegram API URL
        else:
            log_path = str(self._log_base_dir / file_name)
            await telegram_file.download_to_drive(custom_path=log_path)

        logger.debug("bot.get_file(file_id=%s, file_name=%s) -> %s", file_id, file_name, log_path)
        return log_path

    def start_polling(self):
        self._app.run_polling()
