#!/usr/bin/env python3
# This software is distributed under the terms of the MIT License.
# Copyright (c) 2023-2024 Dmitry Ponomarev.
# Author: Dmitry Ponomarev <ponomarevda96@gmail.com>
"""
Docstring is under construction
"""
import time
import logging
from pathlib import Path
from typing import Optional

from reactions import Reactions
from reports import TelegramReportCreator
from log_parser import LogParser
from utils import run_cmd

logger = logging.getLogger(__name__)
MODULE_DIR = Path(__file__).resolve().parent

class StateLogger:
    def __init__(self, flbot, chat_id, reply_id) -> None:
        self.flbot = flbot
        self.chat_id = chat_id
        self.reply_id = reply_id
        self.msg_id = None

    async def update(self, text, reaction):
        if reaction in [Reactions.IN_PROCESS, Reactions.DONE]:
            logger.info(text)
        else:
            logger.error(text)

        if self.msg_id is None:
            self.msg_id = await self.flbot.send_message(self.chat_id, text, self.reply_id)
        else:
            await self.flbot.edit_message_text(self.chat_id, self.msg_id, text)

        await self.flbot.set_message_reaction(self.chat_id, self.reply_id, reaction.value)

def upload_to_px4_flight_review(log_path: str) -> str:
    script_path = MODULE_DIR / "log_upload.py"
    cmd = [script_path, log_path, '-q']
    res = run_cmd(cmd)
    url = res.stdout[5:]
    return url

class Application:
    def __init__(self, flbot, known_vehicles, flight_review : bool = False):
        self.flbot = flbot
        self.known_vehicles = known_vehicles
        self.flight_review = flight_review
        self.url = "-"
        self.data = {}

    def start_command_handler(self, _: int):
        return __doc__

    def message_handler(self, _chat_id: int, _msg: str, replied_id: Optional[int]) -> dict:
        reaction = Reactions.DONE.value if replied_id else Reactions.IDN.value
        return {
            "text": "I'm starving for .ulog files 🛩️",
            "reaction": reaction,
        }

    async def handle_file(self, chat_id: int, msg_id: int, file_properties: dict):
        logger.debug("handle_file(%s, %s, %s, %s, %s)",
                     chat_id,
                     msg_id,
                     file_properties["id"],
                     file_properties["name"],
                     file_properties["size"],
        )

        state_logger = StateLogger(flbot=self.flbot, chat_id=chat_id, reply_id=msg_id)

        step = "Step 1. Getting file"
        approx_req_time = int(file_properties["size"] / self.flbot.DOWNLOAD_SPEED_MB_PER_SEC)
        text = f"{step}. Estimated: {approx_req_time} sec..."
        await state_logger.update(text, Reactions.IN_PROCESS)
        try:
            start_time = time.time()
            log_path = await self.flbot.get_file(file_properties["id"],
                                                 file_properties["name"])
            logger.info("%s. Done in %s sec", step, time.time() - start_time)
        except (ValueError, Exception) as err:  # pylint: disable=broad-exception-caught
            text = f"{step}. Failed:{type(err).__name__}:{err}"
            await state_logger.update(text, Reactions.ERROR)
            return

        step = "Step 2. Uploading to PX4 Flight Review"
        text = f"{step}..."
        await state_logger.update(text, Reactions.IN_PROCESS)
        try:
            if not self.flight_review:
                self.url = "Skip for a while"
                logger.info("%s. Skipped.", step)
            else:
                self.url = upload_to_px4_flight_review(log_path)
                logger.debug("Done")
        except Exception as err:  # pylint: disable=broad-exception-caught
            self.url = "Failed"
            text = f"{step}. Failed:{type(err).__name__}:{err}"
            await state_logger.update(text, Reactions.ERROR)
            # This is an iptional step, so a failure is not critical

        step = "Step 3. Parsing log"
        approx_req_time = int(file_properties["size"] / self.flbot.PARSING_SPEED_MB_PER_SEC)
        text = f"{step}. Estimated: {approx_req_time} sec..."
        await state_logger.update(text, Reactions.IN_PROCESS)
        try:
            start_time = time.time()
            self.data = LogParser(log_path).parse()
            logger.info("%s. Done in %s sec", step, time.time() - start_time)
        except Exception as err:  # pylint: disable=broad-exception-caught
            text = f"{step}. Failed:{type(err).__name__}:{err}"
            await state_logger.update(text, Reactions.ERROR)
            return

        step = "Step 4. Constructing flight report"
        text = f"{step}..."
        await state_logger.update(text, Reactions.IN_PROCESS)
        try:
            start_time = time.time()
            text = TelegramReportCreator(self.data, self.url, self.known_vehicles).create()
            logger.info("%s. Done in %s sec", step, time.time() - start_time)
        except Exception as err:  # pylint: disable=broad-exception-caught
            text = f"{step}. Failed:{type(err).__name__}:{err}"
            await state_logger.update(text, Reactions.ERROR)
            return

        await state_logger.update(text, Reactions.DONE)
