"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.IntegTesting = void 0;
const codebuild = require("@aws-cdk/aws-codebuild");
const codePipeline = require("@aws-cdk/aws-codepipeline");
const codepipeline_actions = require("@aws-cdk/aws-codepipeline-actions");
const cdk = require("@aws-cdk/core");
const index_1 = require("./index");
class IntegTesting {
    constructor() {
        const app = new cdk.App();
        const env = {
            region: process.env.CDK_DEFAULT_REGION,
            account: process.env.CDK_DEFAULT_ACCOUNT,
        };
        const stack = new cdk.Stack(app, 'my-codepipeline-badge-notification-stack', { env });
        const pipeline = this.createCodePipeline(stack);
        const codePipelineStatus = new index_1.CodePipelineBadgeNotification(stack, 'CodePipelineStatus', {
            pipelineArn: pipeline.pipelineArn,
            gitHubTokenFromSecretsManager: {
                secretsManagerArn: 'arn:aws:secretsmanager:ap-northeast-1:482631629698:secret:codepipeline/lambda/github-token-YWWmII',
                secretKey: 'codepipeline/lambda/github-token',
            },
            notification: {
                stageName: 'production',
                slackWebHookUrl: 'slack url webhook',
            },
        });
        new cdk.CfnOutput(stack, 'BadgeUrl', {
            value: codePipelineStatus.badgeUrl,
        });
        new cdk.CfnOutput(stack, 'CodePipelineLink', {
            value: codePipelineStatus.codePipelineLink,
        });
        app.synth();
        this.stack = [stack];
    }
    createCodePipeline(stack) {
        const pipeline = new codePipeline.Pipeline(stack, 'TestPipeline', {
            pipelineName: 'integTestCodePipeline',
            crossAccountKeys: false,
        });
        const sourceOutput = new codePipeline.Artifact();
        pipeline.addStage({
            stageName: 'Source',
            actions: [
                new codepipeline_actions.CodeStarConnectionsSourceAction({
                    actionName: 'GitHub_Source',
                    owner: 'kimisme9386',
                    repo: 'cdk-codepipeline-status',
                    output: sourceOutput,
                    connectionArn: `arn:aws:codestar-connections:ap-northeast-1:${cdk.Aws.ACCOUNT_ID}:connection/XXxxxxxxxxxx`,
                    variablesNamespace: 'GitHubSourceVariables',
                    branch: 'dev',
                    codeBuildCloneOutput: true,
                }),
            ],
        });
        const project = this.createCodeBuildProjectWithinCodePipeline(stack, 'buildspec.yml');
        const afterBuildArtifact = new codePipeline.Artifact();
        pipeline.addStage({
            stageName: 'Build',
            actions: [
                new codepipeline_actions.CodeBuildAction({
                    actionName: 'AWS_CodeBuild',
                    input: sourceOutput,
                    project: project,
                    type: codepipeline_actions.CodeBuildActionType.BUILD,
                    outputs: [afterBuildArtifact],
                }),
            ],
        });
        return pipeline;
    }
    createCodeBuildProjectWithinCodePipeline(stack, buildSpecPath) {
        const project = new codebuild.PipelineProject(stack, 'CodeBuildWithinCodePipeline', {
            buildSpec: codebuild.BuildSpec.fromSourceFilename(buildSpecPath),
            environment: {
                buildImage: codebuild.LinuxBuildImage.STANDARD_4_0,
                computeType: codebuild.ComputeType.SMALL,
                privileged: true,
            },
            cache: codebuild.Cache.local(codebuild.LocalCacheMode.DOCKER_LAYER),
        });
        return project;
    }
}
exports.IntegTesting = IntegTesting;
new IntegTesting();
//# sourceMappingURL=data:application/json;base64,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