"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * A collection of connected states
 *
 * A StateGraph is used to keep track of all states that are connected (have
 * transitions between them). It does not include the substatemachines in
 * a Parallel's branches: those are their own StateGraphs, but the graphs
 * themselves have a hierarchical relationship as well.
 *
 * By assigning states to a definitive StateGraph, we verify that no state
 * machines are constructed. In particular:
 *
 * - Every state object can only ever be in 1 StateGraph, and not inadvertently
 *   be used in two graphs.
 * - Every stateId must be unique across all states in the entire state
 *   machine.
 *
 * All policy statements in all states in all substatemachines are bubbled so
 * that the top-level StateMachine instantiation can read them all and add
 * them to the IAM Role.
 *
 * You do not need to instantiate this class; it is used internally.
 */
class StateGraph {
    /**
     * @param startState state that gets executed when the state machine is launched
     * @param graphDescription description of the state machine
     */
    constructor(startState, graphDescription) {
        this.startState = startState;
        this.graphDescription = graphDescription;
        /**
         * The accumulated policy statements
         */
        this.policyStatements = new Array();
        /**
         * All states in this graph
         */
        this.allStates = new Set();
        /**
         * A mapping of stateId -> Graph for all states in this graph and subgraphs
         */
        this.allContainedStates = new Map();
        this.allStates.add(startState);
        startState.bindToGraph(this);
    }
    /**
     * Register a state as part of this graph
     *
     * Called by State.bindToGraph().
     */
    registerState(state) {
        this.registerContainedState(state.stateId, this);
        this.allStates.add(state);
    }
    /**
     * Register a Policy Statement used by states in this graph
     */
    registerPolicyStatement(statement) {
        if (this.superGraph) {
            this.superGraph.registerPolicyStatement(statement);
        }
        else {
            this.policyStatements.push(statement);
        }
    }
    /**
     * Register this graph as a child of the given graph
     *
     * Resource changes will be bubbled up to the given graph.
     */
    registerSuperGraph(graph) {
        if (this.superGraph === graph) {
            return;
        }
        if (this.superGraph) {
            throw new Error('Every StateGraph can only be registered into one other StateGraph');
        }
        this.superGraph = graph;
        this.pushContainedStatesUp(graph);
        this.pushPolicyStatementsUp(graph);
    }
    /**
     * Return the Amazon States Language JSON for this graph
     */
    toGraphJson() {
        const states = {};
        for (const state of this.allStates) {
            states[state.stateId] = state.toStateJson();
        }
        return {
            StartAt: this.startState.stateId,
            States: states,
            TimeoutSeconds: this.timeout && this.timeout.toSeconds(),
        };
    }
    /**
     * Return a string description of this graph
     */
    toString() {
        const someNodes = Array.from(this.allStates).slice(0, 3).map(x => x.stateId);
        if (this.allStates.size > 3) {
            someNodes.push('...');
        }
        return `${this.graphDescription} (${someNodes.join(', ')})`;
    }
    /**
     * Register a stateId and graph where it was registered
     */
    registerContainedState(stateId, graph) {
        if (this.superGraph) {
            this.superGraph.registerContainedState(stateId, graph);
        }
        else {
            const existingGraph = this.allContainedStates.get(stateId);
            if (existingGraph) {
                throw new Error(`State with name '${stateId}' occurs in both ${graph} and ${existingGraph}. All states must have unique names.`);
            }
            this.allContainedStates.set(stateId, graph);
        }
    }
    /**
     * Push all contained state info up to the given super graph
     */
    pushContainedStatesUp(superGraph) {
        for (const [stateId, graph] of this.allContainedStates) {
            superGraph.registerContainedState(stateId, graph);
        }
    }
    /**
     * Push all policy statements to into the given super graph
     */
    pushPolicyStatementsUp(superGraph) {
        for (const policyStatement of this.policyStatements) {
            superGraph.registerPolicyStatement(policyStatement);
        }
    }
}
exports.StateGraph = StateGraph;
//# sourceMappingURL=data:application/json;base64,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