import * as cdk from '@aws-cdk/core';
import { IConstruct, Construct } from 'constructs';
import { Condition } from '../condition';
import { StateGraph } from '../state-graph';
import { CatchProps, IChainable, INextable, RetryProps } from '../types';
/**
 * Properties shared by all states.
 *
 * @stability stable
 */
export interface StateProps {
    /**
     * A comment describing this state.
     *
     * @default No comment
     * @stability stable
     */
    readonly comment?: string;
    /**
     * JSONPath expression to select part of the state to be the input to this state.
     *
     * May also be the special value JsonPath.DISCARD, which will cause the effective
     * input to be the empty object {}.
     *
     * @default $
     * @stability stable
     */
    readonly inputPath?: string;
    /**
     * Parameters pass a collection of key-value pairs, either static values or JSONPath expressions that select from the input.
     *
     * @default No parameters
     * @see https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-parameters
     * @stability stable
     */
    readonly parameters?: {
        [name: string]: any;
    };
    /**
     * JSONPath expression to select part of the state to be the output to this state.
     *
     * May also be the special value JsonPath.DISCARD, which will cause the effective
     * output to be the empty object {}.
     *
     * @default $
     * @stability stable
     */
    readonly outputPath?: string;
    /**
     * JSONPath expression to indicate where to inject the state's output.
     *
     * May also be the special value JsonPath.DISCARD, which will cause the state's
     * input to become its output.
     *
     * @default $
     * @stability stable
     */
    readonly resultPath?: string;
}
/**
 * Base class for all other state classes.
 *
 * @stability stable
 */
export declare abstract class State extends cdk.Construct implements IChainable {
    /**
     * Add a prefix to the stateId of all States found in a construct tree.
     *
     * @stability stable
     */
    static prefixStates(root: IConstruct, prefix: string): void;
    /**
     * Find the set of states reachable through transitions from the given start state.
     *
     * This does not retrieve states from within sub-graphs, such as states within a Parallel state's branch.
     *
     * @stability stable
     */
    static findReachableStates(start: State, options?: FindStateOptions): State[];
    /**
     * Find the set of end states states reachable through transitions from the given start state.
     *
     * @stability stable
     */
    static findReachableEndStates(start: State, options?: FindStateOptions): State[];
    /**
     * Return only the states that allow chaining from an array of states.
     *
     * @stability stable
     */
    static filterNextables(states: State[]): INextable[];
    /**
     * First state of this Chainable.
     *
     * @stability stable
     */
    readonly startState: State;
    /**
     * Continuable states of this Chainable.
     *
     * @stability stable
     */
    abstract readonly endStates: INextable[];
    /**
     * @stability stable
     */
    protected readonly comment?: string;
    /**
     * @stability stable
     */
    protected readonly inputPath?: string;
    /**
     * @stability stable
     */
    protected readonly parameters?: object;
    /**
     * @stability stable
     */
    protected readonly outputPath?: string;
    /**
     * @stability stable
     */
    protected readonly resultPath?: string;
    /**
     * @stability stable
     */
    protected readonly branches: StateGraph[];
    /**
     * @stability stable
     */
    protected iteration?: StateGraph;
    /**
     * @stability stable
     */
    protected defaultChoice?: State;
    /**
     * @internal
     */
    protected _next?: State;
    private readonly retries;
    private readonly catches;
    private readonly choices;
    private readonly prefixes;
    /**
     * The graph that this state is part of.
     *
     * Used for guaranteeing consistency between graphs and graph components.
     */
    private containingGraph?;
    /**
     * States with references to this state.
     *
     * Used for finding complete connected graph that a state is part of.
     */
    private readonly incomingStates;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: StateProps);
    /**
     * Descriptive identifier for this chainable.
     *
     * @stability stable
     */
    get id(): string;
    /**
     * Tokenized string that evaluates to the state's ID.
     *
     * @stability stable
     */
    get stateId(): string;
    /**
     * Add a prefix to the stateId of this state.
     *
     * @stability stable
     */
    addPrefix(x: string): void;
    /**
     * Register this state as part of the given graph.
     *
     * Don't call this. It will be called automatically when you work
     * with states normally.
     *
     * @stability stable
     */
    bindToGraph(graph: StateGraph): void;
    /**
     * Render the state as JSON.
     *
     * @stability stable
     */
    abstract toStateJson(): object;
    /**
     * Add a retrier to the retry list of this state
     * @internal
     */
    protected _addRetry(props?: RetryProps): void;
    /**
     * Add an error handler to the catch list of this state
     * @internal
     */
    protected _addCatch(handler: State, props?: CatchProps): void;
    /**
     * Make the indicated state the default transition of this state.
     *
     * @stability stable
     */
    protected makeNext(next: State): void;
    /**
     * Add a choice branch to this state.
     *
     * @stability stable
     */
    protected addChoice(condition: Condition, next: State): void;
    /**
     * Add a paralle branch to this state.
     *
     * @stability stable
     */
    protected addBranch(branch: StateGraph): void;
    /**
     * Add a map iterator to this state.
     *
     * @stability stable
     */
    protected addIterator(iteration: StateGraph): void;
    /**
     * Make the indicated state the default choice transition of this state.
     *
     * @stability stable
     */
    protected makeDefault(def: State): void;
    /**
     * Render the default next state in ASL JSON format.
     *
     * @stability stable
     */
    protected renderNextEnd(): any;
    /**
     * Render the choices in ASL JSON format.
     *
     * @stability stable
     */
    protected renderChoices(): any;
    /**
     * Render InputPath/Parameters/OutputPath in ASL JSON format.
     *
     * @stability stable
     */
    protected renderInputOutput(): any;
    /**
     * Render parallel branches in ASL JSON format.
     *
     * @stability stable
     */
    protected renderBranches(): any;
    /**
     * Render map iterator in ASL JSON format.
     *
     * @stability stable
     */
    protected renderIterator(): any;
    /**
     * Render error recovery options in ASL JSON format.
     *
     * @stability stable
     */
    protected renderRetryCatch(): any;
    /**
     * Called whenever this state is bound to a graph.
     *
     * Can be overridden by subclasses.
     *
     * @stability stable
     */
    protected whenBoundToGraph(graph: StateGraph): void;
    /**
     * Add a state to the incoming list
     */
    private addIncoming;
    /**
     * Return all states this state can transition to
     */
    private outgoingTransitions;
}
/**
 * Options for finding reachable states.
 *
 * @stability stable
 */
export interface FindStateOptions {
    /**
     * Whether or not to follow error-handling transitions.
     *
     * @default false
     * @stability stable
     */
    readonly includeErrorHandlers?: boolean;
}
/**
 * Render a list or return undefined for an empty list
 */
export declare function renderList<T>(xs: T[], mapFn: (x: T) => any, sortFn?: (a: T, b: T) => number): any;
/**
 * Render JSON path, respecting the special value DISCARD
 */
export declare function renderJsonPath(jsonPath?: string): undefined | null | string;
