# AUTOGENERATED! DO NOT EDIT! File to edit: nbs/dev-02-eda.ipynb (unless otherwise specified).

__all__ = ['load_EI_df', 'load_DE_df', 'clean_EI_df_for_plot', 'rgb_2_plt_tuple',
           'convert_fuel_colour_dict_to_plt_tuple', 'hide_spines', 'stacked_fuel_plot', 'clean_EC_df_for_plot']

# Cell
import pandas as pd

import matplotlib.pyplot as plt
import matplotlib.transforms as mtf

# Cell
def load_EI_df(EI_fp):
    """Loads the electric insights data and returns a DataFrame"""
    df = pd.read_csv(EI_fp)

    df['local_datetime'] = pd.to_datetime(df['local_datetime'], utc=True)
    df = df.set_index('local_datetime')

    return df

# Cell
def load_DE_df(EC_fp, ENTSOE_fp):
    """Loads the energy-charts and ENTSOE data and returns a DataFrame"""
    # Energy-Charts
    df_DE = pd.read_csv(EC_fp)

    df_DE['local_datetime'] = pd.to_datetime(df_DE['local_datetime'], utc=True)
    df_DE = df_DE.set_index('local_datetime')

    # ENTSOE
    df_ENTSOE = pd.read_csv(ENTSOE_fp)

    df_ENTSOE['local_datetime'] = pd.to_datetime(df_ENTSOE['local_datetime'], utc=True)
    df_ENTSOE = df_ENTSOE.set_index('local_datetime')

    # Combining data
    df_DE['demand'] = df_DE.sum(axis=1)

    s_price = df_ENTSOE['DE_price']
    df_DE['price'] = s_price[~s_price.index.duplicated(keep='first')]

    return df_DE

# Cell
def clean_EI_df_for_plot(df, freq='7D'):
    """Cleans the electric insights dataframe for plotting"""
    fuel_order = ['Imports & Storage', 'nuclear', 'biomass', 'gas', 'coal', 'hydro', 'wind', 'solar']
    interconnectors = ['french', 'irish', 'dutch', 'belgian', 'ireland', 'northern_ireland']

    df = (df
          .copy()
          .assign(imports_storage=df[interconnectors+['pumped_storage']].sum(axis=1))
          .rename(columns={'imports_storage':'Imports & Storage'})
          .drop(columns=interconnectors+['demand', 'pumped_storage'])
          [fuel_order]
         )

    df_resampled = df.astype('float').resample(freq).mean()
    return df_resampled

# Cell
def rgb_2_plt_tuple(r, g, b):
    """converts a standard rgb set from a 0-255 range to 0-1"""
    plt_tuple = tuple([x/255 for x in (r, g, b)])
    return plt_tuple

def convert_fuel_colour_dict_to_plt_tuple(fuel_colour_dict_rgb):
    """Converts a dictionary of fuel colours to matplotlib colour values"""
    fuel_colour_dict_plt = fuel_colour_dict_rgb.copy()

    fuel_colour_dict_plt = {
        fuel: rgb_2_plt_tuple(*rgb_tuple)
        for fuel, rgb_tuple
        in fuel_colour_dict_plt.items()
    }

    return fuel_colour_dict_plt

# Cell
def hide_spines(ax, positions=["top", "right"]):
    """
    Pass a matplotlib axis and list of positions with spines to be removed

    Parameters:
        ax:          Matplotlib axis object
        positions:   Python list e.g. ['top', 'bottom']
    """
    assert isinstance(positions, list), "Position must be passed as a list "

    for position in positions:
        ax.spines[position].set_visible(False)

def stacked_fuel_plot(
    df,
    ax=None,
    save_path=None,
    dpi=150,
    fuel_colour_dict = {
    'Imports & Storage' : rgb_2_plt_tuple(121,68,149),
    'nuclear' : rgb_2_plt_tuple(77,157,87),
    'biomass' : rgb_2_plt_tuple(168,125,81),
    'gas' : rgb_2_plt_tuple(254,156,66),
    'coal' : rgb_2_plt_tuple(122,122,122),
    'hydro' : rgb_2_plt_tuple(50,120,196),
    'wind' : rgb_2_plt_tuple(72,194,227),
    'solar' : rgb_2_plt_tuple(255,219,65),
}
):
    """Plots the electric insights fuel data as a stacked area graph"""
    df = df[fuel_colour_dict.keys()]

    if ax == None:
        fig = plt.figure(figsize=(10, 5), dpi=dpi)
        ax = plt.subplot()

    ax.stackplot(df.index.values, df.values.T, labels=df.columns.str.capitalize(), linewidth=0.25, edgecolor='white', colors=list(fuel_colour_dict.values()))

    plt.rcParams['axes.ymargin'] = 0
    ax.spines['bottom'].set_position('zero')
    hide_spines(ax)

    ax.set_xlim(df.index.min(), df.index.max())
    ax.legend(ncol=4, bbox_to_anchor=(0.85, 1.15), frameon=False)
    ax.set_ylabel('Generation (GW)')

    if save_path:
        fig.savefig(save_path)

    return ax

# Cell
def clean_EC_df_for_plot(
    df_EC,
    freq='7D',
    fuel_order=['Imports & Storage', 'nuclear', 'biomass',
                'gas', 'coal', 'hydro', 'wind', 'solar']
):
    """Cleans the electric insights dataframe for plotting"""

    df_EC_clean = (pd
                   .DataFrame(index=df_EC.index)
                   .assign(nuclear=df_EC['Uranium'])
                   .assign(biomass=df_EC['Biomass'])
                   .assign(gas=df_EC['Gas'])
                   .assign(coal=df_EC['Brown Coal']+df_EC['Hard Coal'])
                   .assign(hydro=df_EC['Hydro Power'])
                   .assign(wind=df_EC['Wind'])
                   .assign(solar=df_EC['Solar'])
                  )

    df_EC_clean['Imports & Storage'] = df_EC['Pumped Storage'] + df_EC['Seasonal Storage'] + df_EC['Net Balance']
    df_EC_clean = df_EC_clean[fuel_order].interpolate()

    df_EC_resampled = df_EC_clean.astype('float').resample(freq).mean()

    return df_EC_resampled