"use strict";
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
//
// Utilities for fetching the JWKS URI, to get the public keys with which to verify JWTs
Object.defineProperty(exports, "__esModule", { value: true });
exports.fetchJson = exports.SimpleJsonFetcher = void 0;
const https_1 = require("https");
const stream_1 = require("stream");
const util_1 = require("util");
const error_1 = require("./error");
const safe_json_parse_1 = require("./safe-json-parse");
/**
 * HTTPS Fetcher for URIs with JSON body
 *
 * @param defaultRequestOptions - The default RequestOptions to use on individual HTTPS requests
 */
class SimpleJsonFetcher {
    constructor(props) {
        this.defaultRequestOptions = {
            timeout: 500,
            responseTimeout: 1500,
            ...props === null || props === void 0 ? void 0 : props.defaultRequestOptions,
        };
    }
    /**
     * Execute a HTTPS request (with 1 immediate retry in case of errors)
     * @param uri - The URI
     * @param requestOptions - The RequestOptions to use
     * @param data - Data to send to the URI (e.g. POST data)
     * @returns - The response as parsed JSON
     */
    async fetch(uri, requestOptions, data) {
        requestOptions = { ...this.defaultRequestOptions, ...requestOptions };
        try {
            return await fetchJson(uri, requestOptions, data);
        }
        catch (err) {
            if (err instanceof error_1.NonRetryableFetchError) {
                throw err;
            }
            // Retry once, immediately
            return fetchJson(uri, requestOptions, data);
        }
    }
}
exports.SimpleJsonFetcher = SimpleJsonFetcher;
/**
 * Execute a HTTPS request
 * @param uri - The URI
 * @param requestOptions - The RequestOptions to use
 * @param data - Data to send to the URI (e.g. POST data)
 * @returns - The response as parsed JSON
 */
async function fetchJson(uri, requestOptions, data) {
    let responseTimeout;
    return new Promise((resolve, reject) => {
        const req = https_1.request(uri, {
            method: 'GET',
            ...requestOptions,
        }, (response) => {
            // Capture response data
            // @types/node is incomplete so cast to any
            // eslint-disable-next-line @typescript-eslint/no-explicit-any
            stream_1.pipeline([
                response,
                getJsonDestination(uri, response.statusCode, response.headers),
            ], done);
        });
        if (requestOptions === null || requestOptions === void 0 ? void 0 : requestOptions.responseTimeout) {
            responseTimeout = setTimeout(() => done(new error_1.FetchError(uri, `Response time-out (after ${requestOptions.responseTimeout} ms.)`)), requestOptions.responseTimeout);
            responseTimeout.unref(); // Don't block Node from exiting
        }
        function done(...args) {
            var _a;
            if (responseTimeout)
                clearTimeout(responseTimeout);
            if (args[0] == null) {
                resolve(args[1]);
                return;
            }
            // In case of errors, let the Agent (if any) know to abandon the socket
            // This is probably best, because the socket may have become stale
            /* istanbul ignore next */
            (_a = req.socket) === null || _a === void 0 ? void 0 : _a.emit('agentRemove');
            // Turn error into FetchError so the URI is nicely captured in the message
            let error = args[0];
            if (!(error instanceof error_1.FetchError)) {
                error = new error_1.FetchError(uri, error.message);
            }
            req.destroy();
            reject(error);
        }
        // Handle errors while sending request
        req.on('error', done);
        // Signal end of request (include optional data)
        req.end(data);
    });
}
exports.fetchJson = fetchJson;
/**
 * Ensures the HTTPS response contains valid JSON
 *
 * @param uri - The URI you were requesting
 * @param statusCode - The response status code to your HTTPS request
 * @param headers - The response headers to your HTTPS request
 *
 * @returns - Async function that can be used as destination in a stream.pipeline, it will return the JSON, if valid, or throw an error otherwise
 */
function getJsonDestination(uri, statusCode, headers) {
    return async (responseIterable) => {
        var _a;
        if (statusCode === 429) {
            throw new error_1.FetchError(uri, 'Too many requests');
        }
        else if (statusCode !== 200) {
            throw new error_1.NonRetryableFetchError(uri, `Status code is ${statusCode}, expected 200`);
        }
        if (!((_a = headers['content-type']) === null || _a === void 0 ? void 0 : _a.toLowerCase().startsWith('application/json'))) {
            throw new error_1.NonRetryableFetchError(uri, `Content-type is "${headers['content-type']}", expected "application/json"`);
        }
        const collected = [];
        for await (const chunk of responseIterable) {
            collected.push(chunk);
        }
        try {
            return safe_json_parse_1.safeJsonParse(new util_1.TextDecoder('utf8', { fatal: true, ignoreBOM: true }).decode(Buffer.concat(collected)));
        }
        catch (err) {
            throw new error_1.NonRetryableFetchError(uri, err);
        }
    };
}
//# sourceMappingURL=data:application/json;base64,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