"use strict";
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.NonRetryableFetchError = exports.FetchError = exports.JwkInvalidKtyError = exports.JwkInvalidUseError = exports.JwksNotAvailableInCacheError = exports.WaitPeriodNotYetEndedJwkError = exports.KidNotFoundInJwksError = exports.JwtWithoutValidKidError = exports.JwkValidationError = exports.JwksValidationError = exports.Asn1DecodingError = exports.CognitoJwtInvalidClientIdError = exports.CognitoJwtInvalidTokenUseError = exports.CognitoJwtInvalidGroupError = exports.JwtNotBeforeError = exports.JwtExpiredError = exports.JwtInvalidScopeError = exports.JwtInvalidAudienceError = exports.JwtInvalidIssuerError = exports.JwtInvalidClaimError = exports.JwtInvalidSignatureAlgorithmError = exports.JwtInvalidSignatureError = exports.ParameterValidationError = exports.JwtParseError = exports.FailedAssertionError = exports.JwtBaseError = void 0;
/**
 * Base Error for all other errors in this file
 */
class JwtBaseError extends Error {
}
exports.JwtBaseError = JwtBaseError;
/**
 * An error that is raised because an actual value does not match with the expected value
 */
class FailedAssertionError extends JwtBaseError {
    constructor(msg, actual, expected) {
        super(msg);
        this.failedAssertion = {
            actual,
            expected,
        };
    }
}
exports.FailedAssertionError = FailedAssertionError;
/**
 * JWT errors
 */
class JwtParseError extends JwtBaseError {
    constructor(msg, error) {
        const message = error != null ? `${msg}: ${error}` : msg;
        super(message);
    }
}
exports.JwtParseError = JwtParseError;
class ParameterValidationError extends JwtBaseError {
}
exports.ParameterValidationError = ParameterValidationError;
class JwtInvalidSignatureError extends JwtBaseError {
}
exports.JwtInvalidSignatureError = JwtInvalidSignatureError;
class JwtInvalidSignatureAlgorithmError extends FailedAssertionError {
}
exports.JwtInvalidSignatureAlgorithmError = JwtInvalidSignatureAlgorithmError;
class JwtInvalidClaimError extends FailedAssertionError {
    withRawJwt({ header, payload }) {
        this.rawJwt = {
            header,
            payload,
        };
        return this;
    }
}
exports.JwtInvalidClaimError = JwtInvalidClaimError;
class JwtInvalidIssuerError extends JwtInvalidClaimError {
}
exports.JwtInvalidIssuerError = JwtInvalidIssuerError;
class JwtInvalidAudienceError extends JwtInvalidClaimError {
}
exports.JwtInvalidAudienceError = JwtInvalidAudienceError;
class JwtInvalidScopeError extends JwtInvalidClaimError {
}
exports.JwtInvalidScopeError = JwtInvalidScopeError;
class JwtExpiredError extends JwtInvalidClaimError {
}
exports.JwtExpiredError = JwtExpiredError;
class JwtNotBeforeError extends JwtInvalidClaimError {
}
exports.JwtNotBeforeError = JwtNotBeforeError;
/**
 * Amazon Cognito specific erros
 */
class CognitoJwtInvalidGroupError extends JwtInvalidClaimError {
}
exports.CognitoJwtInvalidGroupError = CognitoJwtInvalidGroupError;
class CognitoJwtInvalidTokenUseError extends JwtInvalidClaimError {
}
exports.CognitoJwtInvalidTokenUseError = CognitoJwtInvalidTokenUseError;
class CognitoJwtInvalidClientIdError extends JwtInvalidClaimError {
}
exports.CognitoJwtInvalidClientIdError = CognitoJwtInvalidClientIdError;
/**
 * ASN.1 errors
 */
class Asn1DecodingError extends JwtBaseError {
}
exports.Asn1DecodingError = Asn1DecodingError;
/**
 * JWK errors
 */
class JwksValidationError extends JwtBaseError {
}
exports.JwksValidationError = JwksValidationError;
class JwkValidationError extends JwtBaseError {
}
exports.JwkValidationError = JwkValidationError;
class JwtWithoutValidKidError extends JwtBaseError {
}
exports.JwtWithoutValidKidError = JwtWithoutValidKidError;
class KidNotFoundInJwksError extends JwtBaseError {
}
exports.KidNotFoundInJwksError = KidNotFoundInJwksError;
class WaitPeriodNotYetEndedJwkError extends JwtBaseError {
}
exports.WaitPeriodNotYetEndedJwkError = WaitPeriodNotYetEndedJwkError;
class JwksNotAvailableInCacheError extends JwtBaseError {
}
exports.JwksNotAvailableInCacheError = JwksNotAvailableInCacheError;
class JwkInvalidUseError extends FailedAssertionError {
}
exports.JwkInvalidUseError = JwkInvalidUseError;
class JwkInvalidKtyError extends FailedAssertionError {
}
exports.JwkInvalidKtyError = JwkInvalidKtyError;
/**
 * HTTPS fetch errors
 */
class FetchError extends JwtBaseError {
    constructor(uri, msg) {
        super(`Failed to fetch ${uri}: ${msg}`);
    }
}
exports.FetchError = FetchError;
class NonRetryableFetchError extends FetchError {
}
exports.NonRetryableFetchError = NonRetryableFetchError;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZXJyb3IuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi9zcmMvYXdzLWp3dC12ZXJpZnktcmlwb3V0L2Vycm9yLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQSxxRUFBcUU7QUFDckUsc0NBQXNDOzs7QUFJdEM7O0dBRUc7QUFDSCxNQUFzQixZQUFhLFNBQVEsS0FBSztDQUFJO0FBQXBELG9DQUFvRDtBQXVCcEQ7O0dBRUc7QUFDSCxNQUFhLG9CQUFxQixTQUFRLFlBQVk7SUFLcEQsWUFBWSxHQUFXLEVBQUUsTUFBZSxFQUFFLFFBQTRCO1FBQ3BFLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUNYLElBQUksQ0FBQyxlQUFlLEdBQUc7WUFDckIsTUFBTTtZQUNOLFFBQVE7U0FDVCxDQUFDO0lBQ0osQ0FBQztDQUNGO0FBWkQsb0RBWUM7QUFFRDs7R0FFRztBQUVILE1BQWEsYUFBYyxTQUFRLFlBQVk7SUFDN0MsWUFBWSxHQUFXLEVBQUUsS0FBZTtRQUN0QyxNQUFNLE9BQU8sR0FBRyxLQUFLLElBQUksSUFBSSxDQUFDLENBQUMsQ0FBQyxHQUFHLEdBQUcsS0FBSyxLQUFLLEVBQUUsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDO1FBQ3pELEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQztJQUNqQixDQUFDO0NBQ0Y7QUFMRCxzQ0FLQztBQUVELE1BQWEsd0JBQXlCLFNBQVEsWUFBWTtDQUFJO0FBQTlELDREQUE4RDtBQUU5RCxNQUFhLHdCQUF5QixTQUFRLFlBQVk7Q0FBSTtBQUE5RCw0REFBOEQ7QUFFOUQsTUFBYSxpQ0FBa0MsU0FBUSxvQkFBb0I7Q0FBSTtBQUEvRSw4RUFBK0U7QUFPL0UsTUFBc0Isb0JBQXFCLFNBQVEsb0JBQW9CO0lBRTlELFVBQVUsQ0FFZixFQUFFLE1BQU0sRUFBRSxPQUFPLEVBQVU7UUFFM0IsSUFBSSxDQUFDLE1BQU0sR0FBRztZQUNaLE1BQU07WUFDTixPQUFPO1NBQ1IsQ0FBQztRQUNGLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztDQUNGO0FBWkQsb0RBWUM7QUFFRCxNQUFhLHFCQUFzQixTQUFRLG9CQUFvQjtDQUFJO0FBQW5FLHNEQUFtRTtBQUVuRSxNQUFhLHVCQUF3QixTQUFRLG9CQUFvQjtDQUFJO0FBQXJFLDBEQUFxRTtBQUVyRSxNQUFhLG9CQUFxQixTQUFRLG9CQUFvQjtDQUFJO0FBQWxFLG9EQUFrRTtBQUVsRSxNQUFhLGVBQWdCLFNBQVEsb0JBQW9CO0NBQUk7QUFBN0QsMENBQTZEO0FBRTdELE1BQWEsaUJBQWtCLFNBQVEsb0JBQW9CO0NBQUk7QUFBL0QsOENBQStEO0FBRS9EOztHQUVHO0FBRUgsTUFBYSwyQkFBNEIsU0FBUSxvQkFBb0I7Q0FBSTtBQUF6RSxrRUFBeUU7QUFFekUsTUFBYSw4QkFBK0IsU0FBUSxvQkFBb0I7Q0FBSTtBQUE1RSx3RUFBNEU7QUFFNUUsTUFBYSw4QkFBK0IsU0FBUSxvQkFBb0I7Q0FBSTtBQUE1RSx3RUFBNEU7QUFFNUU7O0dBRUc7QUFFSCxNQUFhLGlCQUFrQixTQUFRLFlBQVk7Q0FBSTtBQUF2RCw4Q0FBdUQ7QUFFdkQ7O0dBRUc7QUFFSCxNQUFhLG1CQUFvQixTQUFRLFlBQVk7Q0FBSTtBQUF6RCxrREFBeUQ7QUFFekQsTUFBYSxrQkFBbUIsU0FBUSxZQUFZO0NBQUk7QUFBeEQsZ0RBQXdEO0FBRXhELE1BQWEsdUJBQXdCLFNBQVEsWUFBWTtDQUFJO0FBQTdELDBEQUE2RDtBQUU3RCxNQUFhLHNCQUF1QixTQUFRLFlBQVk7Q0FBSTtBQUE1RCx3REFBNEQ7QUFFNUQsTUFBYSw2QkFBOEIsU0FBUSxZQUFZO0NBQUk7QUFBbkUsc0VBQW1FO0FBRW5FLE1BQWEsNEJBQTZCLFNBQVEsWUFBWTtDQUFJO0FBQWxFLG9FQUFrRTtBQUVsRSxNQUFhLGtCQUFtQixTQUFRLG9CQUFvQjtDQUFJO0FBQWhFLGdEQUFnRTtBQUVoRSxNQUFhLGtCQUFtQixTQUFRLG9CQUFvQjtDQUFJO0FBQWhFLGdEQUFnRTtBQUVoRTs7R0FFRztBQUVILE1BQWEsVUFBVyxTQUFRLFlBQVk7SUFDMUMsWUFBWSxHQUFXLEVBQUUsR0FBWTtRQUNuQyxLQUFLLENBQUMsbUJBQW1CLEdBQUcsS0FBSyxHQUFHLEVBQUUsQ0FBQyxDQUFDO0lBQzFDLENBQUM7Q0FDRjtBQUpELGdDQUlDO0FBRUQsTUFBYSxzQkFBdUIsU0FBUSxVQUFVO0NBQUk7QUFBMUQsd0RBQTBEIiwic291cmNlc0NvbnRlbnQiOlsiLy8gQ29weXJpZ2h0IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4vLyBTUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogQXBhY2hlLTIuMFxuXG5pbXBvcnQgeyBKd3RIZWFkZXIsIEp3dFBheWxvYWQgfSBmcm9tICcuL2p3dC1tb2RlbCc7XG5cbi8qKlxuICogQmFzZSBFcnJvciBmb3IgYWxsIG90aGVyIGVycm9ycyBpbiB0aGlzIGZpbGVcbiAqL1xuZXhwb3J0IGFic3RyYWN0IGNsYXNzIEp3dEJhc2VFcnJvciBleHRlbmRzIEVycm9yIHsgfVxuXG4vKipcbiAqIEludGVyZmFjZSBmb3IgYW4gZXJyb3IgdGhhdCBpcyByYWlzZWQgYmVjYXVzZSBhbiBhY3R1YWwgdmFsdWUgZG9lcyBub3QgbWF0Y2ggd2l0aCB0aGUgZXhwZWN0ZWQgdmFsdWVcbiAqL1xuaW50ZXJmYWNlIEFzc2VydGlvbkVycm9yIGV4dGVuZHMgSnd0QmFzZUVycm9yIHtcbiAgZmFpbGVkQXNzZXJ0aW9uOiB7XG4gICAgYWN0dWFsOiB1bmtub3duO1xuICAgIGV4cGVjdGVkPzogc3RyaW5nIHwgc3RyaW5nW107XG4gIH07XG59XG5cbi8qKlxuICogQ29uc3RydWN0b3IgaW50ZXJmYWNlIGZvciBBc3NlcnRpb25FcnJvclxuICovXG5leHBvcnQgaW50ZXJmYWNlIEFzc2VydGlvbkVycm9yQ29uc3RydWN0b3Ige1xuICBuZXcoXG4gICAgbXNnOiBzdHJpbmcsXG4gICAgYWN0dWFsOiB1bmtub3duLFxuICAgIGV4cGVjdGVkPzogc3RyaW5nIHwgc3RyaW5nW11cbiAgKTogQXNzZXJ0aW9uRXJyb3I7XG59XG5cbi8qKlxuICogQW4gZXJyb3IgdGhhdCBpcyByYWlzZWQgYmVjYXVzZSBhbiBhY3R1YWwgdmFsdWUgZG9lcyBub3QgbWF0Y2ggd2l0aCB0aGUgZXhwZWN0ZWQgdmFsdWVcbiAqL1xuZXhwb3J0IGNsYXNzIEZhaWxlZEFzc2VydGlvbkVycm9yIGV4dGVuZHMgSnd0QmFzZUVycm9yIHtcbiAgZmFpbGVkQXNzZXJ0aW9uOiB7XG4gICAgYWN0dWFsOiB1bmtub3duO1xuICAgIGV4cGVjdGVkPzogc3RyaW5nIHwgc3RyaW5nW107XG4gIH07XG4gIGNvbnN0cnVjdG9yKG1zZzogc3RyaW5nLCBhY3R1YWw6IHVua25vd24sIGV4cGVjdGVkPzogc3RyaW5nIHwgc3RyaW5nW10pIHtcbiAgICBzdXBlcihtc2cpO1xuICAgIHRoaXMuZmFpbGVkQXNzZXJ0aW9uID0ge1xuICAgICAgYWN0dWFsLFxuICAgICAgZXhwZWN0ZWQsXG4gICAgfTtcbiAgfVxufVxuXG4vKipcbiAqIEpXVCBlcnJvcnNcbiAqL1xuXG5leHBvcnQgY2xhc3MgSnd0UGFyc2VFcnJvciBleHRlbmRzIEp3dEJhc2VFcnJvciB7XG4gIGNvbnN0cnVjdG9yKG1zZzogc3RyaW5nLCBlcnJvcj86IHVua25vd24pIHtcbiAgICBjb25zdCBtZXNzYWdlID0gZXJyb3IgIT0gbnVsbCA/IGAke21zZ306ICR7ZXJyb3J9YCA6IG1zZztcbiAgICBzdXBlcihtZXNzYWdlKTtcbiAgfVxufVxuXG5leHBvcnQgY2xhc3MgUGFyYW1ldGVyVmFsaWRhdGlvbkVycm9yIGV4dGVuZHMgSnd0QmFzZUVycm9yIHsgfVxuXG5leHBvcnQgY2xhc3MgSnd0SW52YWxpZFNpZ25hdHVyZUVycm9yIGV4dGVuZHMgSnd0QmFzZUVycm9yIHsgfVxuXG5leHBvcnQgY2xhc3MgSnd0SW52YWxpZFNpZ25hdHVyZUFsZ29yaXRobUVycm9yIGV4dGVuZHMgRmFpbGVkQXNzZXJ0aW9uRXJyb3IgeyB9XG5cbmludGVyZmFjZSBSYXdKd3Qge1xuICBoZWFkZXI6IEp3dEhlYWRlcjtcbiAgcGF5bG9hZDogSnd0UGF5bG9hZDtcbn1cblxuZXhwb3J0IGFic3RyYWN0IGNsYXNzIEp3dEludmFsaWRDbGFpbUVycm9yIGV4dGVuZHMgRmFpbGVkQXNzZXJ0aW9uRXJyb3Ige1xuICBwdWJsaWMgcmF3Snd0PzogUmF3Snd0O1xuICBwdWJsaWMgd2l0aFJhd0p3dDxUIGV4dGVuZHMgSnd0SW52YWxpZENsYWltRXJyb3I+KFxuICAgIHRoaXM6IFQsXG4gICAgeyBoZWFkZXIsIHBheWxvYWQgfTogUmF3Snd0LFxuICApOiBUIHtcbiAgICB0aGlzLnJhd0p3dCA9IHtcbiAgICAgIGhlYWRlcixcbiAgICAgIHBheWxvYWQsXG4gICAgfTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxufVxuXG5leHBvcnQgY2xhc3MgSnd0SW52YWxpZElzc3VlckVycm9yIGV4dGVuZHMgSnd0SW52YWxpZENsYWltRXJyb3IgeyB9XG5cbmV4cG9ydCBjbGFzcyBKd3RJbnZhbGlkQXVkaWVuY2VFcnJvciBleHRlbmRzIEp3dEludmFsaWRDbGFpbUVycm9yIHsgfVxuXG5leHBvcnQgY2xhc3MgSnd0SW52YWxpZFNjb3BlRXJyb3IgZXh0ZW5kcyBKd3RJbnZhbGlkQ2xhaW1FcnJvciB7IH1cblxuZXhwb3J0IGNsYXNzIEp3dEV4cGlyZWRFcnJvciBleHRlbmRzIEp3dEludmFsaWRDbGFpbUVycm9yIHsgfVxuXG5leHBvcnQgY2xhc3MgSnd0Tm90QmVmb3JlRXJyb3IgZXh0ZW5kcyBKd3RJbnZhbGlkQ2xhaW1FcnJvciB7IH1cblxuLyoqXG4gKiBBbWF6b24gQ29nbml0byBzcGVjaWZpYyBlcnJvc1xuICovXG5cbmV4cG9ydCBjbGFzcyBDb2duaXRvSnd0SW52YWxpZEdyb3VwRXJyb3IgZXh0ZW5kcyBKd3RJbnZhbGlkQ2xhaW1FcnJvciB7IH1cblxuZXhwb3J0IGNsYXNzIENvZ25pdG9Kd3RJbnZhbGlkVG9rZW5Vc2VFcnJvciBleHRlbmRzIEp3dEludmFsaWRDbGFpbUVycm9yIHsgfVxuXG5leHBvcnQgY2xhc3MgQ29nbml0b0p3dEludmFsaWRDbGllbnRJZEVycm9yIGV4dGVuZHMgSnd0SW52YWxpZENsYWltRXJyb3IgeyB9XG5cbi8qKlxuICogQVNOLjEgZXJyb3JzXG4gKi9cblxuZXhwb3J0IGNsYXNzIEFzbjFEZWNvZGluZ0Vycm9yIGV4dGVuZHMgSnd0QmFzZUVycm9yIHsgfVxuXG4vKipcbiAqIEpXSyBlcnJvcnNcbiAqL1xuXG5leHBvcnQgY2xhc3MgSndrc1ZhbGlkYXRpb25FcnJvciBleHRlbmRzIEp3dEJhc2VFcnJvciB7IH1cblxuZXhwb3J0IGNsYXNzIEp3a1ZhbGlkYXRpb25FcnJvciBleHRlbmRzIEp3dEJhc2VFcnJvciB7IH1cblxuZXhwb3J0IGNsYXNzIEp3dFdpdGhvdXRWYWxpZEtpZEVycm9yIGV4dGVuZHMgSnd0QmFzZUVycm9yIHsgfVxuXG5leHBvcnQgY2xhc3MgS2lkTm90Rm91bmRJbkp3a3NFcnJvciBleHRlbmRzIEp3dEJhc2VFcnJvciB7IH1cblxuZXhwb3J0IGNsYXNzIFdhaXRQZXJpb2ROb3RZZXRFbmRlZEp3a0Vycm9yIGV4dGVuZHMgSnd0QmFzZUVycm9yIHsgfVxuXG5leHBvcnQgY2xhc3MgSndrc05vdEF2YWlsYWJsZUluQ2FjaGVFcnJvciBleHRlbmRzIEp3dEJhc2VFcnJvciB7IH1cblxuZXhwb3J0IGNsYXNzIEp3a0ludmFsaWRVc2VFcnJvciBleHRlbmRzIEZhaWxlZEFzc2VydGlvbkVycm9yIHsgfVxuXG5leHBvcnQgY2xhc3MgSndrSW52YWxpZEt0eUVycm9yIGV4dGVuZHMgRmFpbGVkQXNzZXJ0aW9uRXJyb3IgeyB9XG5cbi8qKlxuICogSFRUUFMgZmV0Y2ggZXJyb3JzXG4gKi9cblxuZXhwb3J0IGNsYXNzIEZldGNoRXJyb3IgZXh0ZW5kcyBKd3RCYXNlRXJyb3Ige1xuICBjb25zdHJ1Y3Rvcih1cmk6IHN0cmluZywgbXNnOiB1bmtub3duKSB7XG4gICAgc3VwZXIoYEZhaWxlZCB0byBmZXRjaCAke3VyaX06ICR7bXNnfWApO1xuICB9XG59XG5cbmV4cG9ydCBjbGFzcyBOb25SZXRyeWFibGVGZXRjaEVycm9yIGV4dGVuZHMgRmV0Y2hFcnJvciB7IH0iXX0=