"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.renderJavaScriptOptions = exports.resolveNewProject = exports.renderProjenNewOptions = void 0;
const inventory = require("../inventory");
const PROJEN_NEW = '__new__';
/**
 * Renders options as if the project was created via `projen new` (embeds the __new__ field).
 */
function renderProjenNewOptions(fqn, args) {
    return {
        ...args,
        [PROJEN_NEW]: { fqn, args },
    };
}
exports.renderProjenNewOptions = renderProjenNewOptions;
function resolveNewProject(opts) {
    const f = opts[PROJEN_NEW];
    if (!f) {
        return undefined;
    }
    const type = inventory.resolveProjectType(f.fqn);
    if (!type) {
        throw new Error(`unable to resolve project type for ${f.fqn}`);
    }
    return {
        args: f.args,
        fqn: f.fqn,
        type: type,
    };
}
exports.resolveNewProject = resolveNewProject;
/**
 * Prints all parameters that can be used in a project type, alongside their descriptions.
 *
 * Parameters in `params` that aren't undefined are rendered as defaults,
 * while all other parameters are rendered as commented out.
 */
function renderJavaScriptOptions(opts) {
    var _a, _b, _c, _d;
    // preprocessing
    const renders = {};
    const optionsWithDefaults = [];
    const optionsByModule = {}; // only options without defaults
    for (const option of opts.type.options) {
        if (option.deprecated) {
            continue;
        }
        const optionName = option.name;
        let paramRender;
        if (opts.args[optionName] !== undefined) {
            const value = opts.args[optionName];
            const js = JSON.stringify(value).replace(/^"(.+)"$/, '\'$1\'');
            paramRender = `${optionName}: ${js},`;
            optionsWithDefaults.push(optionName);
        }
        else {
            const defaultValue = ((_a = option.default) === null || _a === void 0 ? void 0 : _a.startsWith('-')) ? undefined : ((_b = option.default) !== null && _b !== void 0 ? _b : undefined);
            paramRender = `// ${optionName}: ${defaultValue === null || defaultValue === void 0 ? void 0 : defaultValue.replace(/"(.+)"/, '\'$1\'')},`; // single quotes
            const parentModule = option.parent;
            optionsByModule[parentModule] = (_c = optionsByModule[parentModule]) !== null && _c !== void 0 ? _c : [];
            optionsByModule[parentModule].push(option);
        }
        renders[optionName] = paramRender;
    }
    const bootstrap = (_d = opts.bootstrap) !== null && _d !== void 0 ? _d : false;
    if (bootstrap) {
        renders[PROJEN_NEW] = `${PROJEN_NEW}: ${JSON.stringify({ args: opts.args, fqn: opts.type.fqn })},`;
        optionsWithDefaults.push(PROJEN_NEW);
    }
    // alphabetize
    const marginSize = Math.max(...Object.values(renders).map(str => str.length));
    optionsWithDefaults.sort();
    for (const parentModule in optionsByModule) {
        optionsByModule[parentModule].sort((o1, o2) => o1.name.localeCompare(o2.name));
    }
    // generate rendering
    const tab = makePadding(2);
    const result = [];
    result.push('{');
    // render options with defaults
    for (const optionName of optionsWithDefaults) {
        result.push(`${tab}${renders[optionName]}`);
    }
    if (result.length > 1) {
        result.push('');
    }
    // render options without defaults
    if (opts.comments) {
        for (const [moduleName, options] of Object.entries(optionsByModule).sort()) {
            result.push(`${tab}/* ${moduleName} */`);
            for (const option of options) {
                const paramRender = renders[option.name];
                result.push(`${tab}${paramRender}${makePadding(marginSize - paramRender.length + 2)}/* ${option.docs} */`);
            }
            result.push('');
        }
    }
    if (result[result.length - 1] === '') {
        result.pop();
    }
    result.push('}');
    return result.join('\n');
}
exports.renderJavaScriptOptions = renderJavaScriptOptions;
function makePadding(paddingLength) {
    return ' '.repeat(paddingLength);
}
//# sourceMappingURL=data:application/json;base64,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