"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const iot = require("@aws-cdk/aws-iot");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const defaults = require("@aws-solutions-konstruk/core");
const core_2 = require("@aws-solutions-konstruk/core");
class IotToLambda extends core_1.Construct {
    /**
     * @summary Constructs a new instance of the IotToLambda class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {IotToLambdaProps} props - user provided props for the construct
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.fn = defaults.buildLambdaFunction(scope, {
            deployLambda: props.deployLambda,
            existingLambdaObj: props.existingLambdaObj,
            lambdaFunctionProps: props.lambdaFunctionProps
        });
        const defaultIotTopicProps = defaults.DefaultCfnTopicRuleProps([{
                lambda: {
                    functionArn: this.fn.functionArn
                }
            }]);
        const iotTopicProps = core_2.overrideProps(defaultIotTopicProps, props.iotTopicRuleProps, true);
        // Create the IoT topic rule
        this.topic = new iot.CfnTopicRule(this, 'IotTopic', iotTopicProps);
        this.fn.addPermission("LambdaInvokePermission", {
            principal: new iam.ServicePrincipal('iot.amazonaws.com'),
            sourceArn: this.topic.attrArn
        });
    }
    /**
     * @summary Retruns an instance of iot.CfnTopicRule created by the construct.
     * @returns {iot.CfnTopicRule} Instance of CfnTopicRule created by the construct
     * @since 0.8.0
     * @access public
     */
    iotTopicRule() {
        return this.topic;
    }
    /**
     * @summary Retruns an instance of lambda.Function created by the construct.
     * @returns {lambda.Function} Instance of lambda.Function created by the construct
     * @since 0.8.0
     * @access public
     */
    lambdaFunction() {
        return this.fn;
    }
}
exports.IotToLambda = IotToLambda;
//# sourceMappingURL=data:application/json;base64,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