def read_data(file_path, **kwargs):
    """Reads data from a file and outputs as a pandas dataframe.
    
    Args:
    file_path (str): The path to the file to be read.
    **kwargs: Additional arguments that are specific to each format type.
    
    Returns:
    pandas.DataFrame: The data read from the file.
    """
    import pandas as pd
    import pyperclip
    import pyarrow.parquet as pq
    import pickle
    import sas7bdat
    import pyreadstat
    
    # Determine file type based on file extension
    file_extension = file_path.split('.')[-1]
    if file_extension == 'csv':
        # Read CSV file
        df = pd.read_csv(file_path, **kwargs)
    elif file_extension == 'xlsx' or file_extension == 'xls':
        # Read Excel file
        df = pd.read_excel(file_path, **kwargs)
    elif file_extension == 'json':
        # Read JSON file
        df = pd.read_json(file_path, **kwargs)
    elif file_extension == 'txt':
        # Read text file (assumes tab-separated values)
        df = pd.read_csv(file_path, delimiter='\t')
    elif file_path == 'clipboard':
        # Get clipboard contents
        clipboard_data = pyperclip.paste()
        
        # Read clipboard contents as CSV
        df = pd.read_csv(pd.compat.StringIO(clipboard_data), delimiter='\t')
    elif file_extension == 'sas':
        # Read SAS file
        with open(file_path, 'rb') as f:
            df = sas7bdat.SAS7BDAT(f).to_data_frame()
    elif file_extension == 'spss':
        # Read SPSS file
        df, meta = pyreadstat.read_sav(file_path, **kwargs)
    elif file_extension == 'parquet':
        # Read Parquet file
        pq_file = pq.ParquetFile(file_path)
        df = pq_file.read().to_pandas()
    elif file_extension == 'pickle':
        # Read Pickle file
        with open(file_path, 'rb') as f:
            df = pickle.load(f)
    else:
        raise ValueError(f'Unsupported file type: {file_extension}')
    
    return df

def read_images(folder_path, image_size):
    """Reads image dataset and returns as numpy array.
    
    Args:
    folder_path (str): The path to the folder containing the image dataset.
    image_size (tuple): The desired size of the images in the dataset. 
    
    Returns:
    numpy.ndarray: The image dataset as a numpy array.
    """
    from PIL import Image
    import numpy as np
    import os
    image_files = [f for f in os.listdir(folder_path) if f.endswith('.jpg') or f.endswith('.png')]
    images = np.zeros((len(image_files), image_size[0], image_size[1], 3), dtype=np.uint8)
    
    for i, file_name in enumerate(image_files):
        file_path = os.path.join(folder_path, file_name)
        image = Image.open(file_path).resize(image_size)
        images[i] = np.array(image, dtype=np.uint8)
    
    return images

def read_images_tf(folder_path, labels='inferred', label_mode='binary', image_size=[128, 128], interpolation='nearest', batch_size=64, shuffle=True,):
    from tensorflow.keras.preprocessing import image_dataset_from_directory
    import tensorflow as tf
    from ..modeling.data_preprocessing import image_to_float

    AUTOTUNE = tf.data.experimental.AUTOTUNE

    return (
        image_dataset_from_directory(
                                     folder_path,
                                     labels=labels,
                                     label_mode=label_mode,
                                     image_size=image_size,
                                     interpolation=interpolation,
                                     batch_size=batch_size,
                                     shuffle=shuffle,
        ).map(image_to_float)
        .cache()
        .prefetch(buffer_size=AUTOTUNE)
    )
    

def read_from_database(database_url, query):
    """Reads data from a SQL database and outputs it as a pandas dataframe.
    
    Args:
    database_url (str): The URL of the database.
    query (str): The SQL query to execute.
    
    Returns:
    pandas.DataFrame: The data read from the database.
    """
    import pandas as pd
    import sqlalchemy as db

    engine = db.create_engine(database_url)
    connection = engine.connect()
    result = connection.execute(query)
    data = result.fetchall()
    connection.close()
    df = pd.DataFrame(data, columns=result.keys())
    return df


def read_from_html(url, table_index=0):
    """Reads tabular data from an HTML website and outputs it as a pandas dataframe.
    
    Args:
    url (str): The URL of the website.
    table_index (int): The index of the HTML table to read (default: 0).
    
    Returns:
    pandas.DataFrame: The tabular data read from the HTML website.
    """
    import pandas as pd
    import requests
    from bs4 import BeautifulSoup
    
    response = requests.get(url)
    soup = BeautifulSoup(response.text, 'html.parser')
    tables = soup.find_all('table')
    table = tables[table_index]
    df = pd.read_html(str(table))[0]
    return df
