"use strict";
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const a = new lib_1.Metric({ namespace: 'Test', metricName: 'ACount' });
const b = new lib_1.Metric({ namespace: 'Test', metricName: 'BCount', statistic: 'Average' });
const c = new lib_1.Metric({ namespace: 'Test', metricName: 'CCount' });
const b99 = new lib_1.Metric({ namespace: 'Test', metricName: 'BCount', statistic: 'p99' });
let stack;
function graphMetricsAre(test, w, metrics) {
    test.deepEqual(stack.resolve(w.toJson()), [{
            type: 'metric',
            width: 6,
            height: 6,
            properties: {
                view: 'timeSeries',
                region: { Ref: 'AWS::Region' },
                metrics,
                yAxis: {},
            },
        }]);
}
function alarmMetricsAre(metrics) {
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudWatch::Alarm', {
        Metrics: metrics,
    }));
}
module.exports = {
    'setUp'(cb) {
        stack = new core_1.Stack();
        cb();
    },
    'can not use invalid variable names in MathExpression'(test) {
        test.throws(() => {
            new lib_1.MathExpression({
                expression: 'HAPPY + JOY',
                usingMetrics: {
                    HAPPY: a,
                    JOY: b,
                },
            });
        }, /Invalid variable names in expression/);
        test.done();
    },
    'cannot reuse variable names in nested MathExpressions'(test) {
        // WHEN
        test.throws(() => {
            new lib_1.MathExpression({
                expression: 'a + e',
                usingMetrics: {
                    a,
                    e: new lib_1.MathExpression({
                        expression: 'a + c',
                        usingMetrics: { a: b, c },
                    }),
                },
            });
        }, /The ID 'a' used for two metrics in the expression: 'BCount' and 'ACount'. Rename one/);
        test.done();
    },
    'can not use invalid period in MathExpression'(test) {
        test.throws(() => {
            new lib_1.MathExpression({
                expression: 'a+b',
                usingMetrics: { a, b },
                period: core_1.Duration.seconds(20),
            });
        }, /'period' must be 1, 5, 10, 30, or a multiple of 60 seconds, received 20/);
        test.done();
    },
    'MathExpression optimization: "with" with the same period returns the same object'(test) {
        const m = new lib_1.MathExpression({ expression: 'SUM(METRICS())', usingMetrics: {}, period: core_1.Duration.minutes(10) });
        // Note: object equality, NOT deep equality on purpose
        test.equals(m.with({}), m);
        test.equals(m.with({ period: core_1.Duration.minutes(10) }), m);
        test.notEqual(m.with({ period: core_1.Duration.minutes(5) }), m);
        test.done();
    },
    'in graphs': {
        'MathExpressions can be added to a graph'(test) {
            // GIVEN
            const graph = new lib_1.GraphWidget({
                left: [
                    new lib_1.MathExpression({
                        expression: 'a + b',
                        usingMetrics: { a, b },
                    }),
                ],
            });
            // THEN
            graphMetricsAre(test, graph, [
                [{ expression: 'a + b', label: 'a + b' }],
                ['Test', 'ACount', { visible: false, id: 'a' }],
                ['Test', 'BCount', { visible: false, id: 'b' }],
            ]);
            test.done();
        },
        'can nest MathExpressions in a graph'(test) {
            // GIVEN
            const graph = new lib_1.GraphWidget({
                left: [
                    new lib_1.MathExpression({
                        expression: 'a + e',
                        usingMetrics: {
                            a,
                            e: new lib_1.MathExpression({
                                expression: 'b + c',
                                usingMetrics: { b, c },
                            }),
                        },
                    }),
                ],
            });
            // THEN
            graphMetricsAre(test, graph, [
                [{ label: 'a + e', expression: 'a + e' }],
                ['Test', 'ACount', { visible: false, id: 'a' }],
                [{ expression: 'b + c', visible: false, id: 'e' }],
                ['Test', 'BCount', { visible: false, id: 'b' }],
                ['Test', 'CCount', { visible: false, id: 'c' }],
            ]);
            test.done();
        },
        'can add the same metric under different ids'(test) {
            const graph = new lib_1.GraphWidget({
                left: [
                    new lib_1.MathExpression({
                        expression: 'a + e',
                        usingMetrics: {
                            a,
                            e: new lib_1.MathExpression({
                                expression: 'b + c',
                                usingMetrics: { b: a, c },
                            }),
                        },
                    }),
                ],
            });
            graphMetricsAre(test, graph, [
                [{ label: 'a + e', expression: 'a + e' }],
                ['Test', 'ACount', { visible: false, id: 'a' }],
                [{ expression: 'b + c', visible: false, id: 'e' }],
                ['Test', 'ACount', { visible: false, id: 'b' }],
                ['Test', 'CCount', { visible: false, id: 'c' }],
            ]);
            test.done();
        },
        'can reuse identifiers in MathExpressions if metrics are the same'(test) {
            const graph = new lib_1.GraphWidget({
                left: [
                    new lib_1.MathExpression({
                        expression: 'a + e',
                        usingMetrics: {
                            a,
                            e: new lib_1.MathExpression({
                                expression: 'a + c',
                                usingMetrics: { a, c },
                            }),
                        },
                    }),
                ],
            });
            // THEN
            graphMetricsAre(test, graph, [
                [{ label: 'a + e', expression: 'a + e' }],
                ['Test', 'ACount', { visible: false, id: 'a' }],
                [{ expression: 'a + c', visible: false, id: 'e' }],
                ['Test', 'CCount', { visible: false, id: 'c' }],
            ]);
            test.done();
        },
        'MathExpression and its constituent metrics can both be added to a graph'(test) {
            const graph = new lib_1.GraphWidget({
                left: [
                    a,
                    new lib_1.MathExpression({
                        expression: 'a + b',
                        usingMetrics: { a, b },
                    }),
                ],
            });
            // THEN
            graphMetricsAre(test, graph, [
                ['Test', 'ACount', { id: 'a' }],
                [{ label: 'a + b', expression: 'a + b' }],
                ['Test', 'BCount', { visible: false, id: 'b' }],
            ]);
            test.done();
        },
        'MathExpression controls period of metrics directly used in it'(test) {
            // Check that if we add A with { period: 10s } to a mathexpression of period 5m
            // then two metric lines are added for A, one at 10s and one at 5m
            const graph = new lib_1.GraphWidget({
                left: [
                    a.with({ period: core_1.Duration.seconds(10) }),
                    new lib_1.MathExpression({
                        expression: 'a + b',
                        usingMetrics: { a: a.with({ period: core_1.Duration.seconds(10) }), b },
                    }),
                ],
            });
            // THEN
            graphMetricsAre(test, graph, [
                ['Test', 'ACount', { period: 10 }],
                [{ label: 'a + b', expression: 'a + b' }],
                ['Test', 'ACount', { visible: false, id: 'a' }],
                ['Test', 'BCount', { visible: false, id: 'b' }],
            ]);
            test.done();
        },
        'MathExpression controls period of metrics transitively used in it'(test) {
            // Same as the previous test, but recursively
            const graph = new lib_1.GraphWidget({
                left: [
                    new lib_1.MathExpression({
                        expression: 'a + e',
                        usingMetrics: {
                            a,
                            e: new lib_1.MathExpression({
                                expression: 'a + b',
                                period: core_1.Duration.minutes(1),
                                usingMetrics: { a, b },
                            }),
                        },
                    }),
                ],
            });
            // THEN
            graphMetricsAre(test, graph, [
                [{ expression: 'a + e', label: 'a + e' }],
                ['Test', 'ACount', { visible: false, id: 'a' }],
                [{ expression: 'a + b', visible: false, id: 'e' }],
                ['Test', 'BCount', { visible: false, id: 'b' }],
            ]);
            test.done();
        },
        'can use percentiles in expression metrics in graphs'(test) {
            // GIVEN
            const graph = new lib_1.GraphWidget({
                left: [
                    new lib_1.MathExpression({
                        expression: 'a + b99',
                        usingMetrics: { a, b99 },
                    }),
                ],
            });
            // THEN
            graphMetricsAre(test, graph, [
                [{ expression: 'a + b99', label: 'a + b99' }],
                ['Test', 'ACount', { visible: false, id: 'a' }],
                ['Test', 'BCount', { visible: false, id: 'b99', stat: 'p99' }],
            ]);
            test.done();
        },
        'can reuse the same metric between left and right axes'(test) {
            // GIVEN
            const graph = new lib_1.GraphWidget({
                left: [
                    new lib_1.MathExpression({
                        expression: 'a + 1',
                        usingMetrics: { a },
                    }),
                ],
                right: [
                    new lib_1.MathExpression({
                        expression: 'a + 2',
                        usingMetrics: { a },
                    }),
                ],
            });
            // THEN
            graphMetricsAre(test, graph, [
                [{ label: 'a + 1', expression: 'a + 1' }],
                ['Test', 'ACount', { visible: false, id: 'a' }],
                [{ label: 'a + 2', expression: 'a + 2', yAxis: 'right' }],
            ]);
            test.done();
        },
        'detect name conflicts between left and right axes'(test) {
            // GIVEN
            const graph = new lib_1.GraphWidget({
                left: [
                    new lib_1.MathExpression({
                        expression: 'm1 + 1',
                        usingMetrics: { m1: a },
                    }),
                ],
                right: [
                    new lib_1.MathExpression({
                        expression: 'm1 + 1',
                        usingMetrics: { m1: b },
                    }),
                ],
            });
            // THEN
            test.throws(() => {
                graphMetricsAre(test, graph, []);
            }, /Cannot have two different metrics share the same id \('m1'\)/);
            test.done();
        },
    },
    'in alarms': {
        'MathExpressions can be used for an alarm'(test) {
            // GIVEN
            new lib_1.Alarm(stack, 'Alarm', {
                threshold: 1,
                evaluationPeriods: 1,
                metric: new lib_1.MathExpression({
                    expression: 'a + b',
                    usingMetrics: { a, b },
                }),
            });
            // THEN
            alarmMetricsAre([
                {
                    Expression: 'a + b',
                    Id: 'expr_1',
                },
                {
                    Id: 'a',
                    MetricStat: {
                        Metric: {
                            MetricName: 'ACount',
                            Namespace: 'Test',
                        },
                        Period: 300,
                        Stat: 'Average',
                    },
                    ReturnData: false,
                },
                {
                    Id: 'b',
                    MetricStat: {
                        Metric: {
                            MetricName: 'BCount',
                            Namespace: 'Test',
                        },
                        Period: 300,
                        Stat: 'Average',
                    },
                    ReturnData: false,
                },
            ]);
            test.done();
        },
        'can nest MathExpressions in an alarm'(test) {
            // GIVEN
            new lib_1.Alarm(stack, 'Alarm', {
                threshold: 1,
                evaluationPeriods: 1,
                metric: new lib_1.MathExpression({
                    expression: 'a + e',
                    usingMetrics: {
                        a,
                        e: new lib_1.MathExpression({
                            expression: 'b + c',
                            usingMetrics: { b, c },
                        }),
                    },
                }),
            });
            // THEN
            alarmMetricsAre([
                {
                    Expression: 'a + e',
                    Id: 'expr_1',
                },
                {
                    Id: 'a',
                    MetricStat: {
                        Metric: {
                            MetricName: 'ACount',
                            Namespace: 'Test',
                        },
                        Period: 300,
                        Stat: 'Average',
                    },
                    ReturnData: false,
                },
                {
                    Expression: 'b + c',
                    Id: 'e',
                    ReturnData: false,
                },
                {
                    Id: 'b',
                    MetricStat: {
                        Metric: {
                            MetricName: 'BCount',
                            Namespace: 'Test',
                        },
                        Period: 300,
                        Stat: 'Average',
                    },
                    ReturnData: false,
                },
                {
                    Id: 'c',
                    MetricStat: {
                        Metric: {
                            MetricName: 'CCount',
                            Namespace: 'Test',
                        },
                        Period: 300,
                        Stat: 'Average',
                    },
                    ReturnData: false,
                },
            ]);
            test.done();
        },
        'MathExpression controls period of metrics transitively used in it with alarms'(test) {
            // GIVEN
            new lib_1.Alarm(stack, 'Alarm', {
                threshold: 1,
                evaluationPeriods: 1,
                metric: new lib_1.MathExpression({
                    expression: 'a + e',
                    usingMetrics: {
                        a,
                        e: new lib_1.MathExpression({
                            expression: 'b + c',
                            usingMetrics: { b, c },
                            period: core_1.Duration.minutes(1),
                        }),
                    },
                    period: core_1.Duration.seconds(30),
                }),
            });
            // THEN
            alarmMetricsAre([
                {
                    Expression: 'a + e',
                    Id: 'expr_1',
                },
                {
                    Id: 'a',
                    MetricStat: {
                        Metric: {
                            MetricName: 'ACount',
                            Namespace: 'Test',
                        },
                        Period: 30,
                        Stat: 'Average',
                    },
                    ReturnData: false,
                },
                {
                    Expression: 'b + c',
                    Id: 'e',
                    ReturnData: false,
                },
                {
                    Id: 'b',
                    MetricStat: {
                        Metric: {
                            MetricName: 'BCount',
                            Namespace: 'Test',
                        },
                        Period: 30,
                        Stat: 'Average',
                    },
                    ReturnData: false,
                },
                {
                    Id: 'c',
                    MetricStat: {
                        Metric: {
                            MetricName: 'CCount',
                            Namespace: 'Test',
                        },
                        Period: 30,
                        Stat: 'Average',
                    },
                    ReturnData: false,
                },
            ]);
            test.done();
        },
        'MathExpression without inner metrics emits its own period'(test) {
            // WHEN
            new lib_1.Alarm(stack, 'Alarm', {
                threshold: 1,
                evaluationPeriods: 1,
                metric: new lib_1.MathExpression({
                    expression: 'INSIGHT_RULE_METRIC("SomeId", UniqueContributors)',
                    usingMetrics: {},
                }),
            });
            // THEN
            alarmMetricsAre([
                {
                    Expression: 'INSIGHT_RULE_METRIC("SomeId", UniqueContributors)',
                    Id: 'expr_1',
                    Period: 300,
                },
            ]);
            test.done();
        },
        'annotation for a mathexpression alarm is calculated based upon constituent metrics'(test) {
            // GIVEN
            const alarm = new lib_1.Alarm(stack, 'Alarm', {
                threshold: 1,
                evaluationPeriods: 1,
                metric: new lib_1.MathExpression({
                    period: core_1.Duration.minutes(10),
                    expression: 'a + b',
                    usingMetrics: { a, b: b.with({ period: core_1.Duration.minutes(20) }) },
                }),
            });
            // WHEN
            const alarmLabel = alarm.toAnnotation().label;
            // THEN
            test.equals(alarmLabel, 'a + b >= 1 for 1 datapoints within 10 minutes');
            test.done();
        },
        'can use percentiles in expression metrics in alarms'(test) {
            // GIVEN
            new lib_1.Alarm(stack, 'Alarm', {
                threshold: 1,
                evaluationPeriods: 1,
                metric: new lib_1.MathExpression({
                    expression: 'a + b99',
                    usingMetrics: { a, b99 },
                }),
            });
            // THEN
            alarmMetricsAre([
                {
                    Expression: 'a + b99',
                    Id: 'expr_1',
                },
                {
                    Id: 'a',
                    MetricStat: {
                        Metric: {
                            MetricName: 'ACount',
                            Namespace: 'Test',
                        },
                        Period: 300,
                        Stat: 'Average',
                    },
                    ReturnData: false,
                },
                {
                    Id: 'b99',
                    MetricStat: {
                        Metric: {
                            MetricName: 'BCount',
                            Namespace: 'Test',
                        },
                        Period: 300,
                        Stat: 'p99',
                    },
                    ReturnData: false,
                },
            ]);
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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