"use strict";
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
/**
 * Returns a property predicate that checks that the given Dashboard has the indicated widgets
 */
function thatHasWidgets(widgets) {
    return (props) => {
        try {
            const actualWidgets = JSON.parse(props.DashboardBody).widgets;
            return assert_1.isSuperObject(actualWidgets, widgets);
        }
        catch (e) {
            // eslint-disable-next-line no-console
            console.error('Error parsing', props);
            throw e;
        }
    };
}
module.exports = {
    'widgets in different adds are laid out underneath each other'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const dashboard = new lib_1.Dashboard(stack, 'Dash');
        // WHEN
        dashboard.addWidgets(new lib_1.TextWidget({
            width: 10,
            height: 2,
            markdown: 'first',
        }));
        dashboard.addWidgets(new lib_1.TextWidget({
            width: 1,
            height: 4,
            markdown: 'second',
        }));
        dashboard.addWidgets(new lib_1.TextWidget({
            width: 4,
            height: 1,
            markdown: 'third',
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudWatch::Dashboard', thatHasWidgets([
            { type: 'text', width: 10, height: 2, x: 0, y: 0, properties: { markdown: 'first' } },
            { type: 'text', width: 1, height: 4, x: 0, y: 2, properties: { markdown: 'second' } },
            { type: 'text', width: 4, height: 1, x: 0, y: 6, properties: { markdown: 'third' } },
        ])));
        test.done();
    },
    'widgets in same add are laid out next to each other'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const dashboard = new lib_1.Dashboard(stack, 'Dash');
        // WHEN
        dashboard.addWidgets(new lib_1.TextWidget({
            width: 10,
            height: 2,
            markdown: 'first',
        }), new lib_1.TextWidget({
            width: 1,
            height: 4,
            markdown: 'second',
        }), new lib_1.TextWidget({
            width: 4,
            height: 1,
            markdown: 'third',
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudWatch::Dashboard', thatHasWidgets([
            { type: 'text', width: 10, height: 2, x: 0, y: 0, properties: { markdown: 'first' } },
            { type: 'text', width: 1, height: 4, x: 10, y: 0, properties: { markdown: 'second' } },
            { type: 'text', width: 4, height: 1, x: 11, y: 0, properties: { markdown: 'third' } },
        ])));
        test.done();
    },
    'tokens in widgets are retained'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const dashboard = new lib_1.Dashboard(stack, 'Dash');
        // WHEN
        dashboard.addWidgets(new lib_1.GraphWidget({ width: 1, height: 1 }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudWatch::Dashboard', {
            DashboardBody: {
                'Fn::Join': ['', [
                        '{"widgets":[{"type":"metric","width":1,"height":1,"x":0,"y":0,"properties":{"view":"timeSeries","region":"',
                        { Ref: 'AWS::Region' },
                        '","yAxis":{}}}]}',
                    ]],
            },
        }));
        test.done();
    },
    'dashboard body includes non-widget fields'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const dashboard = new lib_1.Dashboard(stack, 'Dash', {
            start: '-9H',
            end: '2018-12-17T06:00:00.000Z',
            periodOverride: lib_1.PeriodOverride.INHERIT,
        });
        // WHEN
        dashboard.addWidgets(new lib_1.GraphWidget({ width: 1, height: 1 }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudWatch::Dashboard', {
            DashboardBody: {
                'Fn::Join': ['', [
                        '{"start":"-9H","end":"2018-12-17T06:00:00.000Z","periodOverride":"inherit",\
"widgets":[{"type":"metric","width":1,"height":1,"x":0,"y":0,"properties":{"view":"timeSeries","region":"',
                        { Ref: 'AWS::Region' },
                        '","yAxis":{}}}]}',
                    ]],
            },
        }));
        test.done();
    },
    'DashboardName is set when provided'(test) {
        // GIVEN
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'MyStack');
        // WHEN
        new lib_1.Dashboard(stack, 'MyDashboard', {
            dashboardName: 'MyCustomDashboardName',
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudWatch::Dashboard', {
            DashboardName: 'MyCustomDashboardName',
        }));
        test.done();
    },
    'DashboardName is not generated if not provided'(test) {
        // GIVEN
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'MyStack');
        // WHEN
        new lib_1.Dashboard(stack, 'MyDashboard');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudWatch::Dashboard', {}));
        test.done();
    },
    'throws if DashboardName is not valid'(test) {
        // GIVEN
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'MyStack');
        // WHEN
        const toThrow = () => {
            new lib_1.Dashboard(stack, 'MyDashboard', {
                dashboardName: 'My Invalid Dashboard Name',
            });
        };
        // THEN
        test.throws(() => toThrow(), /field dashboardName contains invalid characters/);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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