"use strict";
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const a = new lib_1.Metric({ namespace: 'Test', metricName: 'ACount' });
let stack1;
let stack2;
function graphMetricsAre(test, stack, w, metrics) {
    test.deepEqual(stack.resolve(w.toJson()), [{
            type: 'metric',
            width: 6,
            height: 6,
            properties: {
                view: 'timeSeries',
                region: { Ref: 'AWS::Region' },
                metrics,
                yAxis: {},
            },
        }]);
}
module.exports = {
    'setUp'(cb) {
        stack1 = new core_1.Stack(undefined, undefined, { env: { region: 'pluto', account: '1234' } });
        stack2 = new core_1.Stack(undefined, undefined, { env: { region: 'mars', account: '5678' } });
        cb();
    },
    'in graphs': {
        'metric attached to stack1 will not render region and account in stack1'(test) {
            // GIVEN
            const graph = new lib_1.GraphWidget({
                left: [
                    a.attachTo(stack1),
                ],
            });
            // THEN
            graphMetricsAre(test, stack1, graph, [
                ['Test', 'ACount'],
            ]);
            test.done();
        },
        'metric attached to stack1 will render region and account in stack2'(test) {
            // GIVEN
            const graph = new lib_1.GraphWidget({
                left: [
                    a.attachTo(stack1),
                ],
            });
            // THEN
            graphMetricsAre(test, stack2, graph, [
                ['Test', 'ACount', { region: 'pluto', accountId: '1234' }],
            ]);
            test.done();
        },
        'metric with explicit account and region will render in environment agnostic stack'(test) {
            // GIVEN
            const graph = new lib_1.GraphWidget({
                left: [
                    a.with({ account: '1234', region: 'us-north-5' }),
                ],
            });
            // THEN
            graphMetricsAre(test, new core_1.Stack(), graph, [
                ['Test', 'ACount', { accountId: '1234', region: 'us-north-5' }],
            ]);
            test.done();
        },
        'metric attached to agnostic stack will not render in agnostic stack'(test) {
            // GIVEN
            const graph = new lib_1.GraphWidget({
                left: [
                    a.attachTo(new core_1.Stack()),
                ],
            });
            // THEN
            graphMetricsAre(test, new core_1.Stack(), graph, [
                ['Test', 'ACount'],
            ]);
            test.done();
        },
    },
    'in alarms': {
        'metric attached to stack1 will not render region and account in stack1'(test) {
            // GIVEN
            new lib_1.Alarm(stack1, 'Alarm', {
                threshold: 1,
                evaluationPeriods: 1,
                metric: a.attachTo(stack1),
            });
            // THEN
            assert_1.expect(stack1).to(assert_1.haveResourceLike('AWS::CloudWatch::Alarm', {
                MetricName: 'ACount',
                Namespace: 'Test',
                Period: 300,
            }));
            test.done();
        },
        'metric attached to stack1 will throw in stack2'(test) {
            // Cross-region/cross-account metrics are supported in Dashboards but not in Alarms
            // GIVEN
            test.throws(() => {
                new lib_1.Alarm(stack2, 'Alarm', {
                    threshold: 1,
                    evaluationPeriods: 1,
                    metric: a.attachTo(stack1),
                });
            }, /Cannot create an Alarm in region 'mars' based on metric 'ACount' in 'pluto'/);
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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