"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Spacer = exports.Column = exports.Row = void 0;
const widget_1 = require("./widget");
// This file contains widgets that exist for layout purposes
/**
 * A widget that contains other widgets in a horizontal row
 *
 * Widgets will be laid out next to each other
 */
class Row {
    constructor(...widgets) {
        /**
         * Relative position of each widget inside this row
         */
        this.offsets = [];
        this.widgets = widgets;
        this.width = 0;
        this.height = 0;
        let x = 0;
        let y = 0;
        for (const widget of widgets) {
            // See if we need to horizontally wrap to add this widget
            if (x + widget.width > widget_1.GRID_WIDTH) {
                y = this.height;
                x = 0;
            }
            this.offsets.push({ x, y });
            this.width = Math.max(this.width, x + widget.width);
            this.height = Math.max(this.height, y + widget.height);
            x += widget.width;
        }
    }
    position(x, y) {
        for (let i = 0; i < this.widgets.length; i++) {
            this.widgets[i].position(x + this.offsets[i].x, y + this.offsets[i].y);
        }
    }
    toJson() {
        const ret = [];
        for (const widget of this.widgets) {
            ret.push(...widget.toJson());
        }
        return ret;
    }
}
exports.Row = Row;
/**
 * A widget that contains other widgets in a vertical column
 *
 * Widgets will be laid out next to each other
 */
class Column {
    constructor(...widgets) {
        this.widgets = widgets;
        // There's no vertical wrapping so this one's a lot easier
        this.width = Math.max(...this.widgets.map(w => w.width));
        this.height = sum(...this.widgets.map(w => w.height));
    }
    position(x, y) {
        let widgetY = y;
        for (const widget of this.widgets) {
            widget.position(x, widgetY);
            widgetY += widget.height;
        }
    }
    toJson() {
        const ret = [];
        for (const widget of this.widgets) {
            ret.push(...widget.toJson());
        }
        return ret;
    }
}
exports.Column = Column;
/**
 * A widget that doesn't display anything but takes up space
 */
class Spacer {
    constructor(props = {}) {
        this.width = props.width || 1;
        this.height = props.height || 1;
    }
    position(_x, _y) {
        // Don't need to do anything, not a physical widget
    }
    toJson() {
        return [];
    }
}
exports.Spacer = Spacer;
/**
 * Return the sum of a list of numbers
 */
function sum(...xs) {
    let ret = 0;
    for (const x of xs) {
        ret += x;
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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