"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AlarmRule = exports.AlarmState = void 0;
/**
 * Enumeration indicates state of Alarm used in building Alarm Rule.
 */
var AlarmState;
(function (AlarmState) {
    /**
     * State indicates resource is in ALARM
     */
    AlarmState["ALARM"] = "ALARM";
    /**
     * State indicates resource is not in ALARM
     */
    AlarmState["OK"] = "OK";
    /**
     * State indicates there is not enough data to determine is resource is in ALARM
     */
    AlarmState["INSUFFICIENT_DATA"] = "INSUFFICIENT_DATA";
})(AlarmState = exports.AlarmState || (exports.AlarmState = {}));
/**
 * Enumeration of supported Composite Alarms operators.
 */
var Operator;
(function (Operator) {
    Operator["AND"] = "AND";
    Operator["OR"] = "OR";
    Operator["NOT"] = "NOT";
})(Operator || (Operator = {}));
/**
 * Class with static functions to build AlarmRule for Composite Alarms.
 */
class AlarmRule {
    /**
     * function to join all provided AlarmRules with AND operator.
     *
     * @param operands IAlarmRules to be joined with AND operator.
     */
    static allOf(...operands) {
        return this.concat(Operator.AND, ...operands);
    }
    /**
     * function to join all provided AlarmRules with OR operator.
     *
     * @param operands IAlarmRules to be joined with OR operator.
     */
    static anyOf(...operands) {
        return this.concat(Operator.OR, ...operands);
    }
    /**
     * function to wrap provided AlarmRule in NOT operator.
     *
     * @param operand IAlarmRule to be wrapped in NOT operator.
     */
    static not(operand) {
        return new class {
            renderAlarmRule() {
                return `(NOT (${operand.renderAlarmRule()}))`;
            }
        };
    }
    /**
     * function to build TRUE/FALSE intent for Rule Expression.
     *
     * @param value boolean value to be used in rule expression.
     */
    static fromBoolean(value) {
        return new class {
            renderAlarmRule() {
                return `${String(value).toUpperCase()}`;
            }
        };
    }
    /**
     * function to build Rule Expression for given IAlarm and AlarmState.
     *
     * @param alarm IAlarm to be used in Rule Expression.
     * @param alarmState AlarmState to be used in Rule Expression.
     */
    static fromAlarm(alarm, alarmState) {
        return new class {
            renderAlarmRule() {
                return `${alarmState}(${alarm.alarmArn})`;
            }
        };
    }
    /**
     * function to build Rule Expression for given Alarm Rule string.
     *
     * @param alarmRule string to be used in Rule Expression.
     */
    static fromString(alarmRule) {
        return new class {
            renderAlarmRule() {
                return alarmRule;
            }
        };
    }
    static concat(operator, ...operands) {
        return new class {
            renderAlarmRule() {
                const expression = operands
                    .map(operand => `${operand.renderAlarmRule()}`)
                    .join(` ${operator} `);
                return `(${expression})`;
            }
        };
    }
}
exports.AlarmRule = AlarmRule;
//# sourceMappingURL=data:application/json;base64,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