"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PrivateSubnet = exports.PublicSubnet = exports.RouterType = exports.Subnet = exports.Vpc = exports.DefaultInstanceTenancy = exports.SubnetType = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const constructs_1 = require("constructs");
const client_vpn_endpoint_1 = require("./client-vpn-endpoint");
const ec2_generated_1 = require("./ec2.generated");
const nat_1 = require("./nat");
const network_acl_1 = require("./network-acl");
const network_util_1 = require("./network-util");
const subnet_1 = require("./subnet");
const util_1 = require("./util");
const vpc_endpoint_1 = require("./vpc-endpoint");
const vpc_flow_logs_1 = require("./vpc-flow-logs");
const vpn_1 = require("./vpn");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("@aws-cdk/core");
const VPC_SUBNET_SYMBOL = Symbol.for('@aws-cdk/aws-ec2.VpcSubnet');
/**
 * The type of Subnet
 */
var SubnetType;
(function (SubnetType) {
    /**
     * Isolated Subnets do not route traffic to the Internet (in this VPC),
     * and as such, do not require NAT gateways.
     *
     * Isolated subnets can only connect to or be connected to from other
     * instances in the same VPC. A default VPC configuration will not include
     * isolated subnets.
     *
     * This can be good for subnets with RDS or Elasticache instances,
     * or which route Internet traffic through a peer VPC.
     *
     * @deprecated use `SubnetType.PRIVATE_ISOLATED`
     */
    SubnetType["ISOLATED"] = "Deprecated_Isolated";
    /**
     * Isolated Subnets do not route traffic to the Internet (in this VPC),
     * and as such, do not require NAT gateways.
     *
     * Isolated subnets can only connect to or be connected to from other
     * instances in the same VPC. A default VPC configuration will not include
     * isolated subnets.
     *
     * This can be good for subnets with RDS or Elasticache instances,
     * or which route Internet traffic through a peer VPC.
     */
    SubnetType["PRIVATE_ISOLATED"] = "Isolated";
    /**
     * Subnet that routes to the internet, but not vice versa.
     *
     * Instances in a private subnet can connect to the Internet, but will not
     * allow connections to be initiated from the Internet. NAT Gateway(s) are
     * required with this subnet type to route the Internet traffic through.
     * If a NAT Gateway is not required or desired, use `SubnetType.PRIVATE_ISOLATED` instead.
     *
     * By default, a NAT gateway is created in every public subnet for maximum availability.
     * Be aware that you will be charged for NAT gateways.
     *
     * Normally a Private subnet will use a NAT gateway in the same AZ, but
     * if `natGateways` is used to reduce the number of NAT gateways, a NAT
     * gateway from another AZ will be used instead.
     *
     * @deprecated use `PRIVATE_WITH_NAT`
     */
    SubnetType["PRIVATE"] = "Deprecated_Private";
    /**
     * Subnet that routes to the internet (via a NAT gateway), but not vice versa.
     *
     * Instances in a private subnet can connect to the Internet, but will not
     * allow connections to be initiated from the Internet. NAT Gateway(s) are
     * required with this subnet type to route the Internet traffic through.
     * If a NAT Gateway is not required or desired, use `SubnetType.PRIVATE_ISOLATED` instead.
     *
     * By default, a NAT gateway is created in every public subnet for maximum availability.
     * Be aware that you will be charged for NAT gateways.
     *
     * Normally a Private subnet will use a NAT gateway in the same AZ, but
     * if `natGateways` is used to reduce the number of NAT gateways, a NAT
     * gateway from another AZ will be used instead.
     */
    SubnetType["PRIVATE_WITH_NAT"] = "Private";
    /**
     * Subnet connected to the Internet
     *
     * Instances in a Public subnet can connect to the Internet and can be
     * connected to from the Internet as long as they are launched with public
     * IPs (controlled on the AutoScalingGroup or other constructs that launch
     * instances).
     *
     * Public subnets route outbound traffic via an Internet Gateway.
     */
    SubnetType["PUBLIC"] = "Public";
})(SubnetType = exports.SubnetType || (exports.SubnetType = {}));
/**
 * A new or imported VPC
 */
class VpcBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        /**
         * Dependencies for NAT connectivity
         *
         * @deprecated - This value is no longer used.
         */
        this.natDependencies = new Array();
        /**
         * If this is set to true, don't error out on trying to select subnets
         */
        this.incompleteSubnetDefinition = false;
    }
    /**
     * Returns IDs of selected subnets
     */
    selectSubnets(selection = {}) {
        const subnets = this.selectSubnetObjects(selection);
        const pubs = new Set(this.publicSubnets);
        return {
            subnetIds: subnets.map(s => s.subnetId),
            get availabilityZones() { return subnets.map(s => s.availabilityZone); },
            internetConnectivityEstablished: tap(new CompositeDependable(), d => subnets.forEach(s => d.add(s.internetConnectivityEstablished))),
            subnets,
            hasPublic: subnets.some(s => pubs.has(s)),
            isPendingLookup: this.incompleteSubnetDefinition,
        };
    }
    /**
     * Adds a VPN Gateway to this VPC
     */
    enableVpnGateway(options) {
        if (this.vpnGatewayId) {
            throw new Error('The VPN Gateway has already been enabled.');
        }
        const vpnGateway = new vpn_1.VpnGateway(this, 'VpnGateway', {
            amazonSideAsn: options.amazonSideAsn,
            type: vpn_1.VpnConnectionType.IPSEC_1,
        });
        this._vpnGatewayId = vpnGateway.gatewayId;
        const attachment = new ec2_generated_1.CfnVPCGatewayAttachment(this, 'VPCVPNGW', {
            vpcId: this.vpcId,
            vpnGatewayId: this._vpnGatewayId,
        });
        // Propagate routes on route tables associated with the right subnets
        const vpnRoutePropagation = options.vpnRoutePropagation ?? [{}];
        const routeTableIds = util_1.allRouteTableIds(util_1.flatten(vpnRoutePropagation.map(s => this.selectSubnets(s).subnets)));
        if (routeTableIds.length === 0) {
            core_1.Annotations.of(this).addError(`enableVpnGateway: no subnets matching selection: '${JSON.stringify(vpnRoutePropagation)}'. Select other subnets to add routes to.`);
        }
        const routePropagation = new ec2_generated_1.CfnVPNGatewayRoutePropagation(this, 'RoutePropagation', {
            routeTableIds,
            vpnGatewayId: this._vpnGatewayId,
        });
        // The AWS::EC2::VPNGatewayRoutePropagation resource cannot use the VPN gateway
        // until it has successfully attached to the VPC.
        // See https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-vpn-gatewayrouteprop.html
        routePropagation.node.addDependency(attachment);
    }
    /**
     * Adds a new VPN connection to this VPC
     */
    addVpnConnection(id, options) {
        return new vpn_1.VpnConnection(this, id, {
            vpc: this,
            ...options,
        });
    }
    /**
     * Adds a new client VPN endpoint to this VPC
     */
    addClientVpnEndpoint(id, options) {
        return new client_vpn_endpoint_1.ClientVpnEndpoint(this, id, {
            ...options,
            vpc: this,
        });
    }
    /**
     * Adds a new interface endpoint to this VPC
     */
    addInterfaceEndpoint(id, options) {
        return new vpc_endpoint_1.InterfaceVpcEndpoint(this, id, {
            vpc: this,
            ...options,
        });
    }
    /**
     * Adds a new gateway endpoint to this VPC
     */
    addGatewayEndpoint(id, options) {
        return new vpc_endpoint_1.GatewayVpcEndpoint(this, id, {
            vpc: this,
            ...options,
        });
    }
    /**
     * Adds a new flow log to this VPC
     */
    addFlowLog(id, options) {
        return new vpc_flow_logs_1.FlowLog(this, id, {
            resourceType: vpc_flow_logs_1.FlowLogResourceType.fromVpc(this),
            ...options,
        });
    }
    /**
     * Returns the id of the VPN Gateway (if enabled)
     */
    get vpnGatewayId() {
        return this._vpnGatewayId;
    }
    /**
     * Return the subnets appropriate for the placement strategy
     */
    selectSubnetObjects(selection = {}) {
        selection = this.reifySelectionDefaults(selection);
        if (selection.subnets !== undefined) {
            return selection.subnets;
        }
        let subnets;
        if (selection.subnetGroupName !== undefined) { // Select by name
            subnets = this.selectSubnetObjectsByName(selection.subnetGroupName);
        }
        else { // Or specify by type
            const type = selection.subnetType || SubnetType.PRIVATE_WITH_NAT;
            subnets = this.selectSubnetObjectsByType(type);
        }
        // Apply all the filters
        subnets = this.applySubnetFilters(subnets, selection.subnetFilters ?? []);
        return subnets;
    }
    applySubnetFilters(subnets, filters) {
        let filtered = subnets;
        // Apply each filter in sequence
        for (const filter of filters) {
            filtered = filter.selectSubnets(filtered);
        }
        return filtered;
    }
    selectSubnetObjectsByName(groupName) {
        const allSubnets = [...this.publicSubnets, ...this.privateSubnets, ...this.isolatedSubnets];
        const subnets = allSubnets.filter(s => util_1.subnetGroupNameFromConstructId(s) === groupName);
        if (subnets.length === 0 && !this.incompleteSubnetDefinition) {
            const names = Array.from(new Set(allSubnets.map(util_1.subnetGroupNameFromConstructId)));
            throw new Error(`There are no subnet groups with name '${groupName}' in this VPC. Available names: ${names}`);
        }
        return subnets;
    }
    selectSubnetObjectsByType(subnetType) {
        const allSubnets = {
            [SubnetType.PRIVATE_ISOLATED]: this.isolatedSubnets,
            [SubnetType.ISOLATED]: this.isolatedSubnets,
            [SubnetType.PRIVATE_WITH_NAT]: this.privateSubnets,
            [SubnetType.PRIVATE]: this.privateSubnets,
            [SubnetType.PUBLIC]: this.publicSubnets,
        };
        const subnets = allSubnets[subnetType];
        // Force merge conflict here with https://github.com/aws/aws-cdk/pull/4089
        // see ImportedVpc
        if (subnets.length === 0 && !this.incompleteSubnetDefinition) {
            const availableTypes = Object.entries(allSubnets).filter(([_, subs]) => subs.length > 0).map(([typeName, _]) => typeName);
            throw new Error(`There are no '${subnetType}' subnet groups in this VPC. Available types: ${availableTypes}`);
        }
        return subnets;
    }
    /**
     * Validate the fields in a SubnetSelection object, and reify defaults if necessary
     *
     * In case of default selection, select the first type of PRIVATE, ISOLATED,
     * PUBLIC (in that order) that has any subnets.
     */
    reifySelectionDefaults(placement) {
        if (placement.subnetName !== undefined) {
            if (placement.subnetGroupName !== undefined) {
                throw new Error('Please use only \'subnetGroupName\' (\'subnetName\' is deprecated and has the same behavior)');
            }
            else {
                core_1.Annotations.of(this).addWarning('Usage of \'subnetName\' in SubnetSelection is deprecated, use \'subnetGroupName\' instead');
            }
            placement = { ...placement, subnetGroupName: placement.subnetName };
        }
        const exclusiveSelections = ['subnets', 'subnetType', 'subnetGroupName'];
        const providedSelections = exclusiveSelections.filter(key => placement[key] !== undefined);
        if (providedSelections.length > 1) {
            throw new Error(`Only one of '${providedSelections}' can be supplied to subnet selection.`);
        }
        if (placement.subnetType === undefined && placement.subnetGroupName === undefined && placement.subnets === undefined) {
            // Return default subnet type based on subnets that actually exist
            let subnetType = this.privateSubnets.length
                ? SubnetType.PRIVATE_WITH_NAT : this.isolatedSubnets.length ? SubnetType.PRIVATE_ISOLATED : SubnetType.PUBLIC;
            placement = { ...placement, subnetType: subnetType };
        }
        // Establish which subnet filters are going to be used
        let subnetFilters = placement.subnetFilters ?? [];
        // Backwards compatibility with existing `availabilityZones` and `onePerAz` functionality
        if (placement.availabilityZones !== undefined) { // Filter by AZs, if specified
            subnetFilters.push(subnet_1.SubnetFilter.availabilityZones(placement.availabilityZones));
        }
        if (!!placement.onePerAz) { // Ensure one per AZ if specified
            subnetFilters.push(subnet_1.SubnetFilter.onePerAz());
        }
        // Overwrite the provided placement filters and remove the availabilityZones and onePerAz properties
        placement = { ...placement, subnetFilters: subnetFilters, availabilityZones: undefined, onePerAz: undefined };
        const { availabilityZones, onePerAz, ...rest } = placement;
        return rest;
    }
}
/**
 * Name tag constant
 */
const NAME_TAG = 'Name';
/**
 * The default tenancy of instances launched into the VPC.
 */
var DefaultInstanceTenancy;
(function (DefaultInstanceTenancy) {
    /**
     * Instances can be launched with any tenancy.
     */
    DefaultInstanceTenancy["DEFAULT"] = "default";
    /**
     * Any instance launched into the VPC automatically has dedicated tenancy, unless you launch it with the default tenancy.
     */
    DefaultInstanceTenancy["DEDICATED"] = "dedicated";
})(DefaultInstanceTenancy = exports.DefaultInstanceTenancy || (exports.DefaultInstanceTenancy = {}));
/**
 * Define an AWS Virtual Private Cloud
 *
 * See the package-level documentation of this package for an overview
 * of the various dimensions in which you can configure your VPC.
 *
 * For example:
 *
 * ```ts
 * const vpc = new ec2.Vpc(this, 'TheVPC', {
 *   cidr: "10.0.0.0/16"
 * })
 *
 * // Iterate the private subnets
 * const selection = vpc.selectSubnets({
 *   subnetType: ec2.SubnetType.PRIVATE_WITH_NAT
 * });
 *
 * for (const subnet of selection.subnets) {
 *   // ...
 * }
 * ```
 *
 * @resource AWS::EC2::VPC
 */
class Vpc extends VpcBase {
    /**
     * Vpc creates a VPC that spans a whole region.
     * It will automatically divide the provided VPC CIDR range, and create public and private subnets per Availability Zone.
     * Network routing for the public subnets will be configured to allow outbound access directly via an Internet Gateway.
     * Network routing for the private subnets will be configured to allow outbound access via a set of resilient NAT Gateways (one per AZ).
     */
    constructor(scope, id, props = {}) {
        super(scope, id);
        /**
         * List of public subnets in this VPC
         */
        this.publicSubnets = [];
        /**
         * List of private subnets in this VPC
         */
        this.privateSubnets = [];
        /**
         * List of isolated subnets in this VPC
         */
        this.isolatedSubnets = [];
        /**
         * Subnet configurations for this VPC
         */
        this.subnetConfiguration = [];
        this._internetConnectivityEstablished = new core_1.ConcreteDependable();
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_VpcProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Vpc);
            }
            throw error;
        }
        const stack = core_1.Stack.of(this);
        // Can't have enabledDnsHostnames without enableDnsSupport
        if (props.enableDnsHostnames && !props.enableDnsSupport) {
            throw new Error('To use DNS Hostnames, DNS Support must be enabled, however, it was explicitly disabled.');
        }
        const cidrBlock = ifUndefined(props.cidr, Vpc.DEFAULT_CIDR_RANGE);
        if (core_1.Token.isUnresolved(cidrBlock)) {
            throw new Error('\'cidr\' property must be a concrete CIDR string, got a Token (we need to parse it for automatic subdivision)');
        }
        this.networkBuilder = new network_util_1.NetworkBuilder(cidrBlock);
        this.dnsHostnamesEnabled = props.enableDnsHostnames == null ? true : props.enableDnsHostnames;
        this.dnsSupportEnabled = props.enableDnsSupport == null ? true : props.enableDnsSupport;
        const instanceTenancy = props.defaultInstanceTenancy || 'default';
        this.internetConnectivityEstablished = this._internetConnectivityEstablished;
        // Define a VPC using the provided CIDR range
        this.resource = new ec2_generated_1.CfnVPC(this, 'Resource', {
            cidrBlock,
            enableDnsHostnames: this.dnsHostnamesEnabled,
            enableDnsSupport: this.dnsSupportEnabled,
            instanceTenancy,
        });
        this.vpcDefaultNetworkAcl = this.resource.attrDefaultNetworkAcl;
        this.vpcCidrBlockAssociations = this.resource.attrCidrBlockAssociations;
        this.vpcCidrBlock = this.resource.attrCidrBlock;
        this.vpcDefaultSecurityGroup = this.resource.attrDefaultSecurityGroup;
        this.vpcIpv6CidrBlocks = this.resource.attrIpv6CidrBlocks;
        core_1.Tags.of(this).add(NAME_TAG, props.vpcName || this.node.path);
        this.availabilityZones = stack.availabilityZones;
        const maxAZs = props.maxAzs ?? 3;
        this.availabilityZones = this.availabilityZones.slice(0, maxAZs);
        this.vpcId = this.resource.ref;
        this.vpcArn = core_1.Arn.format({
            service: 'ec2',
            resource: 'vpc',
            resourceName: this.vpcId,
        }, stack);
        const defaultSubnet = props.natGateways === 0 ? Vpc.DEFAULT_SUBNETS_NO_NAT : Vpc.DEFAULT_SUBNETS;
        this.subnetConfiguration = ifUndefined(props.subnetConfiguration, defaultSubnet);
        const natGatewayPlacement = props.natGatewaySubnets || { subnetType: SubnetType.PUBLIC };
        const natGatewayCount = determineNatGatewayCount(props.natGateways, this.subnetConfiguration, this.availabilityZones.length);
        // subnetConfiguration must be set before calling createSubnets
        this.createSubnets();
        const allowOutbound = this.subnetConfiguration.filter(subnet => (subnet.subnetType !== SubnetType.PRIVATE_ISOLATED && subnet.subnetType !== SubnetType.ISOLATED)).length > 0;
        // Create an Internet Gateway and attach it if necessary
        if (allowOutbound) {
            const igw = new ec2_generated_1.CfnInternetGateway(this, 'IGW', {});
            this.internetGatewayId = igw.ref;
            this._internetConnectivityEstablished.add(igw);
            const att = new ec2_generated_1.CfnVPCGatewayAttachment(this, 'VPCGW', {
                internetGatewayId: igw.ref,
                vpcId: this.resource.ref,
            });
            this.publicSubnets.forEach(publicSubnet => {
                publicSubnet.addDefaultInternetRoute(igw.ref, att);
            });
            // if gateways are needed create them
            if (natGatewayCount > 0) {
                const provider = props.natGatewayProvider || nat_1.NatProvider.gateway();
                this.createNatGateways(provider, natGatewayCount, natGatewayPlacement);
            }
        }
        if (props.vpnGateway && this.publicSubnets.length === 0 && this.privateSubnets.length === 0 && this.isolatedSubnets.length === 0) {
            throw new Error('Can not enable the VPN gateway while the VPC has no subnets at all');
        }
        if ((props.vpnConnections || props.vpnGatewayAsn) && props.vpnGateway === false) {
            throw new Error('Cannot specify `vpnConnections` or `vpnGatewayAsn` when `vpnGateway` is set to false.');
        }
        if (props.vpnGateway || props.vpnConnections || props.vpnGatewayAsn) {
            this.enableVpnGateway({
                amazonSideAsn: props.vpnGatewayAsn,
                type: vpn_1.VpnConnectionType.IPSEC_1,
                vpnRoutePropagation: props.vpnRoutePropagation,
            });
            const vpnConnections = props.vpnConnections || {};
            for (const [connectionId, connection] of Object.entries(vpnConnections)) {
                this.addVpnConnection(connectionId, connection);
            }
        }
        // Allow creation of gateway endpoints on VPC instantiation as those can be
        // immediately functional without further configuration. This is not the case
        // for interface endpoints where the security group must be configured.
        if (props.gatewayEndpoints) {
            const gatewayEndpoints = props.gatewayEndpoints || {};
            for (const [endpointId, endpoint] of Object.entries(gatewayEndpoints)) {
                this.addGatewayEndpoint(endpointId, endpoint);
            }
        }
        // Add flow logs to the VPC
        if (props.flowLogs) {
            const flowLogs = props.flowLogs || {};
            for (const [flowLogId, flowLog] of Object.entries(flowLogs)) {
                this.addFlowLog(flowLogId, flowLog);
            }
        }
    }
    /**
     * Import a VPC by supplying all attributes directly
     *
     * NOTE: using `fromVpcAttributes()` with deploy-time parameters (like a `Fn.importValue()` or
     * `CfnParameter` to represent a list of subnet IDs) sometimes accidentally works. It happens
     * to work for constructs that need a list of subnets (like `AutoScalingGroup` and `eks.Cluster`)
     * but it does not work for constructs that need individual subnets (like
     * `Instance`). See https://github.com/aws/aws-cdk/issues/4118 for more
     * information.
     *
     * Prefer to use `Vpc.fromLookup()` instead.
     */
    static fromVpcAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_VpcAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromVpcAttributes);
            }
            throw error;
        }
        return new ImportedVpc(scope, id, attrs, false);
    }
    /**
     * Import an existing VPC from by querying the AWS environment this stack is deployed to.
     *
     * This function only needs to be used to use VPCs not defined in your CDK
     * application. If you are looking to share a VPC between stacks, you can
     * pass the `Vpc` object between stacks and use it as normal.
     *
     * Calling this method will lead to a lookup when the CDK CLI is executed.
     * You can therefore not use any values that will only be available at
     * CloudFormation execution time (i.e., Tokens).
     *
     * The VPC information will be cached in `cdk.context.json` and the same VPC
     * will be used on future runs. To refresh the lookup, you will have to
     * evict the value from the cache using the `cdk context` command. See
     * https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information.
     */
    static fromLookup(scope, id, options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_VpcLookupOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromLookup);
            }
            throw error;
        }
        if (core_1.Token.isUnresolved(options.vpcId)
            || core_1.Token.isUnresolved(options.vpcName)
            || Object.values(options.tags || {}).some(core_1.Token.isUnresolved)
            || Object.keys(options.tags || {}).some(core_1.Token.isUnresolved)) {
            throw new Error('All arguments to Vpc.fromLookup() must be concrete (no Tokens)');
        }
        const filter = makeTagFilter(options.tags);
        // We give special treatment to some tags
        if (options.vpcId) {
            filter['vpc-id'] = options.vpcId;
        }
        if (options.vpcName) {
            filter['tag:Name'] = options.vpcName;
        }
        if (options.isDefault !== undefined) {
            filter.isDefault = options.isDefault ? 'true' : 'false';
        }
        const overrides = {};
        if (options.region) {
            overrides.region = options.region;
        }
        const attributes = core_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.VPC_PROVIDER,
            props: {
                ...overrides,
                filter,
                returnAsymmetricSubnets: true,
                subnetGroupNameTag: options.subnetGroupNameTag,
            },
            dummyValue: undefined,
        }).value;
        return new LookedUpVpc(scope, id, attributes || DUMMY_VPC_PROPS, attributes === undefined);
        /**
         * Prefixes all keys in the argument with `tag:`.`
         */
        function makeTagFilter(tags) {
            const result = {};
            for (const [name, value] of Object.entries(tags || {})) {
                result[`tag:${name}`] = value;
            }
            return result;
        }
    }
    /**
     * Adds a new S3 gateway endpoint to this VPC
     *
     * @deprecated use `addGatewayEndpoint()` instead
     */
    addS3Endpoint(id, subnets) {
        try {
            jsiiDeprecationWarnings.print("@aws-cdk/aws-ec2.Vpc#addS3Endpoint", "use `addGatewayEndpoint()` instead");
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addS3Endpoint);
            }
            throw error;
        }
        return new vpc_endpoint_1.GatewayVpcEndpoint(this, id, {
            service: vpc_endpoint_1.GatewayVpcEndpointAwsService.S3,
            vpc: this,
            subnets,
        });
    }
    /**
     * Adds a new DynamoDB gateway endpoint to this VPC
     *
     * @deprecated use `addGatewayEndpoint()` instead
     */
    addDynamoDbEndpoint(id, subnets) {
        try {
            jsiiDeprecationWarnings.print("@aws-cdk/aws-ec2.Vpc#addDynamoDbEndpoint", "use `addGatewayEndpoint()` instead");
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addDynamoDbEndpoint);
            }
            throw error;
        }
        return new vpc_endpoint_1.GatewayVpcEndpoint(this, id, {
            service: vpc_endpoint_1.GatewayVpcEndpointAwsService.DYNAMODB,
            vpc: this,
            subnets,
        });
    }
    createNatGateways(provider, natCount, placement) {
        const natSubnets = this.selectSubnetObjects(placement);
        for (const sub of natSubnets) {
            if (this.publicSubnets.indexOf(sub) === -1) {
                throw new Error(`natGatewayPlacement ${placement} contains non public subnet ${sub}`);
            }
        }
        provider.configureNat({
            vpc: this,
            natSubnets: natSubnets.slice(0, natCount),
            privateSubnets: this.privateSubnets,
        });
    }
    /**
     * createSubnets creates the subnets specified by the subnet configuration
     * array or creates the `DEFAULT_SUBNETS` configuration
     */
    createSubnets() {
        const remainingSpaceSubnets = [];
        for (const subnet of this.subnetConfiguration) {
            if (subnet.cidrMask === undefined) {
                remainingSpaceSubnets.push(subnet);
                continue;
            }
            this.createSubnetResources(subnet, subnet.cidrMask);
        }
        const totalRemaining = remainingSpaceSubnets.length * this.availabilityZones.length;
        const cidrMaskForRemaining = this.networkBuilder.maskForRemainingSubnets(totalRemaining);
        for (const subnet of remainingSpaceSubnets) {
            this.createSubnetResources(subnet, cidrMaskForRemaining);
        }
    }
    createSubnetResources(subnetConfig, cidrMask) {
        this.availabilityZones.forEach((zone, index) => {
            if (subnetConfig.reserved === true) {
                // For reserved subnets, just allocate ip space but do not create any resources
                this.networkBuilder.addSubnet(cidrMask);
                return;
            }
            // mapPublicIpOnLaunch true in Subnet.Public, false in Subnet.Private or Subnet.Isolated.
            let mapPublicIpOnLaunch = false;
            if (subnetConfig.subnetType !== SubnetType.PUBLIC && subnetConfig.mapPublicIpOnLaunch !== undefined) {
                throw new Error(`${subnetConfig.subnetType} subnet cannot include mapPublicIpOnLaunch parameter`);
            }
            if (subnetConfig.subnetType === SubnetType.PUBLIC) {
                mapPublicIpOnLaunch = (subnetConfig.mapPublicIpOnLaunch !== undefined)
                    ? subnetConfig.mapPublicIpOnLaunch
                    : true;
            }
            const name = util_1.subnetId(subnetConfig.name, index);
            const subnetProps = {
                availabilityZone: zone,
                vpcId: this.vpcId,
                cidrBlock: this.networkBuilder.addSubnet(cidrMask),
                mapPublicIpOnLaunch: mapPublicIpOnLaunch,
            };
            let subnet;
            switch (subnetConfig.subnetType) {
                case SubnetType.PUBLIC:
                    const publicSubnet = new PublicSubnet(this, name, subnetProps);
                    this.publicSubnets.push(publicSubnet);
                    subnet = publicSubnet;
                    break;
                case SubnetType.PRIVATE_WITH_NAT:
                case SubnetType.PRIVATE:
                    const privateSubnet = new PrivateSubnet(this, name, subnetProps);
                    this.privateSubnets.push(privateSubnet);
                    subnet = privateSubnet;
                    break;
                case SubnetType.PRIVATE_ISOLATED:
                case SubnetType.ISOLATED:
                    const isolatedSubnet = new PrivateSubnet(this, name, subnetProps);
                    this.isolatedSubnets.push(isolatedSubnet);
                    subnet = isolatedSubnet;
                    break;
                default:
                    throw new Error(`Unrecognized subnet type: ${subnetConfig.subnetType}`);
            }
            // These values will be used to recover the config upon provider import
            const includeResourceTypes = [ec2_generated_1.CfnSubnet.CFN_RESOURCE_TYPE_NAME];
            core_1.Tags.of(subnet).add(SUBNETNAME_TAG, subnetConfig.name, { includeResourceTypes });
            core_1.Tags.of(subnet).add(SUBNETTYPE_TAG, subnetTypeTagValue(subnetConfig.subnetType), { includeResourceTypes });
        });
    }
}
exports.Vpc = Vpc;
_a = JSII_RTTI_SYMBOL_1;
Vpc[_a] = { fqn: "@aws-cdk/aws-ec2.Vpc", version: "1.167.0" };
/**
 * The default CIDR range used when creating VPCs.
 * This can be overridden using VpcProps when creating a VPCNetwork resource.
 * e.g. new VpcResource(this, { cidr: '192.168.0.0./16' })
 */
Vpc.DEFAULT_CIDR_RANGE = '10.0.0.0/16';
/**
 * The default subnet configuration
 *
 * 1 Public and 1 Private subnet per AZ evenly split
 */
Vpc.DEFAULT_SUBNETS = [
    {
        subnetType: SubnetType.PUBLIC,
        name: util_1.defaultSubnetName(SubnetType.PUBLIC),
    },
    {
        subnetType: SubnetType.PRIVATE_WITH_NAT,
        name: util_1.defaultSubnetName(SubnetType.PRIVATE_WITH_NAT),
    },
];
/**
 * The default subnet configuration if natGateways specified to be 0
 *
 * 1 Public and 1 Isolated Subnet per AZ evenly split
 */
Vpc.DEFAULT_SUBNETS_NO_NAT = [
    {
        subnetType: SubnetType.PUBLIC,
        name: util_1.defaultSubnetName(SubnetType.PUBLIC),
    },
    {
        subnetType: SubnetType.PRIVATE_ISOLATED,
        name: util_1.defaultSubnetName(SubnetType.PRIVATE_ISOLATED),
    },
];
const SUBNETTYPE_TAG = 'aws-cdk:subnet-type';
const SUBNETNAME_TAG = 'aws-cdk:subnet-name';
function subnetTypeTagValue(type) {
    switch (type) {
        case SubnetType.PUBLIC: return 'Public';
        case SubnetType.PRIVATE_WITH_NAT:
        case SubnetType.PRIVATE:
            return 'Private';
        case SubnetType.PRIVATE_ISOLATED:
        case SubnetType.ISOLATED:
            return 'Isolated';
    }
}
/**
 * Represents a new VPC subnet resource
 *
 * @resource AWS::EC2::Subnet
 */
class Subnet extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * Parts of this VPC subnet
         */
        this.dependencyElements = [];
        this._internetConnectivityEstablished = new core_1.ConcreteDependable();
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_SubnetProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Subnet);
            }
            throw error;
        }
        Object.defineProperty(this, VPC_SUBNET_SYMBOL, { value: true });
        core_1.Tags.of(this).add(NAME_TAG, this.node.path);
        this.availabilityZone = props.availabilityZone;
        this.ipv4CidrBlock = props.cidrBlock;
        const subnet = new ec2_generated_1.CfnSubnet(this, 'Subnet', {
            vpcId: props.vpcId,
            cidrBlock: props.cidrBlock,
            availabilityZone: props.availabilityZone,
            mapPublicIpOnLaunch: props.mapPublicIpOnLaunch,
        });
        this.subnetId = subnet.ref;
        this.subnetVpcId = subnet.attrVpcId;
        this.subnetAvailabilityZone = subnet.attrAvailabilityZone;
        this.subnetIpv6CidrBlocks = subnet.attrIpv6CidrBlocks;
        this.subnetOutpostArn = subnet.attrOutpostArn;
        // subnet.attrNetworkAclAssociationId is the default ACL after the subnet
        // was just created. However, the ACL can be replaced at a later time.
        this._networkAcl = network_acl_1.NetworkAcl.fromNetworkAclId(this, 'Acl', subnet.attrNetworkAclAssociationId);
        this.subnetNetworkAclAssociationId = core_1.Lazy.string({ produce: () => this._networkAcl.networkAclId });
        this.node.defaultChild = subnet;
        const table = new ec2_generated_1.CfnRouteTable(this, 'RouteTable', {
            vpcId: props.vpcId,
        });
        this.routeTable = { routeTableId: table.ref };
        // Associate the public route table for this subnet, to this subnet
        new ec2_generated_1.CfnSubnetRouteTableAssociation(this, 'RouteTableAssociation', {
            subnetId: this.subnetId,
            routeTableId: table.ref,
        });
        this.internetConnectivityEstablished = this._internetConnectivityEstablished;
    }
    static isVpcSubnet(x) {
        return VPC_SUBNET_SYMBOL in x;
    }
    static fromSubnetAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_SubnetAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromSubnetAttributes);
            }
            throw error;
        }
        return new ImportedSubnet(scope, id, attrs);
    }
    /**
     * Import existing subnet from id.
     */
    // eslint-disable-next-line @typescript-eslint/no-shadow
    static fromSubnetId(scope, id, subnetId) {
        return this.fromSubnetAttributes(scope, id, { subnetId });
    }
    /**
     * Create a default route that points to a passed IGW, with a dependency
     * on the IGW's attachment to the VPC.
     *
     * @param gatewayId the logical ID (ref) of the gateway attached to your VPC
     * @param gatewayAttachment the gateway attachment construct to be added as a dependency
     */
    addDefaultInternetRoute(gatewayId, gatewayAttachment) {
        const route = new ec2_generated_1.CfnRoute(this, 'DefaultRoute', {
            routeTableId: this.routeTable.routeTableId,
            destinationCidrBlock: '0.0.0.0/0',
            gatewayId,
        });
        route.node.addDependency(gatewayAttachment);
        // Since the 'route' depends on the gateway attachment, just
        // depending on the route is enough.
        this._internetConnectivityEstablished.add(route);
    }
    /**
     * Network ACL associated with this Subnet
     *
     * Upon creation, this is the default ACL which allows all traffic, except
     * explicit DENY entries that you add.
     *
     * You can replace it with a custom ACL which denies all traffic except
     * the explicit ALLOW entries that you add by creating a `NetworkAcl`
     * object and calling `associateNetworkAcl()`.
     */
    get networkAcl() {
        return this._networkAcl;
    }
    /**
     * Adds an entry to this subnets route table that points to the passed NATGatewayId
     * @param natGatewayId The ID of the NAT gateway
     */
    addDefaultNatRoute(natGatewayId) {
        this.addRoute('DefaultRoute', {
            routerType: RouterType.NAT_GATEWAY,
            routerId: natGatewayId,
            enablesInternetConnectivity: true,
        });
    }
    /**
     * Adds an entry to this subnets route table
     */
    addRoute(id, options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_AddRouteOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addRoute);
            }
            throw error;
        }
        if (options.destinationCidrBlock && options.destinationIpv6CidrBlock) {
            throw new Error('Cannot specify both \'destinationCidrBlock\' and \'destinationIpv6CidrBlock\'');
        }
        const route = new ec2_generated_1.CfnRoute(this, id, {
            routeTableId: this.routeTable.routeTableId,
            destinationCidrBlock: options.destinationCidrBlock || (options.destinationIpv6CidrBlock === undefined ? '0.0.0.0/0' : undefined),
            destinationIpv6CidrBlock: options.destinationIpv6CidrBlock,
            [routerTypeToPropName(options.routerType)]: options.routerId,
        });
        if (options.enablesInternetConnectivity) {
            this._internetConnectivityEstablished.add(route);
        }
    }
    associateNetworkAcl(id, networkAcl) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_INetworkAcl(networkAcl);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.associateNetworkAcl);
            }
            throw error;
        }
        this._networkAcl = networkAcl;
        const scope = core_2.Construct.isConstruct(networkAcl) ? networkAcl : this;
        const other = core_2.Construct.isConstruct(networkAcl) ? this : networkAcl;
        new network_acl_1.SubnetNetworkAclAssociation(scope, id + core_1.Names.nodeUniqueId(other.node), {
            networkAcl,
            subnet: this,
        });
    }
}
exports.Subnet = Subnet;
_b = JSII_RTTI_SYMBOL_1;
Subnet[_b] = { fqn: "@aws-cdk/aws-ec2.Subnet", version: "1.167.0" };
/**
 * Type of router used in route
 */
var RouterType;
(function (RouterType) {
    /**
     * Carrier gateway
     */
    RouterType["CARRIER_GATEWAY"] = "CarrierGateway";
    /**
     * Egress-only Internet Gateway
     */
    RouterType["EGRESS_ONLY_INTERNET_GATEWAY"] = "EgressOnlyInternetGateway";
    /**
     * Internet Gateway
     */
    RouterType["GATEWAY"] = "Gateway";
    /**
     * Instance
     */
    RouterType["INSTANCE"] = "Instance";
    /**
     * Local Gateway
     */
    RouterType["LOCAL_GATEWAY"] = "LocalGateway";
    /**
     * NAT Gateway
     */
    RouterType["NAT_GATEWAY"] = "NatGateway";
    /**
     * Network Interface
     */
    RouterType["NETWORK_INTERFACE"] = "NetworkInterface";
    /**
     * Transit Gateway
     */
    RouterType["TRANSIT_GATEWAY"] = "TransitGateway";
    /**
     * VPC peering connection
     */
    RouterType["VPC_PEERING_CONNECTION"] = "VpcPeeringConnection";
    /**
     * VPC Endpoint for gateway load balancers
     */
    RouterType["VPC_ENDPOINT"] = "VpcEndpoint";
})(RouterType = exports.RouterType || (exports.RouterType = {}));
function routerTypeToPropName(routerType) {
    return ({
        [RouterType.CARRIER_GATEWAY]: 'carrierGatewayId',
        [RouterType.EGRESS_ONLY_INTERNET_GATEWAY]: 'egressOnlyInternetGatewayId',
        [RouterType.GATEWAY]: 'gatewayId',
        [RouterType.INSTANCE]: 'instanceId',
        [RouterType.LOCAL_GATEWAY]: 'localGatewayId',
        [RouterType.NAT_GATEWAY]: 'natGatewayId',
        [RouterType.NETWORK_INTERFACE]: 'networkInterfaceId',
        [RouterType.TRANSIT_GATEWAY]: 'transitGatewayId',
        [RouterType.VPC_PEERING_CONNECTION]: 'vpcPeeringConnectionId',
        [RouterType.VPC_ENDPOINT]: 'vpcEndpointId',
    })[routerType];
}
/**
 * Represents a public VPC subnet resource
 */
class PublicSubnet extends Subnet {
    constructor(scope, id, props) {
        super(scope, id, props);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_PublicSubnetProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, PublicSubnet);
            }
            throw error;
        }
    }
    static fromPublicSubnetAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_PublicSubnetAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromPublicSubnetAttributes);
            }
            throw error;
        }
        return new ImportedSubnet(scope, id, attrs);
    }
    /**
     * Creates a new managed NAT gateway attached to this public subnet.
     * Also adds the EIP for the managed NAT.
     * @returns A ref to the the NAT Gateway ID
     */
    addNatGateway(eipAllocationId) {
        // Create a NAT Gateway in this public subnet
        const ngw = new ec2_generated_1.CfnNatGateway(this, 'NATGateway', {
            subnetId: this.subnetId,
            allocationId: eipAllocationId ?? new ec2_generated_1.CfnEIP(this, 'EIP', {
                domain: 'vpc',
            }).attrAllocationId,
        });
        return ngw;
    }
}
exports.PublicSubnet = PublicSubnet;
_c = JSII_RTTI_SYMBOL_1;
PublicSubnet[_c] = { fqn: "@aws-cdk/aws-ec2.PublicSubnet", version: "1.167.0" };
/**
 * Represents a private VPC subnet resource
 */
class PrivateSubnet extends Subnet {
    constructor(scope, id, props) {
        super(scope, id, props);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_PrivateSubnetProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, PrivateSubnet);
            }
            throw error;
        }
    }
    static fromPrivateSubnetAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_PrivateSubnetAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromPrivateSubnetAttributes);
            }
            throw error;
        }
        return new ImportedSubnet(scope, id, attrs);
    }
}
exports.PrivateSubnet = PrivateSubnet;
_d = JSII_RTTI_SYMBOL_1;
PrivateSubnet[_d] = { fqn: "@aws-cdk/aws-ec2.PrivateSubnet", version: "1.167.0" };
function ifUndefined(value, defaultValue) {
    return value ?? defaultValue;
}
class ImportedVpc extends VpcBase {
    constructor(scope, id, props, isIncomplete) {
        super(scope, id);
        this.internetConnectivityEstablished = new core_1.ConcreteDependable();
        this.vpcId = props.vpcId;
        this.vpcArn = core_1.Arn.format({
            service: 'ec2',
            resource: 'vpc',
            resourceName: this.vpcId,
        }, core_1.Stack.of(this));
        this.cidr = props.vpcCidrBlock;
        this.availabilityZones = props.availabilityZones;
        this._vpnGatewayId = props.vpnGatewayId;
        this.incompleteSubnetDefinition = isIncomplete;
        // None of the values may be unresolved list tokens
        for (const k of Object.keys(props)) {
            if (Array.isArray(props[k]) && core_1.Token.isUnresolved(props[k])) {
                core_1.Annotations.of(this).addWarning(`fromVpcAttributes: '${k}' is a list token: the imported VPC will not work with constructs that require a list of subnets at synthesis time. Use 'Vpc.fromLookup()' or 'Fn.importListValue' instead.`);
            }
        }
        /* eslint-disable max-len */
        const pub = new util_1.ImportSubnetGroup(props.publicSubnetIds, props.publicSubnetNames, props.publicSubnetRouteTableIds, SubnetType.PUBLIC, this.availabilityZones, 'publicSubnetIds', 'publicSubnetNames', 'publicSubnetRouteTableIds');
        const priv = new util_1.ImportSubnetGroup(props.privateSubnetIds, props.privateSubnetNames, props.privateSubnetRouteTableIds, SubnetType.PRIVATE_WITH_NAT, this.availabilityZones, 'privateSubnetIds', 'privateSubnetNames', 'privateSubnetRouteTableIds');
        const iso = new util_1.ImportSubnetGroup(props.isolatedSubnetIds, props.isolatedSubnetNames, props.isolatedSubnetRouteTableIds, SubnetType.PRIVATE_ISOLATED, this.availabilityZones, 'isolatedSubnetIds', 'isolatedSubnetNames', 'isolatedSubnetRouteTableIds');
        /* eslint-enable max-len */
        this.publicSubnets = pub.import(this);
        this.privateSubnets = priv.import(this);
        this.isolatedSubnets = iso.import(this);
    }
    get vpcCidrBlock() {
        if (this.cidr === undefined) {
            throw new Error('Cannot perform this operation: \'vpcCidrBlock\' was not supplied when creating this VPC');
        }
        return this.cidr;
    }
}
class LookedUpVpc extends VpcBase {
    constructor(scope, id, props, isIncomplete) {
        super(scope, id);
        this.internetConnectivityEstablished = new core_1.ConcreteDependable();
        this.vpcId = props.vpcId;
        this.vpcArn = core_1.Arn.format({
            service: 'ec2',
            resource: 'vpc',
            resourceName: this.vpcId,
        }, core_1.Stack.of(this));
        this.cidr = props.vpcCidrBlock;
        this._vpnGatewayId = props.vpnGatewayId;
        this.incompleteSubnetDefinition = isIncomplete;
        const subnetGroups = props.subnetGroups || [];
        const availabilityZones = Array.from(new Set(flatMap(subnetGroups, subnetGroup => {
            return subnetGroup.subnets.map(subnet => subnet.availabilityZone);
        })));
        availabilityZones.sort((az1, az2) => az1.localeCompare(az2));
        this.availabilityZones = availabilityZones;
        this.publicSubnets = this.extractSubnetsOfType(subnetGroups, cxapi.VpcSubnetGroupType.PUBLIC);
        this.privateSubnets = this.extractSubnetsOfType(subnetGroups, cxapi.VpcSubnetGroupType.PRIVATE);
        this.isolatedSubnets = this.extractSubnetsOfType(subnetGroups, cxapi.VpcSubnetGroupType.ISOLATED);
    }
    get vpcCidrBlock() {
        if (this.cidr === undefined) {
            // Value might be cached from an old CLI version, so bumping the CX API protocol to
            // force the value to exist would not have helped.
            throw new Error('Cannot perform this operation: \'vpcCidrBlock\' was not found when looking up this VPC. Use a newer version of the CDK CLI and clear the old context value.');
        }
        return this.cidr;
    }
    extractSubnetsOfType(subnetGroups, subnetGroupType) {
        return flatMap(subnetGroups.filter(subnetGroup => subnetGroup.type === subnetGroupType), subnetGroup => this.subnetGroupToSubnets(subnetGroup));
    }
    subnetGroupToSubnets(subnetGroup) {
        const ret = new Array();
        for (let i = 0; i < subnetGroup.subnets.length; i++) {
            const vpcSubnet = subnetGroup.subnets[i];
            ret.push(Subnet.fromSubnetAttributes(this, `${subnetGroup.name}Subnet${i + 1}`, {
                availabilityZone: vpcSubnet.availabilityZone,
                subnetId: vpcSubnet.subnetId,
                routeTableId: vpcSubnet.routeTableId,
                ipv4CidrBlock: vpcSubnet.cidr,
            }));
        }
        return ret;
    }
}
function flatMap(xs, fn) {
    const ret = new Array();
    for (const x of xs) {
        ret.push(...fn(x));
    }
    return ret;
}
class CompositeDependable {
    constructor() {
        this.dependables = new Array();
        const self = this;
        core_1.DependableTrait.implement(this, {
            get dependencyRoots() {
                const ret = new Array();
                for (const dep of self.dependables) {
                    ret.push(...core_1.DependableTrait.get(dep).dependencyRoots);
                }
                return ret;
            },
        });
    }
    /**
     * Add a construct to the dependency roots
     */
    add(dep) {
        this.dependables.push(dep);
    }
}
/**
 * Invoke a function on a value (for its side effect) and return the value
 */
function tap(x, fn) {
    fn(x);
    return x;
}
class ImportedSubnet extends core_1.Resource {
    constructor(scope, id, attrs) {
        super(scope, id);
        this.internetConnectivityEstablished = new core_1.ConcreteDependable();
        if (!attrs.routeTableId) {
            // The following looks a little weird, but comes down to:
            //
            // * Is the subnetId itself unresolved ({ Ref: Subnet }); or
            // * Was it the accidentally extracted first element of a list-encoded
            //   token? ({ Fn::ImportValue: Subnets } => ['#{Token[1234]}'] =>
            //   '#{Token[1234]}'
            //
            // There's no other API to test for the second case than to the string back into
            // a list and see if the combination is Unresolved.
            //
            // In both cases we can't output the subnetId literally into the metadata (because it'll
            // be useless). In the 2nd case even, if we output it to metadata, the `resolve()` call
            // that gets done on the metadata will even `throw`, because the '#{Token}' value will
            // occur in an illegal position (not in a list context).
            const ref = core_1.Token.isUnresolved(attrs.subnetId) || core_1.Token.isUnresolved([attrs.subnetId])
                ? `at '${constructs_1.Node.of(scope).path}/${id}'`
                : `'${attrs.subnetId}'`;
            // eslint-disable-next-line max-len
            core_1.Annotations.of(this).addWarning(`No routeTableId was provided to the subnet ${ref}. Attempting to read its .routeTable.routeTableId will return null/undefined. (More info: https://github.com/aws/aws-cdk/pull/3171)`);
        }
        this._ipv4CidrBlock = attrs.ipv4CidrBlock;
        this._availabilityZone = attrs.availabilityZone;
        this.subnetId = attrs.subnetId;
        this.routeTable = {
            // Forcing routeTableId to pretend non-null to maintain backwards-compatibility. See https://github.com/aws/aws-cdk/pull/3171
            routeTableId: attrs.routeTableId,
        };
    }
    get availabilityZone() {
        if (!this._availabilityZone) {
            // eslint-disable-next-line max-len
            throw new Error('You cannot reference a Subnet\'s availability zone if it was not supplied. Add the availabilityZone when importing using Subnet.fromSubnetAttributes()');
        }
        return this._availabilityZone;
    }
    get ipv4CidrBlock() {
        if (!this._ipv4CidrBlock) {
            // tslint:disable-next-line: max-line-length
            throw new Error('You cannot reference an imported Subnet\'s IPv4 CIDR if it was not supplied. Add the ipv4CidrBlock when importing using Subnet.fromSubnetAttributes()');
        }
        return this._ipv4CidrBlock;
    }
    associateNetworkAcl(id, networkAcl) {
        const scope = core_2.Construct.isConstruct(networkAcl) ? networkAcl : this;
        const other = core_2.Construct.isConstruct(networkAcl) ? this : networkAcl;
        new network_acl_1.SubnetNetworkAclAssociation(scope, id + core_1.Names.nodeUniqueId(other.node), {
            networkAcl,
            subnet: this,
        });
    }
}
/**
 * Determine (and validate) the NAT gateway count w.r.t. the rest of the subnet configuration
 *
 * We have the following requirements:
 *
 * - NatGatewayCount = 0 ==> there are no private subnets
 * - NatGatewayCount > 0 ==> there must be public subnets
 *
 * Do we want to require that there are private subnets if there are NatGateways?
 * They seem pointless but I see no reason to prevent it.
 */
function determineNatGatewayCount(requestedCount, subnetConfig, azCount) {
    const hasPrivateSubnets = subnetConfig.some(c => (c.subnetType === SubnetType.PRIVATE_WITH_NAT
        || c.subnetType === SubnetType.PRIVATE) && !c.reserved);
    const hasPublicSubnets = subnetConfig.some(c => c.subnetType === SubnetType.PUBLIC);
    const count = requestedCount !== undefined ? Math.min(requestedCount, azCount) : (hasPrivateSubnets ? azCount : 0);
    if (count === 0 && hasPrivateSubnets) {
        // eslint-disable-next-line max-len
        throw new Error('If you do not want NAT gateways (natGateways=0), make sure you don\'t configure any PRIVATE subnets in \'subnetConfiguration\' (make them PUBLIC or ISOLATED instead)');
    }
    if (count > 0 && !hasPublicSubnets) {
        // eslint-disable-next-line max-len
        throw new Error(`If you configure PRIVATE subnets in 'subnetConfiguration', you must also configure PUBLIC subnets to put the NAT gateways into (got ${JSON.stringify(subnetConfig)}.`);
    }
    return count;
}
/**
 * There are returned when the provider has not supplied props yet
 *
 * It's only used for testing and on the first run-through.
 */
const DUMMY_VPC_PROPS = {
    availabilityZones: [],
    vpcCidrBlock: '1.2.3.4/5',
    isolatedSubnetIds: undefined,
    isolatedSubnetNames: undefined,
    isolatedSubnetRouteTableIds: undefined,
    privateSubnetIds: undefined,
    privateSubnetNames: undefined,
    privateSubnetRouteTableIds: undefined,
    publicSubnetIds: undefined,
    publicSubnetNames: undefined,
    publicSubnetRouteTableIds: undefined,
    subnetGroups: [
        {
            name: 'Public',
            type: cxapi.VpcSubnetGroupType.PUBLIC,
            subnets: [
                {
                    availabilityZone: 'dummy1a',
                    subnetId: 's-12345',
                    routeTableId: 'rtb-12345s',
                    cidr: '1.2.3.4/5',
                },
                {
                    availabilityZone: 'dummy1b',
                    subnetId: 's-67890',
                    routeTableId: 'rtb-67890s',
                    cidr: '1.2.3.4/5',
                },
            ],
        },
        {
            name: 'Private',
            type: cxapi.VpcSubnetGroupType.PRIVATE,
            subnets: [
                {
                    availabilityZone: 'dummy1a',
                    subnetId: 'p-12345',
                    routeTableId: 'rtb-12345p',
                    cidr: '1.2.3.4/5',
                },
                {
                    availabilityZone: 'dummy1b',
                    subnetId: 'p-67890',
                    routeTableId: 'rtb-57890p',
                    cidr: '1.2.3.4/5',
                },
            ],
        },
        {
            name: 'Isolated',
            type: cxapi.VpcSubnetGroupType.ISOLATED,
            subnets: [
                {
                    availabilityZone: 'dummy1a',
                    subnetId: 'p-12345',
                    routeTableId: 'rtb-12345p',
                    cidr: '1.2.3.4/5',
                },
                {
                    availabilityZone: 'dummy1b',
                    subnetId: 'p-67890',
                    routeTableId: 'rtb-57890p',
                    cidr: '1.2.3.4/5',
                },
            ],
        },
    ],
    vpcId: 'vpc-12345',
};
//# sourceMappingURL=data:application/json;base64,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