#
# (C) Copyright 2011-2022 Enthought, Inc., Austin, TX
# All right reserved.
#
# This file is open source software distributed according to the terms in LICENSE.txt
#

"""
Storage Events
==============

This module contains asbtract and concrete Event subclasses that support the
Storage API.
"""

from encore.events.api import (BaseEvent, ProgressEvent, ProgressStartEvent,
    ProgressStepEvent, ProgressEndEvent)


class StoreEvent(BaseEvent):
    """ An abstract base class for events generated by a Key-Value Store

    Attributes
    ----------
    source : Store instance
        The key-value store which generated the event.

    """

class StoreTransactionEvent(StoreEvent):
    pass

class StoreTransactionStartEvent(StoreTransactionEvent):
    pass

class StoreTransactionEndEvent(StoreTransactionEvent):
    pass

class StoreKeyEvent(StoreEvent):
    """ An abstract base class for events related to a particular key in the
    store.  This should provide the key and metadata (if available) of the
    modified key.

    Attributes
    ----------
    key : string
        The key which is involved in the event.
    metadata : dict
        The metadata of the key which is involved in the event.

    """


class StoreModificationEvent(StoreKeyEvent):
    """ An abstract base class for modification events generated by a Key-Value Store

    Attributes
    ----------
    key : string
        The key which is involved in the event.
    metadata : dict
        The metadata of the key which is involved in the event.
    action : string
        The modification action that was performed.  One of 'set', 'update' or
        'delete'.

    """


class StoreSetEvent(StoreModificationEvent):
    """ An event generated when a value is set into a Key-Value Store

    Attributes
    ----------
    key : string
        The key which is involved in the event.
    metadata : dict
        The metadata of the key which is involved in the event.
    action : 'set'
        The modification action that was performed.

    """
    action = 'set'


class StoreUpdateEvent(StoreModificationEvent):
    """ An event generated when a value is updated into a Key-Value Store

    Attributes
    ----------
    key : string
        The key which is involved in the event.
    metadata : dict
        The metadata of the key which is involved in the event.
    action : 'update'
        The modification action that was performed.

    """
    action = 'update'


class StoreDeleteEvent(StoreModificationEvent):
    """ An event generated when a value is deleted into a Key-Value Store

    Attributes
    ----------
    key : string
        The key which is involved in the event.
    metadata : dict
        The metadata of the key which is involved in the event.
    action : 'delete'
        The modification action that was performed.

    """
    action = 'delete'


class StoreProgressEvent(ProgressEvent, StoreKeyEvent):
    """ Abstract base class for ProgressEvents generated by a Key-Value Store

    Attributes
    ----------
    operation_id :
        A unique identifier for the operation being performed.
    message : string
        A human-readable describing the operation being performed.
    key : string
        The key which is involved in the event.
    metadata : dict
        The metadata of the key which is involved in the event.

    """


class StoreProgressStartEvent(ProgressStartEvent, StoreProgressEvent):
    """

    Attributes
    ----------
    operation_id :
        A unique identifier for the operation being performed.
    message : string
        A human-readable describing the operation being performed.
    key : string
        The key which is involved in the event.
    metadata : dict
        The metadata of the key which is involved in the event.
    steps : int
        The number of steps in the operation.  If unknown or variable, use -1.

    """

class StoreProgressStepEvent(ProgressStepEvent, StoreProgressEvent):
    """

    Attributes
    ----------
    operation_id :
        A unique identifier for the operation being performed.
    message : string
        A human-readable describing the state of the operation being performed.
    key : string
        The key which is involved in the event.
    metadata : dict
        The metadata of the key which is involved in the event.
    step : int
        The count of the step.  If unknown, use -1.

    """

class StoreProgressEndEvent(ProgressEndEvent, StoreProgressEvent):
    """

    Attributes
    ----------
    operation_id :
        A unique identifier for the operation that is finished.
    message : string
        A human-readable describing the state of the operation that ended.
    key : string
        The key which is involved in the event.
    metadata : dict
        The metadata of the key which is involved in the event.
    exit_state : string
        A constant describing the end state of the operation.  One of ``normal``,
        ``warning``, ``error`` or ``exception``.

    """
