"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecretTargetAttachment = exports.AttachmentTargetType = exports.Secret = exports.SecretStringValueBeta1 = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const policy_1 = require("./policy");
const rotation_schedule_1 = require("./rotation-schedule");
const secretsmanager = require("./secretsmanager.generated");
/**
 * An experimental class used to specify an initial secret value for a Secret.
 *
 * The class wraps a simple string (or JSON representation) in order to provide some safety checks and warnings
 * about the dangers of using plaintext strings as initial secret seed values via CDK/CloudFormation.
 *
 * @stability stable
 */
class SecretStringValueBeta1 {
    constructor(_secretValue) {
        this._secretValue = _secretValue;
    }
    /**
     * Creates a `SecretStringValueBeta1` from a plaintext value.
     *
     * This approach is inherently unsafe, as the secret value may be visible in your source control repository
     * and will also appear in plaintext in the resulting CloudFormation template, including in the AWS Console or APIs.
     * Usage of this method is discouraged, especially for production workloads.
     *
     * @stability stable
     */
    static fromUnsafePlaintext(secretValue) { return new SecretStringValueBeta1(secretValue); }
    /**
     * Creates a `SecretValueValueBeta1` from a string value coming from a Token.
     *
     * The intent is to enable creating secrets from references (e.g., `Ref`, `Fn::GetAtt`) from other resources.
     * This might be the direct output of another Construct, or the output of a Custom Resource.
     * This method throws if it determines the input is an unsafe plaintext string.
     *
     * For example:
     * ```ts
     *      // Creates a new IAM user, access and secret keys, and stores the secret access key in a Secret.
     *      const user = new iam.User(this, 'User');
     *      const accessKey = new iam.CfnAccessKey(this, 'AccessKey', { userName: user.userName });
     *      const secretValue = secretsmanager.SecretStringValueBeta1.fromToken(accessKey.attrSecretAccessKey);
     *      new secretsmanager.Secret(this, 'Secret', {
     *        secretStringBeta1: secretValue,
     *      });
     * ```
     *
     * The secret may also be embedded in a string representation of a JSON structure:
     *      const secretValue = secretsmanager.SecretStringValueBeta1.fromToken(JSON.stringify({
     *        username: user.userName,
     *        database: 'foo',
     *        password: accessKey.attrSecretAccessKey
     *      }));
     *
     * Note that the value being a Token does *not* guarantee safety. For example, a Lazy-evaluated string
     * (e.g., `Lazy.string({ produce: () => 'myInsecurePassword' }))`) is a Token, but as the output is
     * ultimately a plaintext string, and so insecure.
     *
     * @param secretValueFromToken a secret value coming from a Construct attribute or Custom Resource output.
     * @stability stable
     */
    static fromToken(secretValueFromToken) {
        if (!core_1.Token.isUnresolved(secretValueFromToken)) {
            throw new Error('SecretStringValueBeta1 appears to be plaintext (unsafe) string (or resolved Token); use fromUnsafePlaintext if this is intentional');
        }
        return new SecretStringValueBeta1(secretValueFromToken);
    }
    /**
     * Returns the secret value.
     *
     * @stability stable
     */
    secretValue() { return this._secretValue; }
}
exports.SecretStringValueBeta1 = SecretStringValueBeta1;
_a = JSII_RTTI_SYMBOL_1;
SecretStringValueBeta1[_a] = { fqn: "@aws-cdk/aws-secretsmanager.SecretStringValueBeta1", version: "1.138.1" };
/**
 * The common behavior of Secrets. Users should not use this class directly, and instead use ``Secret``.
 */
class SecretBase extends core_1.Resource {
    /**
     * The full ARN of the secret in AWS Secrets Manager, which is the ARN including the Secrets Manager-supplied 6-character suffix.
     *
     * This is equal to `secretArn` in most cases, but is undefined when a full ARN is not available (e.g., secrets imported by name).
     *
     * @stability stable
     */
    get secretFullArn() { return this.secretArn; }
    /**
     * Grants reading the secret value to some role.
     *
     * @stability stable
     */
    grantRead(grantee, versionStages) {
        // @see https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access_identity-based-policies.html
        const result = iam.Grant.addToPrincipalOrResource({
            grantee,
            actions: ['secretsmanager:GetSecretValue', 'secretsmanager:DescribeSecret'],
            resourceArns: [this.arnForPolicies],
            resource: this,
        });
        const statement = result.principalStatement || result.resourceStatement;
        if (versionStages != null && statement) {
            statement.addCondition('ForAnyValue:StringEquals', {
                'secretsmanager:VersionStage': versionStages,
            });
        }
        if (this.encryptionKey) {
            // @see https://docs.aws.amazon.com/kms/latest/developerguide/services-secrets-manager.html
            this.encryptionKey.grantDecrypt(new kms.ViaServicePrincipal(`secretsmanager.${core_1.Stack.of(this).region}.amazonaws.com`, grantee.grantPrincipal));
        }
        // Throw if secret is not imported and it's shared cross account and no KMS key is provided
        if (this instanceof Secret && result.resourceStatement && !this.encryptionKey) {
            throw new Error('KMS Key must be provided for cross account access to Secret');
        }
        return result;
    }
    /**
     * Grants writing and updating the secret value to some role.
     *
     * @stability stable
     */
    grantWrite(grantee) {
        // See https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access_identity-based-policies.html
        const result = iam.Grant.addToPrincipalOrResource({
            grantee,
            actions: ['secretsmanager:PutSecretValue', 'secretsmanager:UpdateSecret'],
            resourceArns: [this.arnForPolicies],
            resource: this,
        });
        if (this.encryptionKey) {
            // See https://docs.aws.amazon.com/kms/latest/developerguide/services-secrets-manager.html
            this.encryptionKey.grantEncrypt(new kms.ViaServicePrincipal(`secretsmanager.${core_1.Stack.of(this).region}.amazonaws.com`, grantee.grantPrincipal));
        }
        // Throw if secret is not imported and it's shared cross account and no KMS key is provided
        if (this instanceof Secret && result.resourceStatement && !this.encryptionKey) {
            throw new Error('KMS Key must be provided for cross account access to Secret');
        }
        return result;
    }
    /**
     * Retrieve the value of the stored secret as a `SecretValue`.
     *
     * @stability stable
     */
    get secretValue() {
        return this.secretValueFromJson('');
    }
    /**
     * Interpret the secret as a JSON object and return a field's value from it as a `SecretValue`.
     *
     * @stability stable
     */
    secretValueFromJson(jsonField) {
        return core_1.SecretValue.secretsManager(this.secretArn, { jsonField });
    }
    /**
     * Adds a rotation schedule to the secret.
     *
     * @stability stable
     */
    addRotationSchedule(id, options) {
        return new rotation_schedule_1.RotationSchedule(this, id, {
            secret: this,
            ...options,
        });
    }
    /**
     * Adds a statement to the IAM resource policy associated with this secret.
     *
     * If this secret was created in this stack, a resource policy will be
     * automatically created upon the first call to `addToResourcePolicy`. If
     * the secret is imported, then this is a no-op.
     *
     * @stability stable
     */
    addToResourcePolicy(statement) {
        if (!this.policy && this.autoCreatePolicy) {
            this.policy = new policy_1.ResourcePolicy(this, 'Policy', { secret: this });
        }
        if (this.policy) {
            this.policy.document.addStatements(statement);
            return { statementAdded: true, policyDependable: this.policy };
        }
        return { statementAdded: false };
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    validate() {
        var _d;
        const errors = super.validate();
        errors.push(...((_d = this.policy) === null || _d === void 0 ? void 0 : _d.document.validateForResourcePolicy()) || []);
        return errors;
    }
    /**
     * Denies the `DeleteSecret` action to all principals within the current account.
     *
     * @stability stable
     */
    denyAccountRootDelete() {
        this.addToResourcePolicy(new iam.PolicyStatement({
            actions: ['secretsmanager:DeleteSecret'],
            effect: iam.Effect.DENY,
            resources: ['*'],
            principals: [new iam.AccountRootPrincipal()],
        }));
    }
    /**
     * Provides an identifier for this secret for use in IAM policies.
     *
     * If there is a full ARN, this is just the ARN;
     * if we have a partial ARN -- due to either importing by secret name or partial ARN --
     * then we need to add a suffix to capture the full ARN's format.
     *
     * @stability stable
     */
    get arnForPolicies() {
        return this.secretFullArn ? this.secretFullArn : `${this.secretArn}-??????`;
    }
    /**
     * Attach a target to this secret.
     *
     * @param target The target to attach.
     * @returns An attached secret
     * @stability stable
     */
    attach(target) {
        const id = 'Attachment';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('Secret is already attached to a target.');
        }
        return new SecretTargetAttachment(this, id, {
            secret: this,
            target,
        });
    }
}
/**
 * Creates a new secret in AWS SecretsManager.
 *
 * @stability stable
 */
class Secret extends SecretBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _d, _e, _f, _g, _h;
        super(scope, id, {
            physicalName: props.secretName,
        });
        this.replicaRegions = [];
        /**
         * @stability stable
         */
        this.autoCreatePolicy = true;
        jsiiDeprecationWarnings._aws_cdk_aws_secretsmanager_SecretProps(props);
        if (props.generateSecretString &&
            (props.generateSecretString.secretStringTemplate || props.generateSecretString.generateStringKey) &&
            !(props.generateSecretString.secretStringTemplate && props.generateSecretString.generateStringKey)) {
            throw new Error('`secretStringTemplate` and `generateStringKey` must be specified together.');
        }
        if (props.generateSecretString && props.secretStringBeta1) {
            throw new Error('Cannot specify both `generateSecretString` and `secretStringBeta1`.');
        }
        const resource = new secretsmanager.CfnSecret(this, 'Resource', {
            description: props.description,
            kmsKeyId: props.encryptionKey && props.encryptionKey.keyArn,
            generateSecretString: (_d = props.generateSecretString) !== null && _d !== void 0 ? _d : (props.secretStringBeta1 ? undefined : {}),
            secretString: (_e = props.secretStringBeta1) === null || _e === void 0 ? void 0 : _e.secretValue(),
            name: this.physicalName,
            replicaRegions: core_1.Lazy.any({ produce: () => this.replicaRegions }, { omitEmptyArray: true }),
        });
        resource.applyRemovalPolicy(props.removalPolicy, {
            default: core_1.RemovalPolicy.DESTROY,
        });
        this.secretArn = this.getResourceArnAttribute(resource.ref, {
            service: 'secretsmanager',
            resource: 'secret',
            resourceName: this.physicalName,
            arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
        });
        this.encryptionKey = props.encryptionKey;
        const parseOwnedSecretName = core_1.FeatureFlags.of(this).isEnabled(cxapi.SECRETS_MANAGER_PARSE_OWNED_SECRET_NAME);
        this.secretName = parseOwnedSecretName
            ? parseSecretNameForOwnedSecret(this, this.secretArn, props.secretName)
            : parseSecretName(this, this.secretArn);
        // @see https://docs.aws.amazon.com/kms/latest/developerguide/services-secrets-manager.html#asm-authz
        const principal = new kms.ViaServicePrincipal(`secretsmanager.${core_1.Stack.of(this).region}.amazonaws.com`, new iam.AccountPrincipal(core_1.Stack.of(this).account));
        (_f = this.encryptionKey) === null || _f === void 0 ? void 0 : _f.grantEncryptDecrypt(principal);
        (_g = this.encryptionKey) === null || _g === void 0 ? void 0 : _g.grant(principal, 'kms:CreateGrant', 'kms:DescribeKey');
        for (const replica of (_h = props.replicaRegions) !== null && _h !== void 0 ? _h : []) {
            this.addReplicaRegion(replica.region, replica.encryptionKey);
        }
    }
    /**
     * @deprecated use `fromSecretCompleteArn` or `fromSecretPartialArn`
     */
    static fromSecretArn(scope, id, secretArn) {
        jsiiDeprecationWarnings.print("@aws-cdk/aws-secretsmanager.Secret#fromSecretArn", "use `fromSecretCompleteArn` or `fromSecretPartialArn`");
        const attrs = arnIsComplete(secretArn) ? { secretCompleteArn: secretArn } : { secretPartialArn: secretArn };
        return Secret.fromSecretAttributes(scope, id, attrs);
    }
    /**
     * Imports a secret by complete ARN.
     *
     * The complete ARN is the ARN with the Secrets Manager-supplied suffix.
     *
     * @stability stable
     */
    static fromSecretCompleteArn(scope, id, secretCompleteArn) {
        return Secret.fromSecretAttributes(scope, id, { secretCompleteArn });
    }
    /**
     * Imports a secret by partial ARN.
     *
     * The partial ARN is the ARN without the Secrets Manager-supplied suffix.
     *
     * @stability stable
     */
    static fromSecretPartialArn(scope, id, secretPartialArn) {
        return Secret.fromSecretAttributes(scope, id, { secretPartialArn });
    }
    /**
     * (deprecated) Imports a secret by secret name;
     *
     * the ARN of the Secret will be set to the secret name.
     * A secret with this name must exist in the same account & region.
     *
     * @deprecated use `fromSecretNameV2`
     */
    static fromSecretName(scope, id, secretName) {
        jsiiDeprecationWarnings.print("@aws-cdk/aws-secretsmanager.Secret#fromSecretName", "use `fromSecretNameV2`");
        return new class extends SecretBase {
            constructor() {
                super(...arguments);
                this.encryptionKey = undefined;
                this.secretArn = secretName;
                this.secretName = secretName;
                this.autoCreatePolicy = false;
            }
            get secretFullArn() { return undefined; }
            // Overrides the secretArn for grant* methods, where the secretArn must be in ARN format.
            // Also adds a wildcard to the resource name to support the SecretsManager-provided suffix.
            get arnForPolicies() {
                return core_1.Stack.of(this).formatArn({
                    service: 'secretsmanager',
                    resource: 'secret',
                    resourceName: this.secretName + '*',
                    arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
                });
            }
        }(scope, id);
    }
    /**
     * Imports a secret by secret name.
     *
     * A secret with this name must exist in the same account & region.
     * Replaces the deprecated `fromSecretName`.
     *
     * @stability stable
     */
    static fromSecretNameV2(scope, id, secretName) {
        return new class extends SecretBase {
            constructor() {
                super(...arguments);
                this.encryptionKey = undefined;
                this.secretName = secretName;
                this.secretArn = this.partialArn;
                this.autoCreatePolicy = false;
            }
            get secretFullArn() { return undefined; }
            // Creates a "partial" ARN from the secret name. The "full" ARN would include the SecretsManager-provided suffix.
            get partialArn() {
                return core_1.Stack.of(this).formatArn({
                    service: 'secretsmanager',
                    resource: 'secret',
                    resourceName: secretName,
                    arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
                });
            }
        }(scope, id);
    }
    /**
     * Import an existing secret into the Stack.
     *
     * @param scope the scope of the import.
     * @param id the ID of the imported Secret in the construct tree.
     * @param attrs the attributes of the imported secret.
     * @stability stable
     */
    static fromSecretAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings._aws_cdk_aws_secretsmanager_SecretAttributes(attrs);
        let secretArn;
        let secretArnIsPartial;
        if (attrs.secretArn) {
            if (attrs.secretCompleteArn || attrs.secretPartialArn) {
                throw new Error('cannot use `secretArn` with `secretCompleteArn` or `secretPartialArn`');
            }
            secretArn = attrs.secretArn;
            secretArnIsPartial = false;
        }
        else {
            if ((attrs.secretCompleteArn && attrs.secretPartialArn) ||
                (!attrs.secretCompleteArn && !attrs.secretPartialArn)) {
                throw new Error('must use only one of `secretCompleteArn` or `secretPartialArn`');
            }
            if (attrs.secretCompleteArn && !arnIsComplete(attrs.secretCompleteArn)) {
                throw new Error('`secretCompleteArn` does not appear to be complete; missing 6-character suffix');
            }
            [secretArn, secretArnIsPartial] = attrs.secretCompleteArn ? [attrs.secretCompleteArn, false] : [attrs.secretPartialArn, true];
        }
        return new class extends SecretBase {
            constructor() {
                super(...arguments);
                this.encryptionKey = attrs.encryptionKey;
                this.secretArn = secretArn;
                this.secretName = parseSecretName(scope, secretArn);
                this.autoCreatePolicy = false;
            }
            get secretFullArn() { return secretArnIsPartial ? undefined : secretArn; }
        }(scope, id, { environmentFromArn: secretArn });
    }
    /**
     * (deprecated) Adds a target attachment to the secret.
     *
     * @returns an AttachedSecret
     * @deprecated use `attach()` instead
     */
    addTargetAttachment(id, options) {
        jsiiDeprecationWarnings.print("@aws-cdk/aws-secretsmanager.Secret#addTargetAttachment", "use `attach()` instead");
        jsiiDeprecationWarnings._aws_cdk_aws_secretsmanager_AttachedSecretOptions(options);
        return new SecretTargetAttachment(this, id, {
            secret: this,
            ...options,
        });
    }
    /**
     * Adds a replica region for the secret.
     *
     * @param region The name of the region.
     * @param encryptionKey The customer-managed encryption key to use for encrypting the secret value.
     * @stability stable
     */
    addReplicaRegion(region, encryptionKey) {
        const stack = core_1.Stack.of(this);
        if (!core_1.Token.isUnresolved(stack.region) && !core_1.Token.isUnresolved(region) && region === stack.region) {
            throw new Error('Cannot add the region where this stack is deployed as a replica region.');
        }
        this.replicaRegions.push({
            region,
            kmsKeyId: encryptionKey === null || encryptionKey === void 0 ? void 0 : encryptionKey.keyArn,
        });
    }
}
exports.Secret = Secret;
_b = JSII_RTTI_SYMBOL_1;
Secret[_b] = { fqn: "@aws-cdk/aws-secretsmanager.Secret", version: "1.138.1" };
/**
 * The type of service or database that's being associated with the secret.
 *
 * @stability stable
 */
var AttachmentTargetType;
(function (AttachmentTargetType) {
    AttachmentTargetType["INSTANCE"] = "AWS::RDS::DBInstance";
    AttachmentTargetType["CLUSTER"] = "AWS::RDS::DBCluster";
    /**
     * AWS::RDS::DBInstance
     */
    AttachmentTargetType["RDS_DB_INSTANCE"] = "AWS::RDS::DBInstance";
    /**
     * AWS::RDS::DBCluster
     */
    AttachmentTargetType["RDS_DB_CLUSTER"] = "AWS::RDS::DBCluster";
    AttachmentTargetType["RDS_DB_PROXY"] = "AWS::RDS::DBProxy";
    AttachmentTargetType["REDSHIFT_CLUSTER"] = "AWS::Redshift::Cluster";
    AttachmentTargetType["DOCDB_DB_INSTANCE"] = "AWS::DocDB::DBInstance";
    AttachmentTargetType["DOCDB_DB_CLUSTER"] = "AWS::DocDB::DBCluster";
})(AttachmentTargetType = exports.AttachmentTargetType || (exports.AttachmentTargetType = {}));
/**
 * An attached secret.
 *
 * @stability stable
 */
class SecretTargetAttachment extends SecretBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * @stability stable
         */
        this.autoCreatePolicy = true;
        jsiiDeprecationWarnings._aws_cdk_aws_secretsmanager_SecretTargetAttachmentProps(props);
        const attachment = new secretsmanager.CfnSecretTargetAttachment(this, 'Resource', {
            secretId: props.secret.secretArn,
            targetId: props.target.asSecretAttachmentTarget().targetId,
            targetType: props.target.asSecretAttachmentTarget().targetType,
        });
        this.encryptionKey = props.secret.encryptionKey;
        this.secretName = props.secret.secretName;
        // This allows to reference the secret after attachment (dependency).
        this.secretArn = attachment.ref;
        this.secretTargetAttachmentSecretArn = attachment.ref;
    }
    /**
     * @stability stable
     */
    static fromSecretTargetAttachmentSecretArn(scope, id, secretTargetAttachmentSecretArn) {
        class Import extends SecretBase {
            constructor() {
                super(...arguments);
                this.secretArn = secretTargetAttachmentSecretArn;
                this.secretTargetAttachmentSecretArn = secretTargetAttachmentSecretArn;
                this.secretName = parseSecretName(scope, secretTargetAttachmentSecretArn);
                this.autoCreatePolicy = false;
            }
        }
        return new Import(scope, id);
    }
}
exports.SecretTargetAttachment = SecretTargetAttachment;
_c = JSII_RTTI_SYMBOL_1;
SecretTargetAttachment[_c] = { fqn: "@aws-cdk/aws-secretsmanager.SecretTargetAttachment", version: "1.138.1" };
/** Parses the secret name from the ARN. */
function parseSecretName(construct, secretArn) {
    const resourceName = core_1.Stack.of(construct).splitArn(secretArn, core_1.ArnFormat.COLON_RESOURCE_NAME).resourceName;
    if (resourceName) {
        // Can't operate on the token to remove the SecretsManager suffix, so just return the full secret name
        if (core_1.Token.isUnresolved(resourceName)) {
            return resourceName;
        }
        // Secret resource names are in the format `${secretName}-${6-character SecretsManager suffix}`
        // If there is no hyphen (or 6-character suffix) assume no suffix was provided, and return the whole name.
        const lastHyphenIndex = resourceName.lastIndexOf('-');
        const hasSecretsSuffix = lastHyphenIndex !== -1 && resourceName.substr(lastHyphenIndex + 1).length === 6;
        return hasSecretsSuffix ? resourceName.substr(0, lastHyphenIndex) : resourceName;
    }
    throw new Error('invalid ARN format; no secret name provided');
}
/**
 * Parses the secret name from the ARN of an owned secret. With owned secrets we know a few things we don't with imported secrets:
 * - The ARN is guaranteed to be a full ARN, with suffix.
 * - The name -- if provided -- will tell us how many hyphens to expect in the final secret name.
 * - If the name is not provided, we know the format used by CloudFormation for auto-generated names.
 *
 * Note: This is done rather than just returning the secret name passed in by the user to keep the relationship
 * explicit between the Secret and wherever the secretName might be used (i.e., using Tokens).
 */
function parseSecretNameForOwnedSecret(construct, secretArn, secretName) {
    const resourceName = core_1.Stack.of(construct).splitArn(secretArn, core_1.ArnFormat.COLON_RESOURCE_NAME).resourceName;
    if (!resourceName) {
        throw new Error('invalid ARN format; no secret name provided');
    }
    // Secret name was explicitly provided, but is unresolved; best option is to use it directly.
    // If it came from another Secret, it should (hopefully) already be properly formatted.
    if (secretName && core_1.Token.isUnresolved(secretName)) {
        return secretName;
    }
    // If no secretName was provided, the name will be automatically generated by CloudFormation.
    // The autogenerated names have the form of `${logicalID}-${random}`.
    // Otherwise, we can use the existing secretName to determine how to parse the resulting resourceName.
    const secretNameHyphenatedSegments = secretName ? secretName.split('-').length : 2;
    // 2 => [0, 1]
    const segmentIndexes = [...new Array(secretNameHyphenatedSegments)].map((_, i) => i);
    // Create the secret name from the resource name by joining all the known segments together.
    // This should have the effect of stripping the final hyphen and SecretManager suffix.
    return core_1.Fn.join('-', segmentIndexes.map(i => core_1.Fn.select(i, core_1.Fn.split('-', resourceName))));
}
/** Performs a best guess if an ARN is complete, based on if it ends with a 6-character suffix. */
function arnIsComplete(secretArn) {
    return core_1.Token.isUnresolved(secretArn) || /-[a-z0-9]{6}$/i.test(secretArn);
}
//# sourceMappingURL=data:application/json;base64,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