"""Functions for saving proset reports to disk.

Copyright by Nikolaus Ruf
Released under the MIT license - see LICENSE file for details
"""

from copy import deepcopy

import numpy as np
import pandas as pd


CELL_FORMAT = {  # format definitions for xlsxwriter
    "header_blue": {"font_name": "Calibri", "bold": True, "bg_color": "#95D0FC", "border": 1},  # light blue
    "header_green": {"font_name": "Calibri", "bold": True, "bg_color": "#90e4c1", "border": 1},  # light teal
    "float": {"font_name": "Calibri", "num_format": "#,##0.00;[Red]-#,##0.00", "border": 1},
    "integer": {"font_name": "Calibri", "num_format": "#,##0", "border": 1},
    "text": {"font_name": "Calibri", "border": 1},
}
COLUMN_FORMAT = {  # assign widths and cell formats to report columns
    "batch": {"width": 10, "header": "header_blue", "body": "integer"},
    "sample": {"width": 10, "header": "header_blue", "body": "integer"},
    "sample name": {"width": 50, "header": "header_blue", "body": "text"},
    "target": {"width": 10, "header": "header_blue", "body": "integer"},
    "prototype weight": {"width": 20, "header": "header_blue", "body": "float"},
    "similarity": {"width": 20, "header": "header_blue", "body": "float"},
    "impact": {"width": 20, "header": "header_blue", "body": "float"},
    "dominant set": {"width": 20, "header": "header_blue", "body": "integer"},
    "DEFAULT": {"width": 20, "header": "header_green", "body": "float"}
    # use for all columns whose name does not match any key in this dictionary
}


def write_report(file_path, report, column_format=None, cell_format=None):  # pragma: no cover
    """Save results of model.Model.export() or model.Model.explain() as formatted xlsx file.

    :param file_path: string; file name with full or relative path
    :param report: pandas data frame as generated by model.Model.export() or model.Model.explain()
    :param column_format: dict or None; if not None, the dict is used to update the default column formats from
        module-level constant COLUMN_FORMAT
    :param cell_format: dict or None; if not None, the dict is used to update the default cell formats from module-level
        constant CELL_FORMAT
    :return: no return value, file created on disk
    """
    column_format = _update_format(format_=column_format, default=COLUMN_FORMAT)
    cell_format = _update_format(format_=cell_format, default=CELL_FORMAT)
    writer = pd.ExcelWriter(file_path)  # pylint: disable=abstract-class-instantiated
    workbook = writer.book
    cell_format = {key: workbook.add_format(value) for key, value in cell_format.items()}
    worksheet = workbook.add_worksheet("export")
    freeze_rows = np.sum(pd.isna(report["batch"])) + 1
    freeze_cols = np.nonzero(np.logical_or(
        report.columns == "prototype weight", report.columns == "dominant set"
    ))[0][-1] + 1
    worksheet.freeze_panes(freeze_rows, freeze_cols)
    worksheet.autofilter(0, 0, report.shape[0], report.shape[1] - 1)
    report = report.fillna("")
    # this converts all values to string but xlsx determines data format independently of Python anyway
    for i, column in enumerate(report.columns):
        if column in column_format:
            format_ = column_format[column]
        else:
            format_ = column_format["DEFAULT"]
        worksheet.set_column(first_col=i, last_col=i, width=format_["width"])
        worksheet.write(0, i, column, cell_format[format_["header"]])
        worksheet.write_column(1, i, report[column], cell_format[format_["body"]])
    workbook.close()


def _update_format(format_, default):
    """Create complete format definition from user input and default.

    :param format_: dict or None
    :param default: dict
    :return: if format_ is None, returns a copy of default; if format_ is not None, returns a copy of default updated
        with format_
    """
    new_format = deepcopy(default)
    if format_ is not None:
        new_format.update(format_)
    return new_format
