"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Api = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_apigateway_1 = require("aws-cdk-lib/aws-apigateway");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const authorizers_1 = require("../authorizers");
const base_1 = require("../integration/base");
const definition_1 = require("./definition");
/**
 * AWS API Gateway REST API defined with OpenApi v3 schema.
 */
class Api extends aws_apigateway_1.SpecRestApi {
    /**
     * Define a new API Gateway REST API using OpenApi v3 Schema definition.
     *
     * @example
     * const fn: IFunction;
     *
     * new openapix.Api(this, 'MyApi', {
     *   source: './schema.yaml',
     *   paths: {
     *     '/foo': {
     *       get: new openapix.MockIntegration(this),
     *     },
     *     '/bar': {
     *       post: new openapix.LambdaIntegration(this, fn),
     *     },
     *   }
     * })
     *
     */
    constructor(scope, id, props) {
        const apiDefinition = new definition_1.ApiDefinition(scope, {
            source: props.source,
            upload: props.upload === true,
            paths: props.paths || {},
            authorizers: props.authorizers || [],
            validators: props.validators || {},
            defaultCors: props.defaultCors,
            injections: props.injections || {},
            rejections: props.rejections || [],
            rejectionsDeep: props.rejectionsDeep || [],
            defaultIntegration: props.defaultIntegration,
        });
        super(scope, id, {
            apiDefinition: apiDefinition,
            endpointTypes: [aws_apigateway_1.EndpointType.REGIONAL],
            ...props.restApiProps,
        });
        // Allow the API Gateway to invoke given Lambda function integrations
        this.grantLambdaInvokes(props.paths);
        // Allow the API Gateway to invoke given Lambda authorizer integrations
        this.grantLambdaAuthorizerInvokes(props.authorizers);
        // Expose the processed OpenApi v3 document.
        // Mainly used for testing.
        this.document = apiDefinition.document;
    }
    /** Allow Lambda invocations to API Gateway instance principal */
    grantLambdaInvokes(paths = {}) {
        // loop through paths
        Object.keys(paths).forEach(path => {
            const methodIntegrations = paths[path];
            // loop through methods
            Object.keys(methodIntegrations).forEach((method) => {
                const methodIntegration = methodIntegrations[method];
                if (!methodIntegration)
                    return;
                if (this.isLambdaIntegration(methodIntegration) && !methodIntegration.xAmazonApigatewayIntegration.credentials) {
                    methodIntegration.grantFunctionInvoke(this, `ImportForGrant${path}${method}`, this.getApiGatewayPrincipal());
                }
            });
        });
    }
    /** Allow Lambda authorizer invocations to API Gateway instance principal */
    grantLambdaAuthorizerInvokes(authorizers) {
        if (!authorizers) {
            return;
        }
        authorizers.forEach(authorizer => {
            if (authorizer instanceof authorizers_1.LambdaAuthorizer && !authorizer.xAmazonApigatewayAuthorizer.authorizerCredentials) {
                authorizer.grantFunctionInvoke(this);
            }
        });
    }
    /** Determine if the integration internal type is `LAMBDA`. */
    isLambdaIntegration(integration) {
        return integration.type === base_1.InternalIntegrationType.LAMBDA;
    }
    /** Resolve ARN Service Principal for given API Gateway instance. */
    getApiGatewayPrincipal() {
        return new aws_iam_1.ServicePrincipal('apigateway.amazonaws.com', {
            conditions: {
                ArnLike: {
                    'aws:SourceArn': this.arnForExecuteApi(),
                },
            },
        });
    }
}
exports.Api = Api;
_a = JSII_RTTI_SYMBOL_1;
Api[_a] = { fqn: "@alma-cdk/openapix.Api", version: "0.0.45" };
//# sourceMappingURL=data:application/json;base64,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