import ctypes
from typing import List

import numpy as np
import pyart

from ..c_wrapper import DataPair, masked_op
from ..c_wrapper.function_alias import aliases
from ..c_wrapper.run_solo import run_solo_function

se_merge_fields = aliases['merge_fields']


def merge_fields_ray(input_list_data_1: List, input_list_data_2, bad: float, dgi_clip_gate: int = None, boundary_mask: List = None):
    """
        Replaces bad values from input_list_data_1 from values in input_list_data2

        Args:
            input_list_data_1: The list of floats to be modified,
            input_list_data_2: The list of floats that replaces values from input_list_data_1,
            bad: A float that represents a missing/invalid data point,
            (optional) dgi_clip_gate: An integer determines the end of the ray (default: length of input_list)
            (optional) boundary_mask: Defines region over which operations will be done. (default: all True).

        Returns:
          Numpy masked array: Contains an array of data, mask, and fill_value of results.

        Throws:
          ValueError: if input_list and input_boundary_mask are not equal in size,

    """

    args = {
        "data1" : DataPair.DataTypeValue(ctypes.POINTER(ctypes.c_float), input_list_data_1),
        "data2" : DataPair.DataTypeValue(ctypes.POINTER(ctypes.c_float), input_list_data_2),
        "newData" : DataPair.DataTypeValue(np.ctypeslib.ndpointer(ctypes.c_float, flags="C_CONTIGUOUS"), None),
        "nGates" : DataPair.DataTypeValue(ctypes.c_size_t, None),
        "bad" : DataPair.DataTypeValue(ctypes.c_float, bad),
        "dgi_clip_gate" : DataPair.DataTypeValue(ctypes.c_size_t, dgi_clip_gate),
        "boundary_mask" : DataPair.DataTypeValue(ctypes.POINTER(ctypes.c_bool), boundary_mask),
    }

    return run_solo_function(se_merge_fields, args)


def merge_fields_masked(masked_array, reference_masked_array, boundary_masks: List = None):
    """
        Replaces bad values from input_list_data_1 from values in input_list_data2

        Args:
            masked_array: A numpy masked array that is to be modified.
            reference_masked_array: A numpy masked array with values that replace masked_array on bad entries.

        Returns:
            Numpy masked array

        Throws:
            ModuleNotFoundError: if numpy is not installed
            AttributeError: if masked_array arg is not a numpy masked array.
    """
    return masked_op.masked_func(merge_fields_ray, masked_array,  boundary_masks = boundary_masks, second_masked_array=reference_masked_array)


def merge_fields_field(radar: pyart.core.Radar, field: str, field_ref: str, new_field: str, boundary_masks=None, sweep: int = 0):

    with masked_op.SweepManager(radar, sweep, field, new_field) as sm:
        sm.new_masked_array = merge_fields_masked(sm.radar_sweep_data, radar.get_field(sweep, field_ref), boundary_masks)
