"""profiling and introspection utilities"""
from __future__ import annotations

import _ctypes
import sys
import warnings
from collections import defaultdict
from functools import partial
from inspect import currentframe, getsourcelines, stack
from itertools import chain
import re
from types import EllipsisType, FrameType, NoneType, NotImplementedType
from typing import (
    Any,
    Callable,
    Collection,
    Hashable,
    Literal,
    Mapping,
    MutableMapping,
    Optional,
    Sequence,
    Union,
)

from cytoolz import keyfilter, valmap
from pympler.asizeof import asizeof

from hostess.monitors import (
    AbstractMonitor,
    make_stat_printer,
    make_stat_records,
    Stopwatch,
)
from hostess.utilities import mb


class Profiler:
    """
    simple profiling object for specific sections of code.

    Examples:
        >>> from array import array
        >>> from hostess.monitors import RAM, Stopwatch
        >>> from hostess.profilers import Profiler

        >>> prof = Profiler({'time': Stopwatch(), 'memory': RAM()})
        >>> with prof.context("f"):
            >>> var1 = array("B", [0 for _ in range(1024**2 * 100)])
        >>> with prof.context("g"):
            >>> var2 = array("B", [0 for _ in range(1024**2 * 250)])
        >>> print(prof)

        general form of expected output (exact results are system-dependent):
        ```
        Profiler
        f
          time: 2.935
          memory: 105.47
        g
          time: 7.171
          memory: 261.76
        ```
    """

    def __init__(self, monitors: MutableMapping[str, AbstractMonitor]):
        """
        Args:
            monitors: dictionary of AbstractMonitor objects
                (see hostess.monitors for examples)
        """
        self.monitors = monitors
        self.printstats = make_stat_printer(self.monitors)
        self.recordstats = make_stat_records(self.monitors)
        for k, v in self.monitors.items():
            v.default_display = "interval"
        self.labels = defaultdict(self._newcaches)

    def start(self):
        """start all the monitors."""
        for v in self.monitors.values():
            v.start()

    def pause(self):
        """pause all the monitors."""
        for v in self.monitors.values():
            v.pause()

    def restart(self):
        """restart all the monitors."""
        for v in self.monitors.values():
            v.restart()

    def context(self, label: str = "") -> PContext:
        """
        create a context manager that profiles a section of code.

        Args:
            label: label for the code section, possibly shared between
                multiple sections. useful when it is desirable to distinguish
                specific steps of a pipeline, 'categories' of activity, etc.
        """
        return PContext(self, label)

    def reset(self):
        """clear this Profiler, removing all existing readings."""
        self.labels = defaultdict(self._newcaches)

    def _newcaches(self) -> dict:
        """internal data structure initialization function."""
        caches = {}
        for k, v in self.monitors.items():
            if v.qualities is not None:
                caches[k] = {q: 0 for q in v.qualities}
            else:
                caches[k] = 0
        return caches

    def __str__(self):
        if len(self.labels) == 0:
            return f"Profiler (no readings)"
        output = "Profiler\n"
        for k, v in self.labels.items():
            output += f"{k:}\n"
            for m, r in v.items():
                output += f"  {m}: {r}\n"
        return output

    def __repr__(self):
        return self.__str__()


class PContext:
    """
    simple context manager for profiling. typically instantiated via a
    Profiler's context() method, though this is not mandatory.
    """

    def __init__(self, profiler: Profiler, label: str = ""):
        """
        Args:
            profiler: associated Profiler; readings generated by this PContext
                will be stored in that Profiler's labels data structure.
            label: optional label for this PContext's profiling results.
        """
        self.profiler = profiler
        self.label = label

    def __enter__(self):
        self.profiler.restart()
        return self.profiler

    def __exit__(self, *args):
        self.profiler.pause()
        record = self.profiler.recordstats()
        for monitor, reading in record.items():
            self._save_reading(monitor, reading)

    def _save_reading(self, monitor: str, reading: Union[int, float, Mapping]):
        """
        internal function called on context block exit. saves profiling
        results to the associated Profiler.
        """
        if isinstance(reading, (float, int)):
            self.profiler.labels[self.label][monitor] += reading
        else:
            for quality, value in reading.items():
                self.profiler.labels[self.label][monitor][quality] += value


def scopedicts(
    frame: FrameType,
    scopes: Sequence[ScopeName] = ("locals", "globals", "builtins"),
) -> tuple[dict, ...]:
    """
    fetch specified scopes from a frame and return them in a tuple. the
    elements of the tuple should be equivalent to the results of calling,
    e.g., locals() within the passed frame.

    WARNING: caller is responsible for clearing references to locals, etc.
    calling this function with no cleanup deep in a call stack may lead to
    undesired dangling references.

    Args:
        frame: frame (as generated by, e.g., inspect.currentframe()) from
            which to fetch scopes.
        scopes: names of scopes to fetch from frame.

    Returns:
        tuple of dictionaries representing the specified scopes of frame;
            their keys are variable names and their values are the associated
            objects.
    """
    outscopes = []
    for scope in scopes:
        outscopes.append(getattr(frame, f"f_{scope}"))
    return tuple(outscopes)


def val_ids(mapping: Mapping[Hashable, Any]) -> set[int]:
    """
    get ids of all values in mapping.

    Args:
        mapping: mapping whose values are the objects to id.

    Returns:
        set of ids of all objects in mapping's values.
    """
    return set(map(id, mapping.values()))


def _maybe_release_locals(localdict: MutableMapping, frame: FrameType) -> bool:
    """
    Possibly purge a dictionary, depending on the name of `frame`'s code.

    Tedious description of technical rationale:

    the `locals` dict of the top-level module frame is the same as its
    `globals` dict. retrieving it from a frame gives us the _actual_ `globals`
    `dict`, not a copy of it. we do NOT want to casually delete all members
    of the top-level module while pretending to merely inspect it.

    conversely, `locals` `dicts` retrieved from lower frames are only copies.
    modifying them will not affect the actual namespaces of those frames.
    HOWEVER, references to everything in those copies will hang around forever
    until _that_ frame fully dies, which, in most programs, will badly confuse
    the Python garbage collector and cause horrible memory leaks. clearing the
    copies is the only reliable way to prevent that from happening.

    Args:
        localdict: a dict that might be a copy of `frame`'s locals, or might
            be an actual view into its locals
        frame: frame `localdict` came from

    Returns:
        True if we cleared `localdict`,, False we didn't

    Note:
        Always skipped in Python>=3.13, as the introduction of FrameLocalsProxy
            makes it unnecessary.
    """
    if frame.f_code.co_name != "<module>":
        if hasattr(localdict, "clear"):
            localdict.clear()
            return True
    return False


def namespace_ids(
    frames: Union[FrameType, Collection[FrameType], None] = None,
    include_frame_ids=False,
) -> set[int]:
    """
    find ids of all top-level objects in the combined namespace(s) of
    a frame or frames.
    """
    if frames is None:
        frames = [currentframe().f_back]
    ids = set()
    for frame in frames:
        localdict, globaldict, builtindict = scopedicts(frame)
        ids.update(chain(*map(val_ids, (localdict, globaldict, builtindict))))
        _maybe_release_locals(localdict, frame)
    if include_frame_ids is True:
        ids.update(map(id, frames))
    return ids


def _add_scopedict_ids(frame, ids, lids, scopenames):
    sdicts = {k: v for k, v in zip(scopenames, scopedicts(frame, scopenames))}
    for k, v in sdicts.items():
        sid = id(v)
        ids.add(sid)
        if k == "locals":
            if _maybe_release_locals(v, frame) is True:
                lids.add(sid)


ScopeName = Literal["locals", "globals", "builtins"]
"""
string that gives the name of a Python scope, not including enclosing/nonlocal 
scope.
"""


def scopedict_ids(
    frames: Union[FrameType, Collection[FrameType], None] = None,
    *,
    getstack: bool = False,
    scopenames: Sequence[ScopeName] = ("locals", "globals", "builtins"),
    distinguish_locals: bool = True,
):
    """
    return ids of all 'scopedicts' (locals, globals, builtins) in frames (by
    default, just the caller's frame.) uses include: distinguishing
    references held by namespaces from references held by other objects;
    avoiding accidental 'direct' manipulation of namespaces.

    Args:
        frames: a single frame, a collection of frames, or None. if None,
            get ids of scopedicts of the caller's frame.
        getstack: if True, ignore the frames argument and instead look at
            all levels of the stack above the caller's frame.
        scopenames: names of scopes to fetch.
        distinguish_locals: if True, return a tuple whose elements are:
            [0] all ids
            [1] just local-scope ids below top level
    """
    if getstack is True:
        frames = [s.frame for s in stack()[:-1]]
    frames = frames if frames is not None else [currentframe().f_back]
    ids, lids = set(), set()
    for frame in frames:
        _add_scopedict_ids(frame, ids, lids, scopenames)
    if distinguish_locals is True:
        return ids, lids
    return ids


def lineno() -> int:
    """Returns the current line number in our program."""
    return currentframe().f_back.f_lineno


def def_lineno(obj) -> Optional[int]:
    """Returns the line number where the object was defined, if available."""
    try:
        return getsourcelines(obj)[1]
    except TypeError:
        return None


IdentifyResult = dict[str, Union[int, type, str]]
"""
record representing information about a Python object, as produced by 
`identify` and functions that call it.
"""


def identify(
    obj: Any, maxlen: int = 25, getsize: bool = False
) -> IdentifyResult:
    """
    identify an object.

    Args:
        obj: object to identify
        maxlen: maximum length of string representation of `obj` in return
        getsize: if True, attempt to determine the size of `obj`. may be slow
            or unreliable.

    Returns:
        dict giving id, type, string
            representation (possibly truncated), size in MB (if requested),
            and, if available, __name__, __qualname__, and __module__, and
            line number.
    """
    identifiers = {
        "id": id(obj),
        "type": type(obj),
        "line": def_lineno(obj),
        "r": str(obj)[:maxlen],
    }
    if getsize is True:
        identifiers["size"] = mb(asizeof(obj), 2)
    for attr in ("__name__", "__qualname__", "__module__"):
        if hasattr(obj, attr):
            identifiers[attr] = getattr(obj, attr)
    return identifiers


def describe_frame_contents(frame=None):
    """describe the contents of a frame"""
    frame = frame if frame is not None else currentframe().f_back
    try:
        return {
            "filename": frame.f_code.co_filename,
            "lineno": frame.f_lineno,
            "name": frame.f_code.co_name,
            "locals": valmap(identify, frame.f_locals),
        }
    finally:
        _maybe_release_locals(frame.f_locals, frame)


def describe_stack_contents():
    """describe the contents of the stack"""
    return tuple(map(describe_frame_contents, [s[0] for s in stack()]))


def _yclept_framerec(frame: FrameType):
    """
    return terse information about a frame's name and contents. for yclept.
    """
    return {
        "co_names": frame.f_code.co_names,
        "func": frame.f_code.co_name,
        "qual": frame.f_code.co_qualname,
        "varnames": frame.f_code.co_varnames,
    }


Refnom = tuple[
    IdentifyResult, list[dict[str, Union[tuple[str], str, set[str]]]]
]
LITERAL_TYPES = (
    str,
    float,
    int,
    bool,
    slice,
    EllipsisType,
    NoneType,
    NotImplementedType,
)


def _add_varnames(obj, sdict, rec, scopename):
    """
    helper function for yclept(). creates records describing the variables in
    a given scope.
    """
    for varname, reference in sdict.items():
        if obj is reference:
            rec["names"].append(varname)
            rec["scopes"].append(scopename)


def yclept(obj: Any, terse: bool = True, stepback: int = 1) -> Refnom:
    """
    Find basic identifiers for obj, along with any names for obj in all frames
    in stack, starting stepback frames back from the frame of this function.

    Args:
        obj: object to name
        terse: exclude extended information from output?
        stepback: how many frames to step back (from the frame of this
            function) before looking for obj?
    """
    nytta = []
    frame = currentframe()
    for _ in range(stepback):
        frame = frame.f_back
    while frame is not None:
        rec = _yclept_framerec(frame) | {"names": [], "scopes": []}
        if terse is True:
            rec = keyfilter(
                lambda k: k in ("func", "qual", "names", "scopes"), rec
            )
        localdict, globaldict, builtindict = scopedicts(frame)
        _add_varnames(obj, globaldict, rec, "globals")
        _add_varnames(obj, builtindict, rec, "builtins")

        if frame.f_code.co_name != "<module>":
            # don't bother adding redundant local varnames at top level
            _add_varnames(obj, localdict, rec, "locals")
            if hasattr(localdict, "clear"):
                localdict.clear()  # see _maybe_release_locals
        del globaldict, localdict, builtindict
        if len(rec["names"]) > 0:
            rec["names"] = tuple(rec["names"])
            rec["scopes"] = tuple(rec["scopes"])
            nytta.append(rec)
        frame = frame.f_back
    res = identify(obj, maxlen=55, getsize=False), nytta
    del obj  # explicitly releasing obj clears ref faster
    return res


def _filter_types(
    refs: list,
    permit: Collection[type],
    exclude: Collection[type],
    lids: Collection[int],
) -> list:
    """
    helper function for analyze_references(). provides blocklist/allowlist
    behavior based on object types.

    Args:
        refs: list of referencing objects
        permit: types to explicitly permit. if there is at least one type,
            this functions as a strict allowlist.
        exclude: types to explicitly exclude.
        lids: ids of all known copies of other frames' `locals` dicts.

    Returns:
        filtered list of references.
    """
    outrefs = []
    for ref in refs:
        reftype = type(ref)
        try:
            assert reftype not in exclude
            assert not (len(permit) > 0 and (reftype not in permit))
            outrefs.append(ref)
        except AssertionError:
            if id(ref) in lids:
                ref.clear()
    return outrefs


def history_filter(glb: dict[str, Any]) -> Callable[[Any], bool]:
    """
    generate a predicate function that attempts to filter jupyter/ipython
    history-related objects in the context of a particular global namespace.

    Args:
        glb: relevant globals dict

    Returns:
        function that returns False if its single argument looks like it's an
            ipython/jupyter history-related object or internal, and True if
            not.
    """

    def filterref(item):
        if item.__class__.__name__ == "ZMQShellDisplayHook":
            return False
        if item.__class__.__name__ == "ExecutionResult":
            return False
        try:
            globalname = next(filter(lambda kv: kv[1] is item, glb.items()))[0]
        except StopIteration:
            return True
        if re.match(r"_+(i{1,3})?\d?", globalname):
            return False
        if globalname in ("In", "Out", "_ih", "_oh", "_dh"):
            return False
        return True

    return filterref


def _filter_history(
    refs: list, globals_: Optional[dict], lids: Collection[int]
) -> list:
    """
    helper function for analyze_references(). attempts to remove references to
    ipython/jupyter history variables.

    Args:
        refs: list of reference objects
        globals_: optional specified globals dictionary. if not given, uses
            the globals of the parent frame of the caller.
        lids: ids of all known copies of other frames' locals dicts.

    Returns:
        filtered list of references.
    """
    if globals_ is None:
        globals_ = currentframe().f_back.f_back.f_globals
    outrefs, hfilt = [], history_filter(globals_)
    for ref in refs:
        if history_filter(globals_):
            outrefs.append(ref)
        elif id(ref) in lids:
            ref.clear()
    return outrefs


def _filter_ids(
    refs: list,
    permit: Collection[int],
    exclude: Collection[int],
    lids: Collection[int],
) -> tuple[list[Any], list[Refnom]]:
    """
    helper function for analyze_references(). provides blocklist/allowlist
    behavior based on object ids.

    Args:
        refs: list of reference objects
        permit: list of allowed ids. if any are given, this functions as a
            strict allowlist.
        exclude: list of excluded ids
        lids: ids of all known copies of other frames' locals dicts

    Returns:
        filtered_refs: filtered list of references
        refnoms: Refnom dicts for each member of filtered_refs
    """
    outrefs, refnoms = [], []
    for ref in refs:
        try:
            assert id(ref) not in exclude
            assert (len(permit) == 0) or (id(ref) in permit)
            outrefs.append(ref)
            refnoms.append(yclept(ref, stepback=2))
        except AssertionError:
            if id(ref) in lids:
                ref.clear()
    return refnoms, outrefs


def _get_referencing_scopedicts(
    obj: Any, existing_ids: Collection[int]
) -> list[dict]:
    """
    check for `obj` in the globals and locals dicts of all stack frames above
    the caller's. return all dicts in which obj has a name or names.

    Args:
        obj: object to check for
        existing_ids: ids of scopedicts that should be ignored

    Returns:
        list of locals and globals dicts that reference obj
    """
    outscopes = []
    # if you do NOT slice the stack to -2, it will create a reference cycle
    # from the local namespace of the caller to itself, preventing it from ever
    # being garbage collected.
    for scopedict in chain(*[scopedicts(s.frame) for s in stack()[:-2]]):
        if id(scopedict) in existing_ids:
            continue
        if id(obj) in map(id, scopedict.values()):
            outscopes.append(scopedict)
    return outscopes


# TODO, maybe: option to explicitly check variables of higher level namespaces
#  for references to obj (not references from those namespaces to obj, but to
#  other members of the namespaces)
def analyze_references(
    obj: Any,
    method: Callable[[Any], Collection],
    *,
    filter_primitive: bool = True,
    filter_history: bool = True,
    filter_scopedict: bool = True,
    filter_reflexive: bool = True,
    exclude_ids: Collection[int] = frozenset(),
    exclude_types: Collection[type] = frozenset(),
    permit_ids: Collection[int] = frozenset(),
    permit_types: Collection[type] = frozenset(),
    globals_: Optional[dict[str, Any]] = None,
    return_objects: bool = False,
) -> Union[tuple[list[Refnom], list[Any]], list[Refnom]]:
    """
    analyze 'references' to or from obj. designed for, but not limited to,
    analyzing references tracked by the Python garbage collector.

    Danger: careful use is required to avoid memory leaks
        TAKE SPECIAL CARE WHEN DECORATING THIS FUNCTION OR CALLING IT FROM
        A LAMBDA FUNCTION OR GENERATOR EXPRESSION, NO MATTER HOW HARMLESS-
        LOOKING. These operations may add references that are difficult to
        recognize or interpret. Calls that do not add context are much safer.

    Notes:
        1. All 'exclude', 'permit', and 'filter' operations are implicitly
            connected by boolean AND. Represented as a predicate:

                (~PRIMITIVE(REF) | ~FILTER_PRMITIVE)
                & (~HISTORY(REF) | ~FILTER_HISTORY)
                & (~SCOPEDICT(REF) | ~FILTER_SCOPEDICT)
                & ((ID(REF) != ID(OBJ)) | ~FILTER_REFLEXIVE)
                & ~(ID(REF) ∈ EXCLUDE_IDS)
                & (ID(REF) ∈ PERMIT_IDS | PERMIT_IDS = ∅)
                & ~(TYPE(REF) ∈ EXCLUDE_TYPES)
                & (TYPE(REF) ∈ PERMIT_TYPES | PERMIT_TYPES = ∅)

        2. References from obj to itself are never included. This may change
           in the future.
        3. This function is only completely compatible with CPython.


    Args:
        obj: object of referential analysis
        method: Function whose return values define 'references' of
            obj. gc.get_referents and gc.get_referrers are the intended and
            tested values.
        filter_primitive: ignore 'primitive' (str, bool, &c) objects?
        filter_history: attempt to ignore ipython 'history' objects?\
        filter_scopedict: ignore _direct_ references to the locals, globals,
            and builtins dicts of all frames in stack (_not_ the values of
            these dictionaries?)
        filter_reflexive: ignore references from obj to itself?
        exclude_ids: denylist of reference ids.
        exclude_types: denylist of reference types.
        permit_ids: allowlist of reference ids.
        permit_types: allowlist of reference types.
        return_objects: return objects in set of references, or only
            descriptions of those objects?
        globals_: optional dictionary of globals to use in filtering.
            currently only used in history filtering. If this argument is
            None, history filtering uses the globals of the calling frame.
    """
    refs = list(method(obj))
    objid = id(obj)
    exclude_types, permit_types = set(exclude_types), set(permit_types)
    del obj  # explicitly releasing obj clears reference from frame faster
    if filter_primitive is True:
        exclude_types.update(LITERAL_TYPES)
    # ensure we can always clear copies of locals dicts we might have received
    # from the method(obj) call
    sids, lids = scopedict_ids(getstack=True, distinguish_locals=True)
    # type exclusions are easier to perform here. id exclusions need to come
    # at the end of the function in order to filter the objects in this
    # namespace.
    if len(exclude_types) + len(permit_types) > 0:
        refs = _filter_types(refs, permit_types, exclude_types, lids)
    if filter_history is True:
        refs = _filter_history(refs, globals_, lids)
    exclude_ids = set(exclude_ids)
    if filter_scopedict is True:
        exclude_ids.update(sids)
    # the frame of this function, along with all objects in its namespace, are
    # always excluded from analysis
    exclude_ids.update(namespace_ids(include_frame_ids=True))
    # do this via the negative in case a copy of obj was hanging around here
    # somehow
    if filter_reflexive is False:
        exclude_ids.difference_update({objid})
    # TODO, maybe -- consider also allowing arguments to this function to be
    #  included in analysis
    refnoms, outrefs = _filter_ids(refs, permit_ids, exclude_ids, lids)
    if return_objects is True:
        return refnoms, outrefs
    return refnoms


# noinspection PyUnresolvedReferences
def di(obj_id: int) -> Any:
    """
    backwards `id`. Use with care! Can segfault.

    Args:
        obj_id: id of desired object, as returned by `id(obj)`.

    Returns:
        Object corresponding to `obj_id`.
    """
    return _ctypes.PyObj_FromPtr(obj_id)


def val_refs(refmap: Mapping[str, Any]) -> dict[str, int]:
    """
    produce dict containing refcounts of all values in refmap.

    Args:
        refmap: dict to count.

    Returns:
        `dict` whose keys are the same as `refmap`'s and whose
            values are (sys.refcount(v) - 1) for the associated
            values of `refmap`.
    """
    return {k: sys.getrefcount(v) - 1 for k, v in refmap.items()}


class RefAlarmWarning(UserWarning):
    """warning issued by `RefAlarms` with "warn" `verbosity`"""


# TODO: integrate reference filter workflow; possibly requires
#  refactoring that workflow
def refcounts(
    getstack: bool = False, stepback: int = 1
) -> list[dict[str, Union[str, dict[str, int]]]]:
    """
    get refcounts of named variables in one or all frames in the stack.

    Args:
        getstack: check all frames in the stack above the starting frame?
        stepback: number of frames to step back from the frame of this
            function before counting

    Returns:
        A list of dicts, one for each counted frame (always length 1 if
            `getstack` is `False`). Keys are 'name' (co_name of frame code)
            and 'refs' (dict whose keys are variable names and values are
            refcounts of those variables).
    """
    if getstack is True:
        # always ignore current frame
        frames = [s.frame for s in stack()[stepback:]]
    else:
        frame = currentframe()
        for i in range(stepback):
            frame = frame.f_back
        frames = [frame]
    framerefs = []
    from types import FrameType

    frame: FrameType
    for i, frame in enumerate(frames):
        # TODO: optional globals; need to handle name collisions somehow
        localdict, _, _ = scopedicts(frame)
        counts = val_refs(localdict)
        _maybe_release_locals(localdict, frame)
        framerefs.append({"name": frame.f_code.co_name, "counts": counts})
        del frame  # paranoia
    del frames  # paranoia
    return framerefs


class RefAlarm:
    """Simple reference / dereference alarm."""

    def __init__(
        self,
        getstack: bool = False,
        verbosity: Literal["warn", "print", "quiet"] = "print",
        warn_new: bool = False,
        ignore_dunder: bool = True,
    ):
        """
        Args:
            getstack: if True, check entire stack above frame that
                initializes self.context(); if False, just that frame
            verbosity: "warn" means issue RefAlarmWarnings; "print" means
                call `print`; "quiet" means no output (user must check
                `self.refcaches` to see results)
            warn_new: warn / print newly-assigned variables? does nothing
                if `verbosity == "quiet"`.
            ignore_dunder: ignore variables with "dunder" names?
        """
        self.verbosity = verbosity
        self.warn_new, self.getstack = warn_new, getstack
        self.ignore_dunder = ignore_dunder
        self.refcaches = defaultdict(list)

    def context(self, name: str = "default") -> _RefAlarmContext:
        """
        Produce a context manager related to this object.

        Args:
            name: name for this context, used for verbose reports and
                `refcache` keys

        Returns:
            A `_RefAlarmContext` suitable for use in a `with` statement.
        """
        return _RefAlarmContext(self, name, self.getstack, self.ignore_dunder)

    def receive_context_report(self, results: list[dict], name: str):
        """
        receive a context report. called by `_RefAlarmContexts` produced
        by this object's `context()` method.

        Args:
            results: `results` object from `_RefAlarmContext.__exit__()`
            name: name of context
        """
        self.refcaches[name].append(results)
        if self.verbosity == "quiet":
            return
        if self.verbosity == "print":
            printer = print
        else:
            printer = partial(warnings.warn, category=RefAlarmWarning)
        pre = "" if name == "default" else f"{name}: "
        for i, r in enumerate(results):
            fname, mismatches = r["name"], r["mismatches"]
            title = f"{i} ({fname})"
            for k, v in mismatches.items():
                if v == "new":
                    if self.warn_new is False:
                        continue
                    printer(f"{pre}{title}: {k} is new")
                elif v == "missing":
                    printer(f"{pre}{title}: {k} is missing")
                else:
                    printer(f"{pre}{title}: {k} refcount changed by {v}")


class _RefAlarmContext:
    """
    context manager for reference counting. should be initialized only by
    RefAlarm.context().
    """

    def __init__(
        self,
        refalarm: RefAlarm,
        name: str,
        getstack: bool = False,
        ignore_dunder: bool = True,
    ):
        self.refalarm, self.name = refalarm, name
        self.getstack = getstack
        self.refcache, self.ignore_dunder = None, ignore_dunder

    def __enter__(self):
        # drop refs from this frame
        self.refcache = refcounts(self.getstack, 2)

    def __exit__(self, *_):
        results = []
        # avoid, paranoiacally, calling this inside an expression,
        # and also drop refs from this frame
        counts = refcounts(self.getstack, 2)
        # if the size of the stack changed, something weird happened and
        # we cannot produce accurate results
        if len(self.refcache) != len(counts):
            warnings.warn("stack changed during counting, bailing out")
            return
        for old, new in zip(self.refcache, counts):
            oldname, oldcounts = old["name"], old["counts"]
            newname, newcounts = new["name"], new["counts"]
            # if the code name of a frame changed, something weird happened
            # and we cannot produce accurate results
            if oldname != newname:
                warnings.warn("stack changed during counting, bailing out")
                return
            mismatches = {}
            for k in set(oldcounts.keys()).union(newcounts.keys()):
                if self.ignore_dunder is True and k.startswith("__"):
                    continue
                if k not in oldcounts.keys():
                    mismatches[k] = "new"
                elif k not in newcounts.keys():
                    mismatches[k] = "missing"
                elif (refdiff := newcounts[k] - oldcounts[k]) != 0:
                    mismatches[k] = refdiff
            results.append({"name": newname, "mismatches": mismatches})
        self.refalarm.receive_context_report(results, self.name)


DEFAULT_PROFILER = Profiler({"time": Stopwatch()})
"""convenient shared Profiler that measures execution times for code blocks."""
