import time
from typing import Optional

from botocore.client import BaseClient


def create_gateway_instance(
    ec2_client: BaseClient,
    subnet_id: Optional[str],
    bucket_name: str,
    instance_name: str,
    ssh_key_pub: str,
    machine_type: str = "t2.micro",
) -> dict:
    launch_specification = {}
    if subnet_id:
        launch_specification["NetworkInterfaces"] = [
            {
                "AssociatePublicIpAddress": True,
                "DeviceIndex": 0,
                "SubnetId": subnet_id,
                "Groups": [gateway_security_group_id(ec2_client, subnet_id, bucket_name)],
            },
        ]
    else:
        launch_specification["SecurityGroupIds"] = [
            gateway_security_group_id(ec2_client, subnet_id, bucket_name)
        ]
    tags = [
        {"Key": "Name", "Value": instance_name},
        {"Key": "owner", "Value": "dstack"},
        {"Key": "role", "Value": "gateway"},
        {"Key": "dstack_bucket", "Value": bucket_name},
    ]
    response = ec2_client.run_instances(
        BlockDeviceMappings=[
            {
                "DeviceName": "/dev/sda1",
                "Ebs": {
                    "VolumeSize": 10,
                    "VolumeType": "gp2",
                },
            }
        ],
        ImageId="ami-0cffefff2d52e0a23",  # Ubuntu 22.04 LTS
        InstanceType=machine_type,
        MinCount=1,
        MaxCount=1,
        UserData=gateway_user_data_script(ssh_key_pub),
        TagSpecifications=[
            {
                "ResourceType": "instance",
                "Tags": tags,
            },
        ],
        **launch_specification,
    )
    return wait_till_running(ec2_client, response["Instances"][0])


def gateway_security_group_id(
    ec2_client: BaseClient, subnet_id: Optional[str], bucket_name: str
) -> str:
    name_parts = ["dstack_gateway_sg"]
    if subnet_id:
        name_parts.append(subnet_id.replace("-", "_"))
    name_parts.append(bucket_name.replace("-", "_"))
    security_group_name = "_".join(name_parts)
    response = ec2_client.describe_security_groups(
        Filters=[
            {
                "Name": "group-name",
                "Values": [
                    security_group_name,
                ],
            },
        ],
    )
    if response.get("SecurityGroups"):
        return response["SecurityGroups"][0]["GroupId"]

    group_specification = {}
    if subnet_id:
        subnets_response = ec2_client.describe_subnets(SubnetIds=[subnet_id])
        group_specification["VpcId"] = subnets_response["Subnets"][0]["VpcId"]
    security_group = ec2_client.create_security_group(
        Description="Generated by dstack",
        GroupName=security_group_name,
        TagSpecifications=[
            {
                "ResourceType": "security-group",
                "Tags": [
                    {"Key": "owner", "Value": "dstack"},
                    {"Key": "role", "Value": "gateway"},
                ],
            },
        ],
        **group_specification,
    )
    security_group_id = security_group["GroupId"]
    ip_permissions = [
        {
            "FromPort": 0,
            "ToPort": 65535,
            "IpProtocol": "tcp",
            "IpRanges": [{"CidrIp": "0.0.0.0/0"}],
        }
    ]
    ec2_client.authorize_security_group_ingress(
        GroupId=security_group_id, IpPermissions=ip_permissions
    )
    ec2_client.authorize_security_group_egress(
        GroupId=security_group_id,
        IpPermissions=[
            {
                "IpProtocol": "-1",
            }
        ],
    )
    return security_group_id


def wait_till_running(
    ec2_client: BaseClient, instance: dict, delay: int = 5, attempts: int = 30
) -> dict:
    instance_id = instance["InstanceId"]
    attempt = 0
    while instance["State"]["Name"] != "running":
        if attempt >= attempts:
            raise RuntimeError(f"Instance {instance_id} is not running")
        time.sleep(delay)
        attempt += 1
        desc = ec2_client.describe_instances(InstanceIds=[instance_id])
        instance = desc["Reservations"][0]["Instances"][0]
    return instance


def get_instance_id(ec2_client: BaseClient, instance_name: str) -> str:
    desc = ec2_client.describe_instances(
        Filters=[
            {
                "Name": "tag:Name",
                "Values": [instance_name],
            }
        ]
    )
    return desc["Reservations"][0]["Instances"][0]["InstanceId"]


def gateway_user_data_script(ssh_key_pub: str) -> str:
    return f"""#!/bin/bash
sudo apt-get update
DEBIAN_FRONTEND=noninteractive sudo apt-get install -y -q nginx
UBUNTU_UID=$(id -u ubuntu)
UBUNTU_GID=$(id -g ubuntu)
install -m 700 -o $UBUNTU_UID -g $UBUNTU_GID -d /home/ubuntu/.ssh
install -m 600 -o $UBUNTU_UID -g $UBUNTU_GID /dev/null /home/ubuntu/.ssh/authorized_keys
echo "{ssh_key_pub}" > /home/ubuntu/.ssh/authorized_keys
WWW_UID=$(id -u www-data)
WWW_GID=$(id -g www-data)
install -m 700 -o $WWW_UID -g $WWW_GID -d /var/www/.ssh
install -m 600 -o $WWW_UID -g $WWW_GID /dev/null /var/www/.ssh/authorized_keys"""
