#!/usr/bin/env python3
import sys
from pathlib import Path

# Добавляем parent (src) в sys.path для локального запуска
sys.path.insert(0, str(Path(__file__).resolve().parents[1]))

# Сначала импортируем настройки без импорта логгера
from penguin_tamer.config_manager import config


# Простой ленивый логгер
_logger = None


def _get_logger():
    global _logger
    if _logger is None:
        from penguin_tamer.logger import configure_logger
        _logger = configure_logger(config.get("logging"))
    return _logger


# Простой класс-заглушка для логгера
class LazyLogger:
    def info(self, msg):
        _get_logger().info(msg)

    def debug(self, msg):
        _get_logger().debug(msg)

    def error(self, msg):
        _get_logger().error(msg)

    def critical(self, msg, exc_info=None):
        _get_logger().critical(msg, exc_info=exc_info)

logger = LazyLogger()

# Простой декоратор без импорта
def log_execution_time(func):
    """Простой декоратор времени выполнения"""
    return func  # Временно отключаем для ускорения загрузки


# Ленивый импорт i18n
_i18n_initialized = False


def _ensure_i18n():
    global _i18n_initialized, t, translator
    if not _i18n_initialized:
        from penguin_tamer.i18n import t, translator
        # Initialize translator language from config (default 'en')
        try:
            translator.set_language(getattr(config, 'language', 'en'))
        except Exception:
            pass
        _i18n_initialized = True


def t_lazy(text, **kwargs):
    """Ленивая загрузка переводчика"""
    _ensure_i18n()
    return t(text, **kwargs)


# Используем t_lazy вместо t для отложенной инициализации
t = t_lazy

# Ленивые импорты для ускорения загрузки
_prompt_toolkit_imported = False
_rich_console = None
_script_executor = None
_formatter_text = None

def _ensure_prompt_toolkit():
    """Ленивый импорт prompt_toolkit"""
    global _prompt_toolkit_imported
    if not _prompt_toolkit_imported:
        global HTML, prompt, FileHistory, Style, Lexer, PygmentsLexer
        from prompt_toolkit import HTML, prompt
        from prompt_toolkit.history import FileHistory
        from prompt_toolkit.styles import Style
        from prompt_toolkit.lexers import Lexer, PygmentsLexer
        from prompt_toolkit.layout.processors import Processor, Transformation
        _prompt_toolkit_imported = True


def _get_console():
    """Ленивый импорт rich.console"""
    global _rich_console
    if _rich_console is None:
        from rich.console import Console
        _rich_console = Console
    return _rich_console


def _get_script_executor():
    """Ленивый импорт script_executor"""
    global _script_executor
    if _script_executor is None:
        from penguin_tamer.script_executor import run_code_block
        _script_executor = run_code_block
    return _script_executor


def _get_execute_handler():
    """Ленивый импорт execute_and_handle_result для выполнения команд"""
    global _execute_handler
    if '_execute_handler' not in globals() or _execute_handler is None:
        from penguin_tamer.script_executor import execute_and_handle_result
        _execute_handler = execute_and_handle_result
    return _execute_handler


def _create_dot_command_lexer():
    """Создает безопасный лексер для подсветки команд с точкой"""
    try:
        from prompt_toolkit.lexers import Lexer
        
        class DotHighlightLexer(Lexer):
            """Безопасный лексер для подсветки точки серым, команды желтым"""
            
            def lex_document(self, document):
                def get_line_tokens(line_number):
                    if line_number != 0:  # Работаем только с первой строкой
                        return []
                    
                    text = document.text
                    if not text:
                        return []
                    
                    # Если начинается с точки
                    if text.startswith('.'):
                        result = []
                        # Точка серая
                        result.append((0, 'class:dot'))
                        # Остальное желтое  
                        for i in range(1, len(text)):
                            result.append((i, 'class:command'))
                        return result
                    else:
                        return []
                
                return get_line_tokens
        
        return DotHighlightLexer()
        
    except ImportError:
        logger.debug("prompt_toolkit.lexers недоступен")
        return None
    except Exception as e:
        logger.debug(f"Ошибка создания DotHighlightLexer: {e}")
        return None


def _get_formatter_text():
    """Ленивый импорт formatter_text"""
    global _formatter_text
    if _formatter_text is None:
        from penguin_tamer.formatter_text import extract_labeled_code_blocks
        _formatter_text = extract_labeled_code_blocks
    return _formatter_text

# Импортируем только самое необходимое для быстрого старта
from penguin_tamer.llm_client import OpenRouterClient
from penguin_tamer.arguments import parse_args
from penguin_tamer.error_messages import connection_error


STREAM_OUTPUT_MODE: bool = config.get("global", "stream_output_mode")
logger.info(f"Settings - Stream output mode: {STREAM_OUTPUT_MODE}")

# Ленивый импорт Markdown из rich (легкий модуль) для ускорения загрузки
_markdown = None


def _get_markdown():
    global _markdown
    if _markdown is None:
        from rich.markdown import Markdown
        _markdown = Markdown
    return _markdown

educational_text = (
    "ALWAYS number code blocks in your replies so the user can reference them. "
    "Numbering format: [Code #1]\n```bash ... ```, [Code #2]\n```bash ... ```, "
    "etc. Insert the numbering BEFORE the block "
    "If there are multiple code blocks, number them sequentially. "
    "In each new reply, start numbering from 1 again. Do not discuss numbering; just do it automatically."
)
EDUCATIONAL_CONTENT = [{'role': 'user', 'content': educational_text}]

@log_execution_time
def get_system_content() -> str:
    """Construct system prompt content with lazy system info loading"""
    user_content = config.get("global", "user_content", "")
    json_mode = config.get("global", "json_mode", False)

    if json_mode:
        additional_content_json = (
            "You must always respond with a single JSON object containing fields 'cmd' and 'info'. "
        )
    else:
        additional_content_json = ""

    # Базовая информация без вызова медленной системной информации
    additional_content_main = (
        "Your name is Ai-eBash, a sysadmin assistant. "
        "You and the user always work in a terminal. "
        "Respond based on the user's environment and commands. "
    )
    
    system_content = f"{user_content} {additional_content_json} {additional_content_main}".strip()
    return system_content


# === Основная логика ===
@log_execution_time
def run_single_query(chat_client: OpenRouterClient, query: str, console) -> None:
    """Run a single query (optionally streaming)"""
    logger.info(f"Running query: '{query[:50]}'...")
    try:
        if STREAM_OUTPUT_MODE:
            reply = chat_client.ask_stream(query)
        else:
            reply = chat_client.ask(query)
            console.print(_get_markdown()(reply))
    except Exception as e:
        console.print(connection_error(e))
        logger.error(f"Connection error: {e}")


@log_execution_time
def run_dialog_mode(chat_client: OpenRouterClient, console, initial_user_prompt: str = None) -> None:
    """Interactive dialog mode"""
    
    # Загружаем prompt_toolkit только когда нужен диалоговый режим
    _ensure_prompt_toolkit()
    
    # Импортируем необходимые модули для подсветки
    from prompt_toolkit import HTML, prompt
    from prompt_toolkit.history import FileHistory
    from prompt_toolkit.styles import Style
    from prompt_toolkit.layout.processors import Processor, Transformation

    # История команд хранится рядом с настройками в пользовательской папке
    history_file_path = config.user_config_dir / "cmd_history"
    history = FileHistory(str(history_file_path))

    logger.info("Starting dialog mode")

    # Use module global EDUCATIONAL_CONTENT inside the function
    global EDUCATIONAL_CONTENT

    last_code_blocks = []  # code blocks from the last AI answer

    # If there is an initial prompt, process it
    if initial_user_prompt:
        initial_user_prompt
        try:
            if STREAM_OUTPUT_MODE:
                reply = chat_client.ask_stream(initial_user_prompt, educational_content=EDUCATIONAL_CONTENT)
                console.print(_get_markdown()(reply))
            else:
                reply = chat_client.ask(initial_user_prompt, educational_content=EDUCATIONAL_CONTENT)
                console.print(_get_markdown()(reply))
            EDUCATIONAL_CONTENT = []  # clear educational content after first use
            last_code_blocks = _get_formatter_text()(reply)
        except Exception as e:
            console.print(connection_error(e))
            logger.error(f"Connection error: {e}")
        console.print()

    # Main dialog loop
    while True:
        try:

            # Define prompt styles с поддержкой подсветки команд
            style = Style.from_dict({
                "prompt": "bold fg:green",
                "dot": "fg:gray",        # Серая точка
                "command": "fg:cyan",       # Светло-синий текст команды
                "text": "",                 # Стандартный цвет консоли
            })
            
            # Создаем процессор для real-time подсветки
            class DotCommandProcessor(Processor):
                """Процессор для real-time подсветки команд с точкой"""
                
                def apply_transformation(self, transformation_input):
                    """Применяет трансформацию к вводу"""
                    text = transformation_input.document.text
                    
                    if text.startswith('.') and len(text) > 0:
                        # Создаем форматированный текст для команд с точкой
                        formatted_fragments = [
                            ('class:dot', '.'),
                            ('class:command', text[1:])
                        ]
                    else:
                        # Обычный текст
                        formatted_fragments = [('class:text', text)]
                    
                    return Transformation(
                        formatted_fragments,
                        source_to_display=lambda i: i,
                        display_to_source=lambda i: i
                    )
            
            # Создаем экземпляр процессора
            dot_processor = DotCommandProcessor()
            logger.debug("DotCommandProcessor created for real-time highlighting")
            
            if last_code_blocks:
                placeholder = HTML(t("<i><gray>Number of the code block to execute or the next question... Ctrl+C - exit</gray></i>"))
            else:
                placeholder = HTML(t("<i><gray>Your question... Ctrl+C - exit</gray></i>"))

            # Создаем кастомную функцию для динамической подсветки
            def get_prompt_tokens():
                """Возвращает токены промпта с подсветкой в зависимости от введенного текста"""
                return [("class:prompt", ">>> ")]
            
            # Пытаемся использовать prompt_toolkit, если не получается - fallback на input()
            try:
                # Настраиваем параметры prompt с процессором для подсветки
                prompt_kwargs = {
                    'placeholder': placeholder,
                    'history': history,
                    'style': style,
                    'multiline': False,
                    'wrap_lines': True,
                    'enable_history_search': True,
                    'input_processors': [dot_processor]  # Добавляем процессор для real-time подсветки
                }

                user_prompt = prompt(get_prompt_tokens, **prompt_kwargs)
                    
            except Exception as e:
                # Fallback на стандартный input() если prompt_toolkit не работает
                logger.debug(f"prompt_toolkit failed, using fallback input(): {e}")
                console.print("[dim]>>> [/dim]", end="")
                user_prompt = input().strip()
            # Disallow empty input
            if not user_prompt:
                continue

            # Exit commands
            if user_prompt.lower() in ['exit', 'quit', 'q']:
                break

            # Command execution: if input starts with dot ".", execute as direct command
            if user_prompt.startswith('.'):
                command_to_execute = user_prompt[1:].strip()  # Remove the dot and strip spaces
                if command_to_execute:  # Only execute if there's something after the dot
                    console.print(f"[dim]>>> Executing command:[/dim] {command_to_execute}")
                    _get_execute_handler()(console, command_to_execute)
                    console.print()
                    continue
                else:
                    console.print("[dim]Empty command after '.' - skipping.[/dim]")
                    continue

            # If a number is entered
            if user_prompt.isdigit():
                block_index = int(user_prompt)
                if 1 <= block_index <= len(last_code_blocks):
                    _get_script_executor()(console, last_code_blocks, block_index)
                    console.print()
                    continue
                else:
                    console.print(f"[dim]Code block #{user_prompt} not found.[/dim]")
                    continue

            # Если введен текст, отправляем как запрос к AI
            if STREAM_OUTPUT_MODE:
                reply = chat_client.ask_stream(user_prompt, educational_content=EDUCATIONAL_CONTENT)
            else:
                reply = chat_client.ask(user_prompt, educational_content=EDUCATIONAL_CONTENT)
                console.print(_get_markdown()(reply))
            EDUCATIONAL_CONTENT = []  # clear educational content after first use
            last_code_blocks = _get_formatter_text()(reply)
            console.print()  # new line after answer

        except KeyboardInterrupt:
            break
        except Exception as e:
            console.print(connection_error(e))
            logger.error(f"Connection error: {e}")


def _create_chat_client(console):
    """Ленивое создание LLM клиента только когда он действительно нужен"""
    logger.info("Initializing OpenRouterChat client")

    llm_config = config.get_current_llm_config()
    
    chat_client = OpenRouterClient(
        console=console,
        logger=logger,
        api_key=llm_config["api_key"],
        api_url=llm_config["api_url"],
        model=llm_config["model"],
        system_content=get_system_content(),
        temperature=config.get("global", "temperature", 0.7)
    )
    logger.info("OpenRouterChat client created: " + f"{chat_client}")
    return chat_client


@log_execution_time
def main() -> None:

    try:
        args = parse_args()

        # Settings mode - не нужен LLM клиент
        if args.settings:
            logger.info("Starting configuration mode")
            from penguin_tamer.config_menu import main_menu
            main_menu()
            logger.info("Configuration mode finished")
            return 0

        # Создаем консоль и клиент только если они нужны для AI операций
        console = _get_console()()
        chat_client = _create_chat_client(console)

        # Determine execution mode
        dialog_mode: bool = args.dialog
        prompt_parts: list = args.prompt or []
        prompt: str = " ".join(prompt_parts).strip()

        if dialog_mode or not prompt:
            # Dialog mode
            logger.info("Starting in dialog mode")
            run_dialog_mode(chat_client, console, prompt if prompt else None)
        else:
            # Single query mode
            logger.info("Starting in single-query mode")

            run_single_query(chat_client, prompt, console)

    except KeyboardInterrupt:
        logger.info("Interrupted by user")
        return 130
    except Exception as e:
        logger.critical(f"Unhandled error: {e}", exc_info=True)
        return 1
    finally:
        print()  # print empty line anyway

    logger.info("Program finished successfully")
    return 0


if __name__ == "__main__":
    sys.exit(main())
