import pandas
import numpy
import datetime
import random
from datetime import datetime
import datetime as dt

def TimeProfile(time_stopper: list, name: str = '', show: bool = False, estimate_for: int = 0):
    """
    Simple code profiler.

    How to use:

    Create a list ->  time_stopper = []

    Put a -> time_stopper.append(['time_init',time.perf_counter()]) at the beginning.

    Put time_stopper.append(['Func_01',time.perf_counter()]) after the code block with the first parameter being
    a name and the second being the time.

    Call this function at the end.

    Example:

    time_stopper.append(['time_init',time.perf_counter()])

    func1()
    time_stopper.append(['func1',time.perf_counter()])
    func2()
    time_stopper.append(['func2',time.perf_counter()])
    func3()
    time_stopper.append(['func3',time.perf_counter()])
    func4()
    time_stopper.append(['func4',time.perf_counter()])

    TimeProfile(time_stopper,'My Profiler',show=True,estimate_for=500)

    The estimate_for parameter makes the calculation as if you would run x times the code analyzed.


    :param time_stopper: A List that will hold all the stop times.
    :type time_stopper: list

    :param name: A name for this instance of time profile. Defaults to empty.
    :type name: str, optional

    :param show: If True shows the data on the console. Defaults to False.
    :type show: bool, optional

    :param estimate_for: A multiplier to be applied at the end. Takes the whole
    time analyzed and multiplies by "estimate_for".
    :type estimate_for: int

    :return: None
    :rtype: None

    """

    if(show):
        print("Profile: " + name)
        time_stopper = pandas.DataFrame(time_stopper, columns=['Type', 'time'])
        # time_stopper['time'] = time_stopper['time']-time_stopper['time'].min()
        time_stopper['Delta'] = time_stopper['time'] - time_stopper['time'].shift(periods=1, fill_value=0)
        time_stopper = time_stopper.iloc[1:, :]
        time_stopper['%'] = numpy.round(100 * time_stopper['Delta'] / time_stopper['Delta'].sum(), 2)
        total_estimate = time_stopper['Delta'].sum()
        time_stopper = pandas.concat((time_stopper,
                                      pandas.DataFrame([['Total', numpy.nan, time_stopper['Delta'].sum(), 100]],
                                                       columns=['Type', 'time', 'Delta', '%'])))
        print(time_stopper)
        if estimate_for != 0:
            print(
                f"Estimation for {estimate_for} "
                f"runs: {numpy.round(total_estimate * estimate_for / (60 * 60), 2)} hours.")

    return


# BUG Some sample_freq have trouble lol.
def DataSynchronization(x_in: pandas.core.frame.DataFrame,
                        start_date_dt: datetime,
                        end_date_dt: datetime,
                        sample_freq: int = 5,
                        sample_time_base: str = 'm') -> pandas.core.frame.DataFrame:
    """
    Makes the Data Synchronization between the columns (time series) of the data provided.

    Theory background.:

    The time series synchronization is the first step in processing the dataset. The synchronization is vital
    since the alignment between phases (φa, φb, φv) of the same quantity, between quantities (V, I, pf) of the
    same feeder, and between feeders, provides many advantages. The first one being the ability to combine all
    nine time series, the three-phase voltage, current, and power factor of each feeder to calculate the secondary
    quantities (Pactive/Preactive, Eactive/Ereactive).

    Furthermore, the synchronization between feeders provides the capability to analyze the iteration between them,
    for instance, in load transfers for scheduled maintenance and to estimate substation’s transformers quantities
    by the sum of all feeders.

    Most of the functions in this module assumes that the time series are "Clean" to a certain sample_freq. Therefore,
    this function must be executed first on the dataset.


    :param x_in: A pandas.core.frame.DataFrame where the index is of type "pandas.core.indexes.datetime.DatetimeIndex"
    and each column contain an electrical quantity time series.
    :type x_in: pandas.core.frame.DataFrame

    :param start_date_dt: The start date where the synchronization should start.
    :type start_date_dt: datetime

    :param end_date_dt: The end date where the synchronization will consider samples.
    :type end_date_dt: datetime

    :param sample_freq: The sample frequency of the time series. Defaults to 5.
    :type sample_freq: int,optional

    :param sample_time_base: The base time of the sample frequency. Specify if the sample frequency is in (D)ay,
    (M)onth, (Y)ear, (h)ours, (m)inutes, or (s)econds. Defaults to (m)inutes.
    :type sample_time_base: srt,optional


    :raises Exception: if x_in has no DatetimeIndex.
    :raises Exception: if start_date_dt not in datetime format.
    :raises Exception: if end_date_dt not in datetime format.
    :raises Exception: if sample_time_base is not in (D)ay, (M)onth, (Y)ear, (h)ours, (m)inutes, or (s)econds.


    :return: Y: The synchronized pandas.core.frame.DataFrame
    :rtype: Y: pandas.core.frame.DataFrame

    """

    #  BASIC INPUT CHECK

    if not (isinstance(x_in.index, pandas.DatetimeIndex)):
        raise Exception("x_in DataFrame has no DatetimeIndex.")
    if not (isinstance(start_date_dt, datetime)):
        raise Exception("start_date_dt Date not in datetime format.")
    if not (isinstance(end_date_dt, datetime)):
        raise Exception("end_date_dt Date not in datetime format.")
    if sample_time_base not in ['s', 'm', 'h', 'D', 'M', 'Y']:
        raise Exception("sample_time_base not valid. Ex. ['s','m','h','D','M','Y'] ")

    added_dic = {'s': 'ms', 'm': 's', 'h': 'm', 'D': 'h', 'M': 'D', 'Y': 'M'}
    floor_dic = {'s': 'S', 'm': 'T', 'h': 'H', 'D': 'D', 'M': 'M', 'Y': 'Y'}

    x_in.index = x_in.index.tz_localize(None)  # Makes the datetimeIndex naive (no time zone)

    '''
    Creates a base vector that contains all the samples 
    between start_date_dt and end_date_dt filled timestamp and with nan
    '''

    qty_data = len(x_in.columns)

    time_array = numpy.arange(start_date_dt, end_date_dt, numpy.timedelta64(sample_freq, sample_time_base),
                              dtype='datetime64')
    time_array = time_array + numpy.timedelta64(1, added_dic[
        sample_time_base])  # ADD a second/Minute/Hour/Day/Month to the end so during the sort
    # this samples will be at last (HH:MM:01)

    vet_samples = pandas.DataFrame(index=time_array, columns=range(qty_data), dtype=object)
    vet_samples.index.name = 'timestamp'

    # Creates the output dataframe which is the same but without the added second.

    df_y = vet_samples.copy(deep=True)
    df_y.index = df_y.index.floor(floor_dic[sample_time_base])  # Flush the seconds

    # Saves the name of the columns
    save_columns_name = x_in.columns.values

    # Start to process each column

    phase_list = numpy.arange(0, x_in.shape[1])

    for phase in phase_list:

        x = x_in.copy(deep=True)
        x.columns = df_y.columns
        x = x.loc[~x.iloc[:, phase].isnull(), phase]  # Gets only samples on the phase of interest
        x = x[numpy.logical_and(x.index < end_date_dt,
                                x.index >= start_date_dt)]

        if x.shape[0] != 0:

            # Process samples that are multiple of sample_freq
            df_x = x.copy(deep=True)
            df_vet_samples = vet_samples[phase]

            # remove seconds (00:00:00) to put this specific samples at the beginning during sort
            df_x = df_x.sort_index(ascending=True)  # Ensures the sequence of timestamps
            df_x.index = df_x.index.round(
                '1' + floor_dic[sample_time_base])  # Remove seconds, rounding to the nearest minute
            df_x = df_x[
                df_x.index.minute % sample_freq == 0]  # Samples that are multiple of sample_freq have preference

            if not df_x.empty:
                df_x = df_x[~df_x.index.duplicated(keep='first')]  # Remove unnecessary duplicates

                # joins both vectors
                df_aux = pandas.concat([df_x, df_vet_samples])
                df_aux = df_aux.sort_index(ascending=True)  # Ensures the sequence of timestamps

                '''
                Remove sec. (00:00:00), and remove duplicates leaving X when there is data 
                and vet amostra where its empty
                '''
                df_aux.index = df_aux.index.floor(floor_dic[sample_time_base])
                df_aux = df_aux[~df_aux.index.duplicated(keep='first')]  # Remove unnecessary duplicates

                # Make sure that any round up that ended up out of the period of study is removed
                df_aux = df_aux[numpy.logical_and(df_aux.index < end_date_dt, df_aux.index >= start_date_dt)]

                df_y.loc[:, phase] = df_aux

            # Process samples that are NOT multiple of sample_freq
            df_x = x.copy(deep=True)
            df_vet_samples = vet_samples[phase]

            # remove seconds (00:00:00) to put this specific samples at the beginning during sort
            df_x = df_x.sort_index(ascending=True)  # Ensures the sequence of timestamps
            df_x.index = df_x.index.round(
                '1' + floor_dic[sample_time_base])  # Remove seconds, rounding to the nearest minute
            df_x = df_x[
                df_x.index.minute % sample_freq != 0]  # Samples that are NOT multiple of sample_freq have preference

            if not df_x.empty:
                df_x.index = df_x.index.round(str(sample_freq) + floor_dic[
                    sample_time_base])  # Approximate sample to the closest multiple of sample_freq

                df_x = df_x[~df_x.index.duplicated(keep='first')]  # Remove unnecessary duplicates

                # joins both vectors
                df_aux = pandas.concat([df_x, df_vet_samples])
                df_aux = df_aux.sort_index(ascending=True)  # Ensures the sequence of timestamps

                '''
                Remove sec. (00:00:00), and remove duplicates leaving X when there is data 
                and vet amostra where its empty
                '''
                df_aux.index = df_aux.index.floor(floor_dic[sample_time_base])
                df_aux = df_aux[~df_aux.index.duplicated(keep='first')]  # Remove unnecessary duplicates

                # Make sure that any round up that ended up out of the period of study is removed
                df_aux = df_aux[numpy.logical_and(df_aux.index < end_date_dt, df_aux.index >= start_date_dt)]

                # Copy data to the output vector only if there is no data there yet.
                df_y.loc[df_y.iloc[:, phase].isnull(), phase] = df_aux.loc[df_y.iloc[:, phase].isnull()]

    # Last operations before the return of Y

    df_y = df_y.astype(float)
    df_y.columns = save_columns_name  # Gives back the original name of the columns in x_in

    return df_y


def IntegrateHour(x_in: pandas.DataFrame, sample_freq: int = 5,
                  sample_time_base: str = 'm') -> pandas.core.frame.DataFrame:
    """
    Integrates the input pandas.core.frame.DataFrame to an hour samples.

    :param x_in: A pandas.core.frame.DataFrame where the index is of type "pandas.core.indexes.datetimes.DatetimeIndex"
    and each column contain an electrical quantity time series.
    :type x_in: pandas.core.frame.DataFrame

    :param sample_freq: The sample frequency of the time series. Defaults to 5.
    :type sample_freq: int,optional

    :param sample_time_base: The base time of the sample frequency. Specify if the sample frequency is in (m)inutes
    or (s)econds. Defaults to (m)inutes.
    :type sample_time_base: srt,optional


    :raises Exception: if x_in has no DatetimeIndex.


    :return: df_y: The pandas.core.frame.DataFrame integrated by hour.
    :rtype: df_y: pandas.core.frame.DataFrame

    """
    hour_divider = {'s': 60 * 60, 'm': 60}

    # -------------------#
    # BASIC INPUT CHECK #
    # -------------------#

    if not (isinstance(x_in.index, pandas.DatetimeIndex)):
        raise Exception("x_in DataFrame has no DatetimeIndex.")

    df_y = x_in.copy(deep=True)

    time_vet_stamp = df_y.index[numpy.arange(0, len(df_y.index), int(hour_divider[sample_time_base] / sample_freq))]
    df_y = df_y.groupby([df_y.index.year, df_y.index.month, df_y.index.day, df_y.index.hour]).mean()
    df_y = df_y.reset_index(drop=True)
    df_y.insert(0, 'timestamp', time_vet_stamp)
    df_y.set_index('timestamp', inplace=True)

    return df_y


def Correlation(x_in: pandas.DataFrame) -> float:
    """
    Calculates the correlation between each column of the DataFrame and outputs the average of all.


    :param x_in: A pandas.core.frame.DataFrame where the index is of type "pandas.core.indexes.datetime.DatetimeIndex"
    and each column contain an electrical quantity time series.
    :type x_in: pandas.core.frame.DataFrame


    :return: corr_value: Value of the correlation
    :rtype: corr_value: float

    """

    corr_value = x_in.corr()[x_in.corr() != 1].mean().mean()

    return corr_value


def DayPeriodMapper(hour: int) -> int:
    """
    Maps a given hour to one of four periods of a day.

    For 0 to 5 (hour) -> 0 night
    For 6 to 11 (hour) -> 1 moorning
    For 12 to 17 (hour) -> 2 afternoon
    For 18 to 23 (hour) -> 3 evening

    :param hour: an hour of the day between 0 and 23.
    :type hour: int

    :return: mapped: Period of the day
    :rtype: mapped: int

    """

    return (
        0 if 0 <= hour < 6
        else
        1 if 6 <= hour < 12
        else
        2 if 12 <= hour < 18
        else
        3
    )


def DayPeriodMapperVet(hour: pandas.core.series.Series) -> pandas.core.series.Series:
    """
    Maps a given hour to one of four periods of a day.

    For 0 to 5 (hour) -> 0 night
    For 6 to 11 (hour) -> 1 moorning
    For 12 to 17 (hour) -> 2 afternoon
    For 18 to 23 (hour) -> 3 evening


    :param hour: A pandas.core.series.Series with values between 0 and 23 to map each hour in the series to a period
    of the day. this is a "vector" format for DayPeriodMapper function.
    :type hour: pandas.core.series.Series

    :return: period_day: The hour pandas.core.series.Series mapped to periods of the day
    :rtype: period_day: pandas.core.series.Series

    """

    map_dict = {0: 0, 1: 0, 2: 0, 3: 0, 4: 0, 5: 0,
                6: 1, 7: 1, 8: 1, 9: 1, 10: 1, 11: 1,
                12: 2, 13: 2, 14: 2, 15: 2, 16: 2, 17: 2,
                18: 3, 19: 3, 20: 3, 21: 3, 22: 3, 23: 3}

    period_day = hour.map(map_dict)

    return period_day


def YearPeriodMapperVet(month: pandas.core.series.Series) -> pandas.core.series.Series:
    """
    Maps a given month to one of two periods of a year, being dry and humid .

    For october to march (month) -> 0 humid
    For april to september (month) -> 1 dry


    :param month: A pandas.core.series.Series with values between 0 and 12 to map each month
    in the series to dry or humid.

    :return: season: The months pandas.core.series.Series mapped to dry or humid.
    :rtype: season: pandas.core.series.Series

    """

    map_dict = {10: 0, 11: 0, 12: 0, 1: 0, 2: 0, 3: 0,
                4: 1, 5: 1, 6: 1, 7: 1, 8: 1, 9: 1}

    season = month.map(map_dict)

    return season


def CountMissingData(x_in: pandas.core.frame.DataFrame, remove_from_process: list = [], show=False) -> float:
    """
    Calculates the number of vacacies on the dataset.


    :param x_in: A pandas.core.frame.DataFrame where the index is of type "pandas.core.indexes.datetime.DatetimeIndex"
    and each column contain an electrical quantity time series.
    :type x_in: pandas.core.frame.DataFrame

    :param remove_from_process: Columns to be kept off the process.
    :type remove_from_process: list,optional

    :param show: Specify if the function should print or not the value that is also returned.
    :type show: bool,optional


    :return: Y: Returns the amount of vacancies.
    :rtype: Y: float

    """
    Y = x_in.loc[:, x_in.columns.difference(remove_from_process)].isnull().sum().sum()
    if show:
        print(f"Total number of missing samples {Y}")

    return Y


def CalcUnbalance(x_in: pandas.core.frame.DataFrame, remove_from_process: list = []) -> pandas.core.frame.DataFrame:
    """
    Calculates the unbalance between phases for every timestamp.

    Equation:
        Y = (MAX-MEAN)/MEAN

    Ref.: Derating of induction motors operating with a combination of unbalanced voltages and over or under-voltages


    :param x_in: A pandas.core.frame.DataFrame where the index is of type "pandas.core.indexes.datetime.DatetimeIndex"
    and each column contain an electrical quantity time series.
    :type x_in: pandas.core.frame.DataFrame

    :param remove_from_process: Columns to be kept off the process.
    :type remove_from_process: list,optional

    :return: Y: A pandas.core.frame.DataFrame with the % of unbalance between columns (phases).
    :rtype: Y: pandas.core.frame.DataFrame

    """

    X = x_in.loc[:, x_in.columns.difference(remove_from_process)]

    Y = pandas.DataFrame([], index=x_in.index)

    Y['Unbalance'] = 100 * (X.max(axis=1) - X.mean(axis=1)) / X.mean(axis=1)

    return Y


def SavePeriod(x_in: pandas.core.frame.DataFrame,
               df_save: pandas.core.frame.DataFrame) -> tuple:
    """
    For a given set of periods (Start->End) returns the data. It also returns the indexes.

    :param x_in: A pandas.core.frame.DataFrame where the index is of type "pandas.core.indexes.datetime.DatetimeIndex" 
    and each column contain an electrical quantity time series.
    :type x_in: pandas.core.frame.DataFrame

    :param df_save: The first column with the start and the second column with the end date.
    :type df_save: pandas.core.frame.DataFrame

    :return: df_values,index_return: The input pandas.core.frame.DataFrame sliced by the df_save periods. it also returns
    the indexes
    :rtype: df_values,index_return: tuple

    """


    df_values = pandas.DataFrame([])
    index_return = pandas.DataFrame([])
    
    for _, row in df_save.iterrows():
        
        if(df_values.size==0):        
            df_values = x_in.loc[numpy.logical_and(x_in.index >= row.iloc[0], x_in.index <= row.iloc[1]), :]
        else:
            df_values = pandas.concat((df_values,x_in.loc[numpy.logical_and(x_in.index >= row.iloc[0], x_in.index <= row.iloc[1]), :]),axis=0)
             
        if(index_return.size==0):        
            index_return = pandas.Series(x_in.index[numpy.logical_and(x_in.index >= row.iloc[0], x_in.index <= row.iloc[1])].values)
        else:
            index_return = pandas.concat((index_return,pandas.Series(x_in.index[numpy.logical_and(x_in.index >= row.iloc[0], x_in.index <= row.iloc[1])].values)),axis=0)


    return df_values, index_return


def MarkNanPeriod(x_in: pandas.core.frame.DataFrame,
                  df_remove: pandas.core.frame.DataFrame,
                  remove_from_process: list = []) -> pandas.core.frame.DataFrame:
    """
    Marks as nan all specified timestamps

    :param x_in: A pandas.core.frame.DataFrame where the index is of type "pandas.core.indexes.datetime.DatetimeIndex"
    and each column contain an electrical quantity time series.
    :type x_in: pandas.core.frame.DataFrame

    :param df_remove: List of periods to mark as nan. The first column with the start and the second column with
    the end date all in datetime.
    :type df_remove: pandas.core.frame.DataFrame

    :param remove_from_process: Columns to be kept off the process;
    :type remove_from_process: list,optional

    :return: Y: The input pandas.core.frame.DataFrame with samples filled based on the proportion between time series.
    :rtype: Y: pandas.core.frame.DataFrame

    """

    Y = x_in.copy(deep=True)

    # Remove the keep out columns
    if len(remove_from_process) > 0:
        Y = Y.drop(remove_from_process, axis=1)

    for index, row in df_remove.iterrows():
        Y.loc[numpy.logical_and(Y.index >= row.iloc[0], Y.index <= row.iloc[1]), Y.columns.difference(
            remove_from_process)] = numpy.nan

    # return the keep out columns
    if len(remove_from_process) > 0:
        Y = pandas.concat([Y, x_in.loc[:, remove_from_process]], axis=1)

    return Y


def ReturnOnlyValidDays(x_in: pandas.core.frame.DataFrame,
                        sample_freq: int = 5,
                        threshold_accept: float = 1.0,
                        sample_time_base: str = 'm',
                        remove_from_process=[]) -> tuple:
    """
    Returns all valid days. A valid day is one with no missing values for any 
    of the timeseries on each column.
    
    
    :param x_in: A pandas.core.frame.DataFrame where the index is of type "pandas.core.indexes.datetime.DatetimeIndex"
    and each column contain an electrical quantity time series.
    :type x_in: pandas.core.frame.DataFrame
    
    :param sample_freq: The sample frequency of the time series. Defaults to 5.  
    :type sample_freq: int,optional
    
    :param threshold_accept: The amount of samples that is required to consider a valid day. Defaults to 1 (100%).  
    :type threshold_accept: float,optional
    
    :param sample_time_base: The base time of the sample frequency. Specify if the sample frequency is in (h)ours,
    (m)inutes, or (s)econds. Defaults to (m)inutes.
    :type sample_time_base: srt,optional
    
    :param remove_from_process: Columns to be kept off the process;  
    :type remove_from_process: list,optional
    
         
    :raises Exception: if x_in has no DatetimeIndex. 
    :raises Exception: if sample_time_base is not in seconds, minutes or hours.
    
    
    :return: Y: A tupole with the pandas.core.frame.DataFrame with samples filled based on the proportion
    between time series and the number of valid days
    :rtype: Y: tuple

    """

    # BASIC INPUT CHECK
    
    if not(isinstance(x_in.index, pandas.core.frame.DatetimeIndex)):
        raise Exception("DataFrame has no DatetimeIndex.")
    if sample_time_base not in ['s', 'm', 'h']:
        raise Exception("The sample_time_base is not in seconds, minutes or hours.")

    X = x_in.copy(deep=True)
    
    if len(remove_from_process) > 0:
        X = X.drop(remove_from_process, axis=1)

    qty_sample_dic = {'s': 24 * 60 * 60, 'm': 24 * 60, 'h': 24}

    df_count = X.groupby([X.index.year, X.index.month, X.index.day]).count() / (
                qty_sample_dic[sample_time_base] / sample_freq)

    time_vet_stamp = X.index[numpy.arange(0, len(X.index), int((qty_sample_dic[sample_time_base] / sample_freq)))]
    df_count = df_count.reset_index(drop=True)
    df_count.insert(0, 'timestamp_day', time_vet_stamp)
    df_count.set_index('timestamp_day', inplace=True)
    df_count = df_count >= threshold_accept
    
    df_count = df_count.sum(axis=1) == df_count.shape[1]
    df_count.name = 'isValid'
    df_count = df_count.reset_index()
    X['timestamp_day'] = X.index.floor("D").values

    keep_X_index = X.index
    X = pandas.merge(X, df_count, on='timestamp_day', how='left')
    X.index = keep_X_index
    X = X.loc[X['isValid'] == True, :]

    X.drop(columns=['isValid', 'timestamp_day'], inplace=True)
    df_count.set_index('timestamp_day', inplace=True)

    return X, df_count


def GetDayMaxMin(x_in: pandas.core.frame.DataFrame, start_date_dt: datetime, end_date_dt:datetime, sample_freq: int =5, threshold_accept:float=1.0, exe_param:str='max'):
    """
    Returns a tuple of pandas.core.frame.DataFrame containing the values of maximum or minimum of each day
    and the timestamp of each occurrence. For each weekday that is not a valid day the maximum or minimum
    is interpolated->ffill->bff. The interpolation is made regarding each weekday.

    :param x_in: A pandas.core.frame.DataFrame where the index is of type "pandas.core.indexes.datetime.DatetimeIndex"
    and each column contain an electrical quantity time series.
    :type x_in: pandas.core.frame.DataFrame

    :param start_date_dt:
    :param end_date_dt:

    :param sample_freq: The sample frequency of the time series. Defaults to 5.
    :type sample_freq: int,optional

    :param threshold_accept: The amount of samples that is required to consider a valid day. Defaults to 1 (100%).
    :type threshold_accept: float,optional

    :param exe_param: 'max' return the maximum and min return the minimum value of each valid day
    (Default value = 'max')
    :type exe_param: srt,optional

    :return: Y: The first parameter is a pandas.core.frame.DataFrame with maximum value for each day
    and the second parameter pandas.core.frame.DataFrame with the timestamps.
    :rtype: Y: tuple
    """

    # BASIC INPUT CHECK
    
    if not(isinstance(x_in.index, pandas.core.frame.DatetimeIndex)):
        raise Exception("DataFrame has no DatetimeIndex.")

    X = x_in.copy(deep=True)

    X, _ = ReturnOnlyValidDays(X, sample_freq, threshold_accept)

    if exe_param == 'max':
        Y = X.groupby([X.index.year, X.index.month, X.index.day]).max()
        vet_idx = X.groupby([X.index.year, X.index.month, X.index.day]).idxmax()
    else:
        Y = X.groupby([X.index.year, X.index.month, X.index.day]).min()
        vet_idx = X.groupby([X.index.year, X.index.month, X.index.day]).idxmin()

    # redo the timestamp index
    vet_idx.index.rename(['Year', 'Month', 'Day'], inplace=True)
    vet_idx = vet_idx.reset_index(drop=False)

    time_vet_stamp = pandas.to_datetime(
        vet_idx['Year'].astype(str) + '-' + vet_idx['Month'].astype(str) + '-' + vet_idx['Day'].astype(str))

    vet_idx.drop(columns=['Year', 'Month', 'Day'], axis=1, inplace=True)
    vet_idx = vet_idx.reset_index(drop=True)
    vet_idx.insert(0, 'timestamp_day', time_vet_stamp)
    vet_idx.set_index('timestamp_day', inplace=True)

    # redo the timestamp index
    Y.index.rename(['Year', 'Month', 'Day'], inplace=True)
    Y = Y.reset_index(drop=False)

    time_vet_stamp = pandas.to_datetime(Y['Year'].astype(str) + '-' + Y['Month'].astype(str) + '-' + Y['Day'].astype(str))

    Y.drop(columns=['Year', 'Month', 'Day'], axis=1, inplace=True)
    Y = Y.reset_index(drop=True)
    Y.insert(0, 'timestamp_day', time_vet_stamp)
    Y.set_index('timestamp_day', inplace=True)

    Y = DataSynchronization(Y, start_date_dt, end_date_dt, sample_freq=1, sample_time_base='D')

    vet_idx = pandas.merge(vet_idx, Y, left_index=True, right_index=True, how='right', suffixes=('', '_remove'))
    vet_idx.drop(columns=vet_idx.columns[vet_idx.columns.str.contains('_remove')], axis=1, inplace=True)

    # Missing days get midnight as the  hour of max and min
    for col in vet_idx.columns.values:
        vet_idx.loc[vet_idx[col].isna(), col] = vet_idx.index[vet_idx[col].isna()]

    # Interpolate by day of the week
    Y = Y.groupby(Y.index.weekday, group_keys=False).apply(lambda x: x.interpolate())
    Y = Y.groupby(Y.index.weekday, group_keys=False).apply(lambda x: x.ffill())
    Y = Y.groupby(Y.index.weekday, group_keys=False).apply(lambda x: x.bfill())

    return Y, vet_idx


def GetWeekDayCurve(x_in: pandas.core.frame.DataFrame, sample_freq:int=5, threshold_accept:float=1.0, min_sample_per_day:int=3, min_sample_per_workday:int=9):
    """
    Analyzes and normalizes time series data in a DataFrame to compute average curves for each weekday, 
    considering various sampling and validity thresholds.

    :param x_in: Input DataFrame with a DatetimeIndex.
    :type: pandas.core.frame.DataFrame
    :param sample_freq: Sampling frequency in minutes, default is 5.
    :type: int
    :param threshold_accept: Threshold for accepting valid data, default is 1.0.
    :type: float
    :param min_sample_per_day: Minimum samples required per day to consider the data valid, default is 3.
    :type: int
    :param min_sample_per_workday: Minimum samples required per workday (Monday to Friday) to consider the data valid, default is 9.
    :type: int
    
    :raises Exception: If the DataFrame does not have a DatetimeIndex.

    :return: A DataFrame containing the normalized data for each weekday.
    :rtype: pandas.core.frame.DataFrame
    """

    # BASIC INPUT CHECK

    if not (isinstance(x_in.index, pandas.core.frame.DatetimeIndex)):
        raise Exception("DataFrame has no DatetimeIndex.")
   
    
    X = x_in.copy(deep=True)

    Y, df_count = ReturnOnlyValidDays(X, sample_freq, threshold_accept)

    # Get valid data statistics
    df_count = df_count.loc[df_count['isValid'], :]
    df_stats = df_count.groupby(df_count.index.weekday).count()

    # fill days that does not exist with count zero.
    for i_day in range(0,7):
        if i_day not in df_stats.index.values:
            print(f'Weekday {i_day} does not exist.')
            df_stats.loc[i_day] = 0

    # Has enough data do use ?
    if numpy.min(df_stats['isValid'].values) >= min_sample_per_day:
        print('Can calculate a curve for every weekday')

        Y = Y.groupby([Y.index.weekday, Y.index.hour, Y.index.minute]).mean()
        Y.index.names = ['WeekDay', 'Hour', 'Min']
        Y = Y.reset_index()

        # Normalization max min each day
        grouper = Y.groupby([Y.WeekDay])
        maxes = grouper.transform('max')
        mins = grouper.transform('min')

        Y.iloc[:, 3:] = (Y.iloc[:, 3:] - mins.iloc[:, 2:]) / (maxes.iloc[:, 2:] - mins.iloc[:, 2:])
        
    else:
        work_days = df_stats.loc[df_stats.index <= 4, 'isValid'].sum()
        sat_qty = df_stats.loc[df_stats.index == 5, 'isValid'].sum()
        sun_qty = df_stats.loc[df_stats.index == 6, 'isValid'].sum()

        if (work_days >= min_sample_per_workday) and sun_qty >= min_sample_per_day and sat_qty >= min_sample_per_day:
            print('Can calculate a curve for every weekday and use Sat. and Sun.')

            Y['WeekDay'] = Y.index.weekday.values
            Y['Hour'] = Y.index.hour.values
            Y['Min'] = Y.index.minute.values
            Y = Y.reset_index(drop=True)
            Y.loc[Y['WeekDay'] <= 4, 'WeekDay'] = 0

            Y = Y.groupby([Y.WeekDay, Y.Hour, Y.Min]).mean()
            Y.index.names = ['WeekDay', 'Hour', 'Min']
            Y = Y.reset_index()

            # Normalization max min each day
            grouper = Y.groupby([Y.WeekDay])
            maxes = grouper.transform('max')
            mins = grouper.transform('min')

            Y.iloc[:, 3:] = (Y.iloc[:, 3:] - mins.iloc[:, 2:]) / (maxes.iloc[:, 2:] - mins.iloc[:, 2:])

            for i_day in [1, 2, 3, 4]:
                Y_day_aux = Y.loc[Y.WeekDay == 0, :].copy(deep=True)
                Y_day_aux.WeekDay = i_day
                Y = pandas.concat((Y, Y_day_aux))
            Y = Y.reset_index(drop=True)

        else:
            print('Not enough data using default curve.')
            Y = DefaultWeekDayCurve()

    return Y

def CurrentDummyData(qty_weeks:int = 12*4,start_date_dt:datetime = datetime(2023,1,1)):
    """
    Generates a DataFrame containing dummy time series data.
    
    This function creates a pandas DataFrame representing time series data over a specified number of weeks, starting from a given date. The data includes artificial variations to simulate different patterns, including seasonal variations and random noise. The DataFrame includes columns 'IA', 'IB', 'IV', and 'IN', each containing modified time series data. The index of the DataFrame is set to timestamps at 5-minute intervals, starting from the specified start date.
    
    Parameters
    ----------
    qty_weeks : int, optional
        The number of weeks to generate data for, by default 48 weeks (12*4).
    start_date_dt : datetime, optional
        The start date for the time series data, by default datetime(2023,1,1).
    
    Returns
    -------
    pandas.DataFrame
        A DataFrame containing the generated time series data with columns 'IA', 'IB', 'IV', and 'IN', and a timestamp index.
    
    Examples
    --------
    >>> dummy_data = CurrentDummyData(24, datetime(2023,1,1))
    >>> dummy_data.head()
    """
    
    dummy_week = pandas.DataFrame([[133.4,128.7,122.3,5.7],
                                [131.3,129.2,120.9,4.7],
                                [126.5,124.7,120.9,4.7],
                                [129.7,126.6,121.1,4.7],
                                [128.1,130.2,121.6,5.6],
                                [128.1,130.2,121.6,5.6],
                                [126.6,124.5,119.1,5.3],
                                [126.6,124.5,119.1,5.3],
                                [127.3,125.0,121.3,4.9],
                                [125.0,126.0,120.1,5.0],
                                [125.0,126.0,120.1,5.0],
                                [125.8,123.9,120.2,5.0],
                                [125.8,123.9,120.2,5.0],
                                [121.5,119.9,114.5,4.7],
                                [122.2,120.9,115.8,4.8],
                                [126.6,125.1,114.5,4.8],
                                [121.6,120.1,115.3,4.6],
                                [121.6,120.1,115.3,4.6],
                                [119.8,120.3,114.5,5.0],
                                [121.4,118.8,112.7,4.7],
                                [121.4,118.8,112.7,4.7],
                                [125.2,117.9,111.9,5.0],
                                [125.2,117.9,114.2,5.0],
                                [120.2,119.4,112.0,5.3],
                                [120.8,118.4,113.1,5.4],
                                [120.8,118.4,113.1,5.4],
                                [122.4,118.5,113.3,5.1],
                                [122.4,118.5,113.3,5.1],
                                [121.7,117.0,113.2,4.9],
                                [120.0,117.2,111.5,4.9],
                                [120.0,117.2,111.5,4.9],
                                [120.0,115.2,111.5,4.6],
                                [120.0,115.2,111.5,4.6],
                                [118.1,115.4,110.9,4.7],
                                [118.5,116.9,112.3,4.7],
                                [118.5,116.9,109.2,4.7],
                                [116.6,113.1,109.3,4.8],
                                [116.6,113.1,109.3,4.8],
                                [118.9,114.0,109.9,4.7],
                                [119.0,115.4,110.9,5.0],
                                [119.0,115.4,110.9,5.0],
                                [118.6,115.0,110.0,5.0],
                                [118.6,115.0,110.0,5.0],
                                [117.6,113.3,109.6,5.0],
                                [116.7,113.0,108.4,4.9],
                                [116.7,113.0,108.4,4.9],
                                [117.4,115.5,110.5,4.8],
                                [117.4,115.5,110.5,4.8],
                                [116.6,113.4,109.3,4.6],
                                [115.3,113.8,108.4,4.9],
                                [115.3,113.8,108.4,4.9],
                                [116.2,112.9,108.1,4.9],
                                [116.2,114.6,108.1,4.9],
                                [117.9,113.9,109.7,4.8],
                                [115.8,110.3,106.4,4.4],
                                [115.8,110.3,106.4,4.4],
                                [116.2,113.0,106.6,4.7],
                                [116.2,113.0,106.6,4.7],
                                [115.5,112.0,108.3,4.9],
                                [118.8,112.8,111.6,4.5],
                                [118.8,112.8,114.3,4.5],
                                [115.1,112.5,108.1,4.3],
                                [115.1,109.5,104.2,4.3],
                                [113.2,111.7,107.3,4.2],
                                [109.3,109.2,105.0,4.4],
                                [109.3,109.2,105.0,4.4],
                                [108.7,108.1,104.1,5.1],
                                [110.1,108.1,104.1,5.1],
                                [108.5,108.0,105.2,4.7],
                                [108.0,108.4,103.6,4.7],
                                [108.0,109.5,103.6,4.7],
                                [109.0,105.0,103.6,4.4],
                                [105.0,105.0,99.0,4.4],
                                [110.1,109.6,104.3,4.5],
                                [112.1,113.0,108.0,4.5],
                                [112.1,113.0,108.0,4.9],
                                [115.2,112.5,107.0,4.4],
                                [110.2,110.4,104.5,4.4],
                                [113.4,111.8,107.0,4.5],
                                [112.9,111.6,105.7,4.5],
                                [112.9,111.6,105.7,4.5],
                                [112.1,108.4,106.8,4.4],
                                [115.9,115.4,110.2,3.9],
                                [115.4,114.9,112.0,4.9],
                                [115.2,115.8,111.9,4.2],
                                [115.2,115.8,115.3,4.2],
                                [115.5,113.2,111.3,4.3],
                                [115.5,113.2,115.4,4.3],
                                [117.2,115.7,111.0,3.9],
                                [117.0,117.4,111.4,4.1],
                                [117.0,115.7,111.4,4.1],
                                [117.7,117.4,110.1,3.9],
                                [119.6,119.3,116.8,3.9],
                                [122.2,123.9,116.7,4.0],
                                [125.2,123.5,115.8,4.7],
                                [127.7,126.8,118.3,4.7],
                                [128.5,125.3,117.9,4.4],
                                [124.9,121.8,122.8,4.4],
                                [131.3,127.3,121.0,4.9],
                                [132.0,127.0,125.9,4.3],
                                [128.2,127.2,123.4,4.3],
                                [131.8,126.2,118.7,4.5],
                                [136.5,130.0,124.6,4.0],
                                [136.1,143.1,123.3,5.1],
                                [136.4,132.0,128.2,4.0],
                                [136.2,132.6,130.0,4.0],
                                [135.1,131.4,131.3,4.2],
                                [129.6,128.2,125.8,4.2],
                                [130.6,129.8,126.7,4.3],
                                [133.8,132.9,129.0,4.2],
                                [132.1,130.4,127.4,4.8],
                                [133.7,129.8,127.4,4.3],
                                [131.5,131.9,128.5,4.6],
                                [140.9,139.7,131.7,4.1],
                                [142.6,137.0,132.5,3.6],
                                [135.7,137.5,132.3,3.6],
                                [139.0,141.3,132.8,3.9],
                                [140.3,141.4,132.9,3.9],
                                [142.3,141.7,133.4,4.4],
                                [137.3,135.4,133.0,3.7],
                                [144.3,136.7,133.9,3.7],
                                [152.7,155.2,133.4,3.7],
                                [145.1,138.6,138.2,3.7],
                                [146.4,142.0,133.7,3.9],
                                [153.3,146.7,141.2,3.9],
                                [146.2,145.4,140.9,4.1],
                                [143.1,142.3,135.1,3.7],
                                [151.5,149.1,141.2,3.7],
                                [151.4,144.0,144.3,4.0],
                                [149.1,147.3,144.9,4.0],
                                [145.7,143.0,139.3,4.1],
                                [142.1,141.8,137.7,3.9],
                                [142.5,142.5,139.5,3.9],
                                [144.3,144.2,137.3,3.6],
                                [155.1,149.2,142.5,3.6],
                                [149.6,150.1,148.1,4.2],
                                [153.3,151.5,143.3,4.1],
                                [146.5,142.8,142.6,3.9],
                                [149.2,147.0,140.5,4.0],
                                [153.0,149.9,146.0,4.2],
                                [155.8,155.2,147.7,3.9],
                                [153.1,148.7,144.2,3.5],
                                [153.4,154.3,143.0,3.5],
                                [159.1,150.3,149.7,4.0],
                                [155.2,153.0,146.3,4.0],
                                [156.7,154.6,147.5,3.9],
                                [158.5,153.1,147.1,4.1],
                                [156.8,156.6,148.2,4.1],
                                [152.8,153.7,147.2,3.9],
                                [155.1,150.2,147.2,3.9],
                                [153.2,152.1,150.5,5.0],
                                [155.4,153.4,147.2,4.5],
                                [155.6,148.4,147.2,4.0],
                                [156.2,151.6,147.4,4.1],
                                [151.7,151.6,147.4,4.1],
                                [151.5,152.8,144.9,4.1],
                                [152.1,148.2,143.6,4.1],
                                [162.6,158.9,153.8,4.1],
                                [156.2,153.4,150.2,4.0],
                                [152.5,149.0,148.9,4.0],
                                [157.6,154.0,148.9,4.2],
                                [159.9,155.4,152.0,4.2],
                                [159.0,157.2,152.6,4.4],
                                [157.2,155.9,150.8,4.5],
                                [161.0,154.8,152.6,4.3],
                                [165.5,161.8,158.2,4.3],
                                [163.3,160.6,152.8,4.3],
                                [161.6,155.4,153.0,4.3],
                                [166.9,160.6,156.3,4.2],
                                [170.9,168.5,155.1,4.2],
                                [162.7,156.5,151.1,4.5],
                                [162.8,160.2,156.9,4.5],
                                [163.3,157.5,156.2,4.5],
                                [162.6,158.5,152.6,5.3],
                                [162.5,157.8,153.4,4.3],
                                [159.0,158.3,156.3,4.6],
                                [162.0,160.3,156.5,4.7],
                                [166.0,161.3,156.5,4.7],
                                [162.9,159.9,151.3,4.2],
                                [167.1,159.8,160.2,4.2],
                                [161.9,161.3,155.1,4.3],
                                [161.9,160.2,155.4,4.1],
                                [163.5,158.2,154.5,4.1],
                                [164.7,158.3,152.1,4.3],
                                [165.5,162.9,155.6,4.3],
                                [164.7,161.6,155.5,4.2],
                                [168.4,167.9,159.3,5.5],
                                [163.4,158.1,154.9,4.5],
                                [164.5,162.7,156.0,4.7],
                                [166.4,166.9,157.4,4.7],
                                [165.2,163.0,153.4,4.3],
                                [161.6,160.8,152.4,4.3],
                                [162.0,161.0,154.2,4.3],
                                [166.7,164.9,156.7,4.2],
                                [162.0,159.5,149.4,4.2],
                                [163.9,162.6,155.2,4.1],
                                [165.2,161.5,155.2,4.3],
                                [161.3,164.1,155.3,4.3],
                                [161.3,161.9,157.0,3.8],
                                [161.7,162.6,155.1,3.5],
                                [160.0,161.1,152.2,4.0],
                                [158.8,156.1,152.1,3.6],
                                [162.8,161.2,157.4,4.5],
                                [155.2,155.2,148.4,4.1],
                                [157.2,153.3,149.1,4.1],
                                [156.2,153.6,147.9,4.0],
                                [155.5,152.4,148.2,4.2],
                                [154.9,150.9,148.5,4.2],
                                [150.4,151.6,144.6,3.8],
                                [155.3,150.6,145.1,3.8],
                                [150.3,151.2,146.5,4.2],
                                [156.6,152.9,146.2,4.7],
                                [156.0,151.9,143.8,4.5],
                                [156.0,153.1,144.9,4.8],
                                [156.2,154.2,145.4,4.8],
                                [151.1,153.8,145.9,4.1],
                                [151.2,149.2,144.0,4.1],
                                [145.9,144.1,139.0,4.3],
                                [151.1,149.1,137.4,5.1],
                                [146.0,149.1,137.4,5.1],
                                [145.8,144.9,139.8,4.8],
                                [152.1,146.4,139.7,5.5],
                                [153.0,153.3,142.5,5.2],
                                [154.2,151.3,144.4,5.0],
                                [158.1,158.0,152.7,5.0],
                                [163.1,158.2,147.6,5.3],
                                [165.7,160.1,152.7,5.3],
                                [166.0,162.8,155.3,5.1],
                                [165.4,162.8,155.1,6.2],
                                [167.1,160.7,155.7,6.2],
                                [164.5,164.5,156.7,5.8],
                                [174.3,164.8,164.5,5.2],
                                [171.9,170.6,162.3,5.1],
                                [173.7,170.6,160.7,5.3],
                                [174.4,165.6,160.7,6.2],
                                [172.2,171.9,162.4,5.4],
                                [173.3,171.9,166.3,5.4],
                                [178.4,178.1,164.1,5.5],
                                [173.9,168.6,161.8,5.9],
                                [169.1,168.9,162.2,5.9],
                                [169.4,166.8,160.6,5.1],
                                [175.7,167.1,164.0,5.1],
                                [171.0,168.7,163.7,4.9],
                                [172.3,167.1,163.9,5.0],
                                [166.9,164.5,161.7,5.0],
                                [172.0,167.5,162.2,4.8],
                                [171.7,162.0,159.6,4.8],
                                [172.4,167.1,164.8,5.4],
                                [168.4,166.8,158.5,5.8],
                                [172.5,168.8,161.5,5.8],
                                [167.5,165.1,162.7,5.3],
                                [169.9,163.5,156.4,5.3],
                                [169.8,161.5,161.4,6.3],
                                [164.7,163.6,156.9,5.8],
                                [171.8,166.5,161.5,5.3],
                                [170.6,162.4,157.3,5.7],
                                [170.6,162.4,157.3,5.7],
                                [166.6,165.0,154.9,6.3],
                                [166.9,160.5,155.7,5.4],
                                [166.9,162.1,157.6,6.3],
                                [169.2,167.3,156.3,5.7],
                                [169.2,167.3,156.3,5.3],
                                [164.2,162.3,152.9,5.5],
                                [170.8,166.3,158.7,6.3],
                                [165.7,165.9,158.7,6.3],
                                [165.6,162.7,155.9,5.9],
                                [166.4,166.2,158.3,5.9],
                                [166.9,161.2,153.1,5.7],
                                [166.3,161.5,156.0,5.6],
                                [164.0,161.5,156.0,6.4],
                                [162.6,159.4,154.9,5.8],
                                [162.6,159.4,149.9,5.8],
                                [162.6,161.5,154.9,6.0],
                                [164.8,163.7,153.3,5.9],
                                [165.9,163.6,157.9,5.9],
                                [166.0,164.5,157.3,5.4],
                                [166.0,159.2,154.6,5.4],
                                [162.5,165.4,153.5,5.8],
                                [161.9,162.2,152.5,6.4],
                                [164.2,159.9,153.9,6.4],
                                [160.2,160.2,151.5,5.9],
                                [165.4,159.9,153.0,5.9],
                                [160.5,159.4,148.9,6.0],
                                [158.1,155.7,148.6,5.8],
                                [158.1,155.1,146.4,5.8],
                                [155.7,150.6,146.9,5.3],
                                [157.3,155.7,146.9,6.6],
                                [157.0,155.7,148.3,5.8],
                                [157.0,156.0,146.7,5.8],
                                [157.0,156.0,146.7,6.5],
                                [154.6,154.7,145.4,6.2],
                                [154.6,150.8,145.4,5.4],
                                [152.7,152.6,143.8,5.6],
                                [151.7,153.7,141.5,6.5],
                                [151.7,148.6,140.3,5.5],
                                [151.4,149.4,141.1,5.6],
                                [152.6,151.9,138.9,5.6],
                                [150.6,146.9,140.1,5.6],
                                [152.3,148.7,140.2,5.2],
                                [147.3,148.7,140.2,5.2],
                                [147.2,146.9,137.2,5.6],
                                [147.7,143.6,133.7,5.6],
                                [146.9,143.1,133.6,5.2],
                                [148.0,145.2,136.4,5.4],
                                [148.0,145.2,135.7,5.5],
                                [144.6,141.3,133.8,5.5],
                                [142.3,140.5,132.0,5.5],
                                [143.7,141.0,134.0,5.6],
                                [142.3,140.4,131.3,5.7],
                                [143.3,140.4,132.9,6.7],
                                [140.8,138.8,131.9,6.1],
                                [140.8,136.0,131.9,5.7],
                                [141.3,137.1,127.9,5.6],
                                [142.3,137.8,130.1,5.6],
                                [139.1,138.6,130.3,5.6],
                                [141.1,139.8,132.6,5.8],
                                [141.1,139.8,127.6,5.8],
                                [139.7,134.8,130.8,5.4],
                                [139.3,135.1,128.6,5.4],
                                [139.3,136.9,130.9,5.4],
                                [138.7,136.7,129.5,5.3],
                                [138.7,136.7,129.5,5.3],
                                [136.0,131.7,126.8,5.6],
                                [137.7,133.5,126.1,5.6],
                                [137.0,134.9,128.4,5.6],
                                [137.5,133.3,128.5,5.4],
                                [137.5,133.3,123.5,5.4],
                                [135.5,129.8,126.3,5.4],
                                [132.7,129.6,123.4,5.2],
                                [133.5,129.6,126.8,5.2],
                                [133.8,131.0,124.2,5.3],
                                [134.8,133.6,121.7,5.3],
                                [133.0,129.3,124.2,5.4],
                                [133.2,130.3,122.9,5.3],
                                [133.2,132.2,124.5,5.3],
                                [129.8,129.2,123.5,5.3],
                                [130.0,129.2,125.8,5.3],
                                [135.1,128.8,123.5,5.5],
                                [131.2,127.8,121.9,5.1],
                                [131.3,127.8,123.5,5.1],
                                [131.5,130.4,120.9,5.5],
                                [131.4,126.9,118.4,5.5],
                                [129.5,126.9,122.3,5.4],
                                [132.8,129.4,122.0,5.3],
                                [132.8,129.4,122.0,5.3],
                                [128.9,125.8,121.1,5.0],
                                [128.9,125.8,121.1,5.0],
                                [132.7,126.3,119.5,5.2],
                                [128.1,124.5,120.2,5.0],
                                [122.9,121.3,115.5,5.0],
                                [124.1,122.9,112.8,5.3],
                                [124.1,122.4,115.5,4.7],
                                [122.3,120.7,113.4,4.9],
                                [118.9,119.4,110.3,5.0],
                                [118.9,117.3,110.3,5.0],
                                [118.3,117.3,110.7,5.1],
                                [120.0,117.8,112.6,4.7],
                                [116.8,117.6,107.5,4.9],
                                [118.5,119.9,110.6,5.0],
                                [118.5,118.8,114.2,5.0],
                                [115.1,113.9,109.4,4.6],
                                [116.2,114.5,109.4,4.6],
                                [117.2,121.2,114.9,4.7],
                                [119.8,120.1,114.6,5.3],
                                [123.9,122.5,116.0,4.7],
                                [123.0,121.5,115.5,5.2],
                                [125.1,122.3,115.5,4.7],
                                [122.6,118.9,114.5,4.5],
                                [120.3,121.6,112.1,4.6],
                                [121.0,121.6,112.7,4.6],
                                [121.7,122.5,117.0,4.9],
                                [121.7,122.5,117.9,4.9],
                                [126.2,125.7,113.4,4.8],
                                [129.0,126.4,123.5,4.4],
                                [126.7,125.9,120.7,4.4],
                                [130.0,130.4,121.8,4.7],
                                [130.1,130.3,122.2,4.7],
                                [128.4,126.4,123.4,4.5],
                                [131.4,130.2,129.5,4.5],
                                [136.6,136.9,130.9,4.1],
                                [133.6,133.5,124.7,4.4],
                                [133.6,131.3,127.7,4.4],
                                [131.1,136.9,128.1,4.4],
                                [136.2,132.1,128.5,4.4],
                                [135.4,137.9,127.5,4.5],
                                [139.6,140.9,138.2,4.3],
                                [144.3,142.0,128.6,4.3],
                                [143.6,145.3,134.6,4.3],
                                [147.1,145.4,133.1,4.3],
                                [141.2,139.6,132.9,4.4],
                                [142.2,142.0,135.9,4.4],
                                [145.4,141.8,136.2,4.4],
                                [141.3,141.9,138.9,4.3],
                                [141.3,142.3,139.0,4.3],
                                [142.0,141.0,133.4,4.4],
                                [143.4,139.7,134.9,4.4],
                                [146.1,141.4,136.0,4.4],
                                [148.0,148.2,143.4,4.1],
                                [149.7,143.1,139.6,4.1],
                                [152.0,147.6,140.5,3.6],
                                [150.5,148.8,139.8,4.4],
                                [156.2,151.7,144.0,4.4],
                                [151.0,151.7,139.6,3.7],
                                [153.4,150.0,146.8,3.7],
                                [155.6,150.2,144.7,4.3],
                                [153.3,152.2,143.1,4.3],
                                [152.0,148.5,143.8,4.1],
                                [155.1,152.9,146.1,3.7],
                                [155.4,150.7,148.0,3.7],
                                [152.3,148.9,145.9,4.5],
                                [157.3,153.9,147.9,4.5],
                                [155.6,156.7,148.4,4.1],
                                [161.4,156.5,147.7,3.8],
                                [159.4,159.4,149.6,3.8],
                                [157.9,156.0,146.4,4.3],
                                [157.9,156.3,151.7,4.4],
                                [158.5,155.9,146.7,3.9],
                                [156.9,156.2,145.5,4.1],
                                [159.2,155.3,150.2,4.1],
                                [160.3,153.8,150.5,4.0],
                                [155.1,154.6,155.6,4.0],
                                [156.2,158.0,150.5,4.0],
                                [158.9,154.0,148.7,4.1],
                                [159.8,157.3,153.5,4.1],
                                [154.7,152.3,148.2,4.1],
                                [159.8,157.9,153.3,4.1],
                                [160.7,157.8,150.5,4.3],
                                [162.6,162.1,157.8,4.2],
                                [164.5,159.9,155.3,4.2],
                                [167.7,164.0,157.6,4.7],
                                [162.1,164.0,152.6,4.7],
                                [165.1,162.7,157.7,3.8],
                                [166.2,165.7,157.9,4.1],
                                [168.1,164.4,154.8,4.1],
                                [168.3,167.5,158.5,4.1],
                                [164.3,164.6,153.6,4.1],
                                [166.1,167.3,158.8,4.2],
                                [169.7,164.2,158.2,4.0],
                                [164.6,161.3,155.3,4.0],
                                [161.1,161.1,155.4,4.0],
                                [161.1,156.4,154.9,4.0],
                                [162.6,159.1,155.2,4.1],
                                [162.5,158.9,154.5,4.3],
                                [165.4,159.6,156.6,4.4],
                                [163.3,159.5,153.2,3.7],
                                [165.5,159.7,158.8,3.7],
                                [165.8,159.7,159.2,3.8],
                                [172.9,167.7,164.3,3.8],
                                [164.0,163.6,156.4,3.8],
                                [168.4,167.0,161.4,3.8],
                                [171.4,167.1,161.4,3.8],
                                [172.6,167.8,159.0,3.8],
                                [168.7,168.3,158.4,4.2],
                                [169.6,163.5,158.4,4.5],
                                [171.0,164.2,161.2,4.3],
                                [165.9,164.3,161.7,3.6],
                                [169.9,165.2,159.8,3.7],
                                [170.0,166.1,157.3,3.9],
                                [172.9,168.1,160.0,3.9],
                                [169.9,164.1,158.7,4.9],
                                [174.8,179.8,156.9,4.2],
                                [173.7,167.8,161.5,4.1],
                                [170.9,168.8,161.2,3.9],
                                [173.0,170.4,162.7,3.9],
                                [174.0,167.6,158.5,4.3],
                                [170.5,171.0,162.7,4.3],
                                [171.0,168.6,158.8,4.4],
                                [166.8,169.2,160.3,4.5],
                                [169.1,169.8,164.2,4.5],
                                [170.0,167.7,166.1,4.3],
                                [168.9,169.3,161.1,4.3],
                                [173.6,172.9,163.1,4.6],
                                [165.0,166.2,159.3,4.1],
                                [168.6,166.6,160.1,4.1],
                                [171.9,165.1,161.9,4.3],
                                [171.3,168.4,165.8,4.3],
                                [173.2,171.6,164.2,4.1],
                                [169.7,165.5,160.3,4.1],
                                [169.5,167.0,162.0,4.1],
                                [176.3,179.4,154.2,4.3],
                                [168.3,163.5,159.8,3.8],
                                [168.6,164.5,154.6,3.9],
                                [170.0,165.6,157.3,3.9],
                                [168.7,167.1,155.1,3.9],
                                [171.7,168.0,163.2,4.2],
                                [171.3,166.6,163.0,4.0],
                                [167.4,164.4,155.0,4.1],
                                [174.2,178.9,158.3,3.8],
                                [166.8,165.1,160.2,4.5],
                                [167.4,165.9,159.7,3.9],
                                [165.6,163.9,153.1,4.2],
                                [162.4,161.6,154.5,3.4],
                                [161.8,159.9,154.1,3.4],
                                [157.7,154.6,148.9,3.3],
                                [156.8,156.6,147.1,3.9],
                                [158.8,155.6,149.4,3.9],
                                [155.5,155.7,149.1,4.0],
                                [150.4,151.1,144.4,4.0],
                                [150.3,150.0,144.4,4.1],
                                [156.0,147.9,143.7,4.1],
                                [156.7,160.7,149.9,4.1],
                                [157.7,154.0,149.2,4.2],
                                [157.7,155.2,149.2,5.0],
                                [157.0,154.7,150.0,4.0],
                                [152.0,149.6,145.4,4.1],
                                [146.9,149.3,144.6,4.1],
                                [151.3,149.3,139.3,4.5],
                                [151.3,154.4,144.4,5.0],
                                [151.9,149.4,143.3,4.5],
                                [151.9,152.6,143.5,4.8],
                                [156.4,149.5,149.4,4.8],
                                [154.5,154.7,147.1,5.1],
                                [161.6,160.7,154.6,5.1],
                                [167.3,165.7,153.8,6.1],
                                [165.8,164.7,158.1,5.8],
                                [165.8,164.7,158.1,5.8],
                                [167.9,170.6,160.9,5.5],
                                [172.3,170.6,160.9,6.1],
                                [170.9,175.7,164.9,5.7],
                                [177.3,174.3,166.0,5.5],
                                [177.3,174.3,166.0,5.5],
                                [177.2,176.1,165.1,5.9],
                                [177.4,176.1,165.1,5.9],
                                [177.7,174.9,165.0,5.6],
                                [178.3,179.7,171.2,5.8],
                                [178.3,181.2,171.2,5.8],
                                [177.8,176.2,165.1,5.6],
                                [177.8,176.2,170.5,5.6],
                                [179.6,177.9,170.7,5.7],
                                [182.6,176.0,168.6,5.6],
                                [177.7,176.0,170.5,5.6],
                                [177.7,175.9,170.4,5.7],
                                [182.7,175.9,170.4,5.7],
                                [177.6,173.3,165.5,5.8],
                                [177.0,172.8,167.6,5.9],
                                [177.0,172.8,167.2,5.9],
                                [177.1,174.8,167.6,5.7],
                                [177.1,174.8,167.6,5.7],
                                [178.2,173.4,166.5,5.9],
                                [173.6,171.0,162.8,5.9],
                                [172.5,171.0,162.8,5.9],
                                [172.7,168.8,161.4,5.3],
                                [172.7,171.0,161.4,5.3],
                                [171.9,165.9,160.6,5.2],
                                [170.2,169.9,160.8,5.8],
                                [170.2,169.9,160.8,6.2],
                                [173.2,168.5,157.0,5.5],
                                [167.4,168.5,157.0,5.5],
                                [170.1,171.0,158.8,5.7],
                                [169.6,165.9,162.6,6.2],
                                [169.6,171.0,162.6,7.4],
                                [170.8,170.3,161.6,6.6],
                                [170.8,165.8,161.6,6.6],
                                [173.0,169.6,162.6,6.3],
                                [170.6,167.2,160.3,6.2],
                                [172.8,167.2,160.3,6.2],
                                [171.8,171.1,163.0,5.6],
                                [167.8,165.9,157.1,5.6],
                                [167.8,170.4,161.0,6.6],
                                [169.8,165.7,158.2,5.9],
                                [172.9,171.0,162.3,5.9],
                                [170.6,167.9,160.3,6.0],
                                [170.6,166.0,160.3,6.0],
                                [170.4,166.1,157.3,6.0],
                                [169.5,166.6,157.4,6.3],
                                [169.5,166.6,157.4,6.3],
                                [169.0,171.0,157.1,6.6],
                                [168.3,168.8,157.3,6.5],
                                [168.3,166.0,157.3,6.5],
                                [166.2,163.9,156.0,6.3],
                                [162.7,160.8,156.0,6.3],
                                [167.9,165.9,154.5,6.3],
                                [163.5,161.6,153.1,6.2],
                                [162.8,161.6,157.0,6.2],
                                [164.4,160.8,152.0,6.2],
                                [164.4,160.8,152.0,6.2],
                                [161.3,158.4,152.4,6.2],
                                [159.1,156.1,149.6,6.3],
                                [159.1,155.8,149.6,6.3],
                                [158.3,161.2,149.0,5.8],
                                [158.3,156.1,149.0,5.8],
                                [158.1,156.4,148.9,6.0],
                                [157.0,157.5,147.2,6.3],
                                [157.0,157.5,147.2,6.3],
                                [158.1,155.4,147.6,5.9],
                                [152.6,155.4,147.6,5.9],
                                [154.6,151.0,147.3,5.9],
                                [152.7,152.2,144.8,5.8],
                                [152.7,152.2,142.2,5.8],
                                [151.6,145.9,141.6,5.9],
                                [147.2,146.0,141.6,6.4],
                                [150.2,148.5,140.0,5.9],
                                [150.8,148.9,142.9,5.8],
                                [150.8,146.7,142.1,5.8],
                                [151.2,148.0,141.5,6.0],
                                [147.3,148.0,137.0,6.4],
                                [147.0,146.3,139.0,5.4],
                                [146.3,144.3,137.2,5.5],
                                [146.3,144.3,137.2,6.5],
                                [147.9,144.6,137.6,5.6],
                                [147.9,144.6,137.6,5.6],
                                [144.5,142.9,136.6,6.7],
                                [145.5,143.3,136.5,5.9],
                                [145.5,143.3,136.5,5.7],
                                [142.5,142.8,134.2,6.0],
                                [141.2,139.2,134.2,5.7],
                                [142.0,139.9,137.2,5.8],
                                [144.9,141.4,135.4,5.7],
                                [141.7,139.0,135.4,5.7],
                                [143.2,140.3,132.6,6.1],
                                [143.2,140.3,132.6,6.1],
                                [140.7,138.0,130.0,5.7],
                                [143.6,139.4,131.6,5.6],
                                [143.6,139.4,131.6,5.6],
                                [140.7,136.4,129.9,5.5],
                                [140.7,136.4,129.9,5.7],
                                [142.0,136.1,129.7,6.1],
                                [141.3,138.5,129.9,5.8],
                                [141.3,138.5,129.9,5.8],
                                [138.1,135.6,130.1,5.7],
                                [136.6,135.6,130.1,5.7],
                                [139.3,136.3,131.0,5.7],
                                [138.4,135.6,128.8,5.8],
                                [141.6,135.6,128.8,5.8],
                                [138.5,134.0,132.2,5.8],
                                [138.5,134.0,132.1,5.8],
                                [137.4,134.4,129.3,5.9],
                                [136.6,133.5,127.9,5.9],
                                [136.6,133.5,127.9,5.9],
                                [136.0,133.3,126.8,5.5],
                                [136.0,133.3,126.8,5.5],
                                [136.5,133.8,127.5,5.6],
                                [135.0,131.5,125.7,5.9],
                                [135.0,131.5,125.7,5.9],
                                [136.9,132.9,125.5,5.8],
                                [136.9,132.9,125.5,5.8],
                                [131.1,128.9,122.0,5.6],
                                [129.5,126.3,119.3,5.4],
                                [129.5,126.3,119.3,5.4],
                                [124.7,123.8,116.5,5.7],
                                [124.7,128.9,116.5,5.7],
                                [125.6,124.8,118.0,5.2],
                                [120.9,121.7,114.5,5.0],
                                [126.0,123.6,117.1,5.0],
                                [121.4,118.3,114.4,5.1],
                                [121.4,123.5,117.5,5.1],
                                [122.9,118.4,112.5,5.3],
                                [121.2,123.7,116.6,5.0],
                                [121.2,118.7,112.2,5.8],
                                [121.2,124.2,117.3,4.8],
                                [126.6,123.8,117.3,4.8],
                                [127.2,126.2,122.3,4.9],
                                [126.8,127.2,120.7,4.9],
                                [126.8,123.8,117.2,4.9],
                                [132.2,128.6,122.5,4.9],
                                [127.1,128.6,122.5,4.9],
                                [128.6,128.6,122.6,4.9],
                                [128.9,126.8,120.5,4.8],
                                [132.1,128.9,120.5,4.8],
                                [132.8,125.8,122.4,4.6],
                                [132.2,125.8,127.6,4.6],
                                [132.5,128.5,122.3,5.8],
                                [134.5,131.6,124.8,4.8],
                                [137.5,133.7,127.3,4.7],
                                [135.6,134.1,127.2,5.7],
                                [135.6,139.1,132.5,4.7],
                                [132.5,134.1,127.5,4.9],
                                [138.5,137.1,129.8,4.6],
                                [138.5,137.1,129.8,4.6],
                                [141.7,141.5,132.5,5.2],
                                [140.2,138.7,133.2,5.2],
                                [148.0,141.9,133.5,4.7],
                                [143.3,143.1,135.8,4.5],
                                [145.6,142.8,138.0,4.5],
                                [148.6,145.9,137.3,4.7],
                                [145.5,148.1,141.1,4.7],
                                [147.5,145.9,136.4,4.7],
                                [148.7,145.7,140.2,4.5],
                                [149.8,149.8,144.9,4.5],
                                [151.0,148.6,144.3,4.1],
                                [155.7,148.2,144.3,4.1],
                                [157.0,159.2,141.4,4.3],
                                [149.0,147.9,141.5,4.0],
                                [152.4,150.3,148.8,4.4],
                                [150.0,148.9,146.3,4.2],
                                [155.6,149.6,143.9,4.2],
                                [151.7,152.6,145.7,3.9],
                                [154.5,145.1,143.3,3.9],
                                [157.6,154.9,147.1,3.8],
                                [154.6,155.7,147.4,3.9],
                                [154.5,152.6,147.4,3.9],
                                [159.8,155.4,147.5,4.1],
                                [157.1,154.2,147.7,4.1],
                                [157.2,157.0,147.8,4.3],
                                [162.2,158.3,149.1,4.2],
                                [161.8,158.6,149.3,4.2],
                                [161.0,158.4,151.3,4.2],
                                [163.1,160.6,150.2,4.2],
                                [163.1,161.2,156.3,4.1],
                                [163.1,160.7,153.2,4.5],
                                [163.4,155.7,149.8,4.4],
                                [163.4,160.0,153.7,4.1],
                                [164.3,162.8,153.8,4.1],
                                [159.2,158.4,149.4,4.2],
                                [164.4,158.2,149.8,4.2],
                                [164.6,163.4,154.7,4.4],
                                [159.6,158.6,154.9,4.1],
                                [165.7,164.9,157.5,4.1],
                                [161.0,156.9,152.5,4.1],
                                [166.8,158.5,157.9,4.1],
                                [167.5,164.1,153.3,4.0],
                                [170.8,169.2,159.8,4.1],
                                [165.1,165.6,160.1,4.1],
                                [167.5,165.2,156.4,4.3],
                                [167.9,163.3,156.8,4.4],
                                [169.2,169.0,159.4,4.3],
                                [173.0,174.1,161.5,4.6],
                                [170.9,170.3,157.0,4.6],
                                [171.6,169.7,162.0,4.6],
                                [171.5,169.8,157.3,4.6],
                                [169.2,168.6,158.2,4.7],
                                [175.8,175.9,166.4,4.6],
                                [173.5,172.5,166.4,4.6],
                                [172.6,173.5,165.3,4.3],
                                [169.1,169.9,159.7,4.3],
                                [171.5,170.0,165.1,4.5],
                                [170.4,169.3,164.7,4.5],
                                [171.8,171.4,163.4,4.4],
                                [172.2,171.4,168.5,4.4],
                                [168.9,170.2,162.9,4.4],
                                [174.5,170.6,162.2,4.5],
                                [171.7,168.3,164.5,4.6],
                                [165.7,166.8,159.8,4.6],
                                [171.4,172.0,164.8,4.4],
                                [171.4,171.9,164.8,4.4],
                                [174.1,170.9,170.0,4.2],
                                [171.8,170.4,164.7,4.6],
                                [174.8,168.8,167.7,4.6],
                                [175.1,174.6,166.0,4.5],
                                [180.1,175.3,166.7,4.5],
                                [174.6,173.9,167.9,4.6],
                                [181.1,189.9,166.4,4.4],
                                [171.2,173.6,165.1,4.4],
                                [174.4,170.3,166.9,4.6],
                                [178.1,173.9,169.3,4.5],
                                [179.3,174.2,164.2,4.3],
                                [175.0,173.7,165.1,4.5],
                                [173.1,174.7,166.3,4.5],
                                [168.0,169.6,166.5,4.6],
                                [180.1,177.9,171.5,4.6],
                                [175.2,175.9,166.6,4.3],
                                [177.9,178.8,166.7,4.3],
                                [178.0,177.9,167.0,4.3],
                                [176.8,175.1,167.8,4.8],
                                [178.2,175.3,167.1,4.8],
                                [176.8,176.1,171.5,4.3],
                                [175.7,175.6,166.2,4.3],
                                [176.6,175.9,167.7,4.3],
                                [179.5,177.1,169.6,4.2],
                                [177.0,177.1,168.1,4.2],
                                [177.4,177.0,168.5,4.5],
                                [174.7,174.4,168.2,4.7],
                                [173.8,172.9,169.7,4.7],
                                [174.3,175.4,168.3,4.8],
                                [174.3,173.4,169.7,4.8],
                                [174.2,174.0,169.8,4.9],
                                [173.5,177.3,169.2,4.9],
                                [178.5,179.4,172.7,5.0],
                                [176.3,171.7,168.5,4.7],
                                [171.5,171.7,169.0,4.7],
                                [170.6,171.9,164.6,4.5],
                                [170.6,171.9,164.7,4.6],
                                [172.1,172.1,165.2,4.1],
                                [171.1,171.1,163.2,4.6],
                                [174.4,171.8,165.3,4.6],
                                [176.8,172.9,165.0,4.2],
                                [172.0,169.4,165.1,4.2],
                                [172.4,164.7,165.1,4.3],
                                [173.3,170.5,165.5,4.8],
                                [171.2,169.2,159.6,4.8],
                                [168.6,167.1,159.9,4.7],
                                [165.3,165.7,159.8,4.7],
                                [162.7,161.0,154.7,4.5],
                                [166.1,163.8,153.8,4.5],
                                [159.8,158.3,150.9,4.5],
                                [159.7,158.2,150.9,4.8],
                                [159.7,158.2,152.2,4.8],
                                [159.7,158.3,153.6,4.1],
                                [158.1,155.8,149.5,4.5],
                                [164.9,163.5,153.9,4.5],
                                [159.9,160.7,148.8,5.1],
                                [159.2,163.7,154.1,5.1],
                                [159.8,158.4,152.6,4.8],
                                [158.9,154.0,147.3,4.5],
                                [158.8,158.2,147.3,4.5],
                                [155.1,154.9,144.0,4.5],
                                [155.1,154.9,144.1,4.5],
                                [158.9,163.5,146.3,4.7],
                                [160.4,158.3,152.0,4.9],
                                [164.2,163.6,154.0,4.9],
                                [169.3,168.9,159.5,4.7],
                                [174.3,168.9,164.8,5.7],
                                [179.4,174.7,170.3,5.2],
                                [175.3,174.3,165.0,5.8],
                                [175.3,174.3,165.0,5.8],
                                [179.7,173.6,170.4,5.2],
                                [179.7,178.7,170.2,5.2],
                                [179.1,178.7,170.4,5.6],
                                [184.3,184.0,171.1,5.6],
                                [179.2,178.9,171.1,5.6],
                                [179.9,178.3,170.5,5.6],
                                [185.0,178.3,170.7,5.6],
                                [179.9,178.9,170.6,6.1],
                                [178.7,177.3,172.5,5.5],
                                [185.0,177.3,170.6,5.5],
                                [184.3,184.0,175.7,5.7],
                                [184.3,178.7,175.7,5.7],
                                [184.2,179.2,174.6,5.9],
                                [181.9,177.3,171.4,5.9],
                                [181.9,177.3,171.6,5.9],
                                [183.8,177.3,171.8,6.0],
                                [180.1,173.6,171.8,6.0],
                                [178.2,173.5,169.6,5.8],
                                [180.3,178.5,169.9,6.0],
                                [185.1,178.5,175.9,6.0],
                                [182.1,176.9,170.7,6.3],
                                [185.2,176.9,170.6,6.3],
                                [182.0,173.1,171.5,5.7],
                                [177.1,172.4,167.1,5.7],
                                [174.9,172.4,167.1,5.7],
                                [173.9,170.9,162.5,5.8],
                                [180.0,170.9,162.5,5.8],
                                [174.9,172.7,166.0,6.3],
                                [173.5,169.7,163.2,6.8],
                                [175.0,172.8,165.7,6.8],
                                [174.8,167.8,160.9,6.0],
                                [174.8,172.8,166.3,6.0],
                                [176.4,167.7,165.9,6.2],
                                [174.2,168.8,161.2,6.0],
                                [175.0,168.8,161.2,6.0],
                                [173.9,171.4,162.0,6.2],
                                [173.9,172.8,162.0,6.2],
                                [175.1,167.6,162.1,6.4],
                                [173.0,168.8,161.0,6.5],
                                [173.0,168.8,161.0,6.8],
                                [174.4,172.8,161.0,6.6],
                                [174.4,167.8,161.0,6.6],
                                [173.9,171.5,161.5,6.4],
                                [174.6,168.9,161.5,6.6],
                                [174.6,168.9,161.5,6.6],
                                [172.9,169.9,161.7,6.8],
                                [170.0,169.9,161.7,6.8],
                                [175.5,172.9,160.9,6.6],
                                [173.3,169.2,158.9,6.6],
                                [170.4,168.0,158.9,6.6],
                                [168.6,166.2,156.2,6.5],
                                [168.6,166.2,161.4,7.8],
                                [169.5,168.2,156.3,6.9],
                                [168.1,166.6,156.6,6.7],
                                [168.1,166.6,156.6,6.7],
                                [165.1,167.1,157.0,6.7],
                                [165.1,167.1,157.0,6.7],
                                [164.7,164.2,155.2,6.8],
                                [164.7,163.0,155.2,6.8],
                                [165.2,163.0,155.8,6.8],
                                [163.7,162.8,153.2,6.7],
                                [163.7,162.8,151.2,6.7],
                                [160.2,157.7,151.1,6.8],
                                [160.2,157.7,151.1,6.8],
                                [161.9,157.6,150.9,6.6],
                                [159.1,157.2,152.8,6.6],
                                [159.1,157.2,152.8,6.6],
                                [157.1,156.4,147.8,6.3],
                                [157.1,156.4,147.8,6.3],
                                [154.9,152.7,145.9,6.2],
                                [154.7,152.7,146.0,6.1],
                                [154.7,152.7,146.0,5.8],
                                [154.1,152.9,145.0,6.1],
                                [154.1,152.9,145.0,5.9],
                                [153.5,147.8,146.2,5.9],
                                [149.6,147.7,141.6,5.9],
                                [149.5,147.7,141.6,5.9],
                                [149.3,148.2,141.5,6.0],
                                [149.3,148.2,141.5,6.0],
                                [149.9,148.2,140.4,6.0],
                                [150.1,145.1,139.6,5.9],
                                [150.1,145.1,139.6,5.9],
                                [149.2,145.2,140.3,5.9],
                                [149.2,145.2,140.3,5.9],
                                [148.0,147.6,135.3,6.1],
                                [147.6,147.13,134.98,6.1],
                                [147.21,146.66,134.66,6.11],
                                [146.81,146.19,134.35,6.11],
                                [146.41,145.71,134.03,6.11],
                                [146.02,145.24,133.71,6.12],
                                [145.62,144.77,133.39,6.12],
                                [145.23,144.3,133.08,6.13],
                                [144.83,143.83,132.76,6.13],
                                [144.43,143.36,132.44,6.13],
                                [144.04,142.89,132.12,6.14],
                                [143.64,142.41,131.8,6.14],
                                [143.24,141.94,131.49,6.14],
                                [142.85,141.47,131.17,6.15],
                                [142.45,141.0,130.85,6.15],
                                [142.05,140.53,130.53,6.15],
                                [141.66,140.06,130.21,6.16],
                                [141.26,139.59,129.9,6.16],
                                [140.86,139.11,129.58,6.16],
                                [140.47,138.64,129.26,6.17],
                                [140.07,138.17,128.94,6.17],
                                [139.68,137.7,128.63,6.18],
                                [139.28,137.23,128.31,6.18],
                                [138.88,136.76,127.99,6.18],
                                [138.49,136.29,127.67,6.19],
                                [138.09,135.81,127.35,6.19],
                                [137.69,135.34,127.04,6.19],
                                [137.3,134.87,126.72,6.2],
                                [136.9,134.4,126.4,6.2],
                                [135.15,131.9,127.25,6.05],
                                [133.4,129.4,128.1,5.9],
                                [136.3,131.6,122.8,5.4],
                                [134.2,131.6,122.8,5.4],
                                [137.5,132.2,126.4,5.2],
                                [137.5,132.2,126.4,5.2],
                                [134.1,129.6,122.9,5.4],
                                [130.8,127.8,120.4,4.9],
                                [130.8,127.8,117.9,4.9],
                                [128.4,128.1,118.5,5.1],
                                [128.4,124.6,118.5,5.5],
                                [123.6,124.7,112.6,5.0],
                                [124.7,121.9,116.6,5.1],
                                [124.7,121.9,116.6,5.1],
                                [125.5,119.9,113.9,4.8],
                                [121.5,119.9,113.9,4.8],
                                [121.9,125.6,113.4,4.9],
                                [123.1,121.8,115.4,4.9],
                                [123.1,121.8,110.2,5.6],
                                [123.1,122.8,113.7,5.6],
                                [126.1,125.7,119.0,5.6],
                                [124.3,126.5,114.2,4.9],
                                [125.0,123.3,114.9,5.1],
                                [128.0,128.7,120.0,4.6],
                                [125.4,126.2,118.7,4.8],
                                [123.7,123.1,116.7,4.8],
                                [128.4,127.5,118.5,5.6],
                                [127.9,127.0,120.4,5.5],
                                [128.2,128.1,120.2,5.5],
                                [127.1,127.5,120.2,4.9],
                                [125.7,124.2,122.2,4.9],
                                [126.8,129.3,120.9,4.6],
                                [129.9,129.9,123.5,5.2],
                                [129.9,129.8,123.5,4.6],
                                [136.4,135.1,127.9,5.0],
                                [137.8,140.4,127.7,5.7],
                                [143.2,140.5,133.0,4.6],
                                [137.8,139.5,133.3,4.5],
                                [137.3,135.4,128.5,4.5],
                                [135.6,135.6,131.5,4.8],
                                [136.6,137.2,131.6,4.8],
                                [137.0,137.3,132.1,4.5],
                                [139.8,138.4,130.7,4.5],
                                [138.6,134.9,128.6,4.5],
                                [138.7,138.6,130.3,4.4],
                                [144.7,141.3,132.6,4.4],
                                [149.3,145.0,137.8,4.5],
                                [144.1,142.4,134.3,4.5],
                                [146.4,146.3,136.2,4.5],
                                [157.5,164.9,137.6,4.4],
                                [148.5,146.6,139.6,4.4],
                                [145.3,143.7,134.5,3.9],
                                [147.9,147.2,137.2,3.6],
                                [150.4,147.5,139.1,4.5],
                                [152.0,151.4,142.6,4.2],
                                [152.8,150.0,141.2,4.2],
                                [154.9,148.6,138.4,4.1],
                                [154.6,150.3,147.6,4.4],
                                [152.4,149.5,141.6,4.4],
                                [155.6,151.8,145.3,4.2],
                                [152.3,148.8,141.4,4.2],
                                [154.0,154.2,145.6,4.0],
                                [154.7,151.9,146.9,4.0],
                                [154.5,150.4,148.0,4.3],
                                [156.8,154.8,144.4,4.3],
                                [157.4,155.8,146.3,4.1],
                                [163.7,155.5,152.2,4.2],
                                [156.9,156.7,146.4,3.5],
                                [157.7,155.6,144.9,3.7],
                                [158.6,157.4,149.9,3.7],
                                [158.6,157.6,148.4,3.8],
                                [154.2,153.9,151.4,3.8],
                                [159.4,155.7,149.9,3.6],
                                [159.1,155.8,145.1,4.1],
                                [158.0,153.6,145.8,4.1],
                                [158.1,156.3,146.0,4.8],
                                [159.8,156.2,149.1,4.5],
                                [163.1,155.5,149.9,4.8],
                                [167.2,158.6,154.9,4.2],
                                [166.1,160.8,158.7,4.0],
                                [163.1,160.2,155.1,4.0],
                                [165.7,161.6,152.7,4.1],
                                [164.0,164.9,156.0,4.1],
                                [163.7,163.4,153.8,4.4],
                                [165.9,160.1,153.9,4.4],
                                [165.1,164.0,155.7,4.3],
                                [165.1,163.7,155.8,4.6],
                                [168.3,165.9,160.1,4.6],
                                [173.0,169.2,154.7,4.4],
                                [168.1,166.7,159.1,4.9],
                                [166.5,164.7,157.8,4.6],
                                [178.9,181.6,161.2,4.5],
                                [170.7,169.3,157.6,4.5],
                                [166.4,166.2,158.8,4.6],
                                [167.4,169.9,158.8,4.6],
                                [162.3,165.1,153.7,5.1],
                                [167.2,167.8,159.3,4.5],
                                [167.2,163.8,160.5,4.5],
                                [165.1,166.3,160.4,4.3],
                                [167.2,166.8,160.4,4.3],
                                [167.2,166.9,155.4,4.5],
                                [165.3,166.1,156.0,4.5],
                                [169.3,167.5,158.3,4.5],
                                [166.2,172.1,162.2,4.9],
                                [171.1,170.5,163.9,4.9],
                                [172.1,172.6,164.4,4.7],
                                [167.0,167.0,159.2,4.4],
                                [172.3,173.0,160.9,4.4],
                                [170.6,171.5,161.5,4.9],
                                [171.5,172.7,166.1,4.9],
                                [176.0,175.6,164.4,4.5],
                                [176.6,177.8,160.5,4.6],
                                [176.3,173.8,165.0,4.6],
                                [175.0,174.4,167.8,4.8],
                                [175.5,174.7,167.0,4.8],
                                [175.8,174.7,166.6,4.8],
                                [173.3,174.1,166.5,4.6],
                                [174.8,172.4,161.4,4.6],
                                [178.2,173.0,166.3,4.4],
                                [173.9,172.9,166.8,4.4],
                                [177.0,171.4,163.0,4.4],
                                [182.1,186.3,164.2,4.6],
                                [177.5,174.9,167.8,4.6],
                                [174.4,176.9,165.2,4.6],
                                [177.7,176.5,166.3,4.6],
                                [177.7,177.0,168.8,4.5],
                                [174.9,175.4,166.7,4.9],
                                [176.0,178.2,171.9,4.9],
                                [174.0,173.1,166.5,4.6],
                                [177.1,173.0,166.9,4.6],
                                [177.2,174.1,170.0,4.5],
                                [175.8,174.0,164.8,4.3],
                                [177.8,174.2,167.2,4.9],
                                [173.9,173.2,163.9,4.2],
                                [175.3,173.5,163.0,5.2],
                                [175.4,174.5,168.1,4.2],
                                [173.9,170.9,168.4,4.2],
                                [176.7,173.3,168.4,4.3],
                                [175.8,174.3,168.6,4.1],
                                [173.5,174.0,167.7,4.1],
                                [173.5,171.2,168.0,4.3],
                                [175.4,173.3,163.7,4.3],
                                [176.8,170.9,162.6,4.1],
                                [171.9,171.3,163.0,4.2],
                                [171.9,168.0,162.6,4.6],
                                [172.2,168.6,164.1,4.1],
                                [168.7,167.7,163.9,4.1],
                                [168.8,169.5,163.7,4.1],
                                [172.0,171.5,165.0,4.3],
                                [165.9,166.4,159.6,4.3],
                                [168.1,165.1,158.8,4.3],
                                [161.7,161.4,153.8,4.3],
                                [159.5,162.7,153.8,4.5],
                                [160.2,156.1,149.6,4.3],
                                [160.9,159.7,154.5,4.3],
                                [155.7,155.1,149.3,4.6],
                                [152.4,151.3,149.3,4.6],
                                [162.7,156.4,154.6,4.4],
                                [157.0,157.4,147.6,4.6],
                                [157.0,161.5,147.6,4.6],
                                [158.7,156.5,151.6,4.4],
                                [158.7,156.5,151.6,4.4],
                                [152.6,151.5,144.6,4.6],
                                [149.4,150.2,143.4,4.4],
                                [152.5,150.2,144.5,4.4],
                                [147.4,146.5,139.4,4.9],
                                [147.4,146.5,139.4,4.9],
                                [149.1,151.5,139.4,4.8],
                                [152.2,153.7,144.1,5.4],
                                [157.6,156.4,150.1,5.4],
                                [157.8,159.8,150.1,5.4],
                                [162.9,166.5,155.4,5.4],
                                [162.9,166.6,155.4,5.4],
                                [165.5,165.2,156.4,5.4],
                                [165.5,165.2,160.4,5.4],
                                [169.9,166.6,160.4,5.7],
                                [169.9,167.1,165.5,5.7],
                                [172.7,170.0,165.7,5.2],
                                [173.9,173.3,165.5,5.9],
                                [178.5,177.1,165.5,5.9],
                                [175.9,175.3,170.7,5.4],
                                [175.9,175.3,165.5,5.4],
                                [178.1,177.3,167.9,5.4],
                                [182.5,181.2,170.9,5.5],
                                [184.0,181.2,170.9,5.5],
                                [179.0,181.1,172.3,5.9],
                                [179.0,177.3,172.3,5.9],
                                [178.9,176.0,165.5,5.6],
                                [178.0,177.0,165.6,6.1],
                                [178.0,177.0,166.6,6.1],
                                [178.2,175.3,167.0,5.6],
                                [178.2,177.6,166.7,5.6],
                                [178.0,174.3,168.3,5.7],
                                [180.4,175.8,167.5,6.1],
                                [180.0,175.8,167.7,6.7],
                                [177.4,175.0,165.6,6.4],
                                [174.9,172.6,162.6,6.4],
                                [174.2,172.2,161.8,6.2],
                                [176.5,173.6,162.9,6.0],
                                [176.5,173.6,162.9,6.0],
                                [174.5,170.7,164.1,6.1],
                                [174.5,170.7,164.1,6.1],
                                [172.3,173.0,163.0,6.1],
                                [176.1,172.1,164.0,6.2],
                                [176.1,172.1,164.0,6.2],
                                [176.6,172.0,164.1,6.0],
                                [176.6,172.0,164.1,5.8],
                                [169.9,168.4,161.5,5.8],
                                [170.4,169.4,161.3,7.2],
                                [175.0,168.0,161.3,6.1],
                                [172.3,170.5,160.8,6.1],
                                [175.2,173.2,160.8,6.1],
                                [174.0,169.2,160.3,6.5],
                                [172.7,171.0,160.8,6.2],
                                [170.1,171.0,160.8,6.2],
                                [170.0,173.3,161.3,6.4],
                                [170.0,168.3,157.6,6.4],
                                [169.6,166.8,162.7,7.2],
                                [171.6,169.7,159.6,6.5],
                                [171.6,169.7,159.6,6.5],
                                [172.2,170.0,161.0,6.7],
                                [172.2,170.0,161.0,6.7],
                                [171.7,171.0,161.8,6.6],
                                [169.7,168.0,157.5,6.7],
                                [169.7,168.0,157.5,6.7],
                                [168.8,168.5,158.8,7.2],
                                [168.8,168.5,158.8,7.2],
                                [165.0,168.2,157.7,6.5],
                                [166.7,165.7,154.2,6.5],
                                [166.7,163.2,152.7,6.5],
                                [164.6,165.2,152.7,6.2],
                                [164.6,165.2,152.8,6.2],
                                [162.6,163.3,153.1,6.2],
                                [165.9,164.6,153.7,6.7],
                                [165.9,164.6,153.7,6.2],
                                [162.7,161.5,153.7,6.3],
                                [159.5,161.5,153.7,6.3],
                                [161.4,162.2,152.7,6.5],
                                [161.4,162.2,152.7,6.5],
                                [160.7,158.2,152.6,6.3],
                                [160.9,159.7,150.9,6.2],
                                [160.9,159.7,150.9,6.2],
                                [160.4,159.6,147.8,6.3],
                                [160.4,159.6,147.8,6.2],
                                [157.5,156.0,147.2,6.3],
                                [157.2,155.9,146.7,6.3],
                                [157.2,155.9,146.7,6.3],
                                [154.4,158.3,146.4,6.2],
                                [154.4,153.3,142.4,6.2],
                                [153.1,152.3,143.6,6.1],
                                [152.5,150.8,143.5,6.2],
                                [152.5,150.8,143.5,6.2],
                                [151.3,150.6,141.7,6.2],
                                [149.3,150.6,141.7,6.2],
                                [151.7,148.1,142.2,6.1],
                                [151.1,149.9,141.4,6.3],
                                [151.1,149.9,141.4,6.3],
                                [147.6,148.7,142.9,6.2],
                                [147.6,148.7,142.9,6.2],
                                [150.2,149.2,140.8,6.4],
                                [149.9,148.5,140.3,6.2],
                                [149.9,148.5,140.3,6.2],
                                [148.0,147.7,139.6,6.2],
                                [148.0,147.7,139.6,6.2],
                                [144.3,145.9,138.9,6.2],
                                [145.7,143.1,135.8,6.1],
                                [145.7,143.1,135.8,6.1],
                                [144.9,144.9,136.3,5.8],
                                [144.9,144.9,136.3,5.8],
                                [143.6,144.5,135.1,6.0],
                                [143.9,144.1,136.1,5.8],
                                [143.9,144.1,132.7,5.8],
                                [142.3,143.5,133.7,6.0],
                                [142.3,143.5,133.7,6.0],
                                [141.8,141.3,132.7,6.2],
                                [140.9,140.2,132.0,5.7],
                                [140.9,140.2,132.0,5.7],
                                [139.3,141.3,132.1,6.1],
                                [139.3,138.0,132.1,6.1],
                                [140.0,139.8,131.0,6.1],
                                [140.0,139.3,131.0,5.6],
                                [140.0,139.3,131.0,5.6],
                                [138.6,138.0,131.8,5.6],
                                [138.6,138.0,131.8,5.6],
                                [138.3,136.2,130.8,6.0],
                                [139.8,138.3,131.6,5.8],
                                [139.8,138.3,127.6,5.8],
                                [137.3,137.3,130.5,6.1],
                                [137.3,137.3,130.5,6.1],
                                [137.9,135.5,128.3,6.2],
                                [138.8,137.3,129.9,5.8],
                                [138.8,137.3,129.9,5.8],
                                [137.9,135.9,128.0,6.3],
                                [137.9,135.9,128.0,6.3],
                                [139.2,136.5,129.8,5.7],
                                [134.5,131.0,123.4,5.3],
                                [129.0,131.0,123.4,5.3],
                                [128.0,127.9,122.3,5.7],
                                [128.0,127.9,117.4,5.7],
                                [124.0,126.8,115.7,5.5],
                                [126.2,126.8,117.9,5.4],
                                [126.2,126.8,117.9,5.4],
                                [122.5,122.8,116.1,5.4],
                                [122.5,122.8,116.1,5.4],
                                [121.4,122.5,116.3,5.1],
                                [121.1,117.6,115.3,5.1],
                                [121.1,122.7,115.3,5.1],
                                [122.7,127.8,117.5,4.9],
                                [129.3,127.8,122.5,4.9],
                                [124.2,127.4,117.5,4.8],
                                [124.6,123.5,118.8,4.8],
                                [129.3,127.8,117.7,4.8],
                                [125.5,125.9,118.9,5.0],
                                [129.4,125.9,122.8,5.4],
                                [130.9,127.9,123.6,5.2],
                                [129.2,127.8,122.1,5.1],
                                [124.4,127.8,117.9,5.1],
                                [129.3,128.6,123.5,5.1],
                                [129.3,128.6,123.1,5.1],
                                [132.1,132.7,124.8,5.3],
                                [131.5,131.2,124.7,5.5],
                                [129.8,131.2,123.2,5.5],
                                [130.6,131.6,128.7,5.2],
                                [135.3,133.5,128.7,5.2],
                                [130.3,132.3,126.4,4.7],
                                [134.0,134.9,129.3,4.6],
                                [134.7,134.9,129.3,4.6],
                                [135.8,136.5,130.8,4.6],
                                [142.8,147.7,131.1,5.3],
                                [138.0,136.7,130.8,5.0],
                                [142.4,142.9,135.9,4.7],
                                [142.4,142.9,135.9,4.7],
                                [146.2,145.0,136.8,4.9],
                                [147.0,144.5,136.6,4.3],
                                [150.3,145.8,137.3,4.6],
                                [154.1,162.6,140.8,4.7],
                                [149.6,148.6,140.4,4.7],
                                [155.6,151.3,146.1,5.4],
                                [156.2,149.8,144.8,4.4],
                                [150.6,146.9,138.0,4.4],
                                [153.7,149.3,141.8,4.1],
                                [150.6,149.3,144.3,4.1],
                                [151.4,148.0,144.0,4.3],
                                [152.0,151.0,144.8,4.3],
                                [155.9,151.2,149.7,4.6],
                                [155.9,152.9,147.4,4.6],
                                [157.2,153.1,148.3,4.3],
                                [157.6,152.9,143.3,4.4],
                                [156.5,151.3,145.3,4.2],
                                [157.2,151.8,145.0,4.4],
                                [158.9,157.5,150.1,4.6],
                                [153.9,152.9,145.8,4.6],
                                [162.3,159.2,151.3,4.5],
                                [157.3,154.1,146.3,4.5],
                                [160.2,155.2,147.9,4.7],
                                [159.8,154.8,146.0,4.7],
                                [166.8,171.8,151.2,4.5],
                                [157.9,155.8,149.5,4.4],
                                [160.6,156.6,148.8,4.4],
                                [162.5,155.7,152.2,4.2],
                                [162.6,161.5,152.9,4.2],
                                [163.9,159.7,153.7,4.5],
                                [163.7,159.8,151.5,4.6],
                                [162.3,159.6,155.2,4.6],
                                [163.2,159.6,155.1,5.0],
                                [163.8,165.1,150.7,5.0],
                                [164.9,156.9,154.3,4.4],
                                [162.3,163.0,154.5,4.5],
                                [163.7,158.5,153.1,4.5],
                                [163.7,164.9,152.0,4.4],
                                [165.8,163.9,153.3,4.4],
                                [168.9,162.2,154.1,4.3],
                                [167.2,164.8,149.5,4.6],
                                [172.5,174.7,156.2,4.6],
                                [160.9,160.1,151.4,4.5],
                                [166.7,166.3,156.5,4.5],
                                [165.6,164.5,156.2,4.5],
                                [166.2,165.5,154.6,4.4],
                                [168.0,164.8,155.4,4.4],
                                [166.7,160.5,155.8,4.4],
                                [162.6,160.5,155.9,4.4],
                                [162.5,160.5,151.1,4.5],
                                [167.0,165.3,152.2,4.5],
                                [166.2,161.5,153.0,4.5],
                                [161.1,161.5,152.3,4.8],
                                [166.1,161.5,152.3,4.6],
                                [166.1,163.7,153.9,4.6],
                                [164.4,163.7,148.9,4.3],
                                [166.0,163.7,154.2,4.3],
                                [165.7,165.5,154.4,4.3],
                                [165.7,162.0,155.2,4.3],
                                [164.6,162.0,155.2,4.2],
                                [165.6,160.8,154.3,4.0],
                                [165.9,163.4,154.3,4.0],
                                [165.9,162.4,153.9,4.3],
                                [162.6,158.2,153.5,4.3],
                                [163.3,158.7,153.5,4.3],
                                [168.9,164.6,153.9,4.7],
                                [168.5,164.3,155.4,4.7],
                                [166.8,165.0,154.7,4.7],
                                [169.9,168.3,157.5,4.7],
                                [167.3,166.6,157.2,4.4],
                                [166.9,165.2,159.8,4.5],
                                [169.1,165.0,156.7,4.5],
                                [158.8,158.8,151.1,4.7],
                                [172.8,177.7,155.4,4.7],
                                [167.8,163.2,156.7,4.6],
                                [167.7,165.5,156.3,4.3],
                                [171.5,168.7,158.9,4.3],
                                [163.8,160.6,156.3,4.7],
                                [163.3,163.1,154.0,4.7],
                                [163.4,160.8,154.9,4.7],
                                [165.3,158.9,155.7,4.6],
                                [160.3,161.7,150.2,4.6],
                                [165.5,160.9,155.2,4.4],
                                [161.3,162.0,155.2,4.4],
                                [161.3,160.8,150.1,4.3],
                                [161.3,159.0,149.5,4.3],
                                [157.7,161.8,149.0,4.3],
                                [162.1,159.6,149.3,4.1],
                                [164.8,161.1,149.9,4.1],
                                [159.7,160.1,150.6,4.4],
                                [162.2,159.9,153.4,4.5],
                                [160.6,159.4,151.3,4.7],
                                [165.3,162.2,153.3,4.3],
                                [160.4,157.2,153.3,4.2],
                                [157.4,155.8,148.7,4.5],
                                [160.8,160.9,154.1,4.4],
                                [168.0,172.3,150.4,4.4],
                                [162.7,159.5,151.1,4.2],
                                [161.7,160.4,156.9,4.2],
                                [161.7,156.5,151.9,5.0],
                                [162.6,161.9,154.9,4.6],
                                [162.6,156.7,152.0,4.6],
                                [159.2,156.9,151.3,4.3],
                                [156.7,156.9,152.1,4.3],
                                [156.7,157.7,152.0,4.0],
                                [158.9,157.5,152.2,4.5],
                                [158.0,152.4,150.3,4.2],
                                [152.2,152.0,146.5,4.1],
                                [158.8,152.7,146.5,3.5],
                                [153.9,153.3,147.1,3.8],
                                [148.9,153.3,142.0,3.8],
                                [154.0,148.3,142.3,3.9],
                                [148.9,150.0,140.4,4.3],
                                [153.9,148.5,140.4,4.3],
                                [148.9,149.3,143.3,4.4],
                                [148.9,148.9,143.3,4.4],
                                [148.8,147.1,140.4,4.7],
                                [149.6,149.2,140.1,4.2],
                                [149.6,148.8,137.3,4.2],
                                [144.1,148.8,137.9,4.6],
                                [149.1,149.0,137.9,5.2],
                                [149.1,149.4,137.9,4.1],
                                [144.9,146.1,138.3,4.2],
                                [149.2,149.8,138.3,4.2],
                                [149.0,149.0,140.8,4.7],
                                [154.6,160.2,148.2,5.1],
                                [156.0,155.5,148.4,4.8],
                                [160.4,159.3,153.4,4.8],
                                [170.3,170.7,158.5,4.8],
                                [168.0,170.8,158.4,4.9],
                                [170.2,170.8,158.4,4.9],
                                [168.9,171.1,158.8,4.9],
                                [170.7,169.0,162.7,4.7],
                                [170.7,169.0,158.7,4.7],
                                [170.8,169.0,160.3,5.5],
                                [175.3,175.8,163.9,5.5],
                                [175.4,173.3,161.2,6.1],
                                [174.2,175.4,160.8,5.9],
                                [174.2,170.8,163.8,5.9],
                                [173.1,170.0,162.4,5.5],
                                [173.1,175.9,162.4,5.5],
                                [170.4,170.9,158.8,5.4],
                                [170.4,170.3,159.9,5.9],
                                [170.8,171.0,159.9,5.9],
                                [171.7,165.9,161.3,5.6],
                                [171.7,165.9,164.2,5.6],
                                [170.1,165.6,159.1,5.3],
                                [170.7,165.6,158.0,5.4],
                                [170.7,165.6,158.0,5.4],
                                [170.3,166.0,160.6,5.7],
                                [170.3,166.0,160.6,5.7],
                                [175.6,170.4,164.1,5.3],
                                [170.4,166.7,156.9,5.4],
                                [170.4,166.7,156.9,5.4],
                                [171.8,168.7,160.2,5.4],
                                [171.8,166.1,160.2,5.4],
                                [165.7,161.1,153.7,5.3],
                                [167.6,160.7,153.8,5.4],
                                [165.7,160.7,153.8,6.1],
                                [173.0,166.5,157.8,5.1],
                                [173.0,166.5,154.1,6.1],
                                [168.9,161.7,154.8,5.4],
                                [165.4,161.0,150.0,5.2],
                                [165.4,161.0,154.1,5.2],
                                [167.1,162.3,153.7,5.5],
                                [167.1,162.3,153.7,5.2],
                                [163.5,160.2,154.1,5.3],
                                [167.4,162.7,155.3,5.3],
                                [167.4,162.7,155.3,5.3],
                                [165.0,159.5,152.6,5.3],
                                [165.0,159.5,152.6,5.3],
                                [164.8,159.6,150.8,5.2],
                                [164.3,161.1,151.8,5.3],
                                [164.3,161.1,151.8,5.3],
                                [165.7,161.2,153.9,5.1],
                                [165.9,161.8,153.9,5.1],
                                [167.3,163.3,153.9,5.1],
                                [167.7,162.9,155.7,5.5],
                                [160.9,162.9,155.7,5.5],
                                [165.9,161.2,153.7,6.3],
                                [165.9,161.2,153.7,6.3],
                                [160.9,156.7,149.0,5.3],
                                [166.0,162.7,152.9,5.9],
                                [160.9,162.7,152.9,6.3],
                                [163.5,162.3,153.1,5.5],
                                [163.5,156.8,149.1,5.5],
                                [155.7,157.3,148.8,6.0],
                                [160.0,157.0,149.3,5.6],
                                [155.8,157.0,149.3,6.4],
                                [159.0,157.0,147.7,5.7],
                                [156.2,151.8,147.7,5.7],
                                [161.5,155.7,147.9,5.9],
                                [157.5,154.0,149.0,5.8],
                                [157.5,154.0,148.5,5.8],
                                [157.9,154.7,148.9,6.3],
                                [157.9,151.6,143.4,6.3],
                                [153.6,150.6,143.7,5.7],
                                [152.7,150.5,143.7,5.9],
                                [155.5,150.5,143.7,5.9],
                                [152.7,149.9,143.9,6.5],
                                [151.5,149.9,143.9,5.5],
                                [153.8,150.0,143.1,5.5],
                                [149.8,147.8,140.3,5.5],
                                [150.1,147.8,140.3,5.5],
                                [149.5,147.6,138.9,5.7],
                                [149.5,147.6,138.9,5.7],
                                [148.4,146.1,137.4,5.5],
                                [147.3,143.5,136.2,5.8],
                                [147.3,146.2,136.2,5.8],
                                [148.3,145.1,138.0,6.7],
                                [148.3,145.1,138.0,6.7],
                                [144.7,143.5,135.4,5.5],
                                [146.9,142.9,136.3,5.7],
                                [146.9,142.9,138.3,5.7],
                                [145.7,141.7,135.6,5.6],
                                [145.7,141.2,138.4,5.6],
                                [145.7,141.8,133.3,5.4],
                                [144.3,140.8,135.2,5.5],
                                [144.3,140.8,135.2,5.5],
                                [143.6,140.6,133.4,5.6],
                                [143.6,140.6,133.4,5.6],
                                [140.1,139.7,131.7,5.6],
                                [143.5,140.8,132.9,5.5],
                                [143.5,140.8,132.9,5.5],
                                [140.5,136.1,131.7,5.5],
                                [144.5,141.2,131.7,5.5],
                                [141.2,136.9,132.4,5.6],
                                [144.7,136.9,132.4,5.6],
                                [141.8,138.3,130.9,5.6],
                                [141.5,138.0,130.6,5.7],
                                [141.5,138.0,130.6,5.7],
                                [140.7,137.4,128.8,5.6],
                                [140.7,137.4,128.8,5.6],
                                [138.0,135.2,128.3,5.7],
                                [138.8,137.1,128.6,5.6],
                                [138.8,137.1,128.6,5.6],
                                [138.5,136.3,127.8,5.4],
                                [134.6,131.0,127.8,5.4],
                                [136.6,136.1,127.9,5.6],
                                [138.9,134.0,127.9,5.4],
                                [138.9,134.0,127.9,5.4],
                                [136.0,132.0,128.2,5.4],
                                [140.5,132.0,128.2,5.4],
                                [139.1,133.9,127.5,5.4],
                                [137.4,135.3,126.8,5.6],
                                [135.5,135.3,126.8,5.6],
                                [135.4,132.6,126.8,5.6],
                                [135.4,132.6,123.2,5.6],
                                [136.9,133.9,123.2,5.5],
                                [134.8,130.2,124.9,5.4],
                                [134.8,130.2,123.7,5.4],
                                [129.7,126.3,123.7,5.4],
                                [129.7,126.3,118.4,5.4],
                                [125.0,125.2,120.4,5.8],
                                [123.4,122.9,116.8,4.8],
                                [123.4,121.3,116.8,5.8],
                                [123.4,122.0,116.6,5.1],
                                [123.4,122.0,116.6,5.1],
                                [124.4,122.5,112.6,6.0],
                                [119.3,120.7,112.5,5.4],
                                [119.3,116.3,112.5,5.0],
                                [118.1,116.4,111.4,5.0],
                                [118.1,116.4,111.4,5.0],
                                [117.4,116.2,111.0,4.3],
                                [121.8,121.0,114.6,4.7],
                                [119.7,121.0,112.4,4.7],
                                [120.4,117.1,113.0,4.6],
                                [120.4,121.5,113.0,4.6],
                                [118.8,116.5,111.7,4.7],
                                [121.5,121.0,114.5,4.9],
                                [124.7,121.0,114.5,4.9],
                                [121.3,121.5,113.8,4.6],
                                [125.0,121.5,117.6,4.6],
                                [125.4,124.2,117.7,5.0],
                                [122.3,120.8,115.5,4.6],
                                [122.3,120.8,115.5,4.6],
                                [130.1,126.6,122.7,5.1],
                                [125.0,126.6,122.7,5.0],
                                [124.9,121.6,117.7,5.0],
                                [123.0,123.1,116.6,4.8],
                                [123.0,123.1,116.6,4.8],
                                [125.5,122.7,117.2,4.8],
                                [125.5,122.7,117.2,4.8],
                                [125.6,126.9,123.1,5.3],
                                [124.0,124.6,118.2,4.7],
                                [126.1,124.1,118.0,4.7],
                                [126.6,126.6,118.0,4.9],
                                [126.9,126.3,119.1,4.9],
                                [129.3,127.4,121.7,4.7],
                                [127.5,128.4,119.4,4.7],
                                [129.2,133.1,124.4,4.7],
                                [130.8,128.1,122.2,4.2],
                                [127.8,128.7,124.3,4.2],
                                [132.9,134.1,125.4,4.4],
                                [131.8,130.6,124.5,4.0],
                                [131.8,130.6,124.5,4.0],
                                [133.6,131.4,125.4,4.2],
                                [132.7,131.1,126.1,4.2],
                                [129.9,130.3,121.1,4.5],
                                [128.9,128.2,123.3,4.3],
                                [130.7,131.7,126.6,4.3],
                                [136.2,131.7,125.9,4.2],
                                [136.4,132.8,125.9,4.2],
                                [136.3,130.5,125.5,4.1],
                                [131.7,129.5,121.9,4.3],
                                [136.3,129.5,126.7,4.3],
                                [136.4,136.0,126.9,4.4],
                                [141.4,136.0,132.0,4.4],
                                [136.4,139.1,126.8,4.4],
                                [135.3,134.3,129.8,4.2],
                                [135.3,134.3,129.8,4.2],
                                [134.4,132.1,124.6,4.2],
                                [134.4,138.0,126.9,4.2],
                                [136.1,135.7,127.8,4.2],
                                [132.7,134.2,125.7,4.3],
                                [136.6,133.0,125.7,4.3],
                                [134.7,132.8,126.0,4.1],
                                [131.6,132.8,126.0,4.1],
                                [138.7,137.6,127.2,4.5],
                                [135.5,136.2,127.3,4.3],
                                [142.1,136.2,132.4,4.3],
                                [141.7,138.2,133.3,4.0],
                                [137.0,138.2,127.2,4.0],
                                [138.5,137.3,127.2,4.2],
                                [138.5,137.3,132.5,4.2],
                                [137.1,135.5,132.5,4.4],
                                [135.2,134.3,132.5,4.0],
                                [137.0,134.3,132.5,4.0],
                                [138.8,137.5,131.9,4.3],
                                [138.8,143.2,131.9,4.3],
                                [136.9,138.1,131.0,4.1],
                                [138.3,137.5,131.0,4.1],
                                [142.2,138.2,131.0,4.1],
                                [141.1,138.1,133.8,4.1],
                                [141.1,143.5,132.5,4.1],
                                [144.0,143.1,137.6,4.6],
                                [142.6,142.9,133.3,4.3],
                                [142.0,142.9,132.8,4.3],
                                [140.0,141.4,132.7,4.5],
                                [147.1,143.1,132.7,4.5],
                                [142.0,141.3,134.4,4.5],
                                [142.1,142.1,133.3,4.3],
                                [142.1,142.1,133.3,4.3],
                                [147.0,144.5,137.9,4.5],
                                [147.0,144.5,137.9,4.5],
                                [145.7,145.7,138.2,4.5],
                                [143.4,142.7,132.9,4.2],
                                [143.4,138.2,132.9,4.2],
                                [142.5,140.4,131.8,4.5],
                                [142.5,143.2,138.0,4.5],
                                [141.5,142.6,132.9,4.6],
                                [142.4,142.3,133.2,4.5],
                                [142.4,142.3,133.2,4.5],
                                [141.2,143.1,132.3,4.7],
                                [141.2,143.1,132.3,4.7],
                                [141.9,141.6,131.9,4.6],
                                [141.0,143.5,133.0,4.7],
                                [141.0,138.6,133.0,4.7],
                                [141.1,144.2,133.0,4.7],
                                [141.1,144.2,133.0,4.7],
                                [142.3,139.0,134.9,4.5],
                                [144.3,145.8,134.8,4.7],
                                [141.7,145.8,132.9,4.7],
                                [140.0,142.0,133.4,4.8],
                                [140.0,142.0,138.5,4.8],
                                [141.7,144.1,133.5,4.1],
                                [140.1,141.3,131.4,4.2],
                                [142.6,144.7,135.2,4.4],
                                [144.0,143.8,135.6,4.3],
                                [144.0,143.8,135.6,4.3],
                                [142.4,142.9,135.9,4.5],
                                [143.5,145.0,136.5,4.1],
                                [142.8,149.8,134.9,4.1],
                                [148.0,145.7,140.6,4.6],
                                [142.8,145.7,135.6,4.6],
                                [143.0,144.9,135.5,4.6],
                                [141.7,141.7,134.2,4.2],
                                [138.0,139.5,130.5,4.2],
                                [139.8,139.7,133.3,4.6],
                                [139.8,144.6,133.3,4.6],
                                [141.1,139.5,134.9,4.1],
                                [142.8,144.6,136.2,4.0],
                                [142.8,139.4,136.2,4.0],
                                [141.1,138.4,133.8,4.2],
                                [143.2,138.4,133.8,4.2],
                                [138.8,138.5,131.6,4.3],
                                [141.9,144.5,135.2,4.4],
                                [141.9,139.5,135.2,4.4],
                                [140.1,139.7,134.6,4.5],
                                [140.1,139.7,134.6,4.5],
                                [143.4,139.8,134.4,4.5],
                                [142.3,139.8,135.0,4.0],
                                [138.0,140.3,130.5,4.0],
                                [142.9,140.2,130.4,4.2],
                                [138.3,135.2,135.5,4.2],
                                [140.2,137.0,130.5,4.2],
                                [140.2,140.6,131.6,4.7],
                                [140.2,137.3,130.9,4.7],
                                [137.7,138.7,131.1,4.4],
                                [137.7,138.7,131.1,4.4],
                                [139.8,137.5,131.1,4.5],
                                [141.2,140.5,133.8,4.5],
                                [141.2,137.5,131.4,4.5],
                                [142.7,141.4,131.2,4.4],
                                [142.7,141.4,136.4,4.4],
                                [143.8,138.1,131.2,4.4],
                                [142.9,140.5,132.0,4.4],
                                [142.9,140.5,132.0,4.4],
                                [138.7,143.7,130.6,4.3],
                                [143.0,143.9,136.4,4.3],
                                [141.8,138.7,134.7,4.6],
                                [144.4,145.3,137.5,5.1],
                                [148.3,149.2,136.4,5.6],
                                [153.3,147.3,139.7,5.3],
                                [158.6,154.3,152.2,5.3],
                                [157.8,155.3,148.4,5.4],
                                [162.9,158.1,150.8,5.2],
                                [162.9,158.1,150.8,5.2],
                                [163.8,158.8,152.3,5.3],
                                [163.8,159.6,156.7,5.3],
                                [166.1,162.5,156.0,5.3],
                                [165.9,158.9,151.2,5.2],
                                [169.0,160.4,156.0,5.2],
                                [166.0,165.5,153.7,5.3],
                                [163.6,160.4,150.6,5.3],
                                [164.5,160.2,150.7,5.2],
                                [165.6,162.8,155.7,5.3],
                                [169.1,165.6,161.0,5.3],
                                [168.2,161.7,156.8,5.3],
                                [169.0,162.0,156.8,5.3],
                                [164.4,157.0,154.3,5.2],
                                [167.2,164.3,158.8,5.8],
                                [167.2,164.3,158.8,5.8],
                                [169.0,163.0,156.4,5.9],
                                [163.9,163.0,156.9,5.9],
                                [168.9,162.1,156.0,5.7],
                                [165.3,161.5,153.9,5.8],
                                [165.3,161.5,157.0,5.8],
                                [168.9,162.9,154.9,5.9],
                                [163.9,162.9,152.0,5.6],
                                [168.2,164.6,157.8,5.5],
                                [164.0,164.6,157.8,5.5],
                                [169.0,160.2,156.5,5.5],
                                [166.3,159.8,151.9,5.6],
                                [158.7,159.8,152.0,5.6],
                                [163.8,156.7,152.5,5.7],
                                [163.8,156.7,152.5,5.7],
                                [162.7,157.1,151.7,5.4],
                                [161.9,157.6,152.4,5.5],
                                [161.9,157.6,147.4,5.5],
                                [163.7,157.9,152.8,5.7],
                                [163.7,157.9,147.8,5.7],
                                [162.5,158.3,149.3,5.8],
                                [162.9,158.1,152.6,5.4],
                                [162.9,158.1,152.6,5.4],
                                [162.3,162.5,151.9,5.6],
                                [163.8,157.4,152.8,5.6],
                                [162.0,162.8,152.1,5.6],
                                [165.2,162.9,151.6,6.3],
                                [165.2,162.6,152.8,6.3],
                                [163.6,157.5,152.8,5.7],
                                [163.6,162.6,152.8,5.7],
                                [164.2,160.0,150.7,5.6],
                                [164.0,162.3,152.0,5.9],
                                [164.0,162.3,152.0,5.9],
                                [162.6,162.7,153.2,6.0],
                                [162.6,162.7,153.2,6.0],
                                [162.1,163.2,155.5,6.0],
                                [163.1,163.2,153.3,5.9],
                                [163.1,163.2,153.3,5.9],
                                [164.2,162.9,153.9,6.4],
                                [164.2,162.9,153.9,6.4],
                                [158.4,163.8,154.2,6.4],
                                [160.8,157.4,150.3,5.7],
                                [160.8,162.6,150.3,6.6],
                                [162.5,163.1,154.1,6.3],
                                [158.4,162.5,152.6,7.7],
                                [158.5,162.3,147.5,6.7],
                                [160.9,160.6,152.6,6.1],
                                [158.7,160.6,152.6,6.1],
                                [158.5,159.2,148.8,6.4],
                                [158.5,157.3,148.8,6.4],
                                [160.5,157.3,148.5,6.3],
                                [157.0,154.8,146.5,6.6],
                                [157.0,154.8,146.5,6.6],
                                [156.9,156.7,148.1,6.8],
                                [153.6,156.7,148.1,6.8],
                                [155.7,152.2,142.4,6.3],
                                [153.4,152.7,142.0,6.7],
                                [153.4,152.7,142.0,6.7],
                                [154.5,153.7,145.2,6.1],
                                [154.5,153.7,145.2,6.1],
                                [155.9,152.5,142.4,6.1],
                                [154.1,153.5,142.4,6.2],
                                [154.1,153.5,142.4,6.2],
                                [153.2,153.4,142.8,6.3],
                                [153.2,147.4,137.4,6.3],
                                [148.2,149.9,137.7,6.4],
                                [149.6,148.5,138.1,6.2],
                                [149.6,148.5,138.1,6.2],
                                [148.6,146.4,136.2,6.1],
                                [148.6,146.4,136.2,6.1],
                                [147.9,146.5,135.6,6.2],
                                [148.0,145.5,137.3,6.1],
                                [148.0,145.5,137.3,6.1],
                                [147.0,146.8,136.2,6.2],
                                [147.0,146.8,136.2,6.2],
                                [147.5,146.0,135.0,6.4],
                                [147.1,145.2,135.9,6.2],
                                [143.2,145.2,132.5,6.2],
                                [145.4,143.8,133.5,6.2],
                                [145.4,143.8,133.5,6.2],
                                [144.6,143.0,131.7,6.1],
                                [143.6,142.1,132.9,6.1],
                                [143.6,142.1,132.9,6.1],
                                [143.6,142.2,131.6,6.2],
                                [143.6,142.2,131.6,6.2],
                                [143.6,142.3,132.2,6.2],
                                [141.4,141.0,131.4,6.0],
                                [141.4,141.0,131.4,6.0],
                                [143.7,141.8,132.8,6.2],
                                [143.7,141.8,132.8,6.2],
                                [141.8,139.4,130.7,6.1],
                                [139.6,138.0,130.6,6.0],
                                [143.1,142.4,130.6,6.7],
                                [140.4,139.1,131.2,5.6],
                                [140.4,137.4,131.2,5.6],
                                [138.0,138.9,127.4,5.8],
                                [143.2,138.8,132.4,5.6],
                                [138.1,138.8,132.4,5.6],
                                [141.2,137.5,130.3,5.6],
                                [137.7,137.5,127.3,5.6],
                                [138.1,138.4,128.2,5.7],
                                [136.7,135.2,127.3,5.8],
                                [136.7,135.2,127.3,5.6],
                                [137.8,136.5,128.6,5.5],
                                [137.8,136.5,128.6,5.5],
                                [137.7,135.1,127.5,5.6],
                                [136.1,134.5,126.8,5.7],
                                [132.7,134.5,122.3,5.7],
                                [127.8,127.2,119.3,5.7],
                                [127.8,127.2,117.2,5.7],
                                [122.5,127.2,117.9,5.8],
                                [125.9,125.7,116.4,5.6],
                                [125.9,122.1,116.4,5.6],
                                [121.3,122.1,114.8,5.3],
                                [121.3,122.1,114.8,5.3],
                                [123.8,123.2,115.6,5.2],
                                [120.7,120.3,113.3,5.1],
                                [120.7,120.3,113.3,5.1],
                                [117.5,117.0,109.8,5.3],
                                [117.5,117.0,109.8,5.6],
                                [116.8,119.5,111.1,5.4],
                                [116.0,117.6,109.7,5.5],
                                [116.0,117.7,109.7,5.5],
                                [118.1,117.7,110.5,5.1],
                                [118.1,117.7,110.5,5.1],
                                [121.2,120.5,114.9,5.3],
                                [118.4,119.1,113.6,5.0],
                                [118.4,119.1,113.6,5.0],
                                [116.2,117.0,110.1,5.4],
                                [116.2,117.0,110.1,5.4],
                                [115.7,117.5,110.8,5.3],
                                [115.7,117.5,110.8,5.3],
                                [117.3,118.5,110.4,5.2],
                                [116.9,123.0,113.2,5.3],
                                [122.8,123.0,117.8,5.3],
                                [117.3,120.8,113.8,5.1],
                                [117.2,120.8,112.7,5.1],
                                [122.3,123.2,117.8,4.6],
                                [120.9,122.3,112.7,4.6],
                                [120.9,123.0,117.9,4.6],
                                [120.4,121.6,118.0,4.4],
                                [122.1,121.6,118.0,4.4],
                                [121.2,123.0,112.9,4.5],
                                [123.0,123.6,116.7,4.5],
                                [123.0,123.6,116.7,4.5],
                                [121.7,122.9,116.8,4.9],
                                [121.7,122.9,113.0,4.9],
                                [122.1,122.9,118.1,4.7],
                                [121.6,121.9,117.8,4.8],
                                [121.6,121.9,117.8,4.8],
                                [124.5,127.3,117.6,4.7],
                                [124.5,127.3,117.6,4.7],
                                [122.0,122.8,118.0,4.7],
                                [123.2,126.3,114.2,4.9],
                                [127.4,122.5,117.8,4.7],
                                [126.2,126.6,117.2,4.8],
                                [122.3,122.5,112.8,4.8],
                                [120.8,121.3,117.9,4.6],
                                [122.3,120.7,114.2,4.6],
                                [122.3,117.5,114.2,4.6],
                                [122.8,120.1,113.1,4.4],
                                [122.8,122.6,113.1,4.4],
                                [119.8,122.5,111.5,4.0],
                                [122.9,121.3,112.7,4.8],
                                [122.9,116.5,112.7,4.8],
                                [122.3,116.4,114.0,4.3],
                                [122.3,116.4,114.0,4.3],
                                [121.9,121.5,112.9,4.2],
                                [125.0,121.8,116.8,4.3],
                                [121.8,121.8,113.4,4.3],
                                [126.9,121.5,116.9,4.4],
                                [126.9,121.5,113.4,4.4],
                                [121.9,121.8,113.4,4.3],
                                [119.5,120.0,111.6,4.3],
                                [119.5,120.0,108.3,4.3],
                                [121.7,119.4,114.3,4.2],
                                [121.7,117.0,114.3,4.2],
                                [121.7,122.1,114.8,4.9],
                                [119.8,119.2,111.4,4.9],
                                [116.8,117.1,111.4,4.9],
                                [119.4,119.8,112.2,4.7],
                                [116.6,116.9,112.2,4.7],
                                [119.3,118.5,113.7,4.7],
                                [123.7,122.4,114.3,4.7],
                                [123.7,122.4,114.3,4.7],
                                [121.0,119.8,113.4,5.0],
                                [121.0,119.8,113.4,5.0],
                                [121.9,120.0,111.7,4.4],
                                [121.8,119.7,113.3,4.5],
                                [121.8,119.7,113.3,4.5],
                                [122.9,121.2,114.0,4.2],
                                [122.9,121.2,114.0,4.2],
                                [122.2,121.6,114.4,4.4],
                                [124.0,122.5,115.9,5.2],
                                [121.9,122.5,115.9,5.2],
                                [123.1,122.2,113.1,4.8],
                                [123.1,122.2,113.1,4.8],
                                [120.9,121.7,111.8,4.6],
                                [119.5,120.4,112.0,4.8],
                                [122.0,120.4,112.0,4.8],
                                [122.8,121.6,113.0,4.8],
                                [127.0,121.6,113.0,4.8],
                                [125.7,124.6,115.3,5.0],
                                [124.3,123.0,116.2,4.7],
                                [124.3,123.0,116.2,4.7],
                                [124.2,121.3,114.4,4.7],
                                [121.9,121.3,113.7,4.7],
                                [123.0,119.6,114.2,4.7],
                                [125.3,122.1,116.4,4.7],
                                [125.3,122.1,116.4,4.7],
                                [122.8,121.7,113.9,4.3],
                                [122.8,121.7,114.2,4.3],
                                [125.3,122.6,116.7,4.5],
                                [125.3,123.0,118.0,5.1],
                                [122.3,123.0,114.5,5.1],
                                [122.3,121.5,118.0,4.6],
                                [122.3,121.5,119.0,4.6],
                                [124.4,122.2,117.9,5.7],
                                [122.2,122.2,117.9,5.7],
                                [123.5,122.4,117.7,5.0],
                                [124.0,121.9,117.1,4.8],
                                [124.0,121.9,117.1,4.8],
                                [124.2,122.3,119.0,4.9],
                                [127.2,122.3,119.0,4.9],
                                [124.0,120.6,117.5,4.9],
                                [129.1,127.4,118.7,5.0],
                                [129.1,121.8,118.7,5.0],
                                [132.5,126.7,124.0,4.9],
                                [132.5,126.7,124.0,4.9],
                                [127.5,127.1,118.9,4.5],
                                [127.4,124.9,121.5,4.9],
                                [127.4,121.6,121.5,4.9],
                                [126.8,121.3,119.7,4.7],
                                [126.8,126.4,119.7,4.7],
                                [131.8,124.2,124.3,4.9],
                                [126.6,125.3,119.9,4.8],
                                [131.7,125.3,124.1,4.8],
                                [130.5,126.6,124.0,4.6],
                                [126.7,126.6,119.1,4.6],
                                [126.8,126.4,118.7,4.6],
                                [127.6,123.3,118.8,4.7],
                                [127.6,126.8,118.8,4.7],
                                [126.9,126.8,124.2,4.7],
                                [131.7,126.8,124.2,4.7],
                                [131.6,125.6,121.9,4.7],
                                [129.1,126.5,121.8,4.4],
                                [129.1,126.5,121.8,4.4],
                                [129.0,126.4,122.7,4.7],
                                [129.0,126.4,122.7,4.7],
                                [129.8,126.9,122.7,4.5],
                                [132.0,130.5,126.9,4.6],
                                [135.8,132.0,129.4,4.6],
                                [136.0,129.7,129.3,4.2],
                                [136.0,129.7,129.3,4.2],
                                [136.1,130.2,123.6,4.3],
                                [140.8,136.5,128.4,4.1],
                                [140.8,135.5,128.4,4.1],
                                [141.4,135.3,128.9,4.5],
                                [141.4,135.7,134.0,4.5],
                                [141.3,140.3,135.4,4.5],
                                [141.3,142.7,132.0,4.7],
                                [141.4,142.7,132.0,4.7],
                                [146.9,144.4,139.1,4.7],
                                [146.9,151.1,139.1,4.7],
                                [146.0,149.2,137.6,4.8],
                                [149.5,149.3,139.2,4.7],
                                [149.5,149.3,139.2,4.7],
                                [151.4,151.1,144.2,4.8],
                                [155.9,157.5,150.0,4.8],
                                [161.1,158.2,150.0,4.4],
                                [166.1,162.5,155.1,4.7],
                                [166.1,162.5,155.1,4.7],
                                [167.5,166.4,155.8,4.7],
                                [168.1,163.9,157.0,4.7],
                                [167.1,163.4,155.6,4.7],
                                [168.6,165.5,157.0,4.7],
                                [173.1,165.5,162.0,4.7],
                                [173.0,168.1,160.8,4.9],
                                [173.0,168.1,160.8,4.9],
                                [168.1,168.4,159.0,5.1],
                                [167.0,163.3,157.3,5.2],
                                [167.0,168.4,157.3,5.2],
                                [173.5,167.8,162.2,5.2],
                                [173.6,167.8,162.2,5.2],
                                [173.8,168.5,160.8,5.2],
                                [171.7,168.2,160.6,5.3],
                                [169.2,168.2,157.1,5.3],
                                [172.9,169.9,163.1,5.2],
                                [168.4,163.3,158.4,5.2],
                                [169.9,168.4,158.8,5.4],
                                [170.2,167.0,162.4,5.4],
                                [168.4,167.0,157.2,5.4],
                                [172.0,169.0,159.5,5.5],
                                [172.0,169.0,162.5,5.5],
                                [169.9,163.4,157.4,5.6],
                                [173.2,169.3,162.0,5.6],
                                [173.2,168.4,162.0,5.6],
                                [169.1,168.6,157.3,5.7],
                                [169.1,163.4,157.3,5.7],
                                [168.7,169.4,156.2,5.7],
                                [171.0,166.8,156.2,5.5],
                                [171.0,166.8,156.2,5.5],
                                [169.7,165.8,157.7,5.9],
                                [163.7,165.8,151.8,5.9],
                                [168.6,162.0,156.9,5.9],
                                [168.6,162.0,156.9,5.9],
                                [167.3,164.4,156.2,6.1],
                                [166.0,163.7,156.1,6.0],
                                [166.0,163.7,156.9,6.0],
                                [168.9,163.9,157.1,5.9],
                                [169.1,166.9,158.3,6.1],
                                [169.1,166.9,158.3,6.1],
                                [169.4,164.6,158.2,6.2],
                                [169.4,169.6,158.2,6.2],
                                [170.4,165.5,158.4,6.1],
                                [170.4,165.5,158.4,6.1],
                                [168.4,164.5,156.9,6.3],
                                [168.6,166.8,160.6,5.9],
                                [168.6,164.7,156.2,5.9],
                                [170.9,166.1,160.8,5.9],
                                [170.9,166.1,156.2,6.3],
                                [167.9,164.9,156.4,6.3],
                                [166.2,164.7,157.6,6.4],
                                [166.2,164.7,161.5,6.4],
                                [166.2,163.7,157.5,7.3],
                                [166.2,163.7,157.5,6.3],
                                [163.5,162.5,155.5,6.3],
                                [167.0,163.2,157.5,6.6],
                                [163.7,163.2,157.5,6.6],
                                [163.4,164.8,154.7,6.5],
                                [163.4,159.8,154.7,6.5],
                                [163.2,159.8,153.5,6.3],
                                [163.1,158.1,152.5,6.4],
                                [163.1,158.1,151.2,6.4],], columns=['IA', 'IB', 'IV', 'IN'])

 

    
    dummy= pandas.concat([dummy_week] * qty_weeks, ignore_index=True)
    
    end_date_dt = start_date_dt  + dt.timedelta(days=qty_weeks*7)
    
    dummy_idx = numpy.arange(start_date_dt, end_date_dt, numpy.timedelta64(5, 'm'), dtype='datetime64')
    
    dummy['timestamp'] = dummy_idx
    dummy.set_index('timestamp', inplace=True)    
    dummy.index = pandas.to_datetime(dummy.index)


    cycles = 0.7 * dummy.shape[0] / (365 * 24 * 12)  # how many sine cycles
    resolution = dummy.shape[0]  # how many datapoints to generate
    length = numpy.pi * 2 * cycles
    season_year = numpy.sin(numpy.arange(0, length, length / resolution))

    cycles = 12 * 4 * dummy.shape[0] / (365 * 24 * 12)  # how many sine cycles
    resolution = dummy.shape[0]  # how many datapoints to generate
    length = numpy.pi * 2 * cycles
    season_week = numpy.sin(numpy.arange(0, length, length / resolution))

    cycles = 12 * dummy.shape[0] / (365 * 24 * 12)  # how many sine cycles
    resolution = dummy.shape[0]  # how many datapoints to generate
    length = numpy.pi * 2 * cycles
    season_month = numpy.sin(numpy.arange(0, length, length / resolution))

    rand_year = random.randint(5, 10)
    rand_month = random.randint(1, 5)
    rand_week = random.randint(1, 3)

    rand_vet = numpy.random.randint(5, 10, size=dummy.shape[0])
    step_vet = numpy.zeros(dummy.shape[0])

    # Load transfer
    for i in range(0, random.randint(1, 4)):
        start = random.randint(0, dummy.shape[0])
        end = start + random.randint(1, 60) * 24 * 12

        if end >= len(step_vet):
            end = len(step_vet)

        step_vet[start:end] = random.randint(-50, -20)

    # Noise
    for i in range(0, random.randint(1, 40)):
        start = random.randint(0, dummy.shape[0])
        end = start + random.randint(1, 12 * 3)

        if end >= len(step_vet):
            end = len(step_vet)

        step_vet[start:end] = random.randint(-300, 300)

    dummy['IA'] = dummy['IA'].values + rand_year * season_year + rand_month * season_month \
                                       + rand_week * season_week + rand_vet + step_vet
    dummy['IB'] = dummy['IB'].values + rand_year * season_year + rand_month * season_month \
                                       + rand_week * season_week + rand_vet + step_vet
    dummy['IV'] = dummy['IV'].values + rand_year * season_year + rand_month * season_month \
                                       + rand_week * season_week + rand_vet + step_vet
    dummy['IN'] = dummy['IN'].values + (rand_year / 10) * season_year \
                                       + (rand_month / 10) * season_month + (rand_week / 10) \
                                                           * season_week + rand_vet / 10

    return dummy



def VoltageDummyData(qty_weeks:int = 12*4,start_date_dt:datetime = datetime(2023,1,1)):
    """
    Generate a DataFrame containing dummy voltage data over a specified number of weeks.

    This function creates a time series of voltage data, simulating variations in voltage values 
    over a given time period. The data includes random noise and step changes to mimic real-world 
    fluctuations in voltage readings.

    Parameters
    ----------
    qty_weeks : int, optional
        The number of weeks over which to generate the data (default is 48 weeks).
    start_date_dt : datetime, optional
        The start date for the data generation (default is January 1, 2023).

    Returns
    -------
    pandas.DataFrame
        A DataFrame with timestamps as index and columns 'VA', 'VB', and 'VV' representing 
        simulated voltage readings for three different phases or measurements. Each column 
        contains voltage values that are affected by random noise and step changes.

    Notes
    -----
    - The voltage values are simulated around a base value of 13.8, adjusted by a random noise 
      factor and step changes.
    - The step changes in voltage are randomly introduced at various points in the time series.
    - The timestamps are spaced 5 minutes apart.

    Examples
    --------
    >>> dummy_data = VoltageDummyData()
    >>> dummy_data.head()
    """
    
    
    end_date_dt = start_date_dt  + dt.timedelta(days=qty_weeks*7)

    dummy = numpy.arange(start_date_dt, end_date_dt, numpy.timedelta64(5, 'm'), dtype='datetime64')
    dummy = pandas.DataFrame(dummy, columns=['timestamp'])
    dummy.set_index('timestamp', inplace=True)

    rand_vet = 0.05 * 13.8 * numpy.random.rand(dummy.shape[0], 1) - 0.025 * 13.8

    step_vet = numpy.zeros((dummy.shape[0], 1))

    #  Noise
    for i in range(0, random.randint(1, 40)):
        start = random.randint(0, dummy.shape[0])
        end = start + random.randint(1, 12 * 3)

        if end >= len(step_vet):
            end = len(step_vet)

        step_vet[start:end] = random.randint(-1, 1)

    dummy['VA'] = 1.03 * 13.8 + rand_vet + step_vet
    dummy['VB'] = 1.03 * 13.8 + rand_vet + step_vet
    dummy['VV'] = 1.03 * 13.8 + rand_vet + step_vet

    return dummy


def PowerFactorDummyData(qty_weeks:int = 12*4,start_date_dt:datetime = datetime(2023,1,1)):
    
    """
    Generates dummy power factor data for a specified number of weeks starting from a given date.
    
    This function creates a pandas DataFrame containing simulated power factor data across three columns: 'FPA', 'FPB', and 'FPV'.
    Each row represents a 5-minute interval within the specified time frame. The data includes base values with added random 
    load transfer and noise effects to simulate real-world fluctuations in power factor measurements.
    
    Parameters
    ----------
    qty_weeks : int, optional
        The quantity of weeks to generate data for, defaults to 48 weeks (approximately one year).
    start_date_dt : datetime, optional
        The start date for the data generation, defaults to January 1, 2023.
    
    Returns
    -------
    pandas.DataFrame
        A DataFrame with a datetime index representing 5-minute intervals and columns 'FPA', 'FPB', and 'FPV' for power factor values.
        The data includes random variations to simulate realistic power factor changes over time.
    
    Notes
    -----
    - The function internally generates a dummy week of data and replicates it for the number of weeks specified.
    - Random load transfers and noise are added to the base values to create variability in the data.
    - The DataFrame's index is set to the timestamp of each record, making it suitable for time series analysis.
    
    Examples
    --------
    >>> import pandas
    >>> from datetime import datetime
    >>> dummy_data = PowerFactorDummyData(qty_weeks=12, start_date_dt=datetime(2023, 1, 1))
    >>> dummy_data.head()
    """

    dummy_week = pandas.DataFrame([[0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,0.99,0.98],
                                        [0.99,0.99,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.99,1.0,0.98],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.96],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.98,0.97],
                                        [0.97,0.98,0.97],
                                        [0.97,0.98,0.96],
                                        [0.97,0.98,0.96],
                                        [0.97,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.97,0.99,0.98],
                                        [0.97,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.99],
                                        [0.98,0.99,0.99],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.99,0.99,0.98],
                                        [0.99,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.99,0.99,0.98],
                                        [0.99,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.99,0.99,0.98],
                                        [0.99,0.99,0.98],
                                        [0.99,0.99,0.98],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,0.99,0.99],
                                        [0.99,0.99,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,0.99,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,1.0,0.97],
                                        [0.98,1.0,0.97],
                                        [0.99,1.0,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.99,0.99,0.97],
                                        [0.98,1.0,0.97],
                                        [0.98,1.0,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.96],
                                        [0.98,0.99,0.96],
                                        [0.98,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.98,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.98,0.96],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.98],
                                        [0.97,0.99,0.98],
                                        [0.96,0.99,0.97],
                                        [0.96,0.99,0.97],
                                        [0.96,0.98,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.96,0.99,0.96],
                                        [0.96,0.99,0.96],
                                        [0.96,0.98,0.96],
                                        [0.96,0.98,0.96],
                                        [0.96,0.98,0.96],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.96,0.98,0.98],
                                        [0.97,0.98,0.97],
                                        [0.97,0.98,0.97],
                                        [0.97,0.98,0.97],
                                        [0.97,0.98,0.97],
                                        [0.97,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.99,1.0,0.99],
                                        [0.98,1.0,0.99],
                                        [0.98,1.0,0.99],
                                        [0.99,0.99,0.99],
                                        [0.99,0.99,0.99],
                                        [0.98,1.0,0.99],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.99,1.0,0.98],
                                        [0.98,1.0,0.97],
                                        [0.98,1.0,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,1.0,0.97],
                                        [0.98,1.0,0.97],
                                        [0.98,1.0,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.98,0.99,0.96],
                                        [0.98,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.96,0.98,0.96],
                                        [0.96,0.98,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.96,0.99,0.96],
                                        [0.96,0.99,0.96],
                                        [0.96,0.99,0.95],
                                        [0.97,0.99,0.95],
                                        [0.97,0.99,0.95],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.98,0.99,0.96],
                                        [0.98,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.98,0.99,0.96],
                                        [0.98,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.96,0.99,0.96],
                                        [0.96,0.99,0.96],
                                        [0.97,0.98,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.96,0.98,0.96],
                                        [0.96,0.98,0.96],
                                        [0.96,0.99,0.96],
                                        [0.96,0.98,0.95],
                                        [0.96,0.98,0.95],
                                        [0.96,0.98,0.96],
                                        [0.96,0.98,0.96],
                                        [0.96,0.98,0.96],
                                        [0.96,0.98,0.96],
                                        [0.96,0.98,0.96],
                                        [0.97,0.98,0.96],
                                        [0.97,0.98,0.96],
                                        [0.97,0.99,0.97],
                                        [0.96,0.98,0.96],
                                        [0.96,0.98,0.96],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.97],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,0.99,0.99],
                                        [0.99,0.99,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,0.99,0.98],
                                        [0.99,0.99,0.98],
                                        [0.99,0.99,0.98],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,0.99,0.98],
                                        [0.99,0.99,0.98],
                                        [0.99,0.99,0.98],
                                        [0.99,0.99,0.98],
                                        [0.99,0.99,0.98],
                                        [0.99,0.99,0.99],
                                        [0.99,0.99,0.99],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.98,0.96],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.96,0.98,0.95],
                                        [0.96,0.98,0.95],
                                        [0.96,0.98,0.95],
                                        [0.96,0.98,0.95],
                                        [0.96,0.98,0.95],
                                        [0.96,0.98,0.96],
                                        [0.96,0.98,0.96],
                                        [0.96,0.98,0.95],
                                        [0.96,0.98,0.95],
                                        [0.95,0.97,0.94],
                                        [0.96,0.98,0.95],
                                        [0.96,0.98,0.95],
                                        [0.95,0.98,0.95],
                                        [0.95,0.98,0.95],
                                        [0.96,0.98,0.96],
                                        [0.96,0.98,0.96],
                                        [0.96,0.98,0.96],
                                        [0.96,0.98,0.97],
                                        [0.96,0.98,0.97],
                                        [0.96,0.98,0.96],
                                        [0.96,0.99,0.97],
                                        [0.96,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.98,0.96],
                                        [0.97,0.98,0.96],
                                        [0.97,0.98,0.96],
                                        [0.98,0.98,0.97],
                                        [0.98,0.98,0.97],
                                        [0.97,0.99,0.96],
                                        [0.98,0.99,0.96],
                                        [0.98,0.99,0.96],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.99,0.99,0.98],
                                        [0.99,0.99,0.98],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.98,1.0,0.99],
                                        [0.98,1.0,0.99],
                                        [0.98,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,0.99,0.98],
                                        [0.99,0.99,0.98],
                                        [0.99,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,0.99,0.98],
                                        [0.99,0.99,0.98],
                                        [0.99,0.99,0.98],
                                        [0.99,0.99,0.98],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,0.99,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.98,0.96],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.98,0.96],
                                        [0.97,0.98,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.98,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.96,0.98,0.96],
                                        [0.96,0.98,0.96],
                                        [0.97,0.99,0.95],
                                        [0.97,0.98,0.96],
                                        [0.97,0.98,0.96],
                                        [0.97,0.98,0.96],
                                        [0.97,0.98,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.98,0.97],
                                        [0.97,0.98,0.97],
                                        [0.97,0.99,0.96],
                                        [0.97,0.98,0.96],
                                        [0.97,0.98,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.98,0.97],
                                        [0.96,0.98,0.96],
                                        [0.96,0.98,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.96,0.98,0.96],
                                        [0.95,0.98,0.95],
                                        [0.95,0.98,0.95],
                                        [0.96,0.98,0.96],
                                        [0.96,0.98,0.96],
                                        [0.96,0.98,0.96],
                                        [0.96,0.98,0.97],
                                        [0.96,0.98,0.97],
                                        [0.97,0.98,0.97],
                                        [0.97,0.98,0.97],
                                        [0.96,0.98,0.97],
                                        [0.96,0.98,0.97],
                                        [0.96,0.98,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.98,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.98],
                                        [0.97,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.99,0.99,0.99],
                                        [0.99,0.99,0.99],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.97],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.98,1.0,0.99],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [0.99,1.0,1.0],
                                        [0.99,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [0.99,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [0.99,1.0,1.0],
                                        [0.99,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.97],
                                        [0.98,0.99,0.96],
                                        [0.98,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.98,0.99,0.96],
                                        [0.98,0.99,0.96],
                                        [0.98,0.99,0.97],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.96],
                                        [0.96,0.98,0.96],
                                        [0.96,0.98,0.96],
                                        [0.96,0.98,0.96],
                                        [0.97,0.98,0.96],
                                        [0.97,0.98,0.96],
                                        [0.96,0.98,0.95],
                                        [0.96,0.98,0.95],
                                        [0.96,0.98,0.95],
                                        [0.96,0.99,0.96],
                                        [0.96,0.99,0.96],
                                        [0.96,0.98,0.96],
                                        [0.96,0.98,0.96],
                                        [0.97,0.98,0.96],
                                        [0.97,0.98,0.97],
                                        [0.97,0.98,0.97],
                                        [0.96,0.98,0.96],
                                        [0.96,0.98,0.96],
                                        [0.97,0.99,0.96],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.98,0.96],
                                        [0.97,0.98,0.96],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.97],
                                        [0.99,0.99,0.98],
                                        [0.99,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [0.99,1.0,1.0],
                                        [0.99,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,1.0],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,1.0],
                                        [0.99,1.0,1.0],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [1.0,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.98,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.98],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.97,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.98,1.0,0.99],
                                        [0.98,1.0,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.98,1.0,0.99],
                                        [0.98,1.0,0.99],
                                        [0.98,1.0,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.98,1.0,0.99],
                                        [0.98,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,0.99,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.97],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.98,0.99,0.98],
                                        [0.99,0.99,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,0.99,0.98],
                                        [0.99,0.99,0.98],
                                        [0.99,0.99,0.98],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.98,1.0,0.98],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.99],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,1.0,0.98],
                                        [0.99,0.99,0.98],
                                        [0.99,0.99,0.98],
                                        [0.98,1.0,0.98],
                                        [0.98,1.0,0.98],
                                        [0.99,0.99,0.98]], columns=['FPA', 'FPB', 'FPV'])

    
    
    dummy= pandas.concat([dummy_week] * qty_weeks, ignore_index=True)
    
    end_date_dt = start_date_dt  + dt.timedelta(days=qty_weeks*7)
    
    dummy_idx = numpy.arange(start_date_dt, end_date_dt, numpy.timedelta64(5, 'm'), dtype='datetime64')
    
    dummy['timestamp'] = dummy_idx
    dummy.set_index('timestamp', inplace=True)    
    dummy.index = pandas.to_datetime(dummy.index)
    

    step_vet = numpy.zeros(dummy.shape[0])

    # Load transfer
    for i in range(0, random.randint(1, 4)):
        start = random.randint(0, dummy.shape[0])
        end = start + random.randint(1, 60) * 24 * 12

        if end >= len(step_vet):
            end = len(step_vet)

        step_vet[start:end] = -0.2 * random.random()

    # Noise
    for i in range(0, random.randint(1, 40)):
        start = random.randint(0, dummy.shape[0])
        end = start + random.randint(1, 12 * 3)

        if end >= len(step_vet):
            end = len(step_vet)

        step_vet[start:end] = random.randint(-10,2)*0.07

    dummy['FPA'] = dummy['FPA'].values + step_vet
    dummy['FPB'] = dummy['FPB'].values + step_vet
    dummy['FPV'] = dummy['FPV'].values + step_vet

    return dummy


def PowerDummyData(qty_weeks:int = 12*4,start_date_dt:datetime = datetime(2023,1,1)):
    """
    Generates dummy power data for a specified number of weeks from a start date.
    
    This function calculates the apparent power (S), active power (P), and reactive power (Q)
    for a given number of weeks starting from a specified date. It uses the CurrentDummyData, 
    VoltageDummyData, and PowerFactorDummyData functions to generate current (I), voltage (V), 
    and power factor (pf) data, respectively. The final DataFrame includes columns for S, P, and Q.
    
    Parameters:
    qty_weeks (int): The quantity of weeks for which to generate data. Default is 48 weeks.
    start_date_dt (datetime): The start date for data generation. Default is January 1, 2023.
    
    Returns:
    pandas.DataFrame: A DataFrame containing the columns 'S' (apparent power), 
                      'P' (active power), and 'Q' (reactive power).
    
    Example:
    >>> PowerDummyData(4, datetime(2023, 1, 1))
    [Output will be a DataFrame with the calculated power data for 4 weeks starting from January 1, 2023]
    """
    
    end_date_dt = start_date_dt  + dt.timedelta(days=qty_weeks*7)
    
    I = CurrentDummyData(qty_weeks, start_date_dt)
    V = VoltageDummyData(qty_weeks, start_date_dt)
    pf = PowerFactorDummyData(qty_weeks, start_date_dt)

    I = I.iloc[:, :-1]

    dummy = pandas.DataFrame([])

    dummy['S'] = V['VA'] / numpy.sqrt(3) * I['IA'] + V['VB'] / numpy.sqrt(3) * I['IB'] \
                                                   + V['VV'] / numpy.sqrt(3) * I['IV']
    dummy['P'] = V['VA'] / numpy.sqrt(3) * I['IA'] * pf['FPA'] + V['VB'] / numpy.sqrt(3) * I['IB'] * pf['FPB'] \
                                                               + V['VV'] / numpy.sqrt(3) * I['IV'] * pf['FPV']
    dummy['Q'] = dummy['S'].pow(2) - dummy['P'].pow(2)
    dummy['Q'] = numpy.sqrt(dummy['Q'].abs())

    return dummy


def EnergyDummyData(qty_weeks:int = 12*4,start_date_dt:datetime = datetime(2023,1,1)):
    """
    Generate a dummy pandas DataFrame containing cumulative energy data.
    
    This function creates a DataFrame with two columns: 'Eactive' and 'Ereactive'.
    'Eactive' is the cumulative sum of the 'P' column from the PowerDummyData function,
    and 'Ereactive' is the absolute cumulative sum of the 'Q' column from the same function.
    
    Parameters
    ----------
    qty_weeks : int, optional
        The quantity of weeks for which to generate the data, default is 48 weeks (12*4).
    start_date_dt : datetime, optional
        The starting date for the data generation, default is January 1, 2023.
    
    Returns
    -------
    pandas.DataFrame
        A DataFrame with two columns 'Eactive' and 'Ereactive' representing the cumulative
        active and reactive energy data respectively.
    
    Examples
    --------
    >>> EnergyDummyData(4, datetime(2023, 1, 1))
    DataFrame with the cumulative energy data for 4 weeks starting from January 1, 2023.
    
    Notes
    -----
    The function relies on PowerDummyData function to generate initial power data
    which is then cumulatively summed to generate energy data.
    """

    dummy_s = PowerDummyData(qty_weeks, start_date_dt)

    dummy = pandas.DataFrame([])

    dummy['Eactive'] = dummy_s['P'].cumsum(skipna=True)

    dummy['Ereactive'] = dummy_s['Q'].abs().cumsum(skipna=True)

    return dummy


def DefaultWeekDayCurve():
    
    default = pandas.DataFrame([[0,0,0,0.452,0.446,0.44,0.815],                                    
                            [0,0,5,0.454,0.444,0.443,0.904],
                            [0,0,10,0.436,0.431,0.431,0.868],
                            [0,0,15,0.433,0.424,0.417,0.893],
                            [0,0,20,0.416,0.408,0.412,0.858],
                            [0,0,25,0.4,0.389,0.398,0.859],
                            [0,0,30,0.393,0.38,0.384,0.899],
                            [0,0,35,0.377,0.369,0.37,0.919],
                            [0,0,40,0.366,0.354,0.357,0.861],
                            [0,0,45,0.357,0.344,0.35,0.829],
                            [0,0,50,0.342,0.33,0.34,0.836],
                            [0,0,55,0.337,0.32,0.33,0.822],
                            [0,1,0,0.323,0.305,0.314,0.857],
                            [0,1,5,0.312,0.292,0.301,0.84],
                            [0,1,10,0.302,0.286,0.291,0.838],
                            [0,1,15,0.294,0.28,0.287,0.861],
                            [0,1,20,0.286,0.276,0.273,0.905],
                            [0,1,25,0.271,0.259,0.266,0.819],
                            [0,1,30,0.262,0.245,0.254,0.799],
                            [0,1,35,0.248,0.233,0.245,0.787],
                            [0,1,40,0.248,0.232,0.241,0.769],
                            [0,1,45,0.245,0.229,0.234,0.794],
                            [0,1,50,0.234,0.213,0.229,0.834],
                            [0,1,55,0.227,0.203,0.226,0.822],
                            [0,2,0,0.224,0.207,0.218,0.773],
                            [0,2,5,0.221,0.211,0.212,0.748],
                            [0,2,10,0.206,0.192,0.2,0.777],
                            [0,2,15,0.202,0.194,0.204,0.764],
                            [0,2,20,0.199,0.182,0.195,0.731],
                            [0,2,25,0.185,0.174,0.182,0.752],
                            [0,2,30,0.177,0.173,0.172,0.75],
                            [0,2,35,0.172,0.164,0.162,0.755],
                            [0,2,40,0.174,0.155,0.158,0.73],
                            [0,2,45,0.168,0.148,0.152,0.7],
                            [0,2,50,0.163,0.142,0.153,0.702],
                            [0,2,55,0.155,0.138,0.147,0.69],
                            [0,3,0,0.153,0.135,0.144,0.728],
                            [0,3,5,0.149,0.128,0.137,0.709],
                            [0,3,10,0.141,0.12,0.13,0.697],
                            [0,3,15,0.14,0.117,0.129,0.7],
                            [0,3,20,0.133,0.114,0.129,0.717],
                            [0,3,25,0.132,0.112,0.121,0.71],
                            [0,3,30,0.128,0.108,0.118,0.686],
                            [0,3,35,0.118,0.1,0.116,0.699],
                            [0,3,40,0.118,0.097,0.112,0.713],
                            [0,3,45,0.114,0.091,0.108,0.713],
                            [0,3,50,0.108,0.087,0.103,0.694],
                            [0,3,55,0.105,0.081,0.093,0.704],
                            [0,4,0,0.102,0.079,0.093,0.637],
                            [0,4,5,0.097,0.078,0.09,0.63],
                            [0,4,10,0.101,0.072,0.093,0.673],
                            [0,4,15,0.1,0.074,0.093,0.69],
                            [0,4,20,0.098,0.068,0.089,0.684],
                            [0,4,25,0.094,0.068,0.081,0.659],
                            [0,4,30,0.097,0.071,0.085,0.632],
                            [0,4,35,0.099,0.076,0.089,0.686],
                            [0,4,40,0.095,0.071,0.081,0.66],
                            [0,4,45,0.095,0.069,0.077,0.632],
                            [0,4,50,0.101,0.076,0.092,0.627],
                            [0,4,55,0.11,0.089,0.104,0.651],
                            [0,5,0,0.106,0.087,0.103,0.606],
                            [0,5,5,0.121,0.109,0.124,0.602],
                            [0,5,10,0.122,0.116,0.133,0.555],
                            [0,5,15,0.107,0.108,0.111,0.575],
                            [0,5,20,0.102,0.096,0.111,0.562],
                            [0,5,25,0.097,0.099,0.1,0.548],
                            [0,5,30,0.082,0.084,0.079,0.526],
                            [0,5,35,0.054,0.049,0.056,0.489],
                            [0,5,40,0.035,0.027,0.041,0.539],
                            [0,5,45,0.017,0.015,0.025,0.492],
                            [0,5,50,0.003,0.002,0.006,0.53],
                            [0,5,55,0.003,0.004,0.0,0.478],
                            [0,6,0,0.0,0.0,0.008,0.414],
                            [0,6,5,0.058,0.073,0.058,0.39],
                            [0,6,10,0.079,0.093,0.097,0.43],
                            [0,6,15,0.072,0.086,0.092,0.402],
                            [0,6,20,0.075,0.086,0.08,0.41],
                            [0,6,25,0.062,0.063,0.071,0.425],
                            [0,6,30,0.064,0.056,0.072,0.327],
                            [0,6,35,0.046,0.042,0.057,0.302],
                            [0,6,40,0.027,0.034,0.043,0.373],
                            [0,6,45,0.036,0.03,0.045,0.361],
                            [0,6,50,0.029,0.033,0.051,0.334],
                            [0,6,55,0.029,0.035,0.054,0.378],
                            [0,7,0,0.023,0.048,0.066,0.297],
                            [0,7,5,0.045,0.068,0.101,0.236],
                            [0,7,10,0.068,0.096,0.12,0.222],
                            [0,7,15,0.09,0.113,0.142,0.226],
                            [0,7,20,0.117,0.142,0.163,0.222],
                            [0,7,25,0.147,0.168,0.188,0.279],
                            [0,7,30,0.148,0.18,0.186,0.269],
                            [0,7,35,0.161,0.188,0.193,0.273],
                            [0,7,40,0.174,0.196,0.198,0.242],
                            [0,7,45,0.176,0.197,0.209,0.25],
                            [0,7,50,0.181,0.198,0.208,0.189],
                            [0,7,55,0.2,0.216,0.23,0.172],
                            [0,8,0,0.182,0.188,0.208,0.17],
                            [0,8,5,0.223,0.233,0.247,0.101],
                            [0,8,10,0.23,0.234,0.265,0.131],
                            [0,8,15,0.24,0.253,0.285,0.141],
                            [0,8,20,0.256,0.263,0.286,0.138],
                            [0,8,25,0.267,0.261,0.289,0.153],
                            [0,8,30,0.275,0.264,0.3,0.085],
                            [0,8,35,0.268,0.264,0.282,0.113],
                            [0,8,40,0.268,0.267,0.291,0.121],
                            [0,8,45,0.258,0.25,0.291,0.1],
                            [0,8,50,0.263,0.269,0.299,0.06],
                            [0,8,55,0.263,0.261,0.302,0.04],
                            [0,9,0,0.275,0.266,0.3,0.034],
                            [0,9,5,0.309,0.298,0.328,0.023],
                            [0,9,10,0.324,0.309,0.352,0.076],
                            [0,9,15,0.328,0.32,0.353,0.1],
                            [0,9,20,0.358,0.345,0.373,0.004],
                            [0,9,25,0.37,0.363,0.388,0.0],
                            [0,9,30,0.379,0.365,0.392,0.085],
                            [0,9,35,0.377,0.375,0.402,0.101],
                            [0,9,40,0.38,0.381,0.411,0.051],
                            [0,9,45,0.36,0.363,0.392,0.068],
                            [0,9,50,0.381,0.381,0.41,0.071],
                            [0,9,55,0.375,0.374,0.412,0.035],
                            [0,10,0,0.398,0.398,0.426,0.085],
                            [0,10,5,0.381,0.386,0.416,0.003],
                            [0,10,10,0.393,0.404,0.432,0.01],
                            [0,10,15,0.417,0.414,0.443,0.03],
                            [0,10,20,0.424,0.431,0.457,0.089],
                            [0,10,25,0.426,0.434,0.461,0.051],
                            [0,10,30,0.417,0.431,0.455,0.084],
                            [0,10,35,0.438,0.44,0.469,0.054],
                            [0,10,40,0.447,0.451,0.468,0.021],
                            [0,10,45,0.47,0.484,0.509,0.025],
                            [0,10,50,0.465,0.474,0.51,0.044],
                            [0,10,55,0.487,0.487,0.526,0.093],
                            [0,11,0,0.484,0.482,0.512,0.063],
                            [0,11,5,0.527,0.533,0.554,0.099],
                            [0,11,10,0.528,0.529,0.557,0.143],
                            [0,11,15,0.526,0.527,0.557,0.121],
                            [0,11,20,0.527,0.531,0.55,0.118],
                            [0,11,25,0.534,0.535,0.554,0.136],
                            [0,11,30,0.532,0.522,0.555,0.131],
                            [0,11,35,0.571,0.566,0.585,0.149],
                            [0,11,40,0.577,0.575,0.596,0.145],
                            [0,11,45,0.588,0.587,0.614,0.142],
                            [0,11,50,0.615,0.621,0.634,0.171],
                            [0,11,55,0.639,0.64,0.669,0.202],
                            [0,12,0,0.63,0.646,0.67,0.23],
                            [0,12,5,0.639,0.639,0.664,0.199],
                            [0,12,10,0.633,0.638,0.663,0.211],
                            [0,12,15,0.627,0.638,0.669,0.184],
                            [0,12,20,0.587,0.59,0.625,0.211],
                            [0,12,25,0.586,0.585,0.618,0.232],
                            [0,12,30,0.543,0.541,0.583,0.174],
                            [0,12,35,0.534,0.534,0.568,0.192],
                            [0,12,40,0.493,0.498,0.543,0.193],
                            [0,12,45,0.489,0.488,0.532,0.224],
                            [0,12,50,0.449,0.446,0.484,0.12],
                            [0,12,55,0.494,0.484,0.528,0.09],
                            [0,13,0,0.487,0.479,0.522,0.114],
                            [0,13,5,0.508,0.498,0.536,0.081],
                            [0,13,10,0.516,0.518,0.552,0.103],
                            [0,13,15,0.546,0.553,0.578,0.07],
                            [0,13,20,0.563,0.569,0.607,0.067],
                            [0,13,25,0.565,0.574,0.603,0.061],
                            [0,13,30,0.56,0.566,0.601,0.07],
                            [0,13,35,0.592,0.592,0.62,0.067],
                            [0,13,40,0.573,0.577,0.613,0.107],
                            [0,13,45,0.597,0.609,0.637,0.145],
                            [0,13,50,0.576,0.587,0.607,0.115],
                            [0,13,55,0.616,0.62,0.637,0.074],
                            [0,14,0,0.621,0.625,0.665,0.088],
                            [0,14,5,0.644,0.643,0.681,0.049],
                            [0,14,10,0.582,0.581,0.626,0.091],
                            [0,14,15,0.594,0.592,0.64,0.098],
                            [0,14,20,0.636,0.641,0.681,0.084],
                            [0,14,25,0.631,0.628,0.666,0.085],
                            [0,14,30,0.602,0.589,0.646,0.063],
                            [0,14,35,0.63,0.619,0.673,0.085],
                            [0,14,40,0.637,0.63,0.678,0.105],
                            [0,14,45,0.619,0.606,0.657,0.085],
                            [0,14,50,0.605,0.596,0.648,0.094],
                            [0,14,55,0.636,0.632,0.682,0.128],
                            [0,15,0,0.629,0.631,0.67,0.103],
                            [0,15,5,0.634,0.637,0.672,0.087],
                            [0,15,10,0.635,0.64,0.672,0.101],
                            [0,15,15,0.636,0.636,0.676,0.107],
                            [0,15,20,0.648,0.657,0.689,0.087],
                            [0,15,25,0.604,0.602,0.644,0.143],
                            [0,15,30,0.649,0.654,0.701,0.119],
                            [0,15,35,0.616,0.619,0.672,0.128],
                            [0,15,40,0.61,0.609,0.662,0.16],
                            [0,15,45,0.631,0.626,0.686,0.094],
                            [0,15,50,0.645,0.645,0.689,0.064],
                            [0,15,55,0.651,0.644,0.674,0.061],
                            [0,16,0,0.673,0.682,0.715,0.116],
                            [0,16,5,0.639,0.641,0.676,0.039],
                            [0,16,10,0.657,0.658,0.706,0.038],
                            [0,16,15,0.667,0.672,0.704,0.035],
                            [0,16,20,0.665,0.685,0.703,0.046],
                            [0,16,25,0.677,0.688,0.709,0.051],
                            [0,16,30,0.664,0.676,0.698,0.053],
                            [0,16,35,0.696,0.706,0.737,0.097],
                            [0,16,40,0.674,0.687,0.72,0.092],
                            [0,16,45,0.684,0.701,0.728,0.109],
                            [0,16,50,0.675,0.689,0.718,0.139],
                            [0,16,55,0.678,0.703,0.725,0.1],
                            [0,17,0,0.664,0.678,0.706,0.104],
                            [0,17,5,0.631,0.649,0.674,0.097],
                            [0,17,10,0.642,0.674,0.689,0.111],
                            [0,17,15,0.644,0.654,0.667,0.193],
                            [0,17,20,0.649,0.663,0.685,0.177],
                            [0,17,25,0.642,0.655,0.687,0.181],
                            [0,17,30,0.653,0.665,0.699,0.249],
                            [0,17,35,0.679,0.698,0.72,0.255],
                            [0,17,40,0.702,0.719,0.751,0.27],
                            [0,17,45,0.735,0.751,0.78,0.317],
                            [0,17,50,0.763,0.779,0.801,0.334],
                            [0,17,55,0.778,0.791,0.817,0.392],
                            [0,18,0,0.766,0.79,0.796,0.441],
                            [0,18,5,0.784,0.795,0.811,0.416],
                            [0,18,10,0.79,0.799,0.828,0.444],
                            [0,18,15,0.822,0.824,0.839,0.491],
                            [0,18,20,0.844,0.854,0.864,0.588],
                            [0,18,25,0.865,0.877,0.894,0.587],
                            [0,18,30,0.885,0.902,0.923,0.644],
                            [0,18,35,0.905,0.913,0.925,0.588],
                            [0,18,40,0.905,0.908,0.931,0.666],
                            [0,18,45,0.928,0.928,0.935,0.707],
                            [0,18,50,0.916,0.917,0.941,0.652],
                            [0,18,55,0.95,0.952,0.973,0.735],
                            [0,19,0,0.967,0.978,0.973,0.717],
                            [0,19,5,0.971,0.978,0.984,0.629],
                            [0,19,10,0.965,0.965,0.965,0.703],
                            [0,19,15,0.965,0.965,0.978,0.668],
                            [0,19,20,0.985,0.994,0.99,0.682],
                            [0,19,25,0.997,0.987,1.0,0.679],
                            [0,19,30,0.974,0.97,0.978,0.71],
                            [0,19,35,0.97,0.981,0.974,0.7],
                            [0,19,40,0.977,0.989,0.977,0.722],
                            [0,19,45,0.992,0.997,0.995,0.793],
                            [0,19,50,1.0,1.0,0.997,0.772],
                            [0,19,55,0.984,0.996,0.98,0.79],
                            [0,20,0,0.948,0.952,0.964,0.722],
                            [0,20,5,0.937,0.931,0.937,0.845],
                            [0,20,10,0.927,0.928,0.924,0.821],
                            [0,20,15,0.916,0.919,0.915,0.805],
                            [0,20,20,0.9,0.91,0.91,0.868],
                            [0,20,25,0.892,0.899,0.9,0.843],
                            [0,20,30,0.886,0.892,0.888,0.844],
                            [0,20,35,0.902,0.909,0.899,0.864],
                            [0,20,40,0.901,0.908,0.903,0.972],
                            [0,20,45,0.87,0.874,0.868,0.952],
                            [0,20,50,0.865,0.873,0.864,0.943],
                            [0,20,55,0.882,0.894,0.89,0.943],
                            [0,21,0,0.872,0.886,0.885,1.0],
                            [0,21,5,0.86,0.869,0.865,0.897],
                            [0,21,10,0.869,0.871,0.872,0.937],
                            [0,21,15,0.834,0.844,0.842,0.98],
                            [0,21,20,0.839,0.852,0.845,0.961],
                            [0,21,25,0.874,0.876,0.873,0.876],
                            [0,21,30,0.833,0.839,0.835,0.908],
                            [0,21,35,0.849,0.853,0.85,0.937],
                            [0,21,40,0.813,0.814,0.818,0.933],
                            [0,21,45,0.821,0.827,0.836,0.914],
                            [0,21,50,0.816,0.82,0.827,0.914],
                            [0,21,55,0.819,0.817,0.822,0.953],
                            [0,22,0,0.796,0.807,0.805,0.977],
                            [0,22,5,0.785,0.795,0.794,0.904],
                            [0,22,10,0.78,0.783,0.781,0.856],
                            [0,22,15,0.78,0.783,0.774,0.949],
                            [0,22,20,0.766,0.77,0.762,0.934],
                            [0,22,25,0.734,0.751,0.73,0.898],
                            [0,22,30,0.749,0.763,0.753,0.936],
                            [0,22,35,0.734,0.748,0.734,0.905],
                            [0,22,40,0.726,0.734,0.724,0.935],
                            [0,22,45,0.72,0.734,0.72,0.901],
                            [0,22,50,0.697,0.713,0.694,0.915],
                            [0,22,55,0.697,0.711,0.699,0.933],
                            [0,23,0,0.677,0.694,0.675,0.947],
                            [0,23,5,0.685,0.695,0.679,0.906],
                            [0,23,10,0.648,0.651,0.636,0.918],
                            [0,23,15,0.655,0.674,0.647,0.862],
                            [0,23,20,0.638,0.65,0.628,0.935],
                            [0,23,25,0.612,0.63,0.613,0.967],
                            [0,23,30,0.592,0.616,0.595,0.967],
                            [0,23,35,0.58,0.598,0.576,0.995],
                            [0,23,40,0.562,0.582,0.557,0.935],
                            [0,23,45,0.547,0.564,0.539,0.948],
                            [0,23,50,0.528,0.544,0.523,0.899],
                            [0,23,55,0.515,0.53,0.519,0.877],
                            [1,0,0,0.445,0.443,0.45,0.856],
                            [1,0,5,0.432,0.428,0.44,0.863],
                            [1,0,10,0.41,0.409,0.426,0.836],
                            [1,0,15,0.395,0.392,0.407,0.831],
                            [1,0,20,0.388,0.381,0.393,0.824],
                            [1,0,25,0.38,0.381,0.382,0.853],
                            [1,0,30,0.37,0.371,0.373,0.84],
                            [1,0,35,0.361,0.35,0.356,0.801],
                            [1,0,40,0.351,0.339,0.347,0.777],
                            [1,0,45,0.337,0.324,0.337,0.846],
                            [1,0,50,0.324,0.316,0.32,0.84],
                            [1,0,55,0.318,0.302,0.311,0.819],
                            [1,1,0,0.311,0.294,0.297,0.801],
                            [1,1,5,0.303,0.285,0.295,0.788],
                            [1,1,10,0.291,0.266,0.281,0.768],
                            [1,1,15,0.272,0.253,0.262,0.784],
                            [1,1,20,0.263,0.249,0.254,0.802],
                            [1,1,25,0.251,0.244,0.248,0.794],
                            [1,1,30,0.241,0.235,0.245,0.772],
                            [1,1,35,0.238,0.217,0.23,0.82],
                            [1,1,40,0.232,0.221,0.227,0.78],
                            [1,1,45,0.226,0.21,0.218,0.78],
                            [1,1,50,0.218,0.201,0.21,0.815],
                            [1,1,55,0.212,0.194,0.203,0.755],
                            [1,2,0,0.213,0.188,0.196,0.722],
                            [1,2,5,0.205,0.188,0.192,0.728],
                            [1,2,10,0.19,0.17,0.174,0.788],
                            [1,2,15,0.187,0.17,0.176,0.735],
                            [1,2,20,0.182,0.165,0.172,0.725],
                            [1,2,25,0.177,0.152,0.166,0.735],
                            [1,2,30,0.179,0.154,0.162,0.709],
                            [1,2,35,0.179,0.147,0.162,0.724],
                            [1,2,40,0.167,0.145,0.151,0.688],
                            [1,2,45,0.157,0.132,0.146,0.695],
                            [1,2,50,0.159,0.131,0.147,0.689],
                            [1,2,55,0.155,0.126,0.136,0.685],
                            [1,3,0,0.147,0.119,0.132,0.675],
                            [1,3,5,0.14,0.117,0.129,0.666],
                            [1,3,10,0.13,0.115,0.124,0.687],
                            [1,3,15,0.126,0.108,0.116,0.643],
                            [1,3,20,0.123,0.106,0.112,0.652],
                            [1,3,25,0.124,0.094,0.11,0.678],
                            [1,3,30,0.118,0.096,0.111,0.651],
                            [1,3,35,0.113,0.093,0.1,0.661],
                            [1,3,40,0.112,0.092,0.098,0.641],
                            [1,3,45,0.103,0.089,0.095,0.648],
                            [1,3,50,0.102,0.082,0.09,0.639],
                            [1,3,55,0.097,0.077,0.085,0.616],
                            [1,4,0,0.093,0.07,0.081,0.638],
                            [1,4,5,0.095,0.069,0.078,0.602],
                            [1,4,10,0.101,0.077,0.088,0.62],
                            [1,4,15,0.102,0.08,0.086,0.633],
                            [1,4,20,0.101,0.076,0.084,0.646],
                            [1,4,25,0.095,0.072,0.084,0.619],
                            [1,4,30,0.103,0.078,0.091,0.621],
                            [1,4,35,0.097,0.067,0.083,0.638],
                            [1,4,40,0.096,0.063,0.08,0.643],
                            [1,4,45,0.093,0.071,0.079,0.627],
                            [1,4,50,0.099,0.075,0.087,0.575],
                            [1,4,55,0.102,0.077,0.087,0.615],
                            [1,5,0,0.115,0.097,0.105,0.568],
                            [1,5,5,0.114,0.097,0.104,0.57],
                            [1,5,10,0.106,0.097,0.104,0.55],
                            [1,5,15,0.103,0.095,0.098,0.545],
                            [1,5,20,0.096,0.097,0.091,0.516],
                            [1,5,25,0.087,0.075,0.083,0.512],
                            [1,5,30,0.062,0.053,0.058,0.514],
                            [1,5,35,0.05,0.04,0.047,0.464],
                            [1,5,40,0.04,0.028,0.029,0.461],
                            [1,5,45,0.028,0.021,0.015,0.457],
                            [1,5,50,0.022,0.014,0.009,0.445],
                            [1,5,55,0.005,0.0,0.0,0.42],
                            [1,6,0,0.004,0.0,0.002,0.414],
                            [1,6,5,0.035,0.039,0.041,0.448],
                            [1,6,10,0.07,0.071,0.075,0.413],
                            [1,6,15,0.07,0.071,0.085,0.326],
                            [1,6,20,0.066,0.068,0.065,0.399],
                            [1,6,25,0.051,0.057,0.058,0.393],
                            [1,6,30,0.028,0.029,0.045,0.323],
                            [1,6,35,0.031,0.032,0.05,0.352],
                            [1,6,40,0.014,0.009,0.026,0.404],
                            [1,6,45,0.008,0.005,0.02,0.392],
                            [1,6,50,0.01,0.005,0.026,0.299],
                            [1,6,55,0.0,0.009,0.019,0.268],
                            [1,7,0,0.012,0.022,0.04,0.262],
                            [1,7,5,0.025,0.046,0.066,0.27],
                            [1,7,10,0.053,0.066,0.097,0.25],
                            [1,7,15,0.076,0.079,0.12,0.234],
                            [1,7,20,0.09,0.099,0.127,0.256],
                            [1,7,25,0.106,0.116,0.134,0.265],
                            [1,7,30,0.106,0.121,0.139,0.243],
                            [1,7,35,0.131,0.147,0.159,0.255],
                            [1,7,40,0.133,0.148,0.161,0.209],
                            [1,7,45,0.133,0.139,0.156,0.206],
                            [1,7,50,0.134,0.135,0.152,0.178],
                            [1,7,55,0.132,0.138,0.157,0.127],
                            [1,8,0,0.153,0.15,0.173,0.156],
                            [1,8,5,0.198,0.192,0.234,0.101],
                            [1,8,10,0.185,0.175,0.22,0.08],
                            [1,8,15,0.187,0.176,0.215,0.101],
                            [1,8,20,0.213,0.207,0.243,0.099],
                            [1,8,25,0.223,0.222,0.25,0.093],
                            [1,8,30,0.244,0.245,0.267,0.071],
                            [1,8,35,0.221,0.221,0.247,0.047],
                            [1,8,40,0.227,0.223,0.257,0.06],
                            [1,8,45,0.289,0.281,0.319,0.056],
                            [1,8,50,0.274,0.265,0.305,0.081],
                            [1,8,55,0.266,0.257,0.3,0.065],
                            [1,9,0,0.265,0.262,0.293,0.081],
                            [1,9,5,0.297,0.289,0.329,0.068],
                            [1,9,10,0.322,0.312,0.346,0.062],
                            [1,9,15,0.328,0.316,0.358,0.022],
                            [1,9,20,0.3,0.291,0.325,0.041],
                            [1,9,25,0.334,0.319,0.351,0.07],
                            [1,9,30,0.343,0.329,0.366,0.065],
                            [1,9,35,0.336,0.329,0.364,0.024],
                            [1,9,40,0.347,0.334,0.37,0.041],
                            [1,9,45,0.339,0.331,0.361,0.041],
                            [1,9,50,0.349,0.344,0.374,0.078],
                            [1,9,55,0.344,0.335,0.367,0.046],
                            [1,10,0,0.355,0.341,0.377,0.055],
                            [1,10,5,0.388,0.389,0.43,0.028],
                            [1,10,10,0.424,0.431,0.468,0.0],
                            [1,10,15,0.424,0.422,0.462,0.02],
                            [1,10,20,0.397,0.404,0.444,0.049],
                            [1,10,25,0.407,0.4,0.438,0.053],
                            [1,10,30,0.431,0.424,0.465,0.068],
                            [1,10,35,0.423,0.422,0.468,0.055],
                            [1,10,40,0.459,0.456,0.489,0.068],
                            [1,10,45,0.476,0.468,0.509,0.051],
                            [1,10,50,0.485,0.478,0.522,0.051],
                            [1,10,55,0.478,0.479,0.528,0.068],
                            [1,11,0,0.514,0.509,0.546,0.093],
                            [1,11,5,0.498,0.487,0.526,0.094],
                            [1,11,10,0.513,0.507,0.551,0.086],
                            [1,11,15,0.5,0.493,0.538,0.115],
                            [1,11,20,0.544,0.542,0.596,0.096],
                            [1,11,25,0.529,0.522,0.571,0.186],
                            [1,11,30,0.536,0.522,0.574,0.174],
                            [1,11,35,0.569,0.561,0.613,0.201],
                            [1,11,40,0.57,0.568,0.615,0.26],
                            [1,11,45,0.606,0.602,0.638,0.189],
                            [1,11,50,0.587,0.591,0.623,0.232],
                            [1,11,55,0.629,0.62,0.679,0.264],
                            [1,12,0,0.623,0.621,0.652,0.237],
                            [1,12,5,0.617,0.622,0.658,0.259],
                            [1,12,10,0.61,0.62,0.648,0.258],
                            [1,12,15,0.599,0.6,0.644,0.257],
                            [1,12,20,0.561,0.558,0.597,0.256],
                            [1,12,25,0.555,0.562,0.602,0.238],
                            [1,12,30,0.539,0.543,0.585,0.294],
                            [1,12,35,0.496,0.507,0.541,0.278],
                            [1,12,40,0.481,0.479,0.517,0.214],
                            [1,12,45,0.508,0.523,0.555,0.209],
                            [1,12,50,0.469,0.474,0.521,0.147],
                            [1,12,55,0.463,0.461,0.512,0.122],
                            [1,13,0,0.471,0.482,0.51,0.13],
                            [1,13,5,0.49,0.484,0.523,0.11],
                            [1,13,10,0.523,0.522,0.558,0.092],
                            [1,13,15,0.554,0.56,0.59,0.08],
                            [1,13,20,0.559,0.562,0.595,0.127],
                            [1,13,25,0.571,0.582,0.614,0.131],
                            [1,13,30,0.61,0.62,0.659,0.12],
                            [1,13,35,0.612,0.623,0.643,0.141],
                            [1,13,40,0.625,0.63,0.66,0.132],
                            [1,13,45,0.607,0.612,0.644,0.122],
                            [1,13,50,0.613,0.623,0.646,0.156],
                            [1,13,55,0.593,0.607,0.642,0.161],
                            [1,14,0,0.622,0.621,0.659,0.094],
                            [1,14,5,0.588,0.59,0.618,0.102],
                            [1,14,10,0.65,0.647,0.689,0.127],
                            [1,14,15,0.636,0.636,0.678,0.151],
                            [1,14,20,0.649,0.653,0.695,0.116],
                            [1,14,25,0.651,0.651,0.689,0.126],
                            [1,14,30,0.62,0.626,0.667,0.108],
                            [1,14,35,0.648,0.64,0.691,0.111],
                            [1,14,40,0.64,0.634,0.684,0.156],
                            [1,14,45,0.641,0.626,0.687,0.138],
                            [1,14,50,0.659,0.65,0.702,0.148],
                            [1,14,55,0.652,0.64,0.692,0.15],
                            [1,15,0,0.66,0.657,0.708,0.118],
                            [1,15,5,0.64,0.635,0.675,0.093],
                            [1,15,10,0.654,0.647,0.69,0.094],
                            [1,15,15,0.637,0.633,0.68,0.119],
                            [1,15,20,0.664,0.658,0.718,0.117],
                            [1,15,25,0.655,0.654,0.706,0.089],
                            [1,15,30,0.669,0.66,0.716,0.069],
                            [1,15,35,0.643,0.642,0.694,0.128],
                            [1,15,40,0.68,0.669,0.717,0.086],
                            [1,15,45,0.667,0.663,0.717,0.092],
                            [1,15,50,0.661,0.656,0.717,0.088],
                            [1,15,55,0.662,0.661,0.713,0.122],
                            [1,16,0,0.684,0.68,0.734,0.111],
                            [1,16,5,0.689,0.691,0.733,0.087],
                            [1,16,10,0.68,0.671,0.724,0.102],
                            [1,16,15,0.711,0.709,0.757,0.098],
                            [1,16,20,0.676,0.67,0.724,0.112],
                            [1,16,25,0.677,0.678,0.732,0.083],
                            [1,16,30,0.709,0.711,0.766,0.085],
                            [1,16,35,0.676,0.677,0.719,0.086],
                            [1,16,40,0.695,0.697,0.747,0.081],
                            [1,16,45,0.707,0.704,0.747,0.084],
                            [1,16,50,0.67,0.677,0.715,0.07],
                            [1,16,55,0.665,0.662,0.708,0.099],
                            [1,17,0,0.661,0.661,0.713,0.081],
                            [1,17,5,0.645,0.652,0.687,0.088],
                            [1,17,10,0.64,0.658,0.691,0.106],
                            [1,17,15,0.652,0.655,0.692,0.17],
                            [1,17,20,0.65,0.663,0.699,0.145],
                            [1,17,25,0.646,0.646,0.689,0.143],
                            [1,17,30,0.679,0.68,0.712,0.196],
                            [1,17,35,0.691,0.693,0.74,0.183],
                            [1,17,40,0.728,0.729,0.779,0.228],
                            [1,17,45,0.76,0.754,0.796,0.265],
                            [1,17,50,0.779,0.779,0.834,0.332],
                            [1,17,55,0.786,0.791,0.845,0.421],
                            [1,18,0,0.79,0.796,0.824,0.43],
                            [1,18,5,0.784,0.791,0.818,0.457],
                            [1,18,10,0.804,0.807,0.826,0.43],
                            [1,18,15,0.811,0.817,0.837,0.474],
                            [1,18,20,0.834,0.839,0.863,0.487],
                            [1,18,25,0.855,0.857,0.883,0.589],
                            [1,18,30,0.857,0.873,0.893,0.594],
                            [1,18,35,0.891,0.885,0.922,0.625],
                            [1,18,40,0.895,0.894,0.922,0.718],
                            [1,18,45,0.897,0.899,0.923,0.721],
                            [1,18,50,0.887,0.887,0.906,0.687],
                            [1,18,55,0.923,0.924,0.931,0.751],
                            [1,19,0,0.94,0.941,0.936,0.78],
                            [1,19,5,0.938,0.941,0.942,0.716],
                            [1,19,10,0.954,0.953,0.956,0.693],
                            [1,19,15,0.95,0.947,0.962,0.666],
                            [1,19,20,0.962,0.955,0.974,0.721],
                            [1,19,25,0.975,0.971,0.982,0.702],
                            [1,19,30,0.992,0.994,1.0,0.697],
                            [1,19,35,0.986,0.994,0.993,0.712],
                            [1,19,40,1.0,1.0,0.994,0.784],
                            [1,19,45,0.987,0.989,0.989,0.728],
                            [1,19,50,0.957,0.964,0.961,0.761],
                            [1,19,55,0.95,0.95,0.951,0.808],
                            [1,20,0,0.964,0.962,0.969,0.73],
                            [1,20,5,0.944,0.933,0.937,0.724],
                            [1,20,10,0.921,0.91,0.92,0.783],
                            [1,20,15,0.898,0.899,0.905,0.762],
                            [1,20,20,0.895,0.896,0.905,0.815],
                            [1,20,25,0.885,0.893,0.887,0.793],
                            [1,20,30,0.878,0.88,0.884,0.869],
                            [1,20,35,0.905,0.909,0.916,0.91],
                            [1,20,40,0.885,0.887,0.892,0.907],
                            [1,20,45,0.888,0.887,0.897,0.925],
                            [1,20,50,0.871,0.867,0.88,0.884],
                            [1,20,55,0.863,0.852,0.872,0.892],
                            [1,21,0,0.852,0.839,0.853,0.897],
                            [1,21,5,0.814,0.804,0.816,0.946],
                            [1,21,10,0.823,0.811,0.828,0.917],
                            [1,21,15,0.833,0.82,0.846,0.896],
                            [1,21,20,0.824,0.822,0.842,0.919],
                            [1,21,25,0.82,0.815,0.832,0.911],
                            [1,21,30,0.803,0.802,0.821,0.943],
                            [1,21,35,0.798,0.796,0.813,0.917],
                            [1,21,40,0.786,0.782,0.803,0.93],
                            [1,21,45,0.77,0.771,0.788,0.918],
                            [1,21,50,0.765,0.761,0.779,0.935],
                            [1,21,55,0.752,0.741,0.769,0.966],
                            [1,22,0,0.746,0.741,0.761,0.952],
                            [1,22,5,0.73,0.718,0.739,0.928],
                            [1,22,10,0.731,0.731,0.745,0.924],
                            [1,22,15,0.736,0.737,0.741,0.96],
                            [1,22,20,0.72,0.724,0.733,0.924],
                            [1,22,25,0.716,0.713,0.726,0.943],
                            [1,22,30,0.707,0.712,0.712,0.941],
                            [1,22,35,0.699,0.708,0.706,0.888],
                            [1,22,40,0.68,0.68,0.692,0.948],
                            [1,22,45,0.682,0.693,0.693,0.892],
                            [1,22,50,0.689,0.695,0.695,0.971],
                            [1,22,55,0.662,0.665,0.662,0.989],
                            [1,23,0,0.659,0.663,0.667,1.0],
                            [1,23,5,0.645,0.653,0.637,0.996],
                            [1,23,10,0.629,0.635,0.63,0.977],
                            [1,23,15,0.622,0.634,0.614,0.955],
                            [1,23,20,0.595,0.601,0.592,0.965],
                            [1,23,25,0.584,0.594,0.576,0.974],
                            [1,23,30,0.58,0.585,0.58,0.954],
                            [1,23,35,0.56,0.564,0.561,0.914],
                            [1,23,40,0.55,0.557,0.555,0.899],
                            [1,23,45,0.532,0.543,0.537,0.873],
                            [1,23,50,0.517,0.53,0.517,0.856],
                            [1,23,55,0.517,0.532,0.519,0.912],
                            [2,0,0,0.491,0.5,0.483,0.915],
                            [2,0,5,0.462,0.468,0.46,0.975],
                            [2,0,10,0.459,0.462,0.457,0.943],
                            [2,0,15,0.451,0.446,0.447,0.888],
                            [2,0,20,0.445,0.444,0.439,0.952],
                            [2,0,25,0.428,0.423,0.419,0.946],
                            [2,0,30,0.411,0.413,0.407,0.923],
                            [2,0,35,0.402,0.403,0.39,0.916],
                            [2,0,40,0.396,0.388,0.381,0.98],
                            [2,0,45,0.386,0.376,0.376,0.947],
                            [2,0,50,0.37,0.357,0.362,0.899],
                            [2,0,55,0.357,0.344,0.352,0.873],
                            [2,1,0,0.345,0.331,0.334,0.827],
                            [2,1,5,0.333,0.323,0.32,0.804],
                            [2,1,10,0.322,0.314,0.307,0.827],
                            [2,1,15,0.311,0.301,0.301,0.872],
                            [2,1,20,0.3,0.295,0.283,0.824],
                            [2,1,25,0.291,0.279,0.277,0.812],
                            [2,1,30,0.283,0.273,0.276,0.775],
                            [2,1,35,0.274,0.257,0.26,0.842],
                            [2,1,40,0.265,0.246,0.248,0.84],
                            [2,1,45,0.252,0.241,0.243,0.791],
                            [2,1,50,0.252,0.24,0.241,0.785],
                            [2,1,55,0.246,0.232,0.234,0.819],
                            [2,2,0,0.242,0.221,0.23,0.802],
                            [2,2,5,0.234,0.217,0.222,0.799],
                            [2,2,10,0.233,0.213,0.216,0.819],
                            [2,2,15,0.228,0.202,0.213,0.82],
                            [2,2,20,0.217,0.198,0.205,0.76],
                            [2,2,25,0.21,0.191,0.2,0.795],
                            [2,2,30,0.203,0.181,0.195,0.787],
                            [2,2,35,0.196,0.175,0.181,0.742],
                            [2,2,40,0.191,0.168,0.176,0.723],
                            [2,2,45,0.186,0.167,0.168,0.707],
                            [2,2,50,0.185,0.159,0.166,0.723],
                            [2,2,55,0.18,0.157,0.163,0.714],
                            [2,3,0,0.169,0.15,0.156,0.688],
                            [2,3,5,0.162,0.143,0.155,0.7],
                            [2,3,10,0.162,0.133,0.149,0.704],
                            [2,3,15,0.157,0.135,0.144,0.736],
                            [2,3,20,0.149,0.129,0.137,0.686],
                            [2,3,25,0.149,0.123,0.134,0.699],
                            [2,3,30,0.146,0.119,0.136,0.709],
                            [2,3,35,0.141,0.116,0.128,0.741],
                            [2,3,40,0.137,0.11,0.123,0.721],
                            [2,3,45,0.126,0.104,0.116,0.705],
                            [2,3,50,0.118,0.093,0.108,0.682],
                            [2,3,55,0.124,0.098,0.107,0.699],
                            [2,4,0,0.118,0.095,0.097,0.683],
                            [2,4,5,0.113,0.086,0.097,0.673],
                            [2,4,10,0.119,0.089,0.106,0.692],
                            [2,4,15,0.116,0.088,0.107,0.661],
                            [2,4,20,0.114,0.089,0.101,0.674],
                            [2,4,25,0.109,0.087,0.095,0.622],
                            [2,4,30,0.112,0.085,0.097,0.63],
                            [2,4,35,0.115,0.077,0.097,0.664],
                            [2,4,40,0.108,0.074,0.096,0.658],
                            [2,4,45,0.106,0.082,0.097,0.611],
                            [2,4,50,0.115,0.089,0.106,0.63],
                            [2,4,55,0.126,0.098,0.114,0.616],
                            [2,5,0,0.123,0.106,0.119,0.617],
                            [2,5,5,0.122,0.101,0.122,0.571],
                            [2,5,10,0.129,0.111,0.124,0.575],
                            [2,5,15,0.133,0.119,0.12,0.563],
                            [2,5,20,0.111,0.098,0.11,0.565],
                            [2,5,25,0.096,0.086,0.097,0.537],
                            [2,5,30,0.073,0.066,0.071,0.544],
                            [2,5,35,0.053,0.048,0.051,0.532],
                            [2,5,40,0.044,0.037,0.042,0.515],
                            [2,5,45,0.022,0.015,0.027,0.475],
                            [2,5,50,0.015,0.009,0.019,0.451],
                            [2,5,55,0.0,0.0,0.0,0.522],
                            [2,6,0,0.008,0.01,0.008,0.457],
                            [2,6,5,0.051,0.056,0.065,0.419],
                            [2,6,10,0.076,0.075,0.103,0.418],
                            [2,6,15,0.071,0.067,0.085,0.334],
                            [2,6,20,0.066,0.063,0.072,0.329],
                            [2,6,25,0.07,0.058,0.078,0.342],
                            [2,6,30,0.055,0.048,0.06,0.315],
                            [2,6,35,0.032,0.021,0.038,0.297],
                            [2,6,40,0.029,0.022,0.039,0.385],
                            [2,6,45,0.015,0.012,0.035,0.351],
                            [2,6,50,0.018,0.023,0.04,0.359],
                            [2,6,55,0.018,0.009,0.038,0.316],
                            [2,7,0,0.02,0.023,0.051,0.248],
                            [2,7,5,0.04,0.046,0.075,0.235],
                            [2,7,10,0.052,0.057,0.096,0.23],
                            [2,7,15,0.077,0.083,0.119,0.272],
                            [2,7,20,0.082,0.096,0.122,0.245],
                            [2,7,25,0.091,0.105,0.134,0.263],
                            [2,7,30,0.087,0.112,0.132,0.234],
                            [2,7,35,0.095,0.113,0.126,0.221],
                            [2,7,40,0.114,0.133,0.144,0.199],
                            [2,7,45,0.114,0.13,0.142,0.206],
                            [2,7,50,0.118,0.13,0.148,0.175],
                            [2,7,55,0.139,0.152,0.17,0.169],
                            [2,8,0,0.169,0.174,0.2,0.166],
                            [2,8,5,0.181,0.173,0.216,0.144],
                            [2,8,10,0.194,0.186,0.22,0.117],
                            [2,8,15,0.185,0.174,0.215,0.148],
                            [2,8,20,0.186,0.177,0.219,0.125],
                            [2,8,25,0.208,0.203,0.231,0.125],
                            [2,8,30,0.206,0.205,0.244,0.126],
                            [2,8,35,0.215,0.206,0.248,0.074],
                            [2,8,40,0.216,0.206,0.248,0.043],
                            [2,8,45,0.211,0.21,0.232,0.024],
                            [2,8,50,0.217,0.202,0.236,0.043],
                            [2,8,55,0.233,0.221,0.26,0.032],
                            [2,9,0,0.242,0.226,0.275,0.028],
                            [2,9,5,0.264,0.251,0.3,0.076],
                            [2,9,10,0.262,0.246,0.291,0.073],
                            [2,9,15,0.3,0.29,0.334,0.06],
                            [2,9,20,0.283,0.269,0.313,0.043],
                            [2,9,25,0.294,0.283,0.323,0.033],
                            [2,9,30,0.309,0.301,0.332,0.038],
                            [2,9,35,0.304,0.293,0.331,0.027],
                            [2,9,40,0.326,0.321,0.355,0.059],
                            [2,9,45,0.325,0.32,0.354,0.054],
                            [2,9,50,0.327,0.31,0.353,0.026],
                            [2,9,55,0.326,0.308,0.361,0.0],
                            [2,10,0,0.295,0.282,0.327,0.066],
                            [2,10,5,0.33,0.331,0.369,0.051],
                            [2,10,10,0.361,0.358,0.387,0.068],
                            [2,10,15,0.384,0.376,0.407,0.002],
                            [2,10,20,0.394,0.395,0.42,0.073],
                            [2,10,25,0.37,0.362,0.397,0.094],
                            [2,10,30,0.405,0.399,0.433,0.109],
                            [2,10,35,0.385,0.381,0.41,0.087],
                            [2,10,40,0.413,0.415,0.449,0.012],
                            [2,10,45,0.419,0.419,0.446,0.069],
                            [2,10,50,0.446,0.44,0.471,0.086],
                            [2,10,55,0.45,0.445,0.487,0.124],
                            [2,11,0,0.469,0.461,0.501,0.132],
                            [2,11,5,0.496,0.493,0.526,0.118],
                            [2,11,10,0.486,0.49,0.516,0.157],
                            [2,11,15,0.507,0.5,0.532,0.129],
                            [2,11,20,0.507,0.504,0.525,0.143],
                            [2,11,25,0.496,0.485,0.509,0.161],
                            [2,11,30,0.51,0.508,0.53,0.129],
                            [2,11,35,0.508,0.509,0.533,0.206],
                            [2,11,40,0.532,0.532,0.56,0.197],
                            [2,11,45,0.527,0.526,0.552,0.192],
                            [2,11,50,0.553,0.553,0.574,0.175],
                            [2,11,55,0.543,0.543,0.575,0.209],
                            [2,12,0,0.55,0.552,0.579,0.241],
                            [2,12,5,0.542,0.55,0.571,0.255],
                            [2,12,10,0.554,0.559,0.588,0.23],
                            [2,12,15,0.553,0.539,0.586,0.223],
                            [2,12,20,0.522,0.513,0.542,0.231],
                            [2,12,25,0.518,0.524,0.547,0.226],
                            [2,12,30,0.486,0.493,0.525,0.206],
                            [2,12,35,0.469,0.469,0.505,0.2],
                            [2,12,40,0.465,0.468,0.495,0.202],
                            [2,12,45,0.426,0.427,0.46,0.159],
                            [2,12,50,0.395,0.395,0.429,0.169],
                            [2,12,55,0.416,0.409,0.437,0.13],
                            [2,13,0,0.438,0.436,0.467,0.14],
                            [2,13,5,0.453,0.458,0.476,0.096],
                            [2,13,10,0.484,0.49,0.514,0.067],
                            [2,13,15,0.505,0.509,0.528,0.108],
                            [2,13,20,0.507,0.516,0.534,0.077],
                            [2,13,25,0.506,0.52,0.53,0.126],
                            [2,13,30,0.523,0.529,0.547,0.132],
                            [2,13,35,0.536,0.54,0.563,0.082],
                            [2,13,40,0.546,0.547,0.563,0.117],
                            [2,13,45,0.582,0.579,0.592,0.11],
                            [2,13,50,0.559,0.571,0.58,0.152],
                            [2,13,55,0.532,0.536,0.554,0.151],
                            [2,14,0,0.574,0.574,0.596,0.139],
                            [2,14,5,0.581,0.586,0.605,0.146],
                            [2,14,10,0.581,0.581,0.611,0.174],
                            [2,14,15,0.584,0.585,0.615,0.196],
                            [2,14,20,0.548,0.548,0.568,0.163],
                            [2,14,25,0.597,0.589,0.627,0.13],
                            [2,14,30,0.579,0.564,0.6,0.073],
                            [2,14,35,0.595,0.593,0.618,0.125],
                            [2,14,40,0.578,0.579,0.599,0.109],
                            [2,14,45,0.583,0.581,0.605,0.099],
                            [2,14,50,0.59,0.587,0.604,0.119],
                            [2,14,55,0.601,0.598,0.622,0.126],
                            [2,15,0,0.63,0.631,0.653,0.113],
                            [2,15,5,0.615,0.619,0.638,0.093],
                            [2,15,10,0.649,0.648,0.668,0.08],
                            [2,15,15,0.612,0.616,0.644,0.1],
                            [2,15,20,0.63,0.634,0.665,0.089],
                            [2,15,25,0.631,0.627,0.666,0.089],
                            [2,15,30,0.646,0.644,0.681,0.094],
                            [2,15,35,0.605,0.603,0.634,0.094],
                            [2,15,40,0.624,0.629,0.661,0.073],
                            [2,15,45,0.632,0.631,0.675,0.057],
                            [2,15,50,0.623,0.619,0.655,0.034],
                            [2,15,55,0.646,0.64,0.68,0.043],
                            [2,16,0,0.632,0.634,0.665,0.086],
                            [2,16,5,0.631,0.635,0.661,0.063],
                            [2,16,10,0.636,0.643,0.663,0.051],
                            [2,16,15,0.614,0.618,0.647,0.101],
                            [2,16,20,0.638,0.656,0.667,0.104],
                            [2,16,25,0.624,0.641,0.661,0.046],
                            [2,16,30,0.647,0.656,0.684,0.049],
                            [2,16,35,0.621,0.639,0.658,0.089],
                            [2,16,40,0.657,0.668,0.693,0.109],
                            [2,16,45,0.664,0.678,0.698,0.118],
                            [2,16,50,0.637,0.652,0.675,0.107],
                            [2,16,55,0.63,0.646,0.669,0.128],
                            [2,17,0,0.618,0.636,0.652,0.122],
                            [2,17,5,0.61,0.62,0.643,0.059],
                            [2,17,10,0.59,0.599,0.619,0.136],
                            [2,17,15,0.624,0.624,0.658,0.112],
                            [2,17,20,0.625,0.633,0.658,0.183],
                            [2,17,25,0.624,0.636,0.665,0.183],
                            [2,17,30,0.655,0.659,0.694,0.18],
                            [2,17,35,0.679,0.67,0.719,0.199],
                            [2,17,40,0.706,0.715,0.751,0.295],
                            [2,17,45,0.729,0.743,0.778,0.346],
                            [2,17,50,0.748,0.762,0.8,0.385],
                            [2,17,55,0.761,0.773,0.804,0.46],
                            [2,18,0,0.781,0.797,0.815,0.465],
                            [2,18,5,0.807,0.814,0.832,0.52],
                            [2,18,10,0.815,0.834,0.848,0.514],
                            [2,18,15,0.814,0.836,0.85,0.572],
                            [2,18,20,0.844,0.869,0.874,0.524],
                            [2,18,25,0.873,0.884,0.902,0.625],
                            [2,18,30,0.875,0.883,0.891,0.615],
                            [2,18,35,0.895,0.9,0.913,0.66],
                            [2,18,40,0.901,0.914,0.924,0.715],
                            [2,18,45,0.914,0.931,0.943,0.733],
                            [2,18,50,0.93,0.935,0.951,0.775],
                            [2,18,55,0.928,0.937,0.948,0.731],
                            [2,19,0,0.951,0.943,0.964,0.726],
                            [2,19,5,0.964,0.956,0.969,0.737],
                            [2,19,10,0.971,0.957,0.979,0.752],
                            [2,19,15,0.974,0.971,0.981,0.761],
                            [2,19,20,0.961,0.967,0.97,0.758],
                            [2,19,25,0.99,0.999,1.0,0.744],
                            [2,19,30,0.99,0.983,0.992,0.711],
                            [2,19,35,0.991,0.989,0.992,0.702],
                            [2,19,40,0.982,0.989,0.983,0.735],
                            [2,19,45,0.957,0.961,0.958,0.739],
                            [2,19,50,1.0,1.0,0.996,0.757],
                            [2,19,55,0.966,0.971,0.969,0.748],
                            [2,20,0,0.929,0.934,0.938,0.74],
                            [2,20,5,0.936,0.937,0.946,0.771],
                            [2,20,10,0.93,0.935,0.935,0.726],
                            [2,20,15,0.91,0.907,0.913,0.823],
                            [2,20,20,0.918,0.92,0.919,0.762],
                            [2,20,25,0.895,0.892,0.896,0.776],
                            [2,20,30,0.9,0.895,0.901,0.815],
                            [2,20,35,0.904,0.899,0.905,0.784],
                            [2,20,40,0.873,0.875,0.882,0.864],
                            [2,20,45,0.888,0.887,0.892,0.849],
                            [2,20,50,0.843,0.845,0.845,0.857],
                            [2,20,55,0.831,0.829,0.839,0.841],
                            [2,21,0,0.812,0.806,0.814,0.843],
                            [2,21,5,0.792,0.794,0.803,0.817],
                            [2,21,10,0.814,0.815,0.821,0.821],
                            [2,21,15,0.807,0.802,0.814,0.799],
                            [2,21,20,0.799,0.793,0.809,0.934],
                            [2,21,25,0.774,0.775,0.783,0.872],
                            [2,21,30,0.766,0.77,0.787,0.862],
                            [2,21,35,0.771,0.768,0.783,0.879],
                            [2,21,40,0.762,0.759,0.772,0.928],
                            [2,21,45,0.74,0.73,0.749,0.864],
                            [2,21,50,0.739,0.733,0.744,0.898],
                            [2,21,55,0.737,0.72,0.733,0.864],
                            [2,22,0,0.716,0.719,0.72,0.969],
                            [2,22,5,0.734,0.735,0.733,0.957],
                            [2,22,10,0.721,0.717,0.719,0.896],
                            [2,22,15,0.693,0.692,0.695,0.897],
                            [2,22,20,0.69,0.694,0.691,0.928],
                            [2,22,25,0.677,0.686,0.682,0.938],
                            [2,22,30,0.668,0.675,0.674,0.938],
                            [2,22,35,0.692,0.696,0.695,0.92],
                            [2,22,40,0.685,0.681,0.688,0.931],
                            [2,22,45,0.667,0.671,0.673,0.956],
                            [2,22,50,0.658,0.666,0.666,0.92],
                            [2,22,55,0.644,0.647,0.641,0.938],
                            [2,23,0,0.638,0.642,0.644,0.922],
                            [2,23,5,0.623,0.624,0.624,0.938],
                            [2,23,10,0.601,0.606,0.611,0.935],
                            [2,23,15,0.591,0.591,0.6,0.961],
                            [2,23,20,0.596,0.592,0.591,0.953],
                            [2,23,25,0.576,0.582,0.571,1.0],
                            [2,23,30,0.569,0.574,0.576,0.931],
                            [2,23,35,0.554,0.555,0.554,0.909],
                            [2,23,40,0.536,0.536,0.529,0.954],
                            [2,23,45,0.523,0.529,0.516,0.926],
                            [2,23,50,0.506,0.506,0.503,0.954],
                            [2,23,55,0.489,0.49,0.492,0.915],
                            [3,0,0,0.491,0.489,0.485,0.932],
                            [3,0,5,0.472,0.465,0.462,0.93],
                            [3,0,10,0.464,0.454,0.452,0.92],
                            [3,0,15,0.449,0.445,0.432,1.0],
                            [3,0,20,0.429,0.421,0.426,0.978],
                            [3,0,25,0.423,0.413,0.413,0.924],
                            [3,0,30,0.407,0.397,0.399,0.94],
                            [3,0,35,0.391,0.386,0.388,0.978],
                            [3,0,40,0.382,0.379,0.371,0.959],
                            [3,0,45,0.369,0.356,0.36,0.968],
                            [3,0,50,0.358,0.345,0.345,0.933],
                            [3,0,55,0.35,0.333,0.332,0.958],
                            [3,1,0,0.341,0.327,0.321,0.915],
                            [3,1,5,0.33,0.316,0.314,0.895],
                            [3,1,10,0.31,0.295,0.3,0.868],
                            [3,1,15,0.297,0.288,0.285,0.902],
                            [3,1,20,0.293,0.27,0.27,0.913],
                            [3,1,25,0.287,0.261,0.261,0.871],
                            [3,1,30,0.271,0.255,0.248,0.859],
                            [3,1,35,0.261,0.244,0.242,0.871],
                            [3,1,40,0.248,0.237,0.237,0.871],
                            [3,1,45,0.246,0.226,0.224,0.857],
                            [3,1,50,0.229,0.212,0.212,0.844],
                            [3,1,55,0.235,0.221,0.22,0.864],
                            [3,2,0,0.231,0.21,0.22,0.84],
                            [3,2,5,0.226,0.202,0.205,0.844],
                            [3,2,10,0.215,0.19,0.2,0.812],
                            [3,2,15,0.205,0.183,0.187,0.852],
                            [3,2,20,0.194,0.178,0.182,0.867],
                            [3,2,25,0.192,0.174,0.18,0.793],
                            [3,2,30,0.189,0.164,0.176,0.84],
                            [3,2,35,0.182,0.156,0.166,0.806],
                            [3,2,40,0.18,0.154,0.164,0.781],
                            [3,2,45,0.159,0.135,0.147,0.712],
                            [3,2,50,0.169,0.146,0.154,0.727],
                            [3,2,55,0.163,0.139,0.152,0.744],
                            [3,3,0,0.157,0.135,0.142,0.75],
                            [3,3,5,0.154,0.132,0.137,0.724],
                            [3,3,10,0.147,0.127,0.137,0.771],
                            [3,3,15,0.142,0.121,0.133,0.742],
                            [3,3,20,0.14,0.118,0.13,0.762],
                            [3,3,25,0.137,0.108,0.118,0.729],
                            [3,3,30,0.13,0.101,0.115,0.69],
                            [3,3,35,0.125,0.098,0.112,0.735],
                            [3,3,40,0.121,0.098,0.107,0.731],
                            [3,3,45,0.117,0.095,0.102,0.724],
                            [3,3,50,0.113,0.091,0.097,0.709],
                            [3,3,55,0.119,0.093,0.098,0.742],
                            [3,4,0,0.113,0.08,0.091,0.707],
                            [3,4,5,0.114,0.087,0.092,0.748],
                            [3,4,10,0.114,0.084,0.096,0.719],
                            [3,4,15,0.114,0.086,0.093,0.736],
                            [3,4,20,0.115,0.081,0.092,0.69],
                            [3,4,25,0.109,0.079,0.086,0.675],
                            [3,4,30,0.107,0.069,0.082,0.672],
                            [3,4,35,0.101,0.065,0.082,0.659],
                            [3,4,40,0.102,0.074,0.083,0.66],
                            [3,4,45,0.102,0.076,0.088,0.648],
                            [3,4,50,0.114,0.086,0.103,0.648],
                            [3,4,55,0.114,0.086,0.097,0.684],
                            [3,5,0,0.125,0.098,0.111,0.669],
                            [3,5,5,0.13,0.108,0.115,0.629],
                            [3,5,10,0.129,0.115,0.117,0.601],
                            [3,5,15,0.118,0.102,0.108,0.569],
                            [3,5,20,0.122,0.106,0.112,0.577],
                            [3,5,25,0.096,0.087,0.096,0.54],
                            [3,5,30,0.08,0.068,0.071,0.6],
                            [3,5,35,0.053,0.052,0.058,0.595],
                            [3,5,40,0.043,0.033,0.042,0.526],
                            [3,5,45,0.027,0.023,0.024,0.603],
                            [3,5,50,0.024,0.018,0.015,0.547],
                            [3,5,55,0.01,0.0,0.002,0.565],
                            [3,6,0,0.0,0.005,0.0,0.498],
                            [3,6,5,0.034,0.038,0.042,0.511],
                            [3,6,10,0.087,0.075,0.084,0.443],
                            [3,6,15,0.078,0.073,0.078,0.396],
                            [3,6,20,0.065,0.067,0.073,0.376],
                            [3,6,25,0.06,0.061,0.064,0.45],
                            [3,6,30,0.065,0.06,0.06,0.404],
                            [3,6,35,0.055,0.053,0.057,0.385],
                            [3,6,40,0.035,0.04,0.045,0.438],
                            [3,6,45,0.017,0.028,0.035,0.428],
                            [3,6,50,0.022,0.027,0.033,0.394],
                            [3,6,55,0.035,0.033,0.056,0.359],
                            [3,7,0,0.029,0.037,0.059,0.321],
                            [3,7,5,0.041,0.048,0.077,0.298],
                            [3,7,10,0.069,0.083,0.119,0.266],
                            [3,7,15,0.067,0.087,0.105,0.228],
                            [3,7,20,0.099,0.122,0.142,0.237],
                            [3,7,25,0.109,0.125,0.147,0.239],
                            [3,7,30,0.106,0.135,0.141,0.236],
                            [3,7,35,0.103,0.136,0.138,0.206],
                            [3,7,40,0.125,0.151,0.16,0.245],
                            [3,7,45,0.137,0.147,0.163,0.296],
                            [3,7,50,0.133,0.142,0.152,0.219],
                            [3,7,55,0.155,0.168,0.18,0.165],
                            [3,8,0,0.187,0.193,0.201,0.177],
                            [3,8,5,0.216,0.22,0.222,0.136],
                            [3,8,10,0.228,0.234,0.25,0.129],
                            [3,8,15,0.234,0.243,0.249,0.163],
                            [3,8,20,0.251,0.248,0.267,0.139],
                            [3,8,25,0.266,0.264,0.271,0.153],
                            [3,8,30,0.234,0.23,0.239,0.152],
                            [3,8,35,0.266,0.259,0.259,0.118],
                            [3,8,40,0.272,0.259,0.26,0.116],
                            [3,8,45,0.27,0.265,0.279,0.117],
                            [3,8,50,0.278,0.264,0.293,0.103],
                            [3,8,55,0.304,0.295,0.318,0.115],
                            [3,9,0,0.298,0.284,0.306,0.098],
                            [3,9,5,0.328,0.327,0.338,0.092],
                            [3,9,10,0.33,0.32,0.342,0.097],
                            [3,9,15,0.318,0.313,0.334,0.122],
                            [3,9,20,0.336,0.343,0.363,0.111],
                            [3,9,25,0.34,0.337,0.356,0.128],
                            [3,9,30,0.338,0.337,0.36,0.107],
                            [3,9,35,0.35,0.348,0.372,0.099],
                            [3,9,40,0.35,0.351,0.367,0.096],
                            [3,9,45,0.352,0.356,0.366,0.044],
                            [3,9,50,0.381,0.373,0.389,0.09],
                            [3,9,55,0.396,0.4,0.407,0.104],
                            [3,10,0,0.414,0.418,0.428,0.073],
                            [3,10,5,0.395,0.397,0.407,0.067],
                            [3,10,10,0.411,0.42,0.429,0.077],
                            [3,10,15,0.431,0.443,0.45,0.052],
                            [3,10,20,0.426,0.434,0.441,0.044],
                            [3,10,25,0.451,0.459,0.461,0.069],
                            [3,10,30,0.449,0.455,0.468,0.081],
                            [3,10,35,0.462,0.467,0.481,0.057],
                            [3,10,40,0.481,0.487,0.499,0.08],
                            [3,10,45,0.492,0.504,0.508,0.075],
                            [3,10,50,0.493,0.488,0.503,0.082],
                            [3,10,55,0.493,0.501,0.511,0.048],
                            [3,11,0,0.532,0.534,0.554,0.043],
                            [3,11,5,0.542,0.539,0.551,0.075],
                            [3,11,10,0.532,0.529,0.545,0.129],
                            [3,11,15,0.543,0.544,0.546,0.174],
                            [3,11,20,0.542,0.544,0.557,0.202],
                            [3,11,25,0.541,0.538,0.557,0.205],
                            [3,11,30,0.572,0.566,0.575,0.198],
                            [3,11,35,0.569,0.566,0.579,0.225],
                            [3,11,40,0.596,0.588,0.6,0.216],
                            [3,11,45,0.628,0.625,0.635,0.287],
                            [3,11,50,0.631,0.633,0.643,0.25],
                            [3,11,55,0.627,0.622,0.636,0.232],
                            [3,12,0,0.6,0.606,0.634,0.234],
                            [3,12,5,0.623,0.604,0.638,0.276],
                            [3,12,10,0.604,0.596,0.615,0.223],
                            [3,12,15,0.61,0.606,0.625,0.267],
                            [3,12,20,0.594,0.591,0.608,0.263],
                            [3,12,25,0.579,0.567,0.602,0.209],
                            [3,12,30,0.542,0.533,0.565,0.21],
                            [3,12,35,0.516,0.505,0.533,0.177],
                            [3,12,40,0.501,0.49,0.514,0.196],
                            [3,12,45,0.5,0.495,0.52,0.18],
                            [3,12,50,0.501,0.497,0.524,0.201],
                            [3,12,55,0.504,0.498,0.53,0.145],
                            [3,13,0,0.529,0.526,0.542,0.122],
                            [3,13,5,0.537,0.544,0.546,0.134],
                            [3,13,10,0.565,0.573,0.585,0.145],
                            [3,13,15,0.573,0.585,0.596,0.129],
                            [3,13,20,0.587,0.591,0.6,0.143],
                            [3,13,25,0.588,0.586,0.596,0.11],
                            [3,13,30,0.618,0.613,0.624,0.118],
                            [3,13,35,0.649,0.645,0.653,0.179],
                            [3,13,40,0.636,0.639,0.648,0.159],
                            [3,13,45,0.651,0.66,0.659,0.167],
                            [3,13,50,0.65,0.662,0.656,0.155],
                            [3,13,55,0.634,0.645,0.639,0.186],
                            [3,14,0,0.639,0.64,0.657,0.187],
                            [3,14,5,0.631,0.631,0.643,0.135],
                            [3,14,10,0.648,0.651,0.663,0.118],
                            [3,14,15,0.687,0.686,0.697,0.169],
                            [3,14,20,0.652,0.649,0.681,0.156],
                            [3,14,25,0.673,0.675,0.694,0.078],
                            [3,14,30,0.667,0.67,0.686,0.143],
                            [3,14,35,0.682,0.689,0.697,0.167],
                            [3,14,40,0.662,0.66,0.666,0.127],
                            [3,14,45,0.648,0.656,0.67,0.194],
                            [3,14,50,0.655,0.664,0.682,0.122],
                            [3,14,55,0.649,0.651,0.665,0.131],
                            [3,15,0,0.696,0.695,0.707,0.082],
                            [3,15,5,0.673,0.677,0.685,0.076],
                            [3,15,10,0.677,0.674,0.695,0.132],
                            [3,15,15,0.681,0.678,0.696,0.088],
                            [3,15,20,0.647,0.644,0.671,0.071],
                            [3,15,25,0.676,0.676,0.697,0.071],
                            [3,15,30,0.684,0.682,0.7,0.083],
                            [3,15,35,0.664,0.657,0.679,0.116],
                            [3,15,40,0.653,0.641,0.674,0.174],
                            [3,15,45,0.675,0.67,0.692,0.114],
                            [3,15,50,0.697,0.685,0.713,0.093],
                            [3,15,55,0.689,0.688,0.71,0.079],
                            [3,16,0,0.678,0.671,0.697,0.073],
                            [3,16,5,0.679,0.672,0.691,0.098],
                            [3,16,10,0.685,0.678,0.706,0.075],
                            [3,16,15,0.685,0.681,0.696,0.044],
                            [3,16,20,0.684,0.68,0.697,0.0],
                            [3,16,25,0.688,0.685,0.703,0.067],
                            [3,16,30,0.67,0.679,0.688,0.043],
                            [3,16,35,0.704,0.707,0.73,0.068],
                            [3,16,40,0.695,0.697,0.718,0.086],
                            [3,16,45,0.713,0.726,0.734,0.089],
                            [3,16,50,0.669,0.679,0.697,0.065],
                            [3,16,55,0.681,0.687,0.702,0.056],
                            [3,17,0,0.68,0.693,0.7,0.138],
                            [3,17,5,0.661,0.675,0.683,0.13],
                            [3,17,10,0.651,0.654,0.652,0.111],
                            [3,17,15,0.647,0.663,0.669,0.148],
                            [3,17,20,0.661,0.682,0.687,0.152],
                            [3,17,25,0.663,0.669,0.696,0.189],
                            [3,17,30,0.661,0.67,0.69,0.243],
                            [3,17,35,0.72,0.717,0.746,0.216],
                            [3,17,40,0.737,0.737,0.766,0.262],
                            [3,17,45,0.761,0.762,0.777,0.365],
                            [3,17,50,0.781,0.78,0.801,0.349],
                            [3,17,55,0.795,0.795,0.81,0.356],
                            [3,18,0,0.814,0.809,0.822,0.443],
                            [3,18,5,0.822,0.816,0.826,0.487],
                            [3,18,10,0.841,0.845,0.845,0.526],
                            [3,18,15,0.863,0.847,0.866,0.504],
                            [3,18,20,0.86,0.853,0.857,0.59],
                            [3,18,25,0.896,0.889,0.898,0.723],
                            [3,18,30,0.886,0.876,0.872,0.727],
                            [3,18,35,0.919,0.928,0.912,0.773],
                            [3,18,40,0.909,0.925,0.899,0.831],
                            [3,18,45,0.92,0.933,0.91,0.766],
                            [3,18,50,0.945,0.95,0.934,0.791],
                            [3,18,55,0.955,0.959,0.938,0.766],
                            [3,19,0,0.971,0.966,0.957,0.856],
                            [3,19,5,0.977,0.969,0.968,0.769],
                            [3,19,10,0.986,0.983,0.981,0.694],
                            [3,19,15,0.983,0.987,0.973,0.7],
                            [3,19,20,0.978,0.985,0.978,0.687],
                            [3,19,25,0.987,0.986,0.99,0.65],
                            [3,19,30,1.0,0.989,1.0,0.699],
                            [3,19,35,0.995,0.976,0.989,0.687],
                            [3,19,40,0.985,0.981,0.971,0.721],
                            [3,19,45,0.999,1.0,0.985,0.774],
                            [3,19,50,0.998,0.999,0.991,0.738],
                            [3,19,55,0.973,0.967,0.972,0.766],
                            [3,20,0,0.979,0.967,0.963,0.774],
                            [3,20,5,0.946,0.928,0.936,0.783],
                            [3,20,10,0.943,0.933,0.939,0.719],
                            [3,20,15,0.942,0.931,0.939,0.695],
                            [3,20,20,0.93,0.921,0.922,0.706],
                            [3,20,25,0.953,0.946,0.939,0.739],
                            [3,20,30,0.898,0.885,0.891,0.799],
                            [3,20,35,0.893,0.885,0.878,0.752],
                            [3,20,40,0.91,0.899,0.886,0.779],
                            [3,20,45,0.911,0.904,0.886,0.825],
                            [3,20,50,0.928,0.919,0.918,0.86],
                            [3,20,55,0.883,0.868,0.877,0.864],
                            [3,21,0,0.873,0.854,0.858,0.891],
                            [3,21,5,0.833,0.815,0.817,0.862],
                            [3,21,10,0.85,0.834,0.828,0.831],
                            [3,21,15,0.843,0.828,0.827,0.806],
                            [3,21,20,0.835,0.821,0.82,0.833],
                            [3,21,25,0.818,0.799,0.81,0.88],
                            [3,21,30,0.795,0.769,0.775,0.927],
                            [3,21,35,0.782,0.768,0.768,0.927],
                            [3,21,40,0.777,0.763,0.763,0.983],
                            [3,21,45,0.765,0.75,0.748,0.99],
                            [3,21,50,0.766,0.744,0.75,0.889],
                            [3,21,55,0.763,0.749,0.758,0.895],
                            [3,22,0,0.735,0.728,0.729,0.942],
                            [3,22,5,0.723,0.725,0.717,0.946],
                            [3,22,10,0.719,0.725,0.708,0.909],
                            [3,22,15,0.727,0.727,0.711,0.972],
                            [3,22,20,0.724,0.724,0.708,0.96],
                            [3,22,25,0.744,0.742,0.727,0.951],
                            [3,22,30,0.724,0.717,0.704,0.906],
                            [3,22,35,0.711,0.717,0.703,0.926],
                            [3,22,40,0.715,0.707,0.699,0.962],
                            [3,22,45,0.704,0.695,0.683,0.929],
                            [3,22,50,0.69,0.685,0.671,0.978],
                            [3,22,55,0.673,0.682,0.66,0.884],
                            [3,23,0,0.657,0.654,0.644,0.889],
                            [3,23,5,0.652,0.651,0.632,0.944],
                            [3,23,10,0.641,0.64,0.621,0.941],
                            [3,23,15,0.63,0.634,0.608,0.867],
                            [3,23,20,0.616,0.619,0.592,0.861],
                            [3,23,25,0.608,0.608,0.593,0.929],
                            [3,23,30,0.589,0.587,0.582,0.915],
                            [3,23,35,0.576,0.578,0.554,0.954],
                            [3,23,40,0.554,0.554,0.54,0.993],
                            [3,23,45,0.541,0.535,0.524,0.938],
                            [3,23,50,0.552,0.549,0.535,0.94],
                            [3,23,55,0.524,0.523,0.511,0.99],
                            [4,0,0,0.531,0.533,0.531,0.947],
                            [4,0,5,0.514,0.515,0.511,0.996],
                            [4,0,10,0.505,0.511,0.514,0.957],
                            [4,0,15,0.481,0.484,0.489,0.894],
                            [4,0,20,0.468,0.476,0.472,0.942],
                            [4,0,25,0.474,0.475,0.474,0.948],
                            [4,0,30,0.442,0.443,0.441,0.925],
                            [4,0,35,0.419,0.417,0.417,0.938],
                            [4,0,40,0.416,0.412,0.406,1.0],
                            [4,0,45,0.409,0.402,0.4,0.948],
                            [4,0,50,0.398,0.394,0.399,0.952],
                            [4,0,55,0.411,0.409,0.408,0.938],
                            [4,1,0,0.389,0.385,0.386,0.911],
                            [4,1,5,0.351,0.345,0.33,0.846],
                            [4,1,10,0.369,0.36,0.357,0.911],
                            [4,1,15,0.361,0.348,0.351,0.908],
                            [4,1,20,0.349,0.335,0.332,0.868],
                            [4,1,25,0.336,0.328,0.324,0.844],
                            [4,1,30,0.318,0.303,0.313,0.853],
                            [4,1,35,0.311,0.295,0.299,0.852],
                            [4,1,40,0.307,0.28,0.292,0.819],
                            [4,1,45,0.302,0.278,0.29,0.866],
                            [4,1,50,0.292,0.269,0.279,0.855],
                            [4,1,55,0.284,0.268,0.27,0.864],
                            [4,2,0,0.275,0.261,0.254,0.783],
                            [4,2,5,0.27,0.252,0.255,0.87],
                            [4,2,10,0.262,0.24,0.246,0.826],
                            [4,2,15,0.253,0.227,0.231,0.839],
                            [4,2,20,0.239,0.218,0.227,0.825],
                            [4,2,25,0.232,0.213,0.221,0.787],
                            [4,2,30,0.227,0.201,0.213,0.79],
                            [4,2,35,0.223,0.206,0.201,0.815],
                            [4,2,40,0.221,0.196,0.196,0.813],
                            [4,2,45,0.217,0.197,0.196,0.795],
                            [4,2,50,0.197,0.174,0.179,0.775],
                            [4,2,55,0.193,0.17,0.175,0.762],
                            [4,3,0,0.189,0.164,0.173,0.77],
                            [4,3,5,0.174,0.15,0.151,0.734],
                            [4,3,10,0.171,0.158,0.159,0.729],
                            [4,3,15,0.175,0.146,0.159,0.751],
                            [4,3,20,0.172,0.148,0.152,0.752],
                            [4,3,25,0.16,0.14,0.147,0.709],
                            [4,3,30,0.161,0.137,0.142,0.694],
                            [4,3,35,0.157,0.133,0.134,0.714],
                            [4,3,40,0.149,0.13,0.126,0.759],
                            [4,3,45,0.142,0.118,0.124,0.772],
                            [4,3,50,0.141,0.111,0.121,0.744],
                            [4,3,55,0.138,0.112,0.121,0.734],
                            [4,4,0,0.134,0.094,0.114,0.693],
                            [4,4,5,0.128,0.093,0.108,0.674],
                            [4,4,10,0.117,0.082,0.096,0.683],
                            [4,4,15,0.132,0.099,0.112,0.647],
                            [4,4,20,0.126,0.094,0.107,0.675],
                            [4,4,25,0.131,0.101,0.117,0.677],
                            [4,4,30,0.122,0.092,0.104,0.686],
                            [4,4,35,0.112,0.079,0.092,0.742],
                            [4,4,40,0.117,0.09,0.096,0.704],
                            [4,4,45,0.12,0.095,0.093,0.65],
                            [4,4,50,0.128,0.104,0.11,0.686],
                            [4,4,55,0.133,0.1,0.11,0.661],
                            [4,5,0,0.132,0.112,0.12,0.632],
                            [4,5,5,0.149,0.132,0.134,0.601],
                            [4,5,10,0.132,0.126,0.133,0.625],
                            [4,5,15,0.124,0.123,0.121,0.575],
                            [4,5,20,0.117,0.106,0.108,0.563],
                            [4,5,25,0.103,0.093,0.099,0.53],
                            [4,5,30,0.081,0.078,0.074,0.502],
                            [4,5,35,0.059,0.056,0.055,0.489],
                            [4,5,40,0.039,0.031,0.037,0.489],
                            [4,5,45,0.034,0.011,0.02,0.473],
                            [4,5,50,0.019,0.0,0.016,0.519],
                            [4,5,55,0.005,0.003,0.0,0.496],
                            [4,6,0,0.0,0.008,0.001,0.467],
                            [4,6,5,0.045,0.053,0.044,0.431],
                            [4,6,10,0.086,0.089,0.099,0.474],
                            [4,6,15,0.072,0.073,0.084,0.367],
                            [4,6,20,0.074,0.079,0.075,0.443],
                            [4,6,25,0.064,0.061,0.061,0.445],
                            [4,6,30,0.027,0.034,0.033,0.388],
                            [4,6,35,0.021,0.015,0.022,0.393],
                            [4,6,40,0.025,0.017,0.024,0.379],
                            [4,6,45,0.017,0.01,0.027,0.378],
                            [4,6,50,0.015,0.019,0.037,0.362],
                            [4,6,55,0.017,0.018,0.045,0.328],
                            [4,7,0,0.019,0.024,0.054,0.268],
                            [4,7,5,0.03,0.034,0.06,0.222],
                            [4,7,10,0.046,0.048,0.086,0.242],
                            [4,7,15,0.07,0.075,0.119,0.279],
                            [4,7,20,0.077,0.08,0.124,0.255],
                            [4,7,25,0.104,0.111,0.144,0.257],
                            [4,7,30,0.094,0.101,0.123,0.279],
                            [4,7,35,0.103,0.129,0.146,0.275],
                            [4,7,40,0.114,0.126,0.143,0.239],
                            [4,7,45,0.13,0.143,0.165,0.201],
                            [4,7,50,0.135,0.148,0.172,0.181],
                            [4,7,55,0.167,0.173,0.19,0.178],
                            [4,8,0,0.194,0.192,0.211,0.166],
                            [4,8,5,0.243,0.241,0.26,0.23],
                            [4,8,10,0.286,0.282,0.296,0.177],
                            [4,8,15,0.261,0.261,0.28,0.134],
                            [4,8,20,0.276,0.281,0.303,0.144],
                            [4,8,25,0.305,0.301,0.32,0.159],
                            [4,8,30,0.286,0.285,0.297,0.141],
                            [4,8,35,0.283,0.275,0.305,0.102],
                            [4,8,40,0.288,0.28,0.308,0.093],
                            [4,8,45,0.288,0.28,0.316,0.09],
                            [4,8,50,0.308,0.299,0.341,0.077],
                            [4,8,55,0.324,0.307,0.344,0.069],
                            [4,9,0,0.313,0.31,0.338,0.042],
                            [4,9,5,0.372,0.362,0.398,0.042],
                            [4,9,10,0.369,0.361,0.384,0.058],
                            [4,9,15,0.377,0.366,0.391,0.075],
                            [4,9,20,0.389,0.386,0.405,0.025],
                            [4,9,25,0.417,0.409,0.436,0.063],
                            [4,9,30,0.402,0.386,0.425,0.046],
                            [4,9,35,0.416,0.405,0.433,0.063],
                            [4,9,40,0.409,0.4,0.433,0.005],
                            [4,9,45,0.417,0.401,0.443,0.06],
                            [4,9,50,0.419,0.414,0.44,0.067],
                            [4,9,55,0.409,0.395,0.429,0.012],
                            [4,10,0,0.391,0.403,0.416,0.044],
                            [4,10,5,0.379,0.395,0.396,0.057],
                            [4,10,10,0.447,0.455,0.474,0.111],
                            [4,10,15,0.457,0.468,0.479,0.063],
                            [4,10,20,0.439,0.449,0.456,0.067],
                            [4,10,25,0.457,0.469,0.489,0.086],
                            [4,10,30,0.437,0.444,0.457,0.106],
                            [4,10,35,0.478,0.484,0.499,0.112],
                            [4,10,40,0.452,0.461,0.468,0.154],
                            [4,10,45,0.483,0.5,0.5,0.106],
                            [4,10,50,0.51,0.524,0.527,0.097],
                            [4,10,55,0.528,0.542,0.549,0.081],
                            [4,11,0,0.556,0.571,0.586,0.126],
                            [4,11,5,0.571,0.585,0.599,0.127],
                            [4,11,10,0.549,0.569,0.57,0.153],
                            [4,11,15,0.589,0.603,0.615,0.157],
                            [4,11,20,0.606,0.625,0.636,0.207],
                            [4,11,25,0.593,0.612,0.628,0.245],
                            [4,11,30,0.589,0.615,0.624,0.215],
                            [4,11,35,0.597,0.613,0.627,0.144],
                            [4,11,40,0.61,0.63,0.644,0.157],
                            [4,11,45,0.612,0.626,0.637,0.22],
                            [4,11,50,0.588,0.601,0.626,0.209],
                            [4,11,55,0.625,0.646,0.659,0.202],
                            [4,12,0,0.625,0.64,0.657,0.212],
                            [4,12,5,0.611,0.625,0.631,0.204],
                            [4,12,10,0.565,0.573,0.599,0.215],
                            [4,12,15,0.586,0.589,0.617,0.226],
                            [4,12,20,0.56,0.582,0.594,0.179],
                            [4,12,25,0.561,0.567,0.6,0.212],
                            [4,12,30,0.571,0.58,0.608,0.192],
                            [4,12,35,0.54,0.545,0.562,0.178],
                            [4,12,40,0.533,0.544,0.56,0.201],
                            [4,12,45,0.54,0.55,0.573,0.21],
                            [4,12,50,0.471,0.484,0.504,0.204],
                            [4,12,55,0.52,0.533,0.556,0.157],
                            [4,13,0,0.541,0.561,0.561,0.215],
                            [4,13,5,0.586,0.6,0.612,0.178],
                            [4,13,10,0.54,0.558,0.573,0.226],
                            [4,13,15,0.549,0.555,0.571,0.16],
                            [4,13,20,0.589,0.617,0.624,0.15],
                            [4,13,25,0.61,0.628,0.638,0.123],
                            [4,13,30,0.612,0.632,0.656,0.146],
                            [4,13,35,0.633,0.646,0.653,0.157],
                            [4,13,40,0.618,0.638,0.653,0.15],
                            [4,13,45,0.627,0.642,0.672,0.127],
                            [4,13,50,0.593,0.613,0.624,0.201],
                            [4,13,55,0.598,0.619,0.629,0.16],
                            [4,14,0,0.633,0.652,0.655,0.18],
                            [4,14,5,0.612,0.636,0.647,0.136],
                            [4,14,10,0.64,0.662,0.676,0.124],
                            [4,14,15,0.648,0.657,0.681,0.109],
                            [4,14,20,0.641,0.646,0.679,0.098],
                            [4,14,25,0.654,0.653,0.687,0.126],
                            [4,14,30,0.625,0.628,0.649,0.143],
                            [4,14,35,0.649,0.669,0.682,0.111],
                            [4,14,40,0.638,0.644,0.664,0.144],
                            [4,14,45,0.623,0.63,0.665,0.15],
                            [4,14,50,0.64,0.656,0.678,0.198],
                            [4,14,55,0.674,0.689,0.706,0.154],
                            [4,15,0,0.632,0.647,0.672,0.116],
                            [4,15,5,0.608,0.627,0.654,0.11],
                            [4,15,10,0.634,0.649,0.67,0.093],
                            [4,15,15,0.666,0.681,0.715,0.056],
                            [4,15,20,0.653,0.655,0.691,0.135],
                            [4,15,25,0.666,0.671,0.706,0.072],
                            [4,15,30,0.653,0.672,0.689,0.096],
                            [4,15,35,0.692,0.713,0.74,0.054],
                            [4,15,40,0.643,0.655,0.687,0.077],
                            [4,15,45,0.701,0.72,0.742,0.107],
                            [4,15,50,0.686,0.711,0.736,0.084],
                            [4,15,55,0.678,0.701,0.722,0.026],
                            [4,16,0,0.677,0.7,0.724,0.04],
                            [4,16,5,0.7,0.711,0.739,0.0],
                            [4,16,10,0.692,0.709,0.736,0.04],
                            [4,16,15,0.706,0.726,0.739,0.056],
                            [4,16,20,0.731,0.751,0.779,0.066],
                            [4,16,25,0.679,0.688,0.73,0.08],
                            [4,16,30,0.676,0.687,0.722,0.084],
                            [4,16,35,0.691,0.705,0.73,0.03],
                            [4,16,40,0.683,0.708,0.74,0.074],
                            [4,16,45,0.667,0.689,0.735,0.087],
                            [4,16,50,0.67,0.685,0.73,0.061],
                            [4,16,55,0.657,0.672,0.711,0.113],
                            [4,17,0,0.637,0.66,0.683,0.036],
                            [4,17,5,0.631,0.648,0.681,0.048],
                            [4,17,10,0.62,0.637,0.663,0.104],
                            [4,17,15,0.634,0.649,0.678,0.093],
                            [4,17,20,0.624,0.649,0.678,0.164],
                            [4,17,25,0.641,0.658,0.698,0.179],
                            [4,17,30,0.683,0.686,0.712,0.278],
                            [4,17,35,0.745,0.743,0.769,0.257],
                            [4,17,40,0.734,0.732,0.77,0.21],
                            [4,17,45,0.76,0.761,0.802,0.268],
                            [4,17,50,0.768,0.789,0.807,0.235],
                            [4,17,55,0.787,0.807,0.827,0.242],
                            [4,18,0,0.79,0.804,0.837,0.298],
                            [4,18,5,0.795,0.804,0.833,0.349],
                            [4,18,10,0.831,0.836,0.853,0.403],
                            [4,18,15,0.863,0.867,0.879,0.432],
                            [4,18,20,0.87,0.879,0.894,0.429],
                            [4,18,25,0.888,0.901,0.917,0.508],
                            [4,18,30,0.915,0.931,0.933,0.634],
                            [4,18,35,0.934,0.941,0.95,0.675],
                            [4,18,40,0.964,0.974,0.984,0.681],
                            [4,18,45,0.949,0.967,0.96,0.705],
                            [4,18,50,0.931,0.944,0.958,0.72],
                            [4,18,55,0.933,0.95,0.951,0.702],
                            [4,19,0,0.951,0.959,0.951,0.743],
                            [4,19,5,0.922,0.938,0.926,0.658],
                            [4,19,10,0.936,0.954,0.945,0.681],
                            [4,19,15,0.95,0.973,0.962,0.704],
                            [4,19,20,0.971,0.979,0.983,0.678],
                            [4,19,25,1.0,1.0,1.0,0.637],
                            [4,19,30,0.959,0.97,0.965,0.657],
                            [4,19,35,0.958,0.969,0.964,0.694],
                            [4,19,40,0.965,0.97,0.961,0.726],
                            [4,19,45,0.966,0.972,0.964,0.658],
                            [4,19,50,0.965,0.975,0.964,0.733],
                            [4,19,55,0.967,0.97,0.967,0.688],
                            [4,20,0,0.955,0.953,0.953,0.721],
                            [4,20,5,0.941,0.934,0.942,0.819],
                            [4,20,10,0.922,0.933,0.929,0.76],
                            [4,20,15,0.92,0.925,0.919,0.671],
                            [4,20,20,0.91,0.901,0.913,0.719],
                            [4,20,25,0.918,0.914,0.904,0.684],
                            [4,20,30,0.906,0.892,0.89,0.725],
                            [4,20,35,0.911,0.899,0.892,0.712],
                            [4,20,40,0.871,0.86,0.863,0.765],
                            [4,20,45,0.884,0.871,0.868,0.762],
                            [4,20,50,0.867,0.861,0.856,0.791],
                            [4,20,55,0.865,0.851,0.838,0.766],
                            [4,21,0,0.839,0.819,0.822,0.812],
                            [4,21,5,0.786,0.781,0.781,0.76],
                            [4,21,10,0.781,0.758,0.765,0.769],
                            [4,21,15,0.765,0.748,0.753,0.822],
                            [4,21,20,0.752,0.742,0.744,0.81],
                            [4,21,25,0.749,0.733,0.742,0.831],
                            [4,21,30,0.735,0.725,0.73,0.75],
                            [4,21,35,0.719,0.716,0.716,0.832],
                            [4,21,40,0.705,0.702,0.695,0.892],
                            [4,21,45,0.691,0.692,0.696,0.828],
                            [4,21,50,0.706,0.704,0.708,0.799],
                            [4,21,55,0.696,0.676,0.692,0.802],
                            [4,22,0,0.682,0.673,0.676,0.788],
                            [4,22,5,0.668,0.653,0.654,0.788],
                            [4,22,10,0.669,0.664,0.667,0.79],
                            [4,22,15,0.671,0.669,0.67,0.862],
                            [4,22,20,0.657,0.663,0.661,0.862],
                            [4,22,25,0.619,0.63,0.627,0.879],
                            [4,22,30,0.631,0.63,0.635,0.815],
                            [4,22,35,0.637,0.631,0.636,0.888],
                            [4,22,40,0.638,0.634,0.636,0.868],
                            [4,22,45,0.629,0.637,0.625,0.838],
                            [4,22,50,0.636,0.628,0.627,0.932],
                            [4,22,55,0.583,0.586,0.593,0.903],
                            [4,23,0,0.581,0.589,0.568,0.889],
                            [4,23,5,0.587,0.598,0.575,0.862],
                            [4,23,10,0.581,0.599,0.573,0.877],
                            [4,23,15,0.559,0.576,0.552,0.954],
                            [4,23,20,0.526,0.525,0.516,0.891],
                            [4,23,25,0.539,0.548,0.537,0.859],
                            [4,23,30,0.535,0.547,0.524,0.844],
                            [4,23,35,0.524,0.521,0.521,0.939],
                            [4,23,40,0.517,0.521,0.519,0.892],
                            [4,23,45,0.485,0.487,0.487,0.931],
                            [4,23,50,0.478,0.477,0.47,0.925],
                            [4,23,55,0.465,0.46,0.457,0.874],
                            [5,0,0,0.611,0.59,0.597,0.902],
                            [5,0,5,0.611,0.593,0.604,0.967],
                            [5,0,10,0.578,0.563,0.567,0.92],
                            [5,0,15,0.597,0.575,0.587,0.892],
                            [5,0,20,0.587,0.567,0.582,0.934],
                            [5,0,25,0.566,0.546,0.572,0.934],
                            [5,0,30,0.568,0.551,0.576,0.92],
                            [5,0,35,0.559,0.536,0.554,0.901],
                            [5,0,40,0.538,0.521,0.535,0.973],
                            [5,0,45,0.523,0.502,0.521,1.0],
                            [5,0,50,0.512,0.491,0.514,0.91],
                            [5,0,55,0.506,0.487,0.503,0.935],
                            [5,1,0,0.487,0.475,0.486,0.89],
                            [5,1,5,0.472,0.462,0.464,0.846],
                            [5,1,10,0.464,0.45,0.456,0.894],
                            [5,1,15,0.451,0.435,0.444,0.908],
                            [5,1,20,0.429,0.41,0.413,0.875],
                            [5,1,25,0.435,0.413,0.412,0.846],
                            [5,1,30,0.425,0.402,0.405,0.829],
                            [5,1,35,0.402,0.384,0.399,0.821],
                            [5,1,40,0.397,0.374,0.392,0.838],
                            [5,1,45,0.391,0.363,0.377,0.837],
                            [5,1,50,0.385,0.359,0.37,0.789],
                            [5,1,55,0.389,0.359,0.369,0.761],
                            [5,2,0,0.375,0.351,0.361,0.755],
                            [5,2,5,0.365,0.335,0.348,0.762],
                            [5,2,10,0.347,0.318,0.324,0.851],
                            [5,2,15,0.344,0.321,0.34,0.811],
                            [5,2,20,0.336,0.307,0.325,0.817],
                            [5,2,25,0.333,0.3,0.316,0.783],
                            [5,2,30,0.324,0.291,0.314,0.806],
                            [5,2,35,0.316,0.285,0.304,0.768],
                            [5,2,40,0.31,0.289,0.299,0.749],
                            [5,2,45,0.308,0.285,0.293,0.756],
                            [5,2,50,0.294,0.269,0.287,0.712],
                            [5,2,55,0.3,0.265,0.284,0.743],
                            [5,3,0,0.291,0.255,0.269,0.739],
                            [5,3,5,0.274,0.243,0.257,0.718],
                            [5,3,10,0.271,0.247,0.258,0.756],
                            [5,3,15,0.266,0.238,0.251,0.786],
                            [5,3,20,0.264,0.24,0.248,0.762],
                            [5,3,25,0.261,0.238,0.247,0.748],
                            [5,3,30,0.254,0.223,0.242,0.724],
                            [5,3,35,0.254,0.224,0.234,0.686],
                            [5,3,40,0.245,0.212,0.224,0.676],
                            [5,3,45,0.246,0.21,0.227,0.717],
                            [5,3,50,0.235,0.203,0.215,0.72],
                            [5,3,55,0.237,0.203,0.216,0.699],
                            [5,4,0,0.237,0.198,0.221,0.724],
                            [5,4,5,0.226,0.193,0.215,0.718],
                            [5,4,10,0.224,0.187,0.209,0.729],
                            [5,4,15,0.229,0.184,0.212,0.697],
                            [5,4,20,0.225,0.182,0.203,0.645],
                            [5,4,25,0.226,0.19,0.212,0.656],
                            [5,4,30,0.214,0.177,0.206,0.693],
                            [5,4,35,0.202,0.157,0.192,0.664],
                            [5,4,40,0.209,0.166,0.184,0.662],
                            [5,4,45,0.205,0.163,0.181,0.686],
                            [5,4,50,0.216,0.175,0.194,0.692],
                            [5,4,55,0.218,0.18,0.2,0.687],
                            [5,5,0,0.218,0.189,0.211,0.656],
                            [5,5,5,0.21,0.186,0.205,0.613],
                            [5,5,10,0.211,0.186,0.196,0.641],
                            [5,5,15,0.194,0.175,0.195,0.613],
                            [5,5,20,0.184,0.161,0.184,0.588],
                            [5,5,25,0.169,0.141,0.161,0.587],
                            [5,5,30,0.147,0.124,0.136,0.57],
                            [5,5,35,0.12,0.113,0.113,0.517],
                            [5,5,40,0.088,0.085,0.072,0.557],
                            [5,5,45,0.061,0.048,0.053,0.51],
                            [5,5,50,0.041,0.04,0.043,0.544],
                            [5,5,55,0.011,0.015,0.01,0.507],
                            [5,6,0,0.0,0.0,0.0,0.493],
                            [5,6,5,0.01,0.004,0.002,0.499],
                            [5,6,10,0.037,0.016,0.032,0.534],
                            [5,6,15,0.025,0.005,0.024,0.467],
                            [5,6,20,0.033,0.016,0.02,0.456],
                            [5,6,25,0.037,0.023,0.025,0.447],
                            [5,6,30,0.014,0.004,0.009,0.407],
                            [5,6,35,0.03,0.009,0.017,0.426],
                            [5,6,40,0.03,0.019,0.022,0.415],
                            [5,6,45,0.038,0.021,0.027,0.334],
                            [5,6,50,0.032,0.02,0.041,0.343],
                            [5,6,55,0.041,0.014,0.046,0.339],
                            [5,7,0,0.047,0.034,0.049,0.379],
                            [5,7,5,0.041,0.032,0.045,0.356],
                            [5,7,10,0.077,0.068,0.083,0.326],
                            [5,7,15,0.075,0.073,0.089,0.366],
                            [5,7,20,0.092,0.087,0.103,0.326],
                            [5,7,25,0.081,0.082,0.089,0.285],
                            [5,7,30,0.086,0.088,0.103,0.311],
                            [5,7,35,0.083,0.096,0.106,0.285],
                            [5,7,40,0.071,0.08,0.091,0.291],
                            [5,7,45,0.071,0.066,0.089,0.287],
                            [5,7,50,0.075,0.072,0.08,0.254],
                            [5,7,55,0.1,0.09,0.098,0.248],
                            [5,8,0,0.124,0.113,0.123,0.241],
                            [5,8,5,0.146,0.129,0.159,0.226],
                            [5,8,10,0.161,0.146,0.184,0.253],
                            [5,8,15,0.174,0.157,0.194,0.212],
                            [5,8,20,0.177,0.164,0.193,0.194],
                            [5,8,25,0.176,0.153,0.19,0.205],
                            [5,8,30,0.2,0.175,0.198,0.219],
                            [5,8,35,0.2,0.182,0.196,0.126],
                            [5,8,40,0.231,0.216,0.247,0.139],
                            [5,8,45,0.229,0.212,0.245,0.169],
                            [5,8,50,0.201,0.185,0.218,0.153],
                            [5,8,55,0.234,0.211,0.242,0.146],
                            [5,9,0,0.217,0.203,0.23,0.086],
                            [5,9,5,0.218,0.206,0.242,0.077],
                            [5,9,10,0.259,0.236,0.278,0.108],
                            [5,9,15,0.25,0.23,0.277,0.089],
                            [5,9,20,0.259,0.239,0.28,0.101],
                            [5,9,25,0.276,0.254,0.299,0.065],
                            [5,9,30,0.258,0.243,0.267,0.059],
                            [5,9,35,0.269,0.24,0.273,0.111],
                            [5,9,40,0.3,0.275,0.316,0.09],
                            [5,9,45,0.263,0.24,0.279,0.05],
                            [5,9,50,0.29,0.273,0.31,0.078],
                            [5,9,55,0.29,0.258,0.296,0.097],
                            [5,10,0,0.278,0.247,0.293,0.031],
                            [5,10,5,0.287,0.267,0.307,0.018],
                            [5,10,10,0.316,0.288,0.336,0.065],
                            [5,10,15,0.306,0.279,0.323,0.077],
                            [5,10,20,0.305,0.278,0.323,0.035],
                            [5,10,25,0.305,0.282,0.325,0.043],
                            [5,10,30,0.308,0.287,0.321,0.1],
                            [5,10,35,0.298,0.273,0.312,0.0],
                            [5,10,40,0.302,0.276,0.317,0.023],
                            [5,10,45,0.302,0.272,0.307,0.017],
                            [5,10,50,0.31,0.273,0.308,0.009],
                            [5,10,55,0.315,0.284,0.315,0.017],
                            [5,11,0,0.338,0.302,0.328,0.02],
                            [5,11,5,0.329,0.291,0.326,0.017],
                            [5,11,10,0.325,0.289,0.321,0.094],
                            [5,11,15,0.343,0.312,0.332,0.14],
                            [5,11,20,0.349,0.335,0.35,0.12],
                            [5,11,25,0.352,0.321,0.354,0.134],
                            [5,11,30,0.351,0.328,0.349,0.104],
                            [5,11,35,0.382,0.351,0.372,0.19],
                            [5,11,40,0.38,0.353,0.389,0.169],
                            [5,11,45,0.389,0.367,0.386,0.11],
                            [5,11,50,0.394,0.377,0.393,0.18],
                            [5,11,55,0.423,0.402,0.429,0.164],
                            [5,12,0,0.411,0.389,0.407,0.169],
                            [5,12,5,0.413,0.399,0.412,0.281],
                            [5,12,10,0.4,0.387,0.407,0.237],
                            [5,12,15,0.41,0.389,0.399,0.281],
                            [5,12,20,0.397,0.391,0.408,0.264],
                            [5,12,25,0.411,0.401,0.423,0.27],
                            [5,12,30,0.402,0.398,0.409,0.251],
                            [5,12,35,0.409,0.385,0.396,0.247],
                            [5,12,40,0.42,0.391,0.428,0.297],
                            [5,12,45,0.425,0.409,0.442,0.234],
                            [5,12,50,0.415,0.396,0.434,0.184],
                            [5,12,55,0.42,0.403,0.424,0.232],
                            [5,13,0,0.408,0.375,0.414,0.27],
                            [5,13,5,0.408,0.372,0.409,0.245],
                            [5,13,10,0.422,0.403,0.43,0.257],
                            [5,13,15,0.412,0.392,0.417,0.295],
                            [5,13,20,0.446,0.436,0.463,0.3],
                            [5,13,25,0.453,0.437,0.466,0.31],
                            [5,13,30,0.432,0.413,0.436,0.287],
                            [5,13,35,0.429,0.406,0.433,0.304],
                            [5,13,40,0.461,0.427,0.454,0.272],
                            [5,13,45,0.462,0.432,0.463,0.276],
                            [5,13,50,0.459,0.437,0.452,0.288],
                            [5,13,55,0.436,0.422,0.435,0.277],
                            [5,14,0,0.455,0.445,0.46,0.293],
                            [5,14,5,0.431,0.425,0.429,0.269],
                            [5,14,10,0.441,0.421,0.441,0.28],
                            [5,14,15,0.445,0.425,0.437,0.256],
                            [5,14,20,0.447,0.426,0.44,0.222],
                            [5,14,25,0.452,0.43,0.447,0.216],
                            [5,14,30,0.447,0.411,0.438,0.247],
                            [5,14,35,0.455,0.434,0.457,0.226],
                            [5,14,40,0.436,0.423,0.446,0.304],
                            [5,14,45,0.44,0.439,0.452,0.262],
                            [5,14,50,0.46,0.447,0.481,0.266],
                            [5,14,55,0.449,0.429,0.468,0.327],
                            [5,15,0,0.469,0.445,0.483,0.249],
                            [5,15,5,0.466,0.441,0.47,0.19],
                            [5,15,10,0.457,0.436,0.473,0.194],
                            [5,15,15,0.456,0.45,0.467,0.202],
                            [5,15,20,0.472,0.464,0.493,0.229],
                            [5,15,25,0.462,0.444,0.476,0.255],
                            [5,15,30,0.441,0.431,0.455,0.239],
                            [5,15,35,0.458,0.444,0.478,0.22],
                            [5,15,40,0.504,0.482,0.502,0.193],
                            [5,15,45,0.49,0.461,0.496,0.214],
                            [5,15,50,0.494,0.473,0.507,0.23],
                            [5,15,55,0.477,0.457,0.486,0.2],
                            [5,16,0,0.477,0.458,0.484,0.216],
                            [5,16,5,0.482,0.482,0.499,0.258],
                            [5,16,10,0.488,0.476,0.506,0.121],
                            [5,16,15,0.466,0.465,0.504,0.164],
                            [5,16,20,0.489,0.486,0.512,0.235],
                            [5,16,25,0.498,0.495,0.519,0.168],
                            [5,16,30,0.515,0.51,0.532,0.209],
                            [5,16,35,0.499,0.504,0.526,0.202],
                            [5,16,40,0.509,0.51,0.531,0.242],
                            [5,16,45,0.51,0.508,0.535,0.273],
                            [5,16,50,0.515,0.506,0.537,0.272],
                            [5,16,55,0.513,0.5,0.529,0.297],
                            [5,17,0,0.527,0.538,0.547,0.25],
                            [5,17,5,0.538,0.548,0.565,0.293],
                            [5,17,10,0.539,0.546,0.577,0.319],
                            [5,17,15,0.553,0.561,0.583,0.292],
                            [5,17,20,0.57,0.573,0.601,0.371],
                            [5,17,25,0.614,0.605,0.63,0.387],
                            [5,17,30,0.641,0.628,0.671,0.438],
                            [5,17,35,0.656,0.663,0.693,0.484],
                            [5,17,40,0.692,0.694,0.711,0.477],
                            [5,17,45,0.739,0.738,0.745,0.521],
                            [5,17,50,0.764,0.784,0.787,0.568],
                            [5,17,55,0.795,0.806,0.816,0.604],
                            [5,18,0,0.853,0.864,0.856,0.628],
                            [5,18,5,0.863,0.879,0.862,0.644],
                            [5,18,10,0.899,0.908,0.888,0.699],
                            [5,18,15,0.925,0.935,0.939,0.666],
                            [5,18,20,0.939,0.951,0.949,0.74],
                            [5,18,25,0.969,0.962,0.973,0.791],
                            [5,18,30,0.971,0.979,0.972,0.681],
                            [5,18,35,0.977,0.976,0.976,0.819],
                            [5,18,40,0.999,1.0,1.0,0.78],
                            [5,18,45,1.0,0.998,0.997,0.794],
                            [5,18,50,0.989,0.986,0.984,0.813],
                            [5,18,55,0.984,0.968,0.999,0.722],
                            [5,19,0,0.984,0.975,0.991,0.621],
                            [5,19,5,0.972,0.957,0.976,0.671],
                            [5,19,10,0.978,0.955,0.973,0.684],
                            [5,19,15,0.974,0.946,0.971,0.619],
                            [5,19,20,0.965,0.948,0.968,0.596],
                            [5,19,25,0.954,0.949,0.96,0.678],
                            [5,19,30,0.94,0.938,0.953,0.6],
                            [5,19,35,0.928,0.918,0.94,0.613],
                            [5,19,40,0.938,0.924,0.945,0.655],
                            [5,19,45,0.94,0.923,0.934,0.747],
                            [5,19,50,0.932,0.927,0.936,0.713],
                            [5,19,55,0.916,0.905,0.922,0.748],
                            [5,20,0,0.884,0.886,0.888,0.699],
                            [5,20,5,0.898,0.889,0.907,0.671],
                            [5,20,10,0.887,0.876,0.895,0.681],
                            [5,20,15,0.873,0.864,0.887,0.697],
                            [5,20,20,0.883,0.871,0.885,0.679],
                            [5,20,25,0.861,0.852,0.853,0.69],
                            [5,20,30,0.851,0.85,0.855,0.709],
                            [5,20,35,0.851,0.845,0.857,0.709],
                            [5,20,40,0.842,0.846,0.863,0.677],
                            [5,20,45,0.847,0.84,0.845,0.74],
                            [5,20,50,0.847,0.836,0.842,0.666],
                            [5,20,55,0.832,0.825,0.835,0.722],
                            [5,21,0,0.824,0.789,0.815,0.706],
                            [5,21,5,0.806,0.772,0.796,0.718],
                            [5,21,10,0.779,0.753,0.775,0.685],
                            [5,21,15,0.749,0.736,0.757,0.714],
                            [5,21,20,0.759,0.746,0.757,0.717],
                            [5,21,25,0.744,0.731,0.749,0.796],
                            [5,21,30,0.733,0.707,0.731,0.774],
                            [5,21,35,0.716,0.706,0.721,0.765],
                            [5,21,40,0.704,0.699,0.71,0.76],
                            [5,21,45,0.709,0.692,0.707,0.773],
                            [5,21,50,0.703,0.674,0.702,0.747],
                            [5,21,55,0.694,0.665,0.681,0.765],
                            [5,22,0,0.685,0.654,0.663,0.763],
                            [5,22,5,0.671,0.651,0.656,0.768],
                            [5,22,10,0.668,0.639,0.658,0.755],
                            [5,22,15,0.666,0.645,0.654,0.758],
                            [5,22,20,0.657,0.634,0.644,0.733],
                            [5,22,25,0.659,0.636,0.648,0.791],
                            [5,22,30,0.656,0.631,0.636,0.764],
                            [5,22,35,0.655,0.631,0.643,0.748],
                            [5,22,40,0.65,0.632,0.639,0.715],
                            [5,22,45,0.644,0.625,0.64,0.804],
                            [5,22,50,0.644,0.622,0.647,0.807],
                            [5,22,55,0.652,0.623,0.646,0.837],
                            [5,23,0,0.639,0.616,0.632,0.898],
                            [5,23,5,0.632,0.616,0.626,0.881],
                            [5,23,10,0.63,0.606,0.616,0.88],
                            [5,23,15,0.624,0.596,0.595,0.826],
                            [5,23,20,0.614,0.579,0.592,0.781],
                            [5,23,25,0.615,0.588,0.596,0.928],
                            [5,23,30,0.601,0.581,0.585,0.853],
                            [5,23,35,0.593,0.568,0.567,0.762],
                            [5,23,40,0.578,0.557,0.566,0.835],
                            [5,23,45,0.566,0.545,0.554,0.835],
                            [5,23,50,0.57,0.547,0.565,0.834],
                            [5,23,55,0.544,0.52,0.532,0.829],
                            [6,0,0,0.586,0.571,0.571,0.868],
                            [6,0,5,0.557,0.556,0.552,0.862],
                            [6,0,10,0.556,0.556,0.563,0.83],
                            [6,0,15,0.543,0.528,0.539,0.792],
                            [6,0,20,0.538,0.525,0.537,0.791],
                            [6,0,25,0.533,0.528,0.528,0.84],
                            [6,0,30,0.527,0.518,0.523,0.806],
                            [6,0,35,0.525,0.513,0.528,0.799],
                            [6,0,40,0.504,0.504,0.508,0.817],
                            [6,0,45,0.493,0.484,0.502,0.782],
                            [6,0,50,0.492,0.473,0.49,0.8],
                            [6,0,55,0.488,0.468,0.478,0.788],
                            [6,1,0,0.475,0.461,0.465,0.809],
                            [6,1,5,0.464,0.455,0.451,0.779],
                            [6,1,10,0.45,0.442,0.44,0.779],
                            [6,1,15,0.444,0.422,0.434,0.768],
                            [6,1,20,0.434,0.416,0.427,0.736],
                            [6,1,25,0.424,0.407,0.417,0.708],
                            [6,1,30,0.415,0.398,0.409,0.729],
                            [6,1,35,0.41,0.395,0.4,0.774],
                            [6,1,40,0.399,0.388,0.392,0.773],
                            [6,1,45,0.395,0.383,0.385,0.755],
                            [6,1,50,0.388,0.379,0.377,0.786],
                            [6,1,55,0.375,0.363,0.372,0.741],
                            [6,2,0,0.387,0.375,0.387,0.776],
                            [6,2,5,0.371,0.359,0.37,0.758],
                            [6,2,10,0.365,0.356,0.359,0.696],
                            [6,2,15,0.361,0.354,0.359,0.716],
                            [6,2,20,0.356,0.345,0.344,0.705],
                            [6,2,25,0.353,0.339,0.34,0.705],
                            [6,2,30,0.343,0.331,0.329,0.67],
                            [6,2,35,0.331,0.322,0.318,0.653],
                            [6,2,40,0.324,0.31,0.315,0.634],
                            [6,2,45,0.325,0.308,0.318,0.674],
                            [6,2,50,0.324,0.304,0.311,0.621],
                            [6,2,55,0.319,0.298,0.31,0.648],
                            [6,3,0,0.311,0.289,0.301,0.693],
                            [6,3,5,0.306,0.287,0.29,0.675],
                            [6,3,10,0.306,0.277,0.29,0.676],
                            [6,3,15,0.302,0.278,0.287,0.699],
                            [6,3,20,0.297,0.278,0.28,0.666],
                            [6,3,25,0.293,0.271,0.275,0.65],
                            [6,3,30,0.286,0.27,0.269,0.655],
                            [6,3,35,0.283,0.266,0.263,0.647],
                            [6,3,40,0.281,0.258,0.257,0.634],
                            [6,3,45,0.27,0.249,0.25,0.7],
                            [6,3,50,0.272,0.252,0.257,0.642],
                            [6,3,55,0.271,0.248,0.252,0.606],
                            [6,4,0,0.268,0.244,0.251,0.625],
                            [6,4,5,0.261,0.238,0.248,0.605],
                            [6,4,10,0.255,0.234,0.237,0.615],
                            [6,4,15,0.254,0.228,0.236,0.596],
                            [6,4,20,0.252,0.225,0.233,0.623],
                            [6,4,25,0.249,0.229,0.232,0.653],
                            [6,4,30,0.247,0.22,0.227,0.642],
                            [6,4,35,0.241,0.217,0.221,0.608],
                            [6,4,40,0.232,0.213,0.22,0.611],
                            [6,4,45,0.211,0.195,0.197,0.634],
                            [6,4,50,0.229,0.209,0.214,0.581],
                            [6,4,55,0.221,0.202,0.209,0.566],
                            [6,5,0,0.217,0.203,0.21,0.612],
                            [6,5,5,0.22,0.204,0.212,0.581],
                            [6,5,10,0.214,0.203,0.205,0.567],
                            [6,5,15,0.195,0.193,0.19,0.554],
                            [6,5,20,0.196,0.193,0.194,0.589],
                            [6,5,25,0.171,0.168,0.165,0.516],
                            [6,5,30,0.147,0.147,0.142,0.52],
                            [6,5,35,0.128,0.122,0.122,0.474],
                            [6,5,40,0.107,0.109,0.104,0.502],
                            [6,5,45,0.125,0.133,0.135,0.463],
                            [6,5,50,0.106,0.107,0.102,0.48],
                            [6,5,55,0.079,0.087,0.079,0.446],
                            [6,6,0,0.069,0.07,0.057,0.495],
                            [6,6,5,0.049,0.045,0.044,0.427],
                            [6,6,10,0.045,0.037,0.029,0.438],
                            [6,6,15,0.035,0.035,0.022,0.445],
                            [6,6,20,0.026,0.03,0.019,0.463],
                            [6,6,25,0.025,0.023,0.022,0.507],
                            [6,6,30,0.018,0.012,0.011,0.407],
                            [6,6,35,0.019,0.017,0.008,0.41],
                            [6,6,40,0.005,0.008,0.003,0.379],
                            [6,6,45,0.006,0.008,0.008,0.365],
                            [6,6,50,0.007,0.002,0.0,0.327],
                            [6,6,55,0.008,0.003,0.008,0.307],
                            [6,7,0,0.0,0.0,0.011,0.323],
                            [6,7,5,0.001,0.01,0.016,0.296],
                            [6,7,10,0.018,0.019,0.026,0.332],
                            [6,7,15,0.025,0.033,0.036,0.346],
                            [6,7,20,0.032,0.047,0.049,0.346],
                            [6,7,25,0.038,0.047,0.05,0.381],
                            [6,7,30,0.045,0.069,0.058,0.37],
                            [6,7,35,0.052,0.065,0.057,0.338],
                            [6,7,40,0.053,0.061,0.062,0.322],
                            [6,7,45,0.059,0.078,0.074,0.326],
                            [6,7,50,0.065,0.083,0.071,0.33],
                            [6,7,55,0.052,0.077,0.058,0.323],
                            [6,8,0,0.064,0.075,0.068,0.274],
                            [6,8,5,0.067,0.072,0.066,0.241],
                            [6,8,10,0.065,0.069,0.075,0.227],
                            [6,8,15,0.076,0.087,0.085,0.189],
                            [6,8,20,0.071,0.075,0.066,0.221],
                            [6,8,25,0.077,0.083,0.073,0.229],
                            [6,8,30,0.082,0.088,0.088,0.191],
                            [6,8,35,0.082,0.09,0.084,0.156],
                            [6,8,40,0.085,0.1,0.09,0.191],
                            [6,8,45,0.081,0.102,0.079,0.171],
                            [6,8,50,0.074,0.095,0.075,0.191],
                            [6,8,55,0.087,0.1,0.085,0.169],
                            [6,9,0,0.089,0.097,0.085,0.175],
                            [6,9,5,0.094,0.099,0.085,0.13],
                            [6,9,10,0.098,0.105,0.091,0.133],
                            [6,9,15,0.085,0.086,0.076,0.143],
                            [6,9,20,0.091,0.088,0.079,0.152],
                            [6,9,25,0.102,0.097,0.082,0.136],
                            [6,9,30,0.088,0.086,0.077,0.133],
                            [6,9,35,0.081,0.08,0.081,0.169],
                            [6,9,40,0.08,0.071,0.071,0.145],
                            [6,9,45,0.092,0.086,0.079,0.104],
                            [6,9,50,0.091,0.084,0.082,0.092],
                            [6,9,55,0.088,0.084,0.079,0.064],
                            [6,10,0,0.07,0.064,0.057,0.088],
                            [6,10,5,0.071,0.063,0.054,0.068],
                            [6,10,10,0.069,0.068,0.061,0.06],
                            [6,10,15,0.078,0.074,0.064,0.0],
                            [6,10,20,0.074,0.061,0.058,0.014],
                            [6,10,25,0.059,0.046,0.052,0.032],
                            [6,10,30,0.07,0.058,0.061,0.094],
                            [6,10,35,0.1,0.087,0.076,0.092],
                            [6,10,40,0.084,0.077,0.063,0.084],
                            [6,10,45,0.074,0.074,0.066,0.077],
                            [6,10,50,0.064,0.07,0.058,0.057],
                            [6,10,55,0.088,0.082,0.083,0.07],
                            [6,11,0,0.079,0.087,0.085,0.106],
                            [6,11,5,0.085,0.098,0.096,0.06],
                            [6,11,10,0.108,0.119,0.115,0.123],
                            [6,11,15,0.1,0.108,0.105,0.144],
                            [6,11,20,0.115,0.122,0.113,0.104],
                            [6,11,25,0.12,0.133,0.125,0.18],
                            [6,11,30,0.115,0.134,0.129,0.182],
                            [6,11,35,0.118,0.13,0.132,0.123],
                            [6,11,40,0.128,0.146,0.152,0.14],
                            [6,11,45,0.136,0.155,0.155,0.163],
                            [6,11,50,0.138,0.16,0.158,0.228],
                            [6,11,55,0.145,0.155,0.147,0.153],
                            [6,12,0,0.104,0.114,0.115,0.212],
                            [6,12,5,0.11,0.12,0.126,0.206],
                            [6,12,10,0.11,0.111,0.121,0.239],
                            [6,12,15,0.097,0.112,0.123,0.248],
                            [6,12,20,0.109,0.122,0.123,0.255],
                            [6,12,25,0.098,0.108,0.107,0.304],
                            [6,12,30,0.099,0.1,0.103,0.248],
                            [6,12,35,0.097,0.106,0.1,0.301],
                            [6,12,40,0.099,0.096,0.105,0.289],
                            [6,12,45,0.118,0.114,0.11,0.308],
                            [6,12,50,0.096,0.092,0.095,0.255],
                            [6,12,55,0.09,0.09,0.106,0.22],
                            [6,13,0,0.086,0.085,0.093,0.293],
                            [6,13,5,0.076,0.07,0.08,0.278],
                            [6,13,10,0.081,0.079,0.084,0.297],
                            [6,13,15,0.104,0.108,0.105,0.277],
                            [6,13,20,0.095,0.086,0.093,0.301],
                            [6,13,25,0.09,0.083,0.088,0.26],
                            [6,13,30,0.09,0.083,0.085,0.294],
                            [6,13,35,0.075,0.076,0.074,0.298],
                            [6,13,40,0.09,0.076,0.082,0.303],
                            [6,13,45,0.08,0.068,0.076,0.273],
                            [6,13,50,0.074,0.06,0.064,0.285],
                            [6,13,55,0.088,0.076,0.08,0.248],
                            [6,14,0,0.082,0.066,0.078,0.335],
                            [6,14,5,0.08,0.065,0.077,0.298],
                            [6,14,10,0.087,0.079,0.086,0.313],
                            [6,14,15,0.078,0.068,0.078,0.259],
                            [6,14,20,0.078,0.062,0.08,0.265],
                            [6,14,25,0.076,0.057,0.085,0.258],
                            [6,14,30,0.073,0.052,0.084,0.274],
                            [6,14,35,0.081,0.055,0.09,0.316],
                            [6,14,40,0.096,0.076,0.1,0.293],
                            [6,14,45,0.092,0.074,0.101,0.294],
                            [6,14,50,0.091,0.081,0.091,0.24],
                            [6,14,55,0.098,0.081,0.096,0.269],
                            [6,15,0,0.105,0.085,0.108,0.269],
                            [6,15,5,0.12,0.102,0.125,0.293],
                            [6,15,10,0.111,0.096,0.117,0.266],
                            [6,15,15,0.12,0.096,0.12,0.273],
                            [6,15,20,0.125,0.103,0.121,0.286],
                            [6,15,25,0.139,0.115,0.138,0.206],
                            [6,15,30,0.145,0.124,0.151,0.266],
                            [6,15,35,0.156,0.127,0.151,0.29],
                            [6,15,40,0.158,0.142,0.168,0.262],
                            [6,15,45,0.16,0.146,0.174,0.273],
                            [6,15,50,0.184,0.174,0.188,0.247],
                            [6,15,55,0.182,0.172,0.197,0.323],
                            [6,16,0,0.207,0.193,0.22,0.277],
                            [6,16,5,0.224,0.215,0.246,0.307],
                            [6,16,10,0.232,0.233,0.249,0.327],
                            [6,16,15,0.234,0.236,0.246,0.284],
                            [6,16,20,0.241,0.239,0.246,0.332],
                            [6,16,25,0.257,0.257,0.26,0.326],
                            [6,16,30,0.283,0.288,0.301,0.288],
                            [6,16,35,0.29,0.294,0.302,0.308],
                            [6,16,40,0.303,0.302,0.31,0.313],
                            [6,16,45,0.324,0.325,0.339,0.368],
                            [6,16,50,0.338,0.341,0.354,0.384],
                            [6,16,55,0.355,0.363,0.376,0.352],
                            [6,17,0,0.41,0.413,0.407,0.429],
                            [6,17,5,0.432,0.438,0.447,0.413],
                            [6,17,10,0.446,0.469,0.466,0.451],
                            [6,17,15,0.486,0.495,0.504,0.46],
                            [6,17,20,0.49,0.499,0.518,0.523],
                            [6,17,25,0.522,0.544,0.55,0.538],
                            [6,17,30,0.559,0.563,0.592,0.519],
                            [6,17,35,0.601,0.608,0.623,0.512],
                            [6,17,40,0.661,0.661,0.684,0.595],
                            [6,17,45,0.698,0.704,0.722,0.591],
                            [6,17,50,0.73,0.736,0.755,0.607],
                            [6,17,55,0.766,0.773,0.784,0.705],
                            [6,18,0,0.82,0.831,0.837,0.731],
                            [6,18,5,0.861,0.864,0.876,0.727],
                            [6,18,10,0.907,0.916,0.918,0.755],
                            [6,18,15,0.919,0.942,0.941,0.742],
                            [6,18,20,0.954,0.962,0.965,0.737],
                            [6,18,25,0.986,0.994,0.993,0.775],
                            [6,18,30,0.984,0.983,0.995,0.859],
                            [6,18,35,1.0,0.999,1.0,0.878],
                            [6,18,40,0.992,1.0,0.987,0.749],
                            [6,18,45,0.985,0.98,0.98,0.783],
                            [6,18,50,0.977,0.966,0.961,0.682],
                            [6,18,55,0.975,0.971,0.972,0.598],
                            [6,19,0,0.972,0.971,0.968,0.65],
                            [6,19,5,0.973,0.97,0.967,0.638],
                            [6,19,10,0.974,0.97,0.959,0.619],
                            [6,19,15,0.96,0.957,0.962,0.605],
                            [6,19,20,0.962,0.953,0.96,0.57],
                            [6,19,25,0.942,0.937,0.949,0.606],
                            [6,19,30,0.953,0.946,0.949,0.578],
                            [6,19,35,0.949,0.93,0.949,0.537],
                            [6,19,40,0.951,0.94,0.949,0.541],
                            [6,19,45,0.938,0.928,0.937,0.631],
                            [6,19,50,0.939,0.937,0.94,0.586],
                            [6,19,55,0.933,0.927,0.938,0.586],
                            [6,20,0,0.929,0.92,0.932,0.578],
                            [6,20,5,0.917,0.913,0.905,0.628],
                            [6,20,10,0.918,0.907,0.905,0.636],
                            [6,20,15,0.903,0.886,0.896,0.649],
                            [6,20,20,0.913,0.893,0.905,0.665],
                            [6,20,25,0.911,0.902,0.907,0.704],
                            [6,20,30,0.886,0.877,0.879,0.742],
                            [6,20,35,0.899,0.895,0.887,0.737],
                            [6,20,40,0.879,0.877,0.868,0.765],
                            [6,20,45,0.874,0.872,0.872,0.784],
                            [6,20,50,0.875,0.861,0.864,0.816],
                            [6,20,55,0.857,0.861,0.859,0.784],
                            [6,21,0,0.837,0.835,0.835,0.739],
                            [6,21,5,0.838,0.838,0.845,0.769],
                            [6,21,10,0.825,0.815,0.83,0.813],
                            [6,21,15,0.827,0.815,0.829,0.819],
                            [6,21,20,0.819,0.804,0.82,0.829],
                            [6,21,25,0.801,0.791,0.807,0.829],
                            [6,21,30,0.806,0.806,0.819,0.852],
                            [6,21,35,0.816,0.81,0.822,0.868],
                            [6,21,40,0.825,0.821,0.832,0.922],
                            [6,21,45,0.818,0.816,0.818,0.94],
                            [6,21,50,0.827,0.818,0.829,0.94],
                            [6,21,55,0.816,0.812,0.825,0.936],
                            [6,22,0,0.831,0.827,0.822,0.905],
                            [6,22,5,0.833,0.826,0.825,0.896],
                            [6,22,10,0.835,0.828,0.831,0.881],
                            [6,22,15,0.803,0.795,0.814,0.942],
                            [6,22,20,0.817,0.816,0.825,0.899],
                            [6,22,25,0.821,0.822,0.82,0.943],
                            [6,22,30,0.807,0.813,0.8,0.969],
                            [6,22,35,0.802,0.803,0.803,0.958],
                            [6,22,40,0.8,0.805,0.807,1.0],
                            [6,22,45,0.777,0.782,0.779,0.941],
                            [6,22,50,0.767,0.778,0.776,0.959],
                            [6,22,55,0.768,0.773,0.767,0.898],
                            [6,23,0,0.752,0.767,0.768,0.939],
                            [6,23,5,0.75,0.759,0.758,0.962],
                            [6,23,10,0.751,0.754,0.743,0.921],
                            [6,23,15,0.743,0.735,0.743,0.965],
                            [6,23,20,0.722,0.719,0.73,0.936],
                            [6,23,25,0.705,0.711,0.717,0.923],
                            [6,23,30,0.697,0.702,0.699,0.985],
                            [6,23,35,0.681,0.687,0.686,0.961],
                            [6,23,40,0.676,0.68,0.667,0.97],
                            [6,23,45,0.657,0.663,0.651,0.912],
                            [6,23,50,0.64,0.648,0.64,0.979],
                            [6,23,55,0.631,0.635,0.627,0.896]],columns =['WeekDay','Hour','Min','IA','IB','IV','IN'])
    
    return default