# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/02_colorizer.ipynb.

# %% auto 0
__all__ = ['system_stream', 'colorize', 'SystemStream', 'ColorStream', 'AnsiColorizer']

# %% ../nbs/02_colorizer.ipynb 4
import contextlib
import io
import re
import sys

from .ansicolor import Fore, Back, Style, AnsiColor


class SystemStream:
    _instance = None

    def __new__(cls):
        if cls._instance is None:
            for file in (sys.stdout, sys.stderr):
                if '__' in repr(file):
                    raise RuntimeError(f'{file} has been modified')
            cls._instance = super().__new__(cls)
            cls.orig_stdout, cls.orig_stderr = sys.stdout, sys.stderr
        return cls._instance
    
    def __init__(self):
        self._affected = False

    def restore(self):
        sys.stdout, sys.stderr = self.orig_stdout, self.orig_stderr
        
    @property    
    def stdout(self): return self.orig_stdout
    
    @property
    def stderr(self): return self.orig_stderr
            
system_stream = SystemStream()


class ColorStream(contextlib.ContextDecorator):
    '''
    Enables context managers to work as decorators 
    to colorize the `sys.stdout` or `sys.stderr`
    
    Some usage:
    
    ```python
        with ColorStream(fore='red'):
            print('text')        

        @ColorStream(fore=Fore.dark_orange)
        def foo():
            print('FOO')
    ```
    '''

    
    def __init__(self, fore=None, back=None, style=None, autoreset=True, streams='stdout'):
        '''
        Parameters
        ----------
        fore : `Fore`, str, int, optional
            Foreground color. Could be hex, rgb string or tuple, `Fore`, 8-bits color

        back : `Back`, str, rgb, int, optional
            Background color, Could be hex, rgb string or tuple, `Back`, 8-bits color

        style : `Style`, str, tuple, optional
            Text style. Seee `Style.available`.
            
        autoreset: bool
        
        streams: str
            One of {stdout, stderr}
        '''
        self.ansi = AnsiColor(fore, back, style)
        
        self._global_flag = False
        self.autoreset = autoreset
        self._ori_reset = autoreset
        
        if streams not in ('stdout', 'stderr'): 
            raise ValueError(f'{streams} is not acceptable')

        self.stream = streams
        self.ori_file = None
        self.file = getattr(sys, streams)

            
    def affect_global_stream(self):
        if not self._global_flag:
            self.__enter__()
            self._ori_reset = self.autoreset
            
        self.autoreset = False
        self._global_flag = True
        
    def unAffect_global_stream(self):
        self.__exit__()
        self._global_flag = False
        self.autoreset = self._ori_reset
        
    def __enter__(self):
        if self.ori_file is None:
            self.ori_file = getattr(system_stream, self.stream)
            setattr(sys, self.stream, self)


    def __exit__(self, *args):
        if self.ori_file:
            setattr(sys, self.stream, self.ori_file)
            self.ori_file = None

    def write(self, text):
        reset = Style.reset_all if self.autoreset else ''
        self.file.write(f'{self.ansi.ansi_fmt}{text}{reset}')
        self.flush()

    def flush(self):
        self.file.flush()
      
    
class AnsiColorizer:
    '''
    
    For `text` parameter,  you can add color tag. Start with \<tag\> end with \</tag\>.
    
    Some usage:
    
    ```python
        text = 'something'

        # 1. blue text tag: 

            f'<blue>{text}</fg>'
            f'<blue>{text}</blue>'

        # 2. specify fg:
            f'<fg red>{text}</fg>'

        # 3. specify bg:
            f'<bg purple>{text}</bg>'

        # 4. style:
            f'<bold>{text}</bold>'

        # 5. support rgb format:
            f'<255, 255, 255>{text}</fg>'
            f'<fg 255, 255, 255>{text}</fg>'
            f'<bg 255, 255, 255>{text}</bg>'

        # 6. support hex format:
            f'<#FFFFFF>{text}</fg>'
            f'<fg #FFFFFF>{text}</fg>'
            f'<bg #FFFFFF>{text}</bg>'

        # 7. support 8-bits color:
            f'<50>{text}</fg>'
            f'<fg 50>{text}</fg>'
            f'<bg 50>{text}</bg>'

        # 8. mix of above is ok:
            f'<fg red>{text}--<bg green>{text}--</fg>{text}--</bg>{text}'
    ```
    '''
    
    _regex_tag = re.compile(r"<([/\w\s,#]*)>")
    
    def __call__(self, text, fore=None, back=None, style=None, raw=False, strip=False):
        '''
        Parameters
        ----------
        text: str
        
        fore : `Fore`, str, int, optional
            Foreground color. Could be hex, rgb string or tuple, `Fore`, 8-bits color

        back : `Back`, str, rgb, int, optional
            Background color, Could be hex, rgb string or tuple, `Back`, 8-bits color

        style : `Style`, str, tuple, optional
            Text style. Seee `Style.available`.
            
        raw : bool
            return raw text
            
        strip : bool
            remove <tag>
        '''  
        if raw:     return text
        elif strip: return self.strip(text)
        elif any((fore, back, style)):
            ansi = AnsiColor(fore, back, style)
            return f'{ansi.ansi_fmt}{text}{Style.end}'
        else: return self.colorize(text)

    def get_ansi(self, tag):
        tag = tag.lower()

        if tag in Style:
            return Style[tag]

        if tag.startswith('fg ') or tag.startswith('bg '):
            st, color = tag[:2], tag[3:]
            if st == 'fg': return Fore[color]
            elif st == 'bg': return Back[color]
        else:
            text = Fore[tag]
            if text: return text
        return ''

    def get_end_ansi(self, tag):
        if tag.startswith('/fg'): return Fore.reset
        if tag.startswith('/bg'): return Back.reset
        
        tag = tag[1:]
        if tag in Style:
            return Style[f'no_{tag}']
        elif tag in Fore:
            return Fore.reset
        else:
            Style.end    
    
    def colorize(self, text):
        position = 0
        tokens = []
        for i, match in enumerate(self._regex_tag.finditer(text)):
            markup, tag = match.group(0), match.group(1)
            tokens.append(text[position: match.start()])
            if tag[0] == '/':
                token = self.get_end_ansi(tag)
            else:
                token = self.get_ansi(tag)
            tokens.append(token)
            position = match.end()

        tokens.append(text[position:])
        return ''.join(tokens)
        
    def strip(self, text):
        return self._regex_tag.sub('', text)
    
    def __repr__(self):
        return 'AnsiColorizer'
    
colorize = AnsiColorizer() 
