import torch
from torch.utils.data import Dataset

import numpy as np
import scipy.sparse as sp
import scanpy as sc


class MyDataset(Dataset):
    
    '''
    construct dataset of model1
    input: adata
    output: Dataset with feature, count and library size.
    '''
    
    def __init__(self, adata, size='explog', normalize=False, hvg=False, scale=False):
        
        super(MyDataset,self).__init__()
        
        if sp.issparse(adata.layers['counts']):
            self.count = torch.from_numpy(adata.layers['counts'].toarray()).float()
        else:
            self.count = torch.from_numpy(adata.layers['counts']).float() 
        
        ### library size vector
        if size == 'explog':
            self.size = torch.exp(torch.log10(self.count.sum(axis=1))).unsqueeze(1)  
        elif size == 'sum':
            self.size = self.count.sum(axis=1).unsqueeze(1)  
        elif size == 'median':
            self.size = (self.count.sum(1) / np.median(self.count.sum(1))).unsqueeze(1) 

        adata.obs['size'] = self.size.squeeze(1).numpy()

        # normalize 
        if normalize:
            sc.pp.normalize_per_cell(adata)
        
        # log1p
        sc.pp.log1p(adata)
        
        if hvg:
            sc.pp.highly_variable_genes(
                                        adata,
                                        n_top_genes=hvg,
                                        subset=True,
                                        layer="counts",
                                        flavor="seurat_v3",
                                    )
            adata = adata[:,:hvg]

        # scale
        if scale:
            sc.pp.scale(adata)
        
        ### count matrix
        if sp.issparse(adata.layers['counts']):
            self.count = torch.from_numpy(adata.layers['counts'].toarray()).float()
        else:
            self.count = torch.from_numpy(adata.layers['counts']).float() 
        
        ### input feature matrix
        if sp.issparse(adata.X):
            self.feature = torch.from_numpy(adata.X.toarray()).float()
        else:
            self.feature = torch.from_numpy(adata.X).float()

        self.all_data = [(self.feature[i], self.count[i], self.size[i]) for i in range(self.feature.shape[0])]


    def __getitem__(self,idx):
        return self.all_data[idx]    
    
    def __len__(self):
        return len(self.all_data)



def calcu_adj(cord, neigh_cal ='knn', n_neigh = 8, n_radius=None, metric='minkowski'):
    '''
    Construct adjacency matrix with coordinates.
    input: cord, np.array
    '''
    from sklearn.neighbors import NearestNeighbors
    if neigh_cal == 'knn':
        n_neigh += 1
        neigh = NearestNeighbors(n_neighbors = n_neigh, metric = metric).fit(cord)
        neigh_index = neigh.kneighbors(cord,return_distance=False)
        index = torch.LongTensor(np.vstack([np.repeat(range(cord.shape[0]),n_neigh),neigh_index.ravel()]))
    if neigh_cal == 'radius':
        neigh = NearestNeighbors(radius=n_radius, metric = metric).fit(cord)
        dist, neigh_index = neigh.radius_neighbors(cord, return_distance=True)
        index = np.array([[],[]], dtype=int)
        for it in range(cord.shape[0]):
            index = np.hstack(((index, np.vstack((np.array([it]*neigh_index[it].shape[0]), neigh_index[it])))))
        index = torch.LongTensor(index)
    value = torch.ones(index.shape[1])
    return torch.sparse.FloatTensor(index, value, torch.Size((cord.shape[0], cord.shape[0])))



def euclidean_distances(x, y, squared=True):
    """
    Compute pairwise (squared) Euclidean distances.
    """
    assert isinstance(x, np.ndarray) and x.ndim == 2
    assert isinstance(y, np.ndarray) and y.ndim == 2
    assert x.shape[1] == y.shape[1]
    x_square = np.sum(x*x, axis=1, keepdims=True)
    if x is y:
        y_square = x_square.T
    else:
        y_square = np.sum(y*y, axis=1, keepdims=True).T
    distances = np.dot(x, y.T)
    # use inplace operation to accelerate
    distances *= -2
    distances += x_square
    distances += y_square
    # result maybe less than 0 due to floating point rounding errors.
    np.maximum(distances, 0, distances)
    if x is y:
        # Ensure that distances between vectors and themselves are set to 0.0.
        # This may not be the case due to floating point rounding errors.
        distances.flat[::distances.shape[0] + 1] = 0.0
    if not squared:
        np.sqrt(distances, distances)
    return distances


from torch_geometric.data import Data

def Batch_Data(adata, cord_keys, latent_key, num_batch_x=1, num_batch_y=1, neigh_cal ='knn', n_neigh = 8, n_radius=None, kernal_thresh=0.5, cat=True, seed=100):
    '''
    Constrcut batch data of model2
    Input: latent: np.array of latent generated by model 1.
           cord: pandas of coordinates.
    Output: List of tuple with Data and kernal similar matrix
    '''
    import random 
    random.seed(seed)
    np.random.seed(seed)
    latent = adata.obsm[latent_key]
    if sp.issparse(adata.X):
        expr = adata.X.toarray()
        count = adata.layers['counts'].toarray()
    else:
        expr = adata.X
        count = adata.layers['counts']
    size = np.array(adata.obs['size'])
    cord = adata.obs[cord_keys]
    cord_ = np.array(cord)
    batch_x_coor = [np.percentile(cord_[:, 0], (1/num_batch_x)*x*100) for x in range(num_batch_x+1)]
    batch_y_coor = [np.percentile(cord_[:, 1], (1/num_batch_y)*x*100) for x in range(num_batch_y+1)]
    Batch_list = []
    i=0
    for it_x in range(num_batch_x):
        for it_y in range(num_batch_y):
            min_x = batch_x_coor[it_x]
            max_x = batch_x_coor[it_x+1]
            min_y = batch_y_coor[it_y]
            max_y = batch_y_coor[it_y+1]
            index_use = np.where(cord.iloc[:,0].map(lambda x: min_x <= x <= max_x) & cord.iloc[:,1].map(lambda y: min_y <= y <= max_y))[0]         
            latent_tmp = latent[index_use,:]
            expr_tmp = expr[index_use,:]
            cord_tmp = cord.iloc[index_use,:]                    
            adj_tmp = calcu_adj(cord_tmp, neigh_cal=neigh_cal, n_neigh=n_neigh, n_radius=n_radius)._indices()
            dist_tmp = euclidean_distances(cord_tmp.values, cord_tmp.values)
            kernal_thresh_tmp = kernal_thresh * num_batch_x * num_batch_y
            dist_tmp = dist_tmp/np.percentile(dist_tmp, kernal_thresh_tmp, axis=0)    
            kernal_tmp = np.exp(-dist_tmp)
            kernal_tmp = torch.from_numpy(kernal_tmp).float()    
            kernal_tmp = 0.5 * (kernal_tmp + kernal_tmp.T)
            data_tmp = Data(x=torch.from_numpy(latent_tmp).float(), edge_index=adj_tmp)
            if cat:
                print(f'Batch {i+1}: Each cell have {adj_tmp.shape[1]/cord_tmp.shape[0]-1} neighbors')
                print(f'Batch {i+1}: Each cell have {((kernal_tmp>0.8).sum()/kernal_tmp.shape[0])-1}similar cells')
            Batch_list.append((data_tmp, kernal_tmp))
            i += 1
    adj = calcu_adj(cord, neigh_cal=neigh_cal, n_neigh=n_neigh, n_radius=n_radius)._indices()
    data = Data(x=torch.from_numpy(latent.copy()).float(), edge_index=adj)
    if cat:
        print(f'All: Each cell have {adj.shape[1]/cord.shape[0]-1} neighbors')
    # print(f'All: Each cell have {(kernal>0.8).sum()/kernal.shape[0]}similar cells')
    return Batch_list, data


