# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['hotspot']

package_data = \
{'': ['*']}

install_requires = \
['anndata>=0.7',
 'matplotlib>=3.0.0',
 'numba>=0.43.1',
 'numpy>=1.16.4',
 'pandas>=0.24.0',
 'pynndescent>=0.5',
 'scikit-learn>=0.21.2',
 'scipy>=1.2.1',
 'seaborn>=0.9.0',
 'statsmodels>=0.9.0',
 'tqdm>=4.32.2']

extras_require = \
{':python_version < "3.8"': ['importlib-metadata>1.0'],
 'docs': ['sphinx-book-theme>=0.2.0', 'nbsphinx', 'ipython'],
 'test': ['pytest>=5.0', 'scanpy']}

setup_kwargs = {
    'name': 'hotspotsc',
    'version': '1.1.1',
    'description': 'Identifying informative genes (and gene modules) in a single-cell dataset.',
    'long_description': '![Build Status](https://github.com/YosefLab/hotspot/workflows/hotspot/badge.svg)\n# Hotspot\n\nHotspot is a tool for identifying informative genes (and gene modules) in a single-cell dataset.\n\nImportantly \'informative\' is decided based on how well a gene\'s variation agrees with some cell metric - some similarity mapping between cells.\n\nGenes which are informative are those whose expression varies in similar way among cells which are nearby in the given metric.\n\nThe choice of metric allows you to evaluate different types of gene modules:\n\n- Spatial:  For spatial single-cell datasets, you can define cell similarity by proximity in physical, 2D/3D space.  When set up this way, Hotspot can be used to identify **spatially-varying genes**.\n- Lineage:  For single-cell datasets with a lineage tracing system, you can infer a cell developmental lineage and use that as the cell metric.  Then Hotspot can be used to identify **heritable patterns of gene expression.**\n- Transcriptional:  The result of a dimensionality reduction procedure can be used create the similarity metric. With this configuration, Hotspot identifies gene modules with local correlation.  This can be particularly useful for interpreting axes of variation in non-linear dimensionality reduction procedures where the mapping between components and genes is not straightforward to evaluate.\n\n**Examples**\n\n- [Spatial Data (Mouse Cerebellum)](https://hotspot.readthedocs.io/en/latest/Spatial_Tutorial.html)\n- [Lineage Data (Mouse Embryogenesis)](https://hotspot.readthedocs.io/en/latest/Lineage_Tutorial.html)\n- [Human CD4 T cells](https://hotspot.readthedocs.io/en/latest/CD4_Tutorial.html)\n\n**Other Links**\n\n- [Publication](https://linkinghub.elsevier.com/retrieve/pii/S2405471221001149)\n- [bioRxiv Preprint](https://www.biorxiv.org/content/10.1101/2020.02.06.937805v1)\n    - [Associated analysis repo](https://github.com/deto/Hotspot_analysis)\n- [Documentation](https://hotspot.readthedocs.io/en/latest/)\n\n# Installation\n\nHotspot is installed directly from PyPI using the following command:\n\n```\npip install hotspotsc\n```\n\nHotspot can be imported as\n\n```\nimport hotspot\n```\n\nStarting with v1.0, Hotspot uses the [AnnData](https://anndata.readthedocs.io/en/latest/) data object. If you\'d like to use the old version of Hotspot then:\n\n```\npip install hotspotsc==0.9.1\n```\n\nThe old documentation for that version can be found [here](https://yoseflab.github.io/Hotspot/), but installation will still be `pip install hotspotsc==0.9.1`.\n\nThe old initialization for v0.9.1 has now moved to `hotspot.Hotspot.legacy_init`, see the docs for usage.\n\n\n# Usage\n\nThe following steps are used when analyzing data in Hotspot:\n\n1. Create the Hotspot object\n2. Compute the KNN graph\n3. Find informative genes (by gene autocorrelation)\n4. Evaluate pair-wise gene associations (gene local correlations)\n5. Group genes into modules\n6. Compute summary per-cell module scores\n\nHere we describe each step in order:\n\n## Create the Hotspot object\n\nTo start an analysis, first create the hotspot object\nWhen creating the object, you need to specify:\n- The gene counts matrix\n- Which background model of gene expression to use\n- The metric space to use to evaluate cell-similarity\n- The per-cell scaling factor\n    - By default, number of umi per cell is used\n\nFor example:\n```\nimport hotspot\nhs = hotspot.Hotspot(\n    adata,\n    layer_key="counts",\n    model=\'danb\',\n    latent_obsm_key="X_pca",\n    umi_counts_obs_key="total_counts"\n)\n```\n\nIn the example above:\n- `adata` is a [AnnData](https://anndata.readthedocs.io/en/latest/) object of cells by genes\n- `layer_key` is the layer of `adata` containing count information\n- `model` \'danb\' selects the umi-adjusted negative binomial model\n- `latent_obsm_key` is the `.obsm` key of `adata` containing Cells x Components matrix (the PCA-reduced space)\n- `umi_counts_obs_key` is the `.obs` key of `adata` with the UMI count for each cell\n\n\n### Alternative choices for \'model\'\n\nThe model is used to fit per-cell expectations for each gene assuming no correlations. This is used as the null model when evaluating autocorrelation and gene-gene local correlations.  The choices are:\n\n- **danb**: \'Depth-adjusted negative binomial\' (aka NBDisp model) from [M3Drop](https://bioconductor.org/packages/release/bioc/html/M3Drop.html)\n- **bernoulli**: Here only the detection probability for each gene is estimated.  Logistic regression on gene bins is used to evaluate this per-gene and per-cell as a function of the cells `umi_count` value.\n- **normal**: Here expression values are assumed to be normally-distributed and scaled by the values in `umi_count`.\n- **none**: With this option, the values are assumed to be pre-standardized\n\n\n### Choosing different metrics\n\nAbove we used `latent_obsm_key` as the input option.  This assumes that cells are in an N-dimensional space and similarity between cells is evaluated by computing euclidean distances in this space.  Either the results of a dimensionality reduction or modeling procedure can be input here, or when working with spatial data, the per-cell coordinates.\n\nAlternately, instead of `latent_obsm_key`, you can specify either `tree` or `distances_obsp_key`.\n\n`tree` is used for a developmental lineage.  In this form, `tree` should be an `ete3.TreeNode` object representing the root of a Tree with each cell as its leaves.  This could be constructed programmatically (see ete3\'s documentation for details) or if your lineage is stored in a Newick file format, you can load it into an ete3.TreeNode object by running `ete3.Tree(\'my_newick.txt\')`.  Note: leaf nodes in the tree must have names that match the column labels in the `counts` input (e.g., cell barcodes).\n\n`distances_obsp_key` is used to specify cell-cell distances directly.  The value entered should be a Cells x Cells matrix in `adata.obsp`.\n\n\n## Compute the KNN graph\n\nThe K-nearest-neighbors graph is computed by running:\n\n```\nhs.create_knn_graph(weighted_graph=False, n_neighbors=30)\n```\n\nInput options:\n- `weighted_graph`: bool, whether or not the graph has weights (scaled by cell-cell distances) or is binary\n- `n_neighbors`:  the number of neighbors per cell to use. Larger neighborhood sizes can result in more robust detection of correlations and autocorrelations at a cost of missing more fine-grained, smaller-scale patterns and increasing run-time\n- `neighborhood_factor`: float, used when `weighted_graph=True`.  Weights decay proportionally to `exp(-distance^2/distance_N^2)` where `distance_N` is the distance to the `n_neighbors/neighborhood_factor`th neighbor.\n\nGenerally, the above defaults should be fine in most cases.\n\n\n## Find informative genes (by gene autocorrelation)\n\nTo compute per-gene autocorrelations, run:\n\n```\nhs_results = hs.compute_autocorrelations()\n```\n\nAn optional argument, `jobs` can be specified to invoke parallel jobs for a speedup on multi-core machines.\n\nThe output is a pandas DataFrame (and is also saved in `hs.results`):\n\n| Gene            |     C |      Z |     Pval |      FDR |\n|:----------------|------:|-------:|---------:|---------:|\n| ENSG00000139644 | 0.069 | 10.527 | 3.21e-26 | 7.45e-25 |\n| ENSG00000179218 | 0.071 | 10.521 | 3.43e-26 | 7.93e-25 |\n| ENSG00000196139 | 0.081 | 10.517 | 3.59e-26 | 8.28e-25 |\n| ENSG00000119801 | 0.062 | 10.515 | 3.68e-26 | 8.48e-25 |\n| ENSG00000233355 | 0.058 | 10.503 | 4.15e-26 | 9.55e-25 |\n|      ...        |  ...  |   ...  |    ...   |    ...   |\n\n\nColumns are:\n- `C`: Scaled -1:1 autocorrelation coeficients\n- `Z`: Z-score for autocorrelation\n- `Pval`:  P-values computed from Z-scores\n- `FDR`:  Q-values using the Benjamini-Hochberg procedure\n\n\n## Evaluate pair-wise gene associations (gene local correlations)\n\nTo group genes into modules, we need to first evaluate their pair-wise local correlations\n\nBetter than regular correlations, these \'local\' correlations also take into accounts associations where one gene, X, is expression \'near\' another gene Y in the map.  This can better resolve correlations between sparsely detected genes.\n\n```\nhs_genes = hs_results.loc[hs_results.FDR < 0.05].index # Select genes\n\nlocal_correlations = hs.compute_local_correlations(hs_genes, jobs=4) # jobs for parallelization\n```\n\nHere we run only on a subset of genes as evaluating all pair-wise correlations is very expensive computationally.   The autocorrelation ordering gives us a natural method to select the informative genes for this purpose.\n\nThe output is a genes x genes pandas DataFrame of Z-scores for the local correlation values between genes.  The output is also stored in `hs.local_correlation_z`.\n\n## Group genes into modules\nNow that pair-wise local correlations are calculated, we can group genes into modules.\n\nTo do this, a convenience method is included `create_modules` which performs\nagglomerative clustering.\n\n```\nmodules = hs.create_modules(\n    min_gene_threshold=30, core_only=True, fdr_threshold=0.05\n)\n```\n\nA note on the arguments - agglomerative clustering proceeds by joining together genes with the highest pair-wise Z-score with the following caveats:\n\n- If the FDR-adjusted p-value of the correlation between two branches exceeds `fdr_threshold`,\nthen the branches are not merged.\n- If two branches are two be merged and they are both have at least `min_gene_threshold` genes,\nthen the branches are not merged.  Further genes that would join to the resulting merged module\nsmaller average correlations between genes, i.e. the least-dense module (if `core_only=False`)\n\nThis method was used to preserved substructure (nested modules) while still giving the analyst\nsome control. However, since there are a lot of ways to do hierarchical clustering, you can also\nmanually cluster using the gene-distances in `hs.local_correlation_z`\n\nThe output is a pandas Series that maps gene to module number.  Unassigned genes are indicated with a module number of `-1`.  The output is also stored in `hs.modules`\n\nCorrelation modules can be visualized by running `hs.plot_local_correlations()`:\n\n<img src="https://hotspot.readthedocs.io/en/latest/_images/CD4_Tutorial_29_0.png" alt="Local Correlation Plot" width="500">\n\n\n## Compute summary per-cell module scores\n\nFinally, summary per-cell scores can be computed for a module.  This is useful for visualizng the general pattern of expression for genes in a module.\n\n```\nmodule_scores = hs.calculate_module_scores()\n```\n\nThe output is a pandas DataFrame (cells x modules) and is also saved in `hs.module_scores`\n\n`module_scores`:\n\n|                    |      1 |      2 |      3 |      4 |      5 |\n|:-------------------|-------:|-------:|-------:|-------:|-------:|\n| AAACCCAAGGCCTAGA-1 |  0.785 | -2.475 | -1.407 | -0.681 | -1.882 |\n| AAACCCAGTCGTGCCA-1 | -5.76  |  5.241 |  6.931 |  1.928 |  4.317 |\n| AAACCCATCGTGCATA-1 | -2.619 |  3.572 |  0.143 |  1.832 |  1.585 |\n| AAACGAAGTAATGATG-1 | -8.778 |  4.012 |  6.927 |  1.181 |  3.882 |\n| AAACGCTCATGCACTA-1 |  2.297 | -2.517 | -1.421 | -1.102 | -1.547 |\n|        ...         |   ...  |   ...  |   ...  |   ...  |   ...  |\n\nThese can then be plotted onto other visual representations of the cells.  For example, for spatial modules (from data in [Rodriques et al, 2019](https://www.ncbi.nlm.nih.gov/pubmed/30923225)) this looks like:\n\n<img src="https://hotspot.readthedocs.io/en/latest/_images/Spatial_Tutorial_23_0.png" alt="Spatial Gene Modules" width="500">\n',
    'author': 'David DeTomaso',
    'author_email': 'David.DeTomaso@berkeley.edu',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/yoseflab/hostpost',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
