"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const appmesh_generated_1 = require("./appmesh.generated");
/**
 * VirtualService represents a service inside an AppMesh
 *
 * It routes traffic either to a Virtual Node or to a Virtual Router.
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_services.html
 */
class VirtualService extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.virtualServiceName || cdk.Lazy.stringValue({ produce: () => this.node.uniqueId })
        });
        if (props.virtualNode && props.virtualRouter) {
            throw new Error('Must provide only one of virtualNode or virtualRouter for the provider');
        }
        this.mesh = props.mesh;
        // Check which provider to use node or router (or neither)
        if (props.virtualRouter) {
            this.virtualServiceProvider = this.addVirtualRouter(props.virtualRouter.virtualRouterName);
        }
        if (props.virtualNode) {
            this.virtualServiceProvider = this.addVirtualNode(props.virtualNode.virtualNodeName);
        }
        const svc = new appmesh_generated_1.CfnVirtualService(this, 'Resource', {
            meshName: this.mesh.meshName,
            virtualServiceName: this.physicalName,
            spec: {
                provider: this.virtualServiceProvider,
            },
        });
        this.virtualServiceName = this.getResourceNameAttribute(svc.attrVirtualServiceName);
        this.virtualServiceArn = this.getResourceArnAttribute(svc.ref, {
            service: 'appmesh',
            resource: `mesh/${props.mesh.meshName}/virtualService`,
            resourceName: this.physicalName,
        });
    }
    /**
     * Import an existing VirtualService given an ARN
     */
    static fromVirtualServiceArn(scope, id, virtualServiceArn) {
        return new ImportedVirtualService(scope, id, {
            virtualServiceArn,
        });
    }
    /**
     * Import an existing VirtualService given mesh and service names
     */
    static fromVirtualServiceName(scope, id, meshName, virtualServiceName) {
        return new ImportedVirtualService(scope, id, {
            meshName,
            virtualServiceName,
        });
    }
    addVirtualRouter(name) {
        return {
            virtualRouter: {
                virtualRouterName: name,
            },
        };
    }
    addVirtualNode(name) {
        return {
            virtualNode: {
                virtualNodeName: name,
            },
        };
    }
}
exports.VirtualService = VirtualService;
/**
 * Returns properties that allows a VirtualService to be imported
 */
class ImportedVirtualService extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.virtualServiceArn) {
            this.virtualServiceArn = props.virtualServiceArn;
            this.virtualServiceName = cdk.Fn.select(2, cdk.Fn.split('/', cdk.Stack.of(scope).parseArn(props.virtualServiceArn).resourceName));
        }
        else if (props.virtualServiceName && props.meshName) {
            this.virtualServiceName = props.virtualServiceName;
            this.virtualServiceArn = cdk.Stack.of(this).formatArn({
                service: 'appmesh',
                resource: `mesh/${props.meshName}/virtualService`,
                resourceName: this.virtualServiceName,
            });
        }
        else {
            throw new Error('Need either arn or both names');
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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