import * as cdk from '@aws-cdk/core';
import { IMesh } from './mesh';
import { IVirtualNode } from './virtual-node';
import { IVirtualRouter } from './virtual-router';
/**
 * Interface for which all Route based classes MUST implement
 */
export interface IRoute extends cdk.IResource {
    /**
     * The name of the route
     *
     * @attribute
     */
    readonly routeName: string;
    /**
     * The Amazon Resource Name (ARN) for the route
     *
     * @attribute
     */
    readonly routeArn: string;
}
/**
 * Base interface properties for all Routes
 */
export interface RouteBaseProps {
    /**
     * The name of the route
     *
     * @default - An automatically generated name
     */
    readonly routeName?: string;
    /**
     * The path prefix to match for the route
     *
     * @default "/" if http otherwise none
     */
    readonly prefix?: string;
    /**
     * Array of weighted route targets
     *
     * @requires minimum of 1
     */
    readonly routeTargets: WeightedTargetProps[];
    /**
     * Weather the route is HTTP based
     *
     * @default - HTTP if `prefix` is given, TCP otherwise
     */
    readonly routeType?: RouteType;
}
/**
 * Type of route
 */
export declare enum RouteType {
    /**
     * HTTP route
     */
    HTTP = "http",
    /**
     * TCP route
     */
    TCP = "tcp"
}
/**
 * Properties for the Weighted Targets in the route
 */
export interface WeightedTargetProps {
    /**
     * The VirtualNode the route points to
     */
    readonly virtualNode: IVirtualNode;
    /**
     * The weight for the target
     *
     * @default 1
     */
    readonly weight?: number;
}
/**
 * Properties to define new Routes
 */
export interface RouteProps extends RouteBaseProps {
    /**
     * The service mesh to define the route in
     */
    readonly mesh: IMesh;
    /**
     * The virtual router in which to define the route
     */
    readonly virtualRouter: IVirtualRouter;
}
/**
 * Route represents a new or existing route attached to a VirtualRouter and Mesh
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/routes.html
 */
export declare class Route extends cdk.Resource implements IRoute {
    /**
     * Import an existing route given an ARN
     */
    static fromRouteArn(scope: cdk.Construct, id: string, routeArn: string): IRoute;
    /**
     * Import an existing route given its name
     */
    static fromRouteName(scope: cdk.Construct, id: string, meshName: string, virtualRouterName: string, routeName: string): IRoute;
    /**
     * The name of the route
     */
    readonly routeName: string;
    /**
     * The Amazon Resource Name (ARN) for the route
     */
    readonly routeArn: string;
    /**
     * The virtual router this route is a part of
     */
    readonly virtualRouter: IVirtualRouter;
    private readonly weightedTargets;
    private readonly httpRoute?;
    private readonly tcpRoute?;
    constructor(scope: cdk.Construct, id: string, props: RouteProps);
    /**
     * Utility method to add weighted route targets to an existing route
     */
    private renderWeightedTargets;
    private renderHttpRoute;
    private renderTcpRoute;
}
